/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.Collection;
import java.util.HashSet;
import java.util.concurrent.Semaphore;

import javax.servlet.http.HttpSessionBindingEvent;
import javax.servlet.http.HttpSessionBindingListener;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.utils.Switchable;

/**
 * 画面遷移の組に対してセマフォを制御する{@link DomExpression}。
 * 同時実行数を越えている場合のリクエストはスレッドをブロックせずにエラーとする。
 * 各セッションに対しひとつの業務にのみ実行権を付与することが可能である。
 * 画面遷移順序を検証する{@link TransactionBuilder}の直接または間接的な委譲先として使用する必要がある。
 * @author nakamura
 *
 */
public class TrySemaphoreTransaction implements DomExpression {
	public static final String KEY = "woolpack.ee.TRY_SEMAPHORE_TRANSACTION";
	
	private final String key;
	private final Switchable<String,Semaphore> startIdSwitchable;
	private final Collection<String> midIdCollection;
	private final Collection<String> endIdCollection;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;
	private final DomExpression releaseExpression;
	
	/**
	 * コンストラクタ。
	 * @param key セマフォを格納する{@link DomContext#getSession()}のキー。
	 * @param startIdSwitchable セッションの開始 id とセマフォの{@link Switchable}。本クラスはこの引数の{@link Semaphore}の状態を変化させる。
	 * @param midIdCollection セッションの途中 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param endIdCollection セッションの終了 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @param releaseExpression パーミットの返却前の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphoreTransaction(
			final String key, 
			final Switchable<String,Semaphore> startIdSwitchable, 
			final Collection<String> midIdCollection, 
			final Collection<String> endIdCollection, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression,
			final DomExpression releaseExpression){
		key.charAt(0);
		startIdSwitchable.getClass();
		midIdCollection.getClass();
		endIdCollection.getClass();
		trueExpression.getClass();
		falseExpression.getClass();
		releaseExpression.getClass();
		
		// key 重複をテストする。
		checkDuplicate(startIdSwitchable.keys(), midIdCollection);
		checkDuplicate(startIdSwitchable.keys(), endIdCollection);
		checkDuplicate(midIdCollection, endIdCollection);
		
		this.key = key;
		this.startIdSwitchable = startIdSwitchable;
		this.midIdCollection = midIdCollection;
		this.endIdCollection = endIdCollection;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
		this.releaseExpression = releaseExpression;
	}
	
	private static void checkDuplicate(final Collection<String> set0, final Collection<String> set1){
		final Collection<String> set = new HashSet<String>(set0);
		set.retainAll(set1);
		if(!set.isEmpty()){
			throw new IllegalArgumentException("duplicate id: " + set);
		}
	}

	/**
	 * コンストラクタ。
	 * セマフォを格納する{@link DomContext#getSession()}のキーとして{@link #KEY}を使用する。
	 * @param startIdSwitchable セッションの開始 id とセマフォの{@link Switchable}。本クラスはこの引数の{@link Semaphore}の状態を変化させる。
	 * @param midIdCollection セッションの途中 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param endIdCollection セッションの終了 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @param releaseExpression パーミットの返却前の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphoreTransaction(
			final Switchable<String,Semaphore> startIdSwitchable, 
			final Collection<String> midIdCollection, 
			final Collection<String> endIdCollection, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression,
			final DomExpression releaseExpression){
		this(KEY, startIdSwitchable, midIdCollection, endIdCollection, trueExpression, falseExpression, releaseExpression);
	}
	
	/**
	 * コンストラクタ。
	 * セマフォを格納する{@link DomContext#getSession()}のキーとして{@link #KEY}を使用する。
	 * @param startIdSwitchable セッションの開始 id とセマフォの{@link Switchable}。本クラスはこの引数の{@link Semaphore}の状態を変化させる。
	 * @param midIdCollection セッションの途中 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param endIdCollection セッションの終了 id の{@link Collection}。本クラスはこの引数の状態を変化させない。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphoreTransaction(
			final Switchable<String,Semaphore> startIdSwitchable, 
			final Collection<String> midIdCollection, 
			final Collection<String> endIdCollection, 
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		this(startIdSwitchable, midIdCollection, endIdCollection, trueExpression, falseExpression, DomConstants.NULL);
	}
	
	private void release(final DomContext context){
		try{
			releaseExpression.interpret(context);
		}finally{
			context.getSession().remove(key);
		}
	}
	
	public void interpret(final DomContext context) {
		final Semaphore semaphore = startIdSwitchable.get(context.getId());
		if(semaphore != null){
			if(context.getSession().containsKey(key)){
				release(context);
			}
			if(semaphore.tryAcquire()){
				context.getSession().put(key, new HttpSessionBindingListener(){
					public void valueBound(final HttpSessionBindingEvent arg0) {
						// do nothing.
					}
					public void valueUnbound(final HttpSessionBindingEvent arg0) {
						semaphore.release();
					}
				});
				try{
					trueExpression.interpret(context);
				}catch(final RuntimeException e){
					release(context);
					throw e;
				}
			}else{
				falseExpression.interpret(context);
			}
		}else if(midIdCollection.contains(context.getId())){
			if(context.getSession().containsKey(key)){
				try{
					trueExpression.interpret(context);
				}catch(final RuntimeException e){
					release(context);
					throw e;
				}
			}else{
				falseExpression.interpret(context);
			}
		}else if(endIdCollection.contains(context.getId())){
			if(context.getSession().containsKey(key)){
				try{
					trueExpression.interpret(context);
				}finally{
					release(context);
				}
			}else{
				falseExpression.interpret(context);
			}
		}else{
			trueExpression.interpret(context);
		}
	}
}
