/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * {@link Map}をキャッシュする{@link ConfigExpression}。
 * 委譲先で生成される{@link Map}が{@link #getCacheKey()}で返されるキーに対して一意であることを想定しているため並行性制御を行っていない。
 * @author nakamura
 *
 */
public class CacheMap implements ConfigExpression {
	private static final Object DEFAULT_KEY = new Object();
	
	private final ConfigExpression mapMaker;
	private final Map<Object,Map<String,Object>> cacheMap;

	/**
	 * コンストラクタ。
	 * @param cacheMap キャッシュ。本クラスはこの引数の状態を変化させる。null が指定された場合はキャッシュを行わない。{@link LinkedHashMap#removeEldestEntry(java.util.Map.Entry)}を指定した{@link LinkedHashMap}を指定するとキャッシュの容量を制御することができる。{@link java.util.HashMap}などの強参照の{@link Map}を指定すると容量制限のないキャッシュになる。{@link java.util.WeakHashMap}を指定すると弱参照によるキャッシュになる。
	 * @param mapMaker {@link Map}を生成し加工する委譲先の{@link ConfigExpression}。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CacheMap(final Map<Object,Map<String,Object>> cacheMap, final ConfigExpression mapMaker){
		mapMaker.getClass();
		this.cacheMap = cacheMap;
		this.mapMaker = mapMaker;
	}

	/**
	 * @throws NullPointerExcpeption 引数が null の場合。
	 */
	public void interpret(final ConfigContext context) {
		if(cacheMap == null){
			mapMaker.interpret(context);
		}else{
			final Object key = getCacheKey();
			Map<String,Object> map = cacheMap.get(key);
			if(map == null){
				mapMaker.interpret(context);
				map = context.getMap();
				cacheMap.put(key, map);
			}
			context.setMap(map);
		}
	}
	
	/**
	 * 委譲先で生成する{@link Map}を一意に識別するためのキーを取得するために呼び出される(called)。
	 * デフォルトは常に同一のインスタンスを返す。
	 * これは{@link java.util.WeakHashMap}が指定されていてもGCにより改修されないオブジェクトである。
	 * {@link java.util.Locale}や端末ごとに{@link Map}をキャッシュする際にオーバライドする。
	 * オーバライドする際は{@link PutResourceBundle#getResourceBundle(String)}と整合性を保つ必要がある。
	 * @return cache 委譲先で生成する{@link Map}を一意に識別するためのキー。
	 */
	public Object getCacheKey(){
		return DEFAULT_KEY;
	}
}
