/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

import ognl.MethodFailedException;
import ognl.Ognl;
import ognl.OgnlException;
import ognl.TypeConverter;
import woolpack.el.ELAdapter;
import woolpack.el.ELTargetRuntimeException;

/**
 * OGNL をクラス化した、オブジェクトグラフ表現(Object Graph Expression)。
 * 本クラスのインスタンスは複数のスレッドで同時に使用できる。
 * 本クラスは OGNL のライブラリを使用する。
 * @author nakamura
 *
 */
public class OGE extends ELAdapter{
	private static final Map<String, Object> cache = new HashMap<String, Object>();
	
	private final String expression;
	private final Object tree;

	private static Object registerToMap(final String expression) throws OgnlException{
		Object myTree = cache.get(expression);
		if(myTree == null){
			myTree = Ognl.parseExpression(expression);
			cache.put(expression, myTree);
		}
		return myTree;
	}
	
	/**
	 * コンストラクタ。
	 * @param expression OGNL の文字列表現。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException ({@link OgnlException})OGNL の解析に失敗した場合。
	 */
	public OGE(final String expression){
		// TODO TypeConverterを引数指定する。
		expression.getClass();
		this.expression = expression;
		try {
			this.tree = registerToMap(expression);
		} catch (final OgnlException exception) {
			throw new RuntimeException(exception);
		}
	}
	
	abstract class OGE2{
		abstract Object execute(Map context) throws MethodFailedException, OgnlException;
	}
	
	private Object execute(final Object root, final OGE2 oge2) throws ELTargetRuntimeException{
		final Map context = Ognl.createDefaultContext(root);
		final TypeConverter defaultTypeConverter = Ognl.getTypeConverter(context);
		final TypeConverter addTypeConverter = new OgnlCollectionTypeConverter(defaultTypeConverter);
		Ognl.setTypeConverter(context, addTypeConverter);
		try {
			return oge2.execute(context);
		}catch(final MethodFailedException e){
			if(e.getReason() instanceof NoSuchMethodException){
				throw new RuntimeException(e);
			}else{
				throw new ELTargetRuntimeException(e.getReason());
			}
		} catch (final OgnlException e) {
			throw new RuntimeException(e);
		}finally{
			Ognl.setTypeConverter(context, defaultTypeConverter);
		}
	}

	/**
	 * @throws ELTargetRuntimeException メソッドがエラーを投げた場合。
	 * @throws RuntimeException ({@link OgnlException})OGNL の解析に失敗した場合。
	 */
	@Override public Object getValue(final Object root, final Class clazz) throws ELTargetRuntimeException{
		return execute(root, new OGE2(){
			@Override 
			Object execute(final Map context) throws MethodFailedException, OgnlException{
				if(Object.class.equals(clazz)){
					return Ognl.getValue(tree, context, root);
				}else{
					return Ognl.getValue(tree, context, root, clazz);
				}
			}
		});
	}

	/**
	 * @throws ELTargetRuntimeException メソッドがエラーを投げた場合。
	 * @throws RuntimeException ({@link OgnlException})OGNL の解析に失敗した場合。
	 */
	@Override public void setValue(final Object root, final Object value) throws ELTargetRuntimeException{
		execute(root, new OGE2(){
			@Override 
			Object execute(final Map context) throws MethodFailedException, OgnlException{
				Ognl.setValue(tree, context, root, value);
				return null;
			}
		});
	}
	
	@Override public int hashCode(){
		return new HashCodeBuilder().append(expression).toHashCode();
	}
	
	@Override public boolean equals(final Object obj){
		if(!(obj instanceof OGE)){
			return false;
		}
		if(this == obj){
			return true;
		}
		final OGE o = (OGE)obj;
		return new EqualsBuilder().append(this.expression, o.expression).isEquals();
	}
	
	@Override public String toString(){
		return new ToStringBuilder(this).append("expression", expression).toString();
	}
}
