/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import woolpack.el.ELConstants;
import woolpack.el.GettingEL;


/**
 * ひとつの遷移先定義を表す。
 * @author nakamura
 *
 */
public class ForwardDef {
	private final String id;
	private final GettingEL componentEL;
	private final ForwardMatchable matchable;

	/**
	 * コンストラクタ。
	 * @param id 遷移先 id。
	 * @param componentEL 遷移先で使用するコンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param matchable マッチ操作。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final GettingEL componentEL, final ForwardMatchable matchable){
		id.charAt(0);
		componentEL.getClass();
		matchable.getClass();
		
		this.id = id;
		this.componentEL = componentEL;
		this.matchable = matchable;
	}

	/**
	 * コンストラクタ。
	 * 遷移先で使用するコンポーネントへの参照を{@link ELConstants#NULL}とする。
	 * @param id 遷移先 id。
	 * @param matchable マッチ操作。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final ForwardMatchable matchable){
		this(id, ELConstants.NULL, matchable);
	}

	/**
	 * コンストラクタ。
	 * マッチ操作を{@link ActionConstants#NOT_THROWABLE}とする。
	 * @param id 遷移先 id。
	 * @param componentEL 遷移先で使用するコンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id, final GettingEL componentEL){
		this(id, componentEL, ActionConstants.NOT_THROWABLE);
	}

	/**
	 * コンストラクタ。
	 * 遷移先で使用するコンポーネントの取得先を{@link ELConstants#NULL}とする。
	 * マッチ操作を{@link ActionConstants#NOT_THROWABLE}とする。
	 * @param id 遷移先 id。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ForwardDef(final String id){
		this(id, ELConstants.NULL, ActionConstants.NOT_THROWABLE);
	}
	
	/**
	 * 遷移先 id を返す。
	 * @return 遷移先 id。
	 */
	public String getId(){
		return id;
	}

	/**
	 * 遷移先で使用するコンポーネントへの参照を返す。
	 * @return 遷移先のコンポーネントへの参照。
	 */
	public GettingEL getComponentEL() {
		return componentEL;
	}
	
	/**
	 * マッチ操作を返す。
	 * @return マッチ操作。
	 */
	public ForwardMatchable getMatchable(){
		return matchable;
	}
}
