/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.crud;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.w3c.dom.Element;

import woolpack.action.ActionConstants;
import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.config.CacheMap;
import woolpack.config.ConfigConstants;
import woolpack.config.ConfigContext;
import woolpack.config.ConfigExpression;
import woolpack.config.ConfigSerial;
import woolpack.config.CopyConfigContext;
import woolpack.config.PutResourceBundle;
import woolpack.config.ToLinkedHashMap;
import woolpack.container.ComponentDef;
import woolpack.container.ComponentScope;
import woolpack.container.ScopeContainer;
import woolpack.dom.CacheNode;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.DumpIfCatch;
import woolpack.dom.Exec;
import woolpack.dom.FormatAttrValue;
import woolpack.dom.FormatId;
import woolpack.dom.RegExpId;
import woolpack.dom.RemoveAttr;
import woolpack.dom.ResetId;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.dom.XmlToNode;
import woolpack.ee.ActionBuilder;
import woolpack.ee.ConfigDomExpression;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletContextMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.ServletRequestAttributeMap;
import woolpack.ee.ValidatorBuilder;
import woolpack.el.EL;
import woolpack.el.PathEL;
import woolpack.el.ToELTargetExceptionEL;
import woolpack.html.HiddenAllToChild;
import woolpack.html.HtmlConstants;
import woolpack.html.MakeRadio;
import woolpack.html.MakeSelect;
import woolpack.html.UpdateValue;
import woolpack.samples.ActionDefMaker;
import woolpack.text.LimitedValueFormat;
import woolpack.text.RegExpFormat;
import woolpack.text.ToIntegerFormat;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.MapBuilder;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.Switchable;
import woolpack.utils.UtilsConstants;
import woolpack.validator.BranchByNameIfExistsValidator;
import woolpack.validator.BranchByNameValidator;
import woolpack.validator.DumpValidator;
import woolpack.validator.IfNotValidator;
import woolpack.validator.IfValidator;
import woolpack.validator.MaxLengthValidator;
import woolpack.validator.MaxValidator;
import woolpack.validator.MessageValidator;
import woolpack.validator.MinValidator;
import woolpack.validator.ParseValidator;
import woolpack.validator.RegExpIdValidator;
import woolpack.validator.SerialValidator;
import woolpack.validator.ValidNamesValidator;
import woolpack.validator.ValidValuesValidator;
import woolpack.validator.ValidatorConstants;
import woolpack.validator.ValidatorExpression;
import woolpack.validator.ValueLoopValidator;

/**
 * CRUD のサンプルウェブアプリケーション。
 * @author nakamura
 *
 */
public class SampleServlet extends HttpServlet {

	public final Map<String,ComponentDef> componentDefMap;
	public final DelegateDomExpression toNode;
	public final DomExpression domExpression;
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private Map<String,ComponentDef> createComponentDefMap(){
		// コンテナの定義。
		final Map<String,ComponentDef> map = new HashMap<String,ComponentDef>();
		// 画面からの入力値を格納する Bean オブジェクト。
		map.put("userBean", new ComponentDef(ComponentScope.REQUEST){
			@Override
			protected Object newInstance() {
				return new SampleBean();
			}
		});
		// ユーザ情報を管理するプログラムで唯一のオブジェクト。
		map.put("userController", new ComponentDef(ComponentScope.APPLICATION){
			@Override
			protected Object newInstance() {
				return new SampleBeanTable();
			}
		});
		return map;
	}
	
	private ConfigExpression createConfigExpression(){
		// 画面表示項目の設定ファイルを読みだす。
		return new CacheMap(0, new ConfigSerial(
				ConfigConstants.MAKE_MAP,
				new CopyConfigContext(new ConfigSerial(
						// キーの重複を検出するため、上書き禁止の Map でラップして操作する。
						ConfigConstants.TO_UNOVERWRITABLE_MAP,
						// 選択肢項目の ResourceBundle のデータをロードする。
						new PutResourceBundle("woolpack.samples.crud.selection"),
						// MakeRadio と MakeSelect で選択肢項目を再生正するため、キーと値のMapを作成する。
						new ToLinkedHashMap("sex", "part.sex.value", "part.sex.label", ","),
						new ToLinkedHashMap("address", "part.address.value", "part.address.label", ","),
						new ToLinkedHashMap("hobby", "part.hobby.value", "part.hobby.label", ","),
						// メッセージを ResourceBundle からロードする。
						new PutResourceBundle("woolpack.samples.crud.messages")
				)),
				// 更新禁止の Map でラップする。
				ConfigConstants.TO_UNMODIFIABLE_MAP
		));
	}
	
	private ValidatorExpression createValidatorExpression(){
		// 値検証の定義。
		return new DumpValidator(new SerialValidator(ValidatorConstants.ANDAND,
				// 受け入れ可能なプロパティ一覧の定義。id の数が多くなるとハッシュ実装のほうが検索効率がよいので HashSet としてプロパティ一覧を扱う。
				new IfNotValidator(
						new ValidNamesValidator(new HashSet<String>(Arrays.asList("userId", "name", "sex", "address", "age", "hobby", "comment"))), 
						new MessageValidator("validator.acceptable")
				),
				// 必須検証以外はアクション id をまたがってプロパティ名で値検証を行う。
				new BranchByNameIfExistsValidator(
						new SwitchBuilder<String,ValidatorExpression>()
						.put("userId", new ValueLoopValidator(
								new IfNotValidator(
										new ParseValidator(new ToIntegerFormat()),
										new MessageValidator("validator.userId.parse"))))
						.put("name", new ValueLoopValidator(
								new IfNotValidator(
										new MaxLengthValidator(40),
										new MessageValidator("validator.name.maxLength"))))
						.put("sex", new ValueLoopValidator(new SerialValidator(
								new IfNotValidator(
										new ValidValuesValidator(Arrays.asList("0", "1")),
										new MessageValidator("validator.sex.acceptable")),
								new IfNotValidator(
										new ParseValidator(new ToIntegerFormat()),
										new MessageValidator("validator.sex.parse")))))
						.put("address", new ValueLoopValidator(
								new IfNotValidator(
										new MaxLengthValidator(80),
										new MessageValidator("validator.address.maxLength"))))
						.put("age", new ValueLoopValidator(new SerialValidator(
								new IfNotValidator(
										new ParseValidator(new ToIntegerFormat()),
										new MessageValidator("validator.age.parse")),
								new IfNotValidator(
										new MinValidator(0),
										new MessageValidator("validator.age.min")),
								new IfNotValidator(
										new MaxValidator(200),
										new MessageValidator("validator.age.max")))))
						.put("hobby", new ValueLoopValidator(
								new IfNotValidator(
										new ValidValuesValidator(Arrays.asList("0", "1", "2", "3")),
										new MessageValidator("validator.hobby.acceptable"))))
						.put("comment", new ValueLoopValidator(
								new IfNotValidator(
										new MaxLengthValidator(200),
										new MessageValidator("validator.comment.maxLength"))))
						.get()),
				new IfValidator(
						new RegExpIdValidator("user_(deleteConfirm|updateInput)"),
						new BranchByNameValidator(
								new SwitchBuilder<String,ValidatorExpression>()
								.put("userId", new ValueLoopValidator(
										new IfNotValidator(
												ValidatorConstants.REQUIRED,
												new MessageValidator("validator.userId.required"))))
								.get())),
				new IfValidator(
						// 必須検証はアクション id に依存するのでアクション id でしぼって検証する。
						new RegExpIdValidator("user_(register|update)(Confirm|Result)"),
						new BranchByNameValidator(
								new SwitchBuilder<String,ValidatorExpression>()
								.put("name", new ValueLoopValidator(
										new IfNotValidator(
												ValidatorConstants.REQUIRED,
												new MessageValidator("validator.name.required"))))
								.put("sex", new ValueLoopValidator(
										new IfNotValidator(
												ValidatorConstants.REQUIRED,
												new MessageValidator("validator.sex.required"))))
								.put("address", new ValueLoopValidator(
										new IfNotValidator(
												ValidatorConstants.REQUIRED,
												new MessageValidator("validator.address.required"))))
								.put("age", new ValueLoopValidator(
										new IfNotValidator(
												ValidatorConstants.REQUIRED,
												new MessageValidator("validator.age.required"))))
								.get()))
				));
	}
	
	private void checkMessages(final ConfigExpression configExpression, final ValidatorExpression validatorExpression){
		// ロケール・ブラウザ別に対応していないメッセージ存在検証。定義(製造)のミスを発見するための仕掛け。
		final ConfigContext configContext = new ConfigContext();
		configExpression.interpret(configContext);
		final Collection<String> retain = new HashSet<String>();
		validatorExpression.addMessageTo(retain);
		retain.removeAll(configContext.getMap().keySet());
		if(!retain.isEmpty()){
			throw new RuntimeException("retain transaction id:" + retain);
		}
	}
	
	private Switchable<String,ActionDef> createActionDefMap(){
		final ActionDefMaker maker = new ActionDefMaker();
		maker.putForward("simple_error");
		maker.putForward("simple_errorValidate");
		maker.putForward("common_top");
		// 登録の指示画面にSampleBeanの初期値をそのまま流し込むための画面遷移定義。
		maker.put(
				"user_registerInput",
				EL.NULL,
				EL.NULL,
				new PathEL("container.userBean"));
		// 入力画面で値検証エラーの場合にリクエストの入力値をそのまま流し込むための画面遷移定義。
		maker.putEcho("user_registerInput_error", "user_registerInput");
		// 入力画面の入力値を確認画面にそのまま流し込むための画面遷移定義。
		maker.putEcho("user_registerConfirm");
		// HTML hidden として送信されたリクエストの入力値を userBeanに設定し、それを引数に userController.register に渡す画面遷移定義。
		// 個の仕掛けによりS2DAOをそのまま使用することが可能。
		maker.put(
				"user_registerResult", 
				new PathEL("container.userBean"), 
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("container.userController.register(container.userBean)")
						final Map container = (Map)((DomContext)root).getContainer();
						((SampleBeanTable)container.get("userController")).register((SampleBean)container.get("userBean"));
						return null;
					}
				},
				EL.NULL);
		maker.putForward("user_searchInput");
		// 検索を実行する画面遷移定義。
		// 入力値(request.getParameterMap())は値の一覧になっているので、ゼロ番めの値を引数に指定している。
		// 検索結果を local スコープに格納し画面に渡している。
		maker.put(
				"user_searchResult", 
				EL.NULL, 
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("local.searchResult = container.userController.searchList(input.name[0])")
						final DomContext context = (DomContext)root;
						final Map container = (Map)context.getContainer();
						context.getLocal().put("searchResult", ((SampleBeanTable)container.get("userController")).searchList(
								(String)((List)context.getInput().get("name")).get(0)
						));
						return null;
					}
				}, 
				new PathEL("local"));
		maker.put(
				"user_updateInput",
				EL.NULL, 
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("local.bean = container.userController.searchOne(input.userId[0])")
						final DomContext context = (DomContext)root;
						final Map container = (Map)context.getContainer();
						context.getLocal().put("bean", ((SampleBeanTable)container.get("userController")).searchOne(
								(Integer)((List)context.getInput().get("userId")).get(0)
						));
						return null;
					}
				},
				new PathEL("local.bean"));
		maker.putEcho("user_updateInput_error", "user_updateInput");
		maker.putEcho("user_updateConfirm");
		maker.put(
				"user_updateResult",
				new PathEL("container.userBean"), 
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("container.userController.update(container.userBean)")
						final DomContext context = (DomContext)root;
						final Map container = (Map)context.getContainer();
						((SampleBeanTable)container.get("userController")).update((SampleBean)container.get("userBean"));
						return null;
					}
				},
				EL.NULL);
		maker.put(
				"user_deleteConfirm",
				EL.NULL,
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("local.bean = container.userController.searchOne(input.userId[0])")
						final DomContext context = (DomContext)root;
						final Map container = (Map)context.getContainer();
						context.getLocal().put("bean", ((SampleBeanTable)container.get("userController")).searchOne(
								(Integer)((List)context.getInput().get("userId")).get(0)
						));
						return null;
					}
				},
				new PathEL("local.bean"));
		maker.put(
				"user_deleteResult",
				EL.NULL,
				new ToELTargetExceptionEL(){
					@Override public Object execute(Object root, Object value) throws Exception {
						// new OGE("container.userController.delete(input.userId[0])")
						final DomContext context = (DomContext)root;
						final Map container = (Map)context.getContainer();
						((SampleBeanTable)container.get("userController")).delete(
								(Integer)((List)context.getInput().get("userId")).get(0)
						);
						return null;
					}
				},
				EL.NULL);
		return maker.get();
	}

	private void checkValidatorId(final Map<String,String> idInputIdMap, final Switchable<String, ActionDef> actionDefMap) {
		// 値検証の画面遷移定義に対する id 存在検証。定義(製造)のミスを発見するための仕掛け。
		{
			final Collection<String> retain =  new HashSet<String>(idInputIdMap.keySet());
			retain.removeAll(actionDefMap.keys());
			if(!retain.isEmpty()){
				throw new RuntimeException("retain target id: " + retain);
			}
		}
		{
			final Collection<String> retain =  new HashSet<String>(idInputIdMap.values());
			retain.removeAll(actionDefMap.keys());
			if(!retain.isEmpty()){
				throw new RuntimeException("retain input id: " + retain);
			}
		}
	}
	
	public SampleServlet(){
		componentDefMap = createComponentDefMap();
		final ConfigExpression configExpression = createConfigExpression();
		final ValidatorExpression validatorExpression = createValidatorExpression();
		checkMessages(configExpression, validatorExpression);

		// 遷移先で値検証の結果が false の場合のエラーを表示する画面 id の定義。
		final Map<String,String> idInputIdMap = MapBuilder.get(new HashMap<String,String>())
		.put("user_registerConfirm", "user_registerInput_error")
		.put("user_updateConfirm", "user_updateInput_error")
		.get();
		
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(validatorExpression);
		
		final Switchable<String,ActionDef> actionDefMap = createActionDefMap();
		checkValidatorId(idInputIdMap, actionDefMap);
		
		final ActionBuilder actionBuilder = new ActionBuilder(
				new ActionInvoker(
						actionDefMap, 
						// 各画面遷移定義でマッチしなかったアクション結果に対するデフォルトの遷移先定義。
						new ForwardDef("simple_error", new PathEL("local"), ActionConstants.ANY)),
						// 自動的に画面に値を設定するための、更新対象の属性名の一覧の定義。
						Arrays.asList("name", "id"));
		
		toNode = new DelegateDomExpression();
		
		domExpression = new DumpIfCatch(new Serial(
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				new ConfigDomExpression(configExpression),
				validatorBuilder.getCheckExpression(
						actionBuilder.getActionExpression(), 
						new Serial(
								// 2番目の引数は値検証エラーが発生したときのエラーを表示するデフォルトの画面 id。
								new FormatId(new LimitedValueFormat(idInputIdMap, "simple_errorValidate")),
								validatorBuilder.getReplaceExpression()
						)),
				// HTML を読み込んで加工した後の静的なHTML雛型をキャッシュする(キャッシュを有効にするには0番目の引数を正の数にする)。
				new CacheNode(0, new Serial(
						toNode,
						HtmlConstants.NORMALIZE,
						// ダミーのノードを削除する。
						new XPath("//*[@id=\"dummy\" or @name=\"dummy\"]", DomConstants.REMOVE_THIS),
						// ToLinkedHashMap で生成した Map を使用してラジオボタンとチェックボックスを再生成する。
						new XPath("//INPUT[@name and (@type=\"radio\" or @type=\"checkbox\")]", 
								new MakeRadio(new ToELTargetExceptionEL(){
									@Override public Object execute(Object root, Object value) throws Exception {
										// new OGE("config[node.getAttribute(\"name\")]")
										final DomContext context = (DomContext)root;
										return context.getConfig().get(((Element)context.getNode()).getAttribute("name"));
									}
								})),
						// ToLinkedHashMap で生成した Map を使用してセレクトを再生成する。
						new XPath("//SELECT[@name]", 
								new MakeSelect(new ToELTargetExceptionEL(){
									@Override public Object execute(Object root, Object value) throws Exception {
										// new OGE("config[node.getAttribute(\"name\")]")
										final DomContext context = (DomContext)root;
										return context.getConfig().get(((Element)context.getNode()).getAttribute("name"));
									}
								})),
						// FORM の action 属性に指定されている URL をアプリケーション用に変換する。
						new XPath("//FORM[@action]", new FormatAttrValue("action", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
						// INPUT の onclick 属性に指定されている URL をアプリケーション用に変換する。
						new XPath("//INPUT[@onclick and (@type=\"button\" or @type=\"submit\")]", 
								new FormatAttrValue("onclick", new RegExpFormat("\\'([^\\.]+)\\.[^\\.]+\\'", "\'$1\'"))),
								// A (アンカー)の href 属性に指定されている URL をアプリケーション用に変換する。
						new XPath("//A[@href]", new FormatAttrValue("href", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
						// 入力値の maxlength などを再設定する。
						new XPath("//INPUT[@name=\"name\"]", new Exec(new ToELTargetExceptionEL(){
							@Override public Object execute(Object root, Object value) throws Exception {
								// new OGE("context.node.setAttribute(\"sise\", \"40\"),context.node.setAttribute(\"maxlength\", \"40\")")
								final Element e = (Element)((DomContext)((Map)root).get("context")).getNode();
								e.setAttribute("size", "40");
								e.setAttribute("maxlength", "40");
								return null;
							}
						})),
						new XPath("//INPUT[@name=\"arg\"]", new Exec(new ToELTargetExceptionEL(){
							@Override public Object execute(Object root, Object value) throws Exception {
								// new OGE("context.node.setAttribute(\"sise\", \"3\"),context.node.setAttribute(\"maxlength\", \"3\")")
								final Element e = (Element)((DomContext)((Map)root).get("context")).getNode();
								e.setAttribute("size", "3");
								e.setAttribute("maxlength", "3");
								return null;
							}
						})),
						new XPath("//INPUT[@name=\"comment\"]", new Exec(new ToELTargetExceptionEL(){
							@Override public Object execute(Object root, Object value) throws Exception {
								// new OGE("context.node.setAttribute(\"rows\", \"3\"),context.node.setAttribute(\"cols\", \"40\")")
								final Element e = (Element)((DomContext)((Map)root).get("context")).getNode();
								e.setAttribute("rows", "3");
								e.setAttribute("cols", "40");
								return null;
							}
						}))
				)),
				actionBuilder.getAutoUpdateExpression(),
				// age 項目の null 値(マイナス値)を画面から削除する。
				new XPath("//INPUT[@name=\"age\" and @value=\"-1\"]", new RemoveAttr("value")),
				// 確認画面の FORM に hidden を自動設定する。
				new RegExpId("^.*Confirm$", new XPath("//FORM", 
						new HiddenAllToChild(new PathEL("input"), Arrays.asList("")))),
				// 入力エラーは actionBuilder.getAutoUpdateExpression() で制御しないため、ここで明示的に埋め込む。
				new XPath("//P[@id=\"errorValidate\"]", new UpdateValue(validatorBuilder.getMessageEL()))
		));
	}

	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/crud/$1.html")),
				new XmlToNode(new InputStreamReaderFactory(new ServletInputStreamFactory(servletConfig.getServletContext()), "UTF-8")))));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException{
		final DomContext domContext = new DomContext();
		domContext.setId(request.getRequestURI());
		domContext.setInput(request.getParameterMap());
		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(UtilsConstants.concurrentMap(new HttpSessionMap(request.getSession()), request.getSession()));
		domContext.setApplication(UtilsConstants.concurrentMap(new ServletContextMap(request.getSession().getServletContext()), request.getSession().getServletContext()));
		domContext.setContainer(new ScopeContainer(domContext.getRequest(), domContext.getSession(), domContext.getApplication(), componentDefMap));
		domExpression.interpret(domContext);
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try{
			DomConstants.write(domContext.getNode(), w);
		}finally{
			w.close();
		}
	}
}
