/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples;

import woolpack.action.ActionDef;
import woolpack.action.ForwardDef;
import woolpack.dom.DomConstants;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.Switchable;

/**
 * 特定のパターンの{@link ActionDef}を{@link SwitchBuilder}に登録する生成器。
 * @author nakamura
 *
 */
public class ActionDefMaker {

	private final SwitchBuilder<String,ActionDef> builder;
	
	/**
	 * コンストラクタ。
	 *
	 */
	public ActionDefMaker(){
		this.builder = new SwitchBuilder<String,ActionDef>();
	}
	
	/**
	 * 生成された{@link Switchable}を返す。
	 * @return 生成された{@link Switchable}。
	 */
	public Switchable<String,ActionDef> get(){
		return builder.get();
	}
	
	/**
	 * アクションの定義を登録する。
	 * @param id id。
	 * @param actionDef アクションの定義。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException 登録済のキーを指定した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ActionDefMaker put(final String id, final ActionDef actionDef){
		id.charAt(0);
		builder.put(id, actionDef);
		return this;
	}
	
	private ActionDefMaker put(
			final String actionId, 
			final GettingEL componentEL, 
			final GettingEL methodEL, 
			final String forwardId, 
			final GettingEL forwardEL){
		return put(actionId, new ActionDef(componentEL, methodEL, new ForwardDef(forwardId, forwardEL)));
	}
	
	/**
	 * アクションの定義を登録する。
	 * メソッドの返却値がエラーでないチェック条件で、かつ遷移先の id をアクション id と同一とした遷移先ひとつを遷移先一覧とする。
	 * @param id id。
	 * @param componentEL コンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param methodEL 実行するメソッドへの参照。コンテキスト役に対して副作用が発生してもよい。
	 * @param forwardEL 遷移先のコンポーネントへの参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException 登録済のキーを指定した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ActionDefMaker put(
			final String id, 
			final GettingEL componentEL, 
			final GettingEL methodEL, 
			final GettingEL forwardEL){
		return put(id, componentEL, methodEL, id, forwardEL);
	}
	
	/**
	 * 遷移先の id をアクション id と同一とし、副作用が発生しないアクション定義を登録する。
	 * @param id id。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException 登録済のキーを指定した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ActionDefMaker putForward(final String id){
		return put(id, EL.NULL, EL.NULL, id, EL.NULL);
	}
	
	/**
	 * 遷移先のコンポーネントへの参照を画面からの入力値そのものとするアクションの定義を登録する。
	 * @param actionId アクション id。
	 * @param forwardId 遷移先の id。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException 登録済のキーを指定した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ActionDefMaker putEcho(final String actionId, final String forwardId){
		return put(actionId, EL.NULL, EL.NULL, forwardId, DomConstants.INPUT_EL);
	}
	
	/**
	 * 遷移先の id をアクション id と同一とし、遷移先のコンポーネントへの参照を画面からの入力値そのものとするアクションの定義を登録する。
	 * @param id id。
	 * @return このオブジェクトへの参照。
	 * @throws IllegalStateException 登録済のキーを指定した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException id が空の場合。
	 */
	public ActionDefMaker putEcho(final String id){
		return put(id, EL.NULL, EL.NULL, id, DomConstants.INPUT_EL);
	}
}
