/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.dom.Count;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.utils.MapBuilder;
import woolpack.utils.UtilsConstants;

public class TransactionBuilderTest extends TestCase {

	public void testConstructor(){
		try{
			new TransactionBuilder(null, Arrays.asList("id0"), 32);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new TransactionBuilder("", Arrays.asList("id0"), 32);
			fail();
		}catch(final StringIndexOutOfBoundsException expected){
		}
		try{
			new TransactionBuilder(null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testGetCheckExpressionTrueExpressionNull(){
		final TransactionBuilder transactionBuilder = new TransactionBuilder("token", Arrays.asList("id0"), 32);
		try{
			transactionBuilder.getCheckExpression(null, DomConstants.NULL);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testGetCheckExpressionFalseExpressionNull(){
		final TransactionBuilder transactionBuilder = new TransactionBuilder(Arrays.asList("id0"));
		try{
			transactionBuilder.getCheckExpression(DomConstants.NULL, null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testGet(){
		final TransactionBuilder transactionBuilder = new TransactionBuilder("token", Arrays.asList("id0"), 32);
		assertEquals("token", transactionBuilder.getKey());
		final Collection<String> expected = Arrays.asList("id0");
		assertTrue(expected.containsAll(transactionBuilder.getIdCollection()));
		assertTrue(transactionBuilder.getIdCollection().containsAll(expected));
	}
	
	public void testGetAppendExpression(){
		final TransactionBuilder transactionBuilder = new TransactionBuilder(Arrays.asList("id0"));
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<A href=\"a0.html\">a0</A>" +
						"<A href=\"a1.html?k0=v0\">a1</A>" +
						"<A href=\"a2.html?k0=v0&amp;k1=v1\">a2</A>"),
				transactionBuilder.getAppendExpression());
			final DomContext context = new DomContext();
			context.setSession(UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object()));
			context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token0");
			expression.interpret(context);

			assertTrue(TestUtils.equalsForm(context, 
					"<A href=\"a0.html?woolpack.transaction.TRANSACTION_TOKEN=token0\">a0</A>" +
					"<A href=\"a1.html?k0=v0&amp;woolpack.transaction.TRANSACTION_TOKEN=token0\">a1</A>" +
					"<A href=\"a2.html?k0=v0&amp;k1=v1&amp;woolpack.transaction.TRANSACTION_TOKEN=token0\">a2</A>" +
					"<INPUT type=\"hidden\" name=\"woolpack.transaction.TRANSACTION_TOKEN\" value=\"token0\" />"));
	}
	
	public void testGetCheckExpression(){
		final Count counter0 = new Count(DomConstants.NULL);
		final Count counter1 = new Count(DomConstants.NULL);
		final TransactionBuilder transactionBuilder = new TransactionBuilder(Arrays.asList("id0"));
		final DomExpression expression = transactionBuilder.getCheckExpression(counter0, counter1);
		final DomContext context = new DomContext();
		context.setInput(new HashMap());
		context.setSession(UtilsConstants.concurrentMap(new HashMap<String,Object>(), new Object()));
		context.setInput(MapBuilder.get(new HashMap<String,Object>()).put("woolpack.transaction.TRANSACTION_TOKEN", "token0").get());

		context.setId("id0");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token0");
		expression.interpret(context);
		assertEquals(1, counter0.getCount());
		assertEquals(0, counter1.getCount());

		System.out.println(context.getSession().get(TransactionBuilder.KEY));
		assertEquals(32, context.getSession().get(TransactionBuilder.KEY).toString().length());

		context.setId("id0");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token1");
		expression.interpret(context);
		assertEquals(1, counter0.getCount());
		assertEquals(1, counter1.getCount());

		context.setId("id1");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token0");
		expression.interpret(context);
		assertEquals(2, counter0.getCount());
		assertEquals(1, counter1.getCount());

		context.setId("id1");
		context.getSession().put("woolpack.transaction.TRANSACTION_TOKEN", "token1");
		expression.interpret(context);
		assertEquals(3, counter0.getCount());
		assertEquals(1, counter1.getCount());

		context.setId("id1");
		context.getSession().remove("woolpack.transaction.TRANSACTION_TOKEN");
		expression.interpret(context);
		assertEquals(4, counter0.getCount());
		assertEquals(1, counter1.getCount());
	}
}
