/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Collection;
import java.util.Iterator;

import woolpack.utils.Switchable;
import woolpack.utils.UtilsConstants;

/**
 * プロパティ名で委譲先を分岐する{@link ValidatorExpression}。
 * プロパティ名に対応する値が存在しない場合も委譲するため、
 * ブラウザから送信されない項目を必須を検証する際に使用可能。
 * @author nakamura
 *
 */
public class BranchByNameValidator extends BranchValidator{
	private final ValidatorIterable validatorIterable;
	
	/**
	 * コンストラクタ。
	 * @param validatorIterable 値検証一覧の呼び出し方。
	 * @param switchable プロパティ名と{@link ValidatorExpression}の{@link Switchable}。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public BranchByNameValidator(final ValidatorIterable validatorIterable, final Switchable<String,ValidatorExpression> switchable){
		super(UtilsConstants.nullToDefaultSwitchIfAbsent(ValidatorConstants.TRUE, switchable));
		validatorIterable.getClass();
		this.validatorIterable = validatorIterable;
	}
	
	/**
	 * コンストラクタ。値検証一覧の呼び出し方に{@link ValidatorConstants#AND}を使用する。
	 * @param switchable プロパティ名と{@link ValidatorExpression}の{@link Switchable}。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public BranchByNameValidator(final Switchable<String,ValidatorExpression> switchable){
		this(ValidatorConstants.AND, switchable);
	}
	
	protected Collection<String> keySet(final ValidatorContext context){
		return getSwitchable().keys();
	}

	/**
	 * @throws NullPointerException 引数が null の場合。
	 */
	@Override public boolean interpret(final ValidatorContext context) {
		return validatorIterable.interpret(context, new Iterable<ValidatorExpression>(){
			public Iterator<ValidatorExpression> iterator() {
				return new Iterator<ValidatorExpression>(){
					private final Iterator<String> it = keySet(context).iterator();
					public boolean hasNext() {
						return it.hasNext();
					}
					public ValidatorExpression next() {
						return new ValidatorExpression(){
							public boolean interpret(final ValidatorContext context) {
								final ValidatorContext newContext = context.copy();
								newContext.setTmpKey(it.next());
								final ValidatorExpression e = getSwitchable().get(newContext.getTmpKey());
								return e.interpret(newContext);
							}
							public void appendTo(final Appendable sb) {
								throw new UnsupportedOperationException();
							}
							public void addMessageTo(final Collection<String> messageCollection) {
								throw new UnsupportedOperationException();
							}
						};
					}
					public void remove() {
						throw new UnsupportedOperationException();
					}
				};
			}
		});
	}
}
