/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;


import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.NodeLoop;
import woolpack.utils.XPathWrapper;

/**
 * 定数・静的メソッドの集まり。
 * @author nakamura
 *
 */
public class HtmlConstants {
	static final XPathWrapper XPATH_OPTION = new XPathWrapper("//OPTION");
	
	/**
	 * 指定された DOM ノードを削除し、さらにその後ろにあるテキストノードを全て削除する{@link DomExpression}。
	 * ラジオボタン・チェックボックスを削除するために使用する。
	 * {@link DomExpression#interpret(DomContext)}は引数または{@link DomContext#getNode()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final DomExpression REMOVE_THIS_AND_TEXTS = new DomExpression(){
		public void interpret(final DomContext context) {
			removeThisAndText(context.getNode());
		}
	};
	
	/**
	 * エレメント名を大文字に/属性名を小文字に変換する{@link DomExpression}。
	 * {@link DomExpression#interpret(DomContext)}は引数が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final DomExpression NORMALIZE = new NodeLoop(){
		@Override 
		public void interpret(final DomContext context) {
			super.interpret(context);
			if(context.getNode().getNodeType() == Node.ELEMENT_NODE){
				final Element e = (Element)context.getNode();
				final Element newE = DomConstants.getDocumentNode(e).createElement(e.getNodeName().toUpperCase());
				{
					final NamedNodeMap attrs = e.getAttributes();
					for(int i=0; i<attrs.getLength(); i++){
						final Attr attr = (Attr)attrs.item(i);
						newE.setAttribute(attr.getName().toLowerCase(), attr.getNodeValue());
					}
				}
				{
					Node child = null;
					while((child = e.getFirstChild()) != null){
						newE.appendChild(child);
					}
				}
				e.getParentNode().replaceChild(newE, e);
			}
		}
	};

	private HtmlConstants(){}
	
	/**
	 * 指定された DOM ノードを削除し、さらにその後ろにあるテキストノードを全て削除する。
	 * ラジオボタン・チェックボックスを削除するために使用する。
	 * @param node 削除対象の基点ノード。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public static void removeThisAndText(final Node node){
		Node nextNode = node.getNextSibling();
		while(nextNode != null && nextNode.getNodeType() == Node.TEXT_NODE){
			DomConstants.removeThis(nextNode);
			nextNode = node.getNextSibling();
		}
		DomConstants.removeThis(node);
	}
}
