/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.ArrayList;
import java.util.Collection;
import java.util.StringTokenizer;


/**
 * ドット区切りのプロパティ名一覧の順に基点をたどっていく{@link EL}。
 * {@link java.util.Map}または get/set メソッドのみをサポートする。
 * @author nakamura
 *
 */
public class PathEL extends ELAdapter {
	private final EL el;
	
	/**
	 * コンストラクタ。
	 * @param expression プロパティ名をドットでつなげた表現。
	 * @param converter 型変換器。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public PathEL(final String expression, final TypeConverter converter){
		final StringTokenizer st = new StringTokenizer(expression, ".");
		final Collection<EL> c = new ArrayList<EL>();
		while(st.hasMoreTokens()){
			c.add(new PropertyEL(st.nextToken(), converter));
		}
		el = new ArrayPathEL(c);
	}

	/**
	 * コンストラクタ。
	 * @param expression プロパティ名をドットでつなげた表現。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public PathEL(final String expression){
		this(expression, ELConstants.COLLECTION_CONVERTER);
	}
	
	@Override public Object getValue(final Object root, final Class clazz) throws ELTargetRuntimeException {
		return el.getValue(root, clazz);
	}

	@Override public void setValue(final Object root, final Object value) throws ELTargetRuntimeException {
		el.setValue(root, value);
	}
	
	@Override public int hashCode(){
		return el.hashCode();
	}
	
	@Override public boolean equals(final Object other){
		if(other instanceof PathEL){
			final PathEL o = (PathEL)other;
			return el.equals(o.el);
		}
		return false;
	}
}
