/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import woolpack.utils.UtilsConstants;

/**
 * 定数の集まり。
 * @author nakamura
 *
 */
public class ConfigConstants {
	private ConfigConstants(){}

	/**
	 * null オブジェクト役の{@link ConfigExpression}。
	 */
	public static final ConfigExpression NULL = new ConfigExpression(){
		public void interpret(final ConfigContext context) {
		}
	};
	
	/**
	 * 書き込み禁止の{@link Map}を{@link ConfigContext#setMap(Map)}に再設定する{@link ConfigExpression}。
	 * {@link ConfigExpression#interpret(ConfigContext)}は引数または{@link ConfigContext#getMap()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final ConfigExpression TO_UNMODIFIABLE_MAP = new ConfigExpression(){
		public void interpret(final ConfigContext context) {
			context.setMap(Collections.unmodifiableMap(context.getMap()));
		}
	};
	
	/**
	 * 上書き禁止の{@link Map}を{@link ConfigContext#setMap(Map)}に再設定する{@link ConfigExpression}。
	 * {@link ConfigExpression#interpret(ConfigContext)}は引数または{@link ConfigContext#getMap()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final ConfigExpression TO_UNOVERWRITABLE_MAP = new ConfigExpression(){
		public void interpret(final ConfigContext context) {
			context.setMap(UtilsConstants.unoverwritableMap(context.getMap()));
		}
	};
	
	/**
	 * {@link HashMap}を生成して{@link ConfigContext#setMap(Map)}に設定する{@link ConfigExpression}。
	 * {@link ConfigExpression#interpret(ConfigContext)}は引数が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final ConfigExpression MAKE_MAP = new ConfigExpression(){
		public void interpret(final ConfigContext context) {
			context.setMap(new HashMap<String,Object>());
		}
	};
}
