/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.transaction;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import woolpack.action.ActionConstants;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.dom.ResetId;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.FormatAttrValue;
import woolpack.dom.FormatId;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.ee.ActionBuilder;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.TransactionBuilder;
import woolpack.ee.ValidatorRuntimeException;
import woolpack.html.ToNode;
import woolpack.samples.ActionDefMaker;
import woolpack.text.FixFormat;
import woolpack.text.RegExpFormat;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.OGE;

/**
 * 画面遷移の順序を検証するサンプル Web アプリケーション。
 * @author nakamura
 *
 */
public class TransactionServlet extends HttpServlet {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	public final DelegateDomExpression toNode;
	public final DomExpression domExpression;
	
	public TransactionServlet(){

		final ActionDefMaker maker = new ActionDefMaker();
		maker.putForward("0");
		maker.putForward("1");
		maker.putForward("2");
		
		final ActionBuilder actionBuilder = new ActionBuilder(
				new ActionInvoker(
						maker.getMap(), 
						new ForwardDef("simple_error", new OGE("local"), ActionConstants.ANY)), 
				Arrays.asList("name", "id"));
		
		toNode = new DelegateDomExpression();

		// id の数が多くなるとハッシュ実装のほうが検索効率がよいので HashSet として id 一覧を扱う。
		final TransactionBuilder transactionBuilder = new TransactionBuilder(new HashSet<String>(Arrays.asList("1", "2")));
		
		{
			// トランザクション定義に対するid存在検証。定義(製造)のミスを発見するための仕掛け。
			final Collection<String> retain =  new HashSet<String>(transactionBuilder.getIdCollection());
			retain.removeAll(maker.getMap().keySet());
			if(!retain.isEmpty()){
				throw new RuntimeException("retain transaction id: " + retain);
			}
		}

		domExpression = new Serial(
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				// トランザクショントークンを検証しトランザクショントークンを生成する。
				transactionBuilder.getCheckExpression(
						// トランザクションが妥当な場合はアクションに処理をさせる。
						actionBuilder.getActionExpression(), 
						// トランザクションが妥当な場合は遷移する画面を切り替える。
						new FormatId(new FixFormat("simple_errorTransaction"))),
				toNode,
				new XPath("//A[@href]", new FormatAttrValue("href", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				new XPath("//FORM[@action]", new FormatAttrValue("action", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1"))),
				actionBuilder.getAutoUpdateExpression(),
				// トランザクショントークンを埋め込む。
				transactionBuilder.getAppendExpression()
		);
	}


	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/transaction/$1.html")),
				new ToNode(new InputStreamReaderFactory(new ServletInputStreamFactory(servletConfig.getServletContext()), "UTF-8")))));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException{
		final DomContext domContext = new DomContext();
		domContext.setId(request.getRequestURI());
		domContext.setInput(request.getParameterMap());
//		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(new HttpSessionMap(request.getSession()));
//		domContext.setApplication(new ServletContextMap(request.getSession().getServletContext()));
		try{
			domExpression.interpret(domContext);
		}catch(final ValidatorRuntimeException e){
			final StringBuilder sb = new StringBuilder();
			e.getContext().appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}catch(final RuntimeException e){
			final StringBuilder sb = new StringBuilder();
			domContext.appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try{
			DomConstants.write(domContext.getNode(), w);
		}finally{
			w.close();
		}
	}
}
