/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.samples.locale;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import woolpack.action.ActionConstants;
import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ForwardDef;
import woolpack.config.ConfigConstants;
import woolpack.config.ConfigExpression;
import woolpack.config.ConfigSerial;
import woolpack.config.CopyConfigContext;
import woolpack.config.ToLinkedHashMap;
import woolpack.container.ComponentDef;
import woolpack.container.ScopeContainer;
import woolpack.dom.DelegateDomExpression;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.FormatAttrValue;
import woolpack.dom.FormatId;
import woolpack.dom.ResetId;
import woolpack.dom.Serial;
import woolpack.dom.XPath;
import woolpack.ee.ActionBuilder;
import woolpack.ee.ConfigDomExpression;
import woolpack.ee.HttpSessionMap;
import woolpack.ee.ServletContextMap;
import woolpack.ee.ServletInputStreamFactory;
import woolpack.ee.ServletRequestAttributeMap;
import woolpack.ee.ValidatorBuilder;
import woolpack.ee.ValidatorRuntimeException;
import woolpack.html.MakeSelect;
import woolpack.html.ToNode;
import woolpack.html.UpdateValue;
import woolpack.locale.BranchByLocaleValidator;
import woolpack.locale.LocaleCacheMap;
import woolpack.locale.LocaleCacheNode;
import woolpack.locale.LocaleId;
import woolpack.locale.LocalePutResourceBundle;
import woolpack.locale.LocaleReaderFactory;
import woolpack.samples.ActionDefMaker;
import woolpack.text.FixFormat;
import woolpack.text.RegExpFormat;
import woolpack.utils.InputStreamFactory;
import woolpack.utils.InputStreamReaderFactory;
import woolpack.utils.MapBuilder;
import woolpack.utils.OGE;
import woolpack.utils.ReaderFactory;
import woolpack.validator.BranchByNameValidator;
import woolpack.validator.IfNotValidator;
import woolpack.validator.IfValidator;
import woolpack.validator.MessageValidator;
import woolpack.validator.RegExpIdValidator;
import woolpack.validator.RegExpValidator;
import woolpack.validator.SerialValidator;
import woolpack.validator.ValidatorConstants;
import woolpack.validator.ValidatorExpression;
import woolpack.validator.ValueLoopValidator;

/**
 * ロケールごとに挙動を変えるサンプルウェブアプリケーション。
 * @author nakamura
 *
 */
public class LocaleServlet extends HttpServlet {

	private final ThreadLocal<Locale> threadLocal;
	
	public final Map<String,ComponentDef> componentDefMap;
	public final DelegateDomExpression toNode;
	public final DomExpression domExpression;
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	
	public LocaleServlet(){
		threadLocal = new ThreadLocal<Locale>();
		
		componentDefMap = new HashMap<String,ComponentDef>();
		
		// ロケールごとに、加工した設定をキャッシュする。
		final ConfigExpression configExpression = new LocaleCacheMap(0, new ConfigSerial(
				ConfigConstants.MAKE_MAP,
				new CopyConfigContext(new ConfigSerial(
						ConfigConstants.TO_UNOVERWRITABLE_MAP,
						// ロケールごとにリソースバンドルを読み込む。
						new LocalePutResourceBundle("woolpack.samples.locale.selection", threadLocal),
						new ToLinkedHashMap("address", "part.address.value", "part.address.label", ","),
						new LocalePutResourceBundle("woolpack.samples.locale.messages", threadLocal)
				)),
				ConfigConstants.TO_UNMODIFIABLE_MAP
		), threadLocal);
		
		final ValidatorExpression validatorExpression = new IfValidator(
				new RegExpIdValidator("bean0_confirm"),
				new BranchByNameValidator(new MapBuilder<String,ValidatorExpression>()
						.put("cache", new ValueLoopValidator(new SerialValidator(
								new IfNotValidator(
										ValidatorConstants.REQUIRED,
										new MessageValidator("validator.cache.required")),
								new IfNotValidator(
										// ロケールごとにフォーマットを設定する。
										new BranchByLocaleValidator(
												new MapBuilder<Locale,ValidatorExpression>()
												.put(Locale.JAPANESE, new RegExpValidator("\\d{3}"))
												.put(Locale.ENGLISH, new RegExpValidator("\\d{2}"))
												.getMap(), threadLocal),
										new MessageValidator("validator.cache.valid"))))
						).getMap()));
		
		final ValidatorBuilder validatorBuilder = new ValidatorBuilder(validatorExpression);
		
		final Map<String,ActionDef> actionDefMap;
		{
			final ActionDefMaker maker = new ActionDefMaker();
			maker.putForward("simple_error");
			maker.putForward("simple_errorValidate");
			maker.putForward("bean0_input");
			maker.putEcho("bean0_confirm");
			actionDefMap = maker.getMap();
		}
		
		final ActionBuilder actionBuilder = new ActionBuilder(
				new ActionInvoker(
						actionDefMap, 
						new ForwardDef("simple_error", new OGE("local"), ActionConstants.ANY)),
						Arrays.asList("name", "id"));
		
		toNode = new DelegateDomExpression();
		
		domExpression = new Serial(
				new FormatId(new RegExpFormat("^.*/([^/]+)$", "$1")),
				new ConfigDomExpression(configExpression),
				validatorBuilder.getCheckExpression(
						actionBuilder.getActionExpression(),
						new Serial(
								// 2番目の引数は値検証エラーが発生したときのエラーを表示するデフォルトの画面 id。
								new FormatId(new FixFormat("simple_errorValidate")),
								validatorBuilder.getReplaceExpression()
						)),
				// ロケールごとに、加工した DOM ノードをキャッシュする。
				new LocaleCacheNode(0, new Serial(
						// ロケール別の HTML をロードする。
						new LocaleId(toNode, threadLocal),
						new XPath("//SELECT[@name]", 
								new MakeSelect(new OGE("config[node.getAttribute(\"name\")]"))),
						new XPath("//FORM[@action]", new FormatAttrValue("action", new RegExpFormat("^([^\\.]+)\\.[^\\.]+$", "$1")))
				), threadLocal),
				actionBuilder.getAutoUpdateExpression(),
				new XPath("//P[@id=\"errorValidate\"]", new UpdateValue(validatorBuilder.getOGE()))
		);
	}

	@Override public void init(final ServletConfig servletConfig) throws ServletException {
		super.init(servletConfig);
		final InputStreamFactory inputStreamFactory = new ServletInputStreamFactory(servletConfig.getServletContext());
		toNode.setExpression(new ResetId(new Serial(
				new FormatId(new RegExpFormat("^(.*)$", "/html/sample/locale/$1.html")),
				// ロケールごとに HTML 読み出しの文字セットを制御する。
				new ToNode(new LocaleReaderFactory(new MapBuilder<Locale,ReaderFactory>()
						.put(Locale.JAPANESE, new InputStreamReaderFactory(inputStreamFactory, "UTF-8"))
						.put(Locale.ENGLISH, new InputStreamReaderFactory(inputStreamFactory, "US-ASCII"))
						.getMap(), threadLocal))
		)));
	}

	@Override protected void service(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException{
		threadLocal.set(request.getLocale());
		
		final DomContext domContext = new DomContext();
		domContext.setId(request.getRequestURI());
		domContext.setInput(request.getParameterMap());
		domContext.setRequest(new ServletRequestAttributeMap(request));
		domContext.setSession(new HttpSessionMap(request.getSession()));
		domContext.setApplication(new ServletContextMap(request.getSession().getServletContext()));
		domContext.setContainer(new ScopeContainer(domContext.getRequest(), domContext.getSession(), domContext.getApplication(), componentDefMap));
		try{
			domExpression.interpret(domContext);
		}catch(final ValidatorRuntimeException e){
			final StringBuilder sb = new StringBuilder();
			e.getContext().appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}catch(final RuntimeException e){
			final StringBuilder sb = new StringBuilder();
			domContext.appendTo(sb);
			throw new RuntimeException(sb.toString(), e);
		}
		final Writer w = new BufferedWriter(new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
		try{
			DomConstants.write(domContext.getNode(), w);
		}finally{
			w.close();
		}
	}
}
