/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.Format;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.text.DelegateNumberFormat;
import woolpack.text.FormatFactory;

public class ParseValidatorTest extends TestCase {

	public void testConstructor(){
		try{
			new ParseValidator((FormatFactory)null);
			fail();
		}catch(final NullPointerException expected){
		}
		try{
			new ParseValidator((Format)null);
			fail();
		}catch(final NullPointerException expected){
		}
	}

	public void testInterpret(){
		final ValidatorExpression expression = new ParseValidator(new DecimalFormat());
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String,List<Object>>());
		context.setTmpKey("key0");
		
		context.setTmpValue("1000");
		assertTrue(expression.interpret(context));
		assertTrue(TestUtils.equals("#{\"key0\":{1000L}}", context.getInputMap()));

		context.setTmpValue("a");
		assertFalse(expression.interpret(context));
		assertTrue(TestUtils.equals("#{\"key0\":{\"a\"}}", context.getInputMap()));

		context.setTmpValue("");
		assertFalse(expression.interpret(context));
		assertTrue(TestUtils.equals("#{\"key0\":{\"\"}}", context.getInputMap()));
		
		try{
			context.setTmpValue(new Integer(1));
			expression.interpret(context);
			fail();
		}catch(final ClassCastException e){
			assertTrue(TestUtils.equals("#{\"key0\":{1}}", context.getInputMap()));
		}
		
		try{
			context.setTmpValue(null);
			expression.interpret(context);
			fail();
		}catch(final NullPointerException e){
			assertTrue(TestUtils.equals("#{\"key0\":{null}}", context.getInputMap()));
		}
	}
	
	public void testAppendTo() throws IOException{
		final ValidatorExpression expression = new ParseValidator(new DelegateNumberFormat(new DecimalFormat(), "new NumberFormat()"));
		final StringBuilder sb = new StringBuilder();
		expression.appendTo(sb);
		assertEquals("new ParseValidator(new NumberFormat())", sb.toString());
	}
	
	public void testAddMessageTo(){
		final ValidatorExpression expression = new ParseValidator(new DecimalFormat());
		final Collection<String> expectedCollection = new ArrayList<String>();
		expression.addMessageTo(expectedCollection);
		assertTrue(expectedCollection.isEmpty());
	}
}
