/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Arrays;

import junit.framework.TestCase;
import woolpack.TestUtils;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.utils.OGE;

public class AutoUpdateTest extends TestCase {

	public void testConstructor(){
		try{
			new AutoUpdate(null, OGE.NULL);
		}catch(final NullPointerException expected){
		}
		try{
			new AutoUpdate(Arrays.asList("id"), null);
		}catch(final NullPointerException expected){
		}
	}

	public void testSimple(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"+ 
						"<DIV id=\"div0\" >value0</DIV>"+ 
						"<SPAN id=\"span0\" >value0</SPAN>"+ 
						"<INPUT name=\"input0\" value=\"value0\" />"+ 
						"<SPAN id=\"span1\" >spanvalue1</SPAN>"),
				new AutoUpdate(Arrays.asList("name", "id"), new OGE("container")));
		final DomContext context = new DomContext();
		new OGE("container = #{\"textarea0\":\"textareavalue0\", \"div0\":\"divvalue0\", \"span0\":\"spanvalue0\", \"input0\":\"inputvalue0\"}").getValue(context);
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, 
				"<TEXTAREA name=\"textarea0\" >textareavalue0</TEXTAREA>"+ 
				"<DIV id=\"div0\" >divvalue0</DIV>"+ 
				"<SPAN id=\"span0\" >spanvalue0</SPAN>"+ 
				"<INPUT name=\"input0\" value=\"inputvalue0\" />"+ 
				"<SPAN id=\"span1\" >spanvalue1</SPAN>"));
	}

	public void testIterate(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<TABLE>" +
						"<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>" +
						"</TABLE>"),
						new AutoUpdate(Arrays.asList("name", "id"), new OGE("local")));
		final DomContext context = new DomContext();
		new OGE("local.iterate0 = {#{\"id0\":\"value0\"}, #{\"id0\":\"value1\"}}").getValue(context);
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, 
				"<TABLE>" +
				"<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>" +
				"<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>" +
				"</TABLE>"));
	}

	public void testIterateEmpty(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<TABLE>" +
						"<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>" +
						"</TABLE>"),
						new AutoUpdate(Arrays.asList("name", "id"), new OGE("local")));
		final DomContext context = new DomContext();
		new OGE("local.iterate0 = {}").getValue(context);
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<TABLE></TABLE>"));
	}

	public void testArray(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeBody(
						"<TABLE>" +
						"<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>" +
						"</TABLE>"),
						new AutoUpdate(Arrays.asList("name", "id"), new OGE("local")));
		final DomContext context = new DomContext();
		new OGE("local.iterate0 = new java.util.HashMap[]{#{\"id0\":\"value0\"}, #{\"id0\":\"value1\"}}").getValue(context);
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, 
				"<TABLE>" +
				"<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>" +
				"<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>" +
				"</TABLE>"));
	}

	public void testNotFound(){
		final DomExpression expression = new Serial(
				TestUtils.getToNodeForm(
						"<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"+ 
						"<DIV id=\"div0\" >value0</DIV>"+ 
						"<SPAN id=\"span0\" >value0</SPAN>"+ 
						"<INPUT name=\"input0\" value=\"value0\" />"+ 
						"<SPAN id=\"span1\" >spanvalue1</SPAN>"),
				new AutoUpdate(Arrays.asList("name", "id"), new OGE("container")));
		final DomContext context = new DomContext();
		new OGE("container = new Object()").getValue(context);
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context, 
				"<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"+ 
				"<DIV id=\"div0\" >value0</DIV>"+ 
				"<SPAN id=\"span0\" >value0</SPAN>"+ 
				"<INPUT name=\"input0\" value=\"value0\" />"+ 
				"<SPAN id=\"span1\" >spanvalue1</SPAN>"));
	}
}
