/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.TestBean;
import woolpack.action.ActionDef;
import woolpack.action.ActionInvoker;
import woolpack.action.ActionResult;
import woolpack.action.ForwardDef;
import woolpack.action.ForwardDefNotFoundException;
import woolpack.action.ForwardMatcher;
import woolpack.action.ActionConstants;
import woolpack.utils.OGE;

public class ActionInvokerTest extends TestCase {

	public void testConstructor(){
		try{
			new ActionInvoker(null, new ForwardDef("forwardId0", OGE.NULL, ActionConstants.ANY));
			fail();
		}catch(final NullPointerException expected){
		}
		final Map<String,ActionDef> map = new HashMap<String,ActionDef>();
		try{
			new ActionInvoker(map, (Iterable<ForwardDef>)null);
			fail();
		}catch(final NullPointerException expected){
		}
	}
	
	public void testFindId(){
		final ForwardDef[] forwardDefArray = new ForwardDef[]{
				new ForwardDef("class", OGE.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("null", OGE.NULL, new ForwardMatcher(null)),
				new ForwardDef("value", OGE.NULL, new ForwardMatcher("myValue"))
		};
		{
			final ActionResult forwardResult = ActionInvoker.findId(Arrays.asList(forwardDefArray), new Integer(3));
			assertEquals(forwardDefArray[0], forwardResult.getForwardDef());
			assertEquals(new Integer(3), forwardResult.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays.asList(forwardDefArray), null);
			assertEquals(forwardDefArray[1], info.getForwardDef());
			assertNull(info.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays.asList(forwardDefArray), "myValue");
			assertEquals(forwardDefArray[2], info.getForwardDef());
			assertEquals("myValue", info.getReturnedObject());
		}
		try{
			ActionInvoker.findId(Arrays.asList(forwardDefArray), "yourValue");
			fail();
		}catch(final ForwardDefNotFoundException expected){
			assertEquals("yourValue", expected.getReturnedObject());
		}
	}
	
	public void testGet0(){
		final ForwardDef[] forwardDefArray = {
				new ForwardDef("forwardId0", OGE.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId1", OGE.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId2", OGE.NULL, ActionConstants.ANY)
		};
		final Map<String,ActionDef> map = new HashMap<String,ActionDef>();
		map.put("id0", new ActionDef(new OGE("myBean"), new OGE("myBean.executeInt()"), forwardDefArray[0]));
		map.put("id1", new ActionDef(new OGE("myBean"), new OGE("myBean.executeVoid()"), forwardDefArray[1]));
		final ActionInvoker defs = new ActionInvoker(map, forwardDefArray[2]);
		assertEquals(map, defs.getActionDefMap());
		final Iterator<ForwardDef> iterator = defs.getForwardDefIterable().iterator();
		assertTrue(iterator.hasNext());
		assertTrue(forwardDefArray[2] == iterator.next());
		assertFalse(iterator.hasNext());
	}
	
	public void testNormal() throws ForwardDefNotFoundException{
		final ForwardDef[] forwardDefs = {
				new ForwardDef("forwardId0", OGE.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId1", OGE.NULL, new ForwardMatcher(Integer.class)),
				new ForwardDef("forwardId2", OGE.NULL, ActionConstants.ANY),
				new ForwardDef("forwardId0", OGE.NULL, new ForwardMatcher(IndexOutOfBoundsException.class))
		};
		final Map<String,ActionDef> map = new HashMap<String,ActionDef>();
		map.put("id0", new ActionDef(new OGE("component0"), new OGE("component0.executeInt()"), forwardDefs[0]));
		map.put("id1", new ActionDef(new OGE("component0"), new OGE("component0.executeVoid()"), forwardDefs[1]));
		map.put("id2", new ActionDef(new OGE("component0"), new OGE("component0.executeException()"), forwardDefs[3]));
		final ActionInvoker defs = new ActionInvoker(map, forwardDefs[2]);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(new Integer(1), forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("yourInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(new Integer(1), forwardResult.getReturnedObject());
			assertEquals(0, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id2", inputMap);
			assertEquals(forwardDefs[3], forwardResult.getForwardDef());
			assertTrue(forwardResult.getReturnedObject() instanceof IndexOutOfBoundsException);
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections.singletonMap("component0", bean), "id1", inputMap);
			assertEquals(forwardDefs[2], forwardResult.getForwardDef());
			assertNull(forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
	}
	
	public void testForwardDefNotFoundException() throws ForwardDefNotFoundException{
		final ForwardDef forwardDefs = new ForwardDef("forwardId0", OGE.NULL, new ForwardMatcher(Integer.class));
		final Map<String,ActionDef> map = new HashMap<String,ActionDef>();
		map.put("id0", new ActionDef(new OGE("component0"), new OGE("component0.executeException()"), forwardDefs));
		final ActionInvoker defs = new ActionInvoker(map, forwardDefs);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			try{
				defs.invoke(Collections.singletonMap("component0", bean), "id0", inputMap);
				fail();
			}catch(final ForwardDefNotFoundException expected){
				assertTrue(expected.getReturnedObject() instanceof IndexOutOfBoundsException);
			}
		}
	}
}
