/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.io.IOException;
import java.util.Collection;
import java.util.LinkedList;

/**
 * {@link ValidatorExpression}の各メソッド呼び出された回数を数えるテスト用の{@link ValidatorExpression}。
 * このクラスはスレッドセーフではない。
 * {@link ValidatorContext}に対して状態を変更する操作を行わない。
 * @author nakamura
 *
 */
public class CountValidator implements ValidatorExpression {
	private final int max;
	private final ValidatorExpression expression;
	private final LinkedList<ValidatorContext> contextList;
	private final LinkedList<Appendable> sbList;
	private final LinkedList<Collection<String>> collectionList;
	private volatile int intepretCount = 0;
	private volatile int appendToCount = 0;
	private volatile int addMessageToCount = 0;
	
	/**
	 * コンストラクタ。
	 * @param max 一覧の最大件数。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CountValidator(final int max, final ValidatorExpression expression){
		expression.getClass();
		
		this.max = max;
		this.expression = expression;
		this.contextList = new LinkedList<ValidatorContext>();
		this.sbList = new LinkedList<Appendable>();
		this.collectionList = new LinkedList<Collection<String>>();
	}
	
	/**
	 * コンストラクタ。
	 * 一覧の最大件数を 4 とする。
	 * @param expression 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CountValidator(final ValidatorExpression expression){
		this(4, expression);
	}

	public boolean interpret(final ValidatorContext context) {
		intepretCount++;
		if(max > 0){
			if(intepretCount > max){
				contextList.removeFirst();
			}
			contextList.add(context);
		}
		return expression.interpret(context);
	}
	
	public void appendTo(final Appendable sb) throws IOException{
		appendToCount++;
		if(max > 0){
			if(appendToCount > max){
				sbList.removeFirst();
			}
			sbList.add(sb);
		}
		expression.appendTo(sb);
	}
	
	public void addMessageTo(final Collection<String> messageCollection) {
		addMessageToCount++;
		if(max > 0){
			if(addMessageToCount > max){
				collectionList.removeFirst();
			}
			collectionList.add(messageCollection);
		}
		expression.addMessageTo(messageCollection);
	}

	/**
	 * {@link #addMessageTo(Collection)}が呼び出された回数を返す。
	 * @return {@link #addMessageTo(Collection)}が呼び出された回数。
	 */
	public int getAddMessageToCount() {
		return addMessageToCount;
	}

	/**
	 * {@link #appendTo(Appendable)}が呼び出された回数を返す。
	 * @return {@link #appendTo(Appendable)}が呼び出された回数。
	 */
	public int getAppendToCount() {
		return appendToCount;
	}

	/**
	 * {@link #interpret(ValidatorContext)}が呼び出された回数を返す。
	 * @return {@link #interpret(ValidatorContext)}が呼び出された回数。
	 */
	public int getIntepretCount() {
		return intepretCount;
	}

	/**
	 * 実行された{@link #addMessageTo(Collection)}の引数の一覧を返す。
	 * @return 実行された{@link #addMessageTo(Collection)}の引数の一覧。
	 */
	public LinkedList<Collection<String>> getCollectionList() {
		return collectionList;
	}

	/**
	 * 実行された{@link #interpret(ValidatorContext)}の引数の一覧を返す。
	 * @return 実行された{@link #interpret(ValidatorContext)}の引数の一覧。
	 */
	public LinkedList<ValidatorContext> getContextList() {
		return contextList;
	}

	/**
	 * 実行された{@link #appendTo(Appendable)}の引数の一覧を返す。
	 * @return 実行された{@link #appendTo(Appendable)}の引数の一覧。
	 */
	public LinkedList<Appendable> getSbList() {
		return sbList;
	}
}
