/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * {@link Map}の生成器。
 * コンストラクタツリー内で{@link Map}の内容を生成する際に使用する。
 * @author nakamura
 *
 * @param <K> キー。
 * @param <V> 値。
 */
public class MapBuilder<K,V> {
	private final Map<K,V> map;
	private V beforeValue;
	
	/**
	 * コンストラクタ。
	 * @param map 作用対象。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public MapBuilder(final Map<K,V> map){
		map.getClass();
		this.map = map;
	}
	
	/**
	 * コンストラクタ。
	 * 内部で{@link LinkedHashMap}を生成する。
	 *
	 */
	public MapBuilder(){
		this(new HashMap<K,V>());
	}
	
	/**
	 * {@link #getMap()}に対し{@link Map#put(K, V)}を実行する。
	 * @param key キー。
	 * @param value 値。
	 * @return このオブジェクトへの参照。
	 */
	public MapBuilder<K,V> put(final K key, final V value){
		map.put(key, value);
		beforeValue = value;
		return this;
	}

	/**
	 * {@link #getMap()}に対し{@link Map#put(K, V)}を実行する。
	 * 前回実行した{@link #put(K, V)}の値を値として格納する。
	 * @param key キー。
	 * @return このオブジェクトへの参照。
	 */
	public MapBuilder<K,V> put(final K key){
		map.put(key, beforeValue);
		return this;
	}
	
	/**
	 * {@link Map}を返す。
	 * @return {@link Map}。
	 */
	public Map<K,V> getMap(){
		return map;
	}
}
