/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import javax.xml.xpath.XPathExpressionException;

import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.utils.XPathWrapper;

/**
 * XPath でノードを検索し委譲先を分岐する{@link DomExpression}。
 * 委譲先で変更されたノードは次の XPath 検索に反映される(スナップショットではない)。
 * @author nakamura
 *
 */
public class XPath implements DomExpression {
	private final XPathWrapper xpath;
	private final DomExpression firstExpression;
	private final DomExpression otherExpression;
	
	/**
	 * コンストラクタ。
	 * @param xpath XPath。
	 * @param firstExpression 最初の検索結果に対する委譲先。
	 * @param otherExpression 2番目以降の検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public XPath(final XPathWrapper xpath, final DomExpression firstExpression, final DomExpression otherExpression){
		xpath.getClass();
		firstExpression.getClass();
		otherExpression.getClass();
		
		this.xpath = xpath;
		this.firstExpression = firstExpression;
		this.otherExpression = otherExpression;
	}

	
	/**
	 * コンストラクタ。
	 * @param xpath XPath。
	 * @param expression 検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public XPath(final XPathWrapper xpath, final DomExpression expression){
		this(xpath, expression, expression);
	}
	
	/**
	 * コンストラクタ。
	 * @param xpath XPath。
	 * @param firstExpression 最初の検索結果に対する委譲先。
	 * @param otherExpression 2番目以降の検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws RuntimeException XPath のコンパイルに失敗した場合。
	 */
	public XPath(final String xpath, final DomExpression firstExpression, final DomExpression otherExpression){
		this(new XPathWrapper(xpath), firstExpression, otherExpression);
	}
	
	/**
	 * コンストラクタ。
	 * @param xpath XPath。
	 * @param expression 検索結果に対する委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 * @throws RuntimeException XPathのコンパイルに失敗した場合。
	 */
	public XPath(final String xpath, final DomExpression expression){
		this(xpath, expression, expression);
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws RuntimeException {@link XPathExpressionException}を catch した場合。
	 */
	public void interpret(final DomContext context) {
		final NodeList nodeList = xpath.evaluateList(context.getNode());
		final Node[] beforeNodeList = new Node[nodeList.getLength()];
		for(int i=0; i<nodeList.getLength(); i++){
			beforeNodeList[i] = nodeList.item(i);
		}
		
		for (int i=0; i < beforeNodeList.length ; i++) {
			final DomContext newContext = context.copy();
			newContext.setNode(beforeNodeList[i]);
			if(i == 0){
				firstExpression.interpret(newContext);
			}else{
				otherExpression.interpret(newContext);
			}
		}
	}
}
