/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.io.Writer;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * 定数・静的メソッドの集まり。
 * @author nakamura
 *
 */
public class DomConstants {
	
	/**
	 * Nullオブジェクト役の{@link DomExpression}。
	 */
	public static final DomExpression NULL = new DomExpression(){
		public void interpret(final DomContext context) {
		}
	};
	
	/**
	 * 子ノードを全て含むDOMノードを削除する{@link DomExpression}。
	 * {@link DomExpression#interpret(DomContext)}は引数または{@link DomContext#getNode()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final DomExpression REMOVE_THIS = new DomExpression(){
		public void interpret(final DomContext context) {
			removeThis(context.getNode());
		}
	};
	
	/**
	 * DOMノードの全ての子ノードを削除する{@link DomExpression}。
	 * {@link DomExpression#interpret(DomContext)}は引数または{@link DomContext#getNode()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final DomExpression REMOVE_CHILDREN = new DomExpression(){
		public void interpret(final DomContext context) {
			removeChildren(context.getNode());
		}
	};
	
	/**
	 * 現在の位置のDOMノードのみを削除する{@link DomExpression}。
	 * 子ノードは指定されたノードの位置に挿入される。
	 * {@link DomExpression#interpret(DomContext)}は引数または{@link DomContext#getNode()}が null の場合に{@link NullPointerException}を投げる。
	 */
	public static final DomExpression RETAIN_CHILDREN = new DomExpression(){
		public void interpret(final DomContext context) {
			retainChildren(context.getNode());
		}
	};

	/**
	 * セッションスコープの全てのキーを削除する{@link DomExpression}。
	 */
	public static final DomExpression CLEAR_SESSION = new DomExpression(){
		public void interpret(DomContext context) {
			context.getSession().clear();
		}
	};
	
	private DomConstants(){
	}
	
	/**
	 * DOMノード(子ノードを全て含む)を削除する。
	 * @param node 削除対象ノードの基点ノード。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public static void removeThis(final Node node){
		node.getParentNode().removeChild(node);
	}
	
	/**
	 * 指定されたDOMノードの全ての子ノードを削除する。
	 * @param node 削除対象ノードの親ノード。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public static void removeChildren(final Node node){
		Node child = null;
		while((child = node.getFirstChild()) != null){
			node.removeChild(child);
		}
	}
	
	/**
	 * 指定されたDOMノードのみを削除する。子ノードは指定されたDOMノードの位置に挿入される。
	 * @param node 削除対象ノード。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public static void retainChildren(final Node node){
		final Node parent = node.getParentNode();
		Node child = null;
		while((child = node.getFirstChild()) != null){
			parent.insertBefore(child, node);
		}
		parent.removeChild(node);
	}
	
	/**
	 * DOMノードの子ノードにテキストノードを追加する。
	 * @param node テキストノードを子ノードに追加するDOMノード。
	 * @param text 追加する文字列。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static void appendText(final Node node, final String text){
		node.appendChild(getDocumentNode(node).createTextNode(text));
	}
	
	/**
	 * DOMドキュメントを返す。
	 * DOMドキュメントに対して{@link Node#getOwnerDocument()}の呼び出しが失敗するためにこのメソッドは存在する。
	 * @param node DOMノード。
	 * @return 引数のDOMドキュメント。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public static Document getDocumentNode(final Node node){
		return (node.getNodeType() == Node.DOCUMENT_NODE)?(Document)node:node.getOwnerDocument();
	}

	/**
	 * DOMノードを{@link Writer}に出力する。
	 * @param node 書き込み対象のDOMノード。
	 * @param writer 書き込み先。
	 * @throws RuntimeException ({@link TransformerException})変換に失敗した場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public static void write(final Node node, final Writer writer) {
		// TransformerFactory はスレッドセーフでないので逐次生成する。
		final TransformerFactory factory = TransformerFactory.newInstance();
		try {
			final Transformer transformer = factory.newTransformer();
			final Source source = new DOMSource(node);
			final Result result = new StreamResult(writer);
			transformer.transform(source, result);
		} catch (final TransformerException e) {
			throw new RuntimeException(e);
		}
	}
}
