/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.w3c.dom.Node;

/**
 * DOM ノードをキャッシュする{@link DomExpression}。
 * 委譲先で生成される{@link Node}が{@link #getCacheKey(String)}に対して一意であることを想定しているため同期化制御を行っていない。
 * @author nakamura
 *
 */
public class CacheNode implements DomExpression {
	private final DomExpression nodeMaker;
	private final Map<Object,Node> cacheMap;

	/**
	 * コンストラクタ。
	 * @param maxEntries キャッシュに保持する最大DOMノード数。
	 * @param nodeMaker DOM ノードを生成し加工する委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CacheNode(final int maxEntries, final DomExpression nodeMaker){
		nodeMaker.getClass();
		this.nodeMaker = nodeMaker;
		
		if(maxEntries > 0){
			cacheMap = new LinkedHashMap<Object,Node>(){
				/**
				 * 
				 */
				private static final long serialVersionUID = 1L;

				@Override protected boolean removeEldestEntry(Entry eldest) {
					return size() > maxEntries;
				}
			};
		}else{
			cacheMap = null;
		}
		
	}

	/**
	 * @throws NullPointerException 引数または生成された{@link DomContext#getNode()}が null の場合。
	 */
	public void interpret(final DomContext context) {
		if(cacheMap == null){
			nodeMaker.interpret(context);
		}else{
			final Object key = getCacheKey(context.getId());
			Node node = cacheMap.get(key);
			if(node == null){
				nodeMaker.interpret(context);
				node = context.getNode();
				cacheMap.put(key, node);
			}
			context.setNode(node.cloneNode(true));
		}
	}
	
	/**
	 * 生成した DOM ノードを一意に識別するキーを取得するために呼び出される(called)。
	 * デフォルトは常に引数の id を返す。
	 * {@link Locale}や端末ごとに DOM ノードをキャッシュする際にオーバライドする。
	 * オーバライドする際は DOM ノード生成クラスと整合性を保つ必要がある。
	 * @param id DOM ノードを生成するのに使用した id。
	 * @return 生成した DOM ノードを一意に識別するキー。
	 */
	public Object getCacheKey(final String id){
		return id;
	}
}
