/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;

import ognl.MethodFailedException;
import woolpack.utils.UtilsConstants;

/**
 * アクション定義の一覧と遷移先定義の一覧の集合。
 * @author nakamura
 *
 */
public class ActionInvoker {
	private final Map<String, ActionDef> actionDefMap;
	private final Iterable<ForwardDef> forwardDefIterable;
	
	/**
	 * コンストラクタ。
	 * @param actionDefMap アクション id とアクション定義の{@link Map}。ステートレスであるべき。
	 * @param forwardDefIterable 遷移先定義の一覧。ステートレスであるべき。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ActionInvoker(final Map<String,ActionDef> actionDefMap, final Iterable<ForwardDef> forwardDefIterable){
		actionDefMap.getClass();
		forwardDefIterable.getClass();

		this.actionDefMap = actionDefMap;
		this.forwardDefIterable = forwardDefIterable;
	}

	public ActionInvoker(final Map<String,ActionDef> actionDefMap, final ForwardDef... forwardDefIterable){
		this(actionDefMap, Arrays.asList(forwardDefIterable));
	}
	
	
	/**
	 * アクションを実行する。
	 * このメソッドは、
	 * アクションidに対するアクション定義を検索し、
	 * root から{@link ActionDef#getComponentOGE()}でコンポーネントを取得し、
	 * inputMap の値をコンポーネントに設定し、
	 * root のメソッド{@link ActionDef#getMethodOGE()}を実行し、
	 * {@link ActionDef#forwardDefIterable}から遷移先定義を検索する。
	 * アクション定義に遷移先定義が見つからない場合、{@link #getForwardDefIterable()}から遷移先定義を検索する。
	 * @param root コンポーネント・メソッド検索の基点。
	 * @param id アクション id。
	 * @param inputMap 入力値のMap。{@link ServletRequest#getParameterMap()}またはそれを{@link ValidatorConstants#convert(Map)}で変換した結果が渡されることを想定している。OGNLに渡すだけなので汎用化型を指定していない。
	 * @return 実行結果。
	 * @throws RuntimeException ({@link ForwardDefNotFoundException})該当する遷移先定義が見つからない場合。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public ActionResult invoke(final Object root, final String id, final Map inputMap){
		final ActionDef actionDef = actionDefMap.get(id);
		actionDef.getClass();
		try{
			actionDef.getComponentOGE().setValues(root, inputMap);
		}catch(final RuntimeException exception){
			// TODO 無視以外の方法がないか考察する。
		}
		try {
			try{
				final Object result = actionDef.getMethodOGE().invoke(root);
				return findId(actionDef.getForwardDefIterable(), result);
			}catch(final MethodFailedException e){
				return findId(actionDef.getForwardDefIterable(), e.getReason());
			}
		} catch (final ForwardDefNotFoundException e) {
			return findId(forwardDefIterable, e.getReturnedObject());
		}
	}

	/**
	 * 遷移先定義を検索して実行結果を返す。
	 * @param iterable 遷移先定義の一覧。
	 * @param aReturnedObject メソッドが返却したオブジェクト。
	 * @return 実行結果。
	 * @throws ForwardDefNotFoundException 該当する遷移先定義が見つからない場合。
	 */
	public static ActionResult findId(
			final Iterable<ForwardDef> iterable, 
			final Object aReturnedObject) throws ForwardDefNotFoundException{
		for(final ForwardDef forwardDef:iterable){
			if(forwardDef.getMatchable().matches(aReturnedObject)){
				return new ActionResult(forwardDef, aReturnedObject);
			}
		}
		throw new ForwardDefNotFoundException(aReturnedObject);
	}

	/**
	 * アクション定義のMapを返す。
	 * テスト用。
	 * @return アクション定義のMap。
	 */
	public Map<String, ActionDef> getActionDefMap() {
		return Collections.unmodifiableMap(actionDefMap);
	}

	/**
	 * 遷移先定義の一覧を返す。
	 * テスト用。
	 * @return 遷移先定義の一覧。
	 */
	public Iterable<ForwardDef> getForwardDefIterable() {
		return UtilsConstants.unmodifiableIterable(forwardDefIterable);
	}
}
