﻿using System;
using System.Collections.Generic;
using System.Text;

namespace com.ast8.fw.forms
{
    /// <summary>
    /// テキストが変更されようとしている時に発生するイベントパラメータ
    /// </summary>
    public class BeforeTextChangedEventArgs : System.ComponentModel.CancelEventArgs
    {
        /// <summary>InputStringプロパティの実体</summary>
        private string m_inputString;
        /// <summary>Operationプロパティの実体</summary>
        private readonly TextBoxOperation m_operation;

        /// <summary>
        /// 入力された文字列を取得または設定する
        /// </summary>
        /// <remarks>
        /// 本プロパティだけでテキストボックスの挙動がわかるわけではない。
        /// 入力モードや選択範囲によって挙動も違えば、
        /// この文字列が空である場合にも削除などが実行されている可能性もある。
        /// </remarks>
        public string InputString
        {
            get { return this.m_inputString; }
            set { this.m_inputString = value; }
        }

        public TextBoxOperation Operation
        {
            get { return this.m_operation; }
        }
        
        /// <summary>
        /// デフォルトコンストラクタ。空文字列と<seealso cref="TextBoxOperation.InputString"/>を渡してインスタンスを生成する
        /// </summary>
        public BeforeTextChangedEventArgs()
            : this("", TextBoxOperation.InputString)
        {
        }

        /// <summary>
        /// 入力された文字列を<seealso cref="TextBoxOperation.InputString"/>を渡してインスタンスを生成する
        /// </summary>
        /// <param name="inputString">入力された文字列</param>
        public BeforeTextChangedEventArgs(string inputString)
            : this(inputString, TextBoxOperation.InputString)
        {
        }

        /// <summary>
        /// 対象の操作を渡してインスタンスを生成する
        /// </summary>
        /// <param name="operation">対象の操作</param>
        public BeforeTextChangedEventArgs(TextBoxOperation operation)
            : this("", operation)
        {
        }

        /// <summary>
        /// パラメータを指定してインスタンスを生成する
        /// </summary>
        /// <param name="inputString">入力された文字列</param>
        /// <param name="operation">対象の操作</param>
        public BeforeTextChangedEventArgs(string inputString, TextBoxOperation operation)
        {
            this.m_inputString = inputString;
            this.m_operation = operation;
        }
    }

    /// <summary>
    /// 古い値へのキャンセル処理が可能なイベントのイベントパラメータ（汎用）
    /// </summary>
    /// <typeparam name="T">OldValueの型</typeparam>
    public class OldValueCancelEventArgs<T> : System.ComponentModel.CancelEventArgs
    {
        /// <summary>OldValueプロパティの実体</summary>
        private readonly T m_oldValue;

        /// <summary>
        /// 変更前の値を取得する。<seealso cref="Cancel"/>プロパティをtrueにするとこの値に戻る事が保証されている。
        /// </summary>
        public T OldValue
        {
            get
            {
                return this.m_oldValue;
            }
        }

        /// <summary>
        /// デフォルトコンストラクタ。
        /// <seealso cref="OldValue"/>プロパティの値を初期化せずインスタンスを生成するため、本コンストラクタの使用は推奨されない
        /// </summary>
        public OldValueCancelEventArgs()
        {
        }

        /// <summary>
        /// <seealso cref="OldValue"/>プロパティの値を指定してインスタンスを生成する
        /// </summary>
        /// <param name="oldValue"><seealso cref="OldValue"/>プロパティの値</param>
        public OldValueCancelEventArgs(T oldValue)
        {
            this.m_oldValue = oldValue;
        }
    }

    /// <summary>
    /// テキストが変更されようとしている時に発生するイベントハンドラ
    /// </summary>
    /// <param name="sender">イベント送出オブジェクト</param>
    /// <param name="e">イベントパラメータ</param>
    public delegate void BeforeTextChangedEventHandler(object sender, BeforeTextChangedEventArgs e);

    /// <summary>
    /// CursorModeプロパティが変更された時に発生するイベントハンドラ
    /// </summary>
    /// <param name="sender">イベント送出オブジェクト</param>
    /// <param name="e">イベントパラメータ</param>
    public delegate void CursorModeChangedEventHandler(object sender, OldValueCancelEventArgs<CursorMode> e);

    /// <summary>
    /// InputModeプロパティが変更された時に発生するイベントハンドラ
    /// </summary>
    /// <param name="sender">イベント送出オブジェクト</param>
    /// <param name="e">イベントパラメータ</param>
    public delegate void InputModeChangedEventHandler(object sender, OldValueCancelEventArgs<TextInputMode> e);

    /// <summary>
    /// SelectModeプロパティが変更された時に発生するイベントハンドラ
    /// </summary>
    /// <param name="sender">イベント送出オブジェクト</param>
    /// <param name="e">イベントパラメータ</param>
    public delegate void SelectModeChangedEventHandler(object sender, OldValueCancelEventArgs<SelectMode> e);
}
