/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    aq.c - Audio queue.
	      Written by Masanao Izumo <mo@goice.co.jp>
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif


#include "timidity.h"
#include "common.h"
#include "output.h"
#include "aq.h"
#include "timer.h"
#include "controls.h"
#include "miditrace.h"
#include "instrum.h"
#include "playmidi.h"

#include "aq_prv.h"

#define TEST_SPARE_RATE 0.9
#define MAX_BUCKET_TIME 0.2
#define MAX_FILLED_TIME 2.0




/* effect.c */
//extern void init_effect(tmdy_struct_ex_t *tmdy_struct);
//extern void do_effect(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count);

int aq_calc_fragsize(tmdy_struct_ex_t *tmdy_struct)
{
    int ch, bps, bs;
    double dq, bt;

    if((TMDY_OUTPUT->play_mode)->encoding & PE_MONO)
	ch = 1;
    else
	ch = 2;
    if((TMDY_OUTPUT->play_mode)->encoding & PE_16BIT)
	bps = ch * 2;
    else
	bps = ch;

    bs = audio_buffer_size * bps;
    dq = (TMDY_OUTPUT->play_mode)->rate * MAX_FILLED_TIME * bps;
    while(bs * 2 > dq)
	bs /= 2;

    bt = (double)bs / bps / (TMDY_OUTPUT->play_mode)->rate;
    while(bt > MAX_BUCKET_TIME)
    {
	bs /= 2;
	bt = (double)bs / bps / (TMDY_OUTPUT->play_mode)->rate;
    }

    return bs;
}

void aq_setup(tmdy_struct_ex_t *tmdy_struct)
{
    int ch;

    /* Initialize Bps, bucket_size, device_qsize, and bucket_time */

    if((TMDY_OUTPUT->play_mode)->encoding & PE_MONO)
	ch = 1;
    else
	ch = 2;
    if((TMDY_OUTPUT->play_mode)->encoding & PE_16BIT)
	(TMDY_AQ->Bps) = 2 * ch;
    else
	(TMDY_AQ->Bps) = ch;

    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_GETFRAGSIZ, &(TMDY_AQ->bucket_size)) == -1)
	(TMDY_AQ->bucket_size) = audio_buffer_size * (TMDY_AQ->Bps);
    (TMDY_AQ->bucket_time) = (double)(TMDY_AQ->bucket_size) / (TMDY_AQ->Bps) / (TMDY_OUTPUT->play_mode)->rate;

    if(IS_STREAM_TRACE)
    {
	if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_GETQSIZ, &(TMDY_AQ->device_qsize)) == -1)
	    (TMDY_AQ->device_qsize) = estimate_queue_size(tmdy_struct);
	if((TMDY_AQ->bucket_size) * 2 > (TMDY_AQ->device_qsize)) {
	  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
		    "Warning: Audio buffer is too small.");
	  (TMDY_AQ->device_qsize) = 0;
	} else {
	  (TMDY_AQ->device_qsize) -= (TMDY_AQ->device_qsize) % (TMDY_AQ->Bps); /* Round Bps */
	  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		    "Audio device queue size: %d bytes", (TMDY_AQ->device_qsize));
	  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
		    "Write bucket size: %d bytes (%d msec)",
		    (TMDY_AQ->bucket_size), (int)((TMDY_AQ->bucket_time) * 1000 + 0.5));
	}
    }
    else
    {
	(TMDY_AQ->device_qsize) = 0;
	if((TMDY_AQ->base_buckets))
	{
	    free((TMDY_AQ->base_buckets)[0].data);
	    free((TMDY_AQ->base_buckets));
	    (TMDY_AQ->base_buckets) = NULL;
	}
	(TMDY_AQ->nbuckets) = 0;
    }

    init_effect(tmdy_struct);
    (TMDY_AQ->aq_add_count) = 0;

}

void aq_set_soft_queue(tmdy_struct_ex_t *tmdy_struct, double soft_buff_time, double fill_start_time)
{
//    static double last_soft_buff_time, last_fill_start_time;
    int nb;

	
    /* for re-initialize */
    if(soft_buff_time < 0)
	soft_buff_time = TMDY_AQ->last_soft_buff_time;
    if(fill_start_time < 0)
	fill_start_time = TMDY_AQ->last_fill_start_time;

    nb = (int)(soft_buff_time / (TMDY_AQ->bucket_time));
    if(nb == 0)
	(TMDY_AQ->aq_start_count) = 0;
    else
	(TMDY_AQ->aq_start_count) = (int32)(fill_start_time * (TMDY_OUTPUT->play_mode)->rate);
    (TMDY_AQ->aq_fill_buffer_flag) = ((TMDY_AQ->aq_start_count) > 0);

    if((TMDY_AQ->nbuckets) != nb)
    {
	(TMDY_AQ->nbuckets) = nb;
	alloc_soft_queue(tmdy_struct);
    }

    TMDY_AQ->last_soft_buff_time = soft_buff_time;
    TMDY_AQ->last_fill_start_time = fill_start_time;

}

/* Estimates the size of audio device queue.
 * About sun audio, there are long-waiting if buffer of device audio is full.
 * So it is impossible to completely estimate the size.
 */
static int32 estimate_queue_size(tmdy_struct_ex_t *tmdy_struct)
{
    char *nullsound;
    double tb, init_time, chunktime;
    int32 qbytes, max_qbytes;
    int ntries;

    nullsound = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, (TMDY_AQ->bucket_size));
    memset(nullsound, 0, (TMDY_AQ->bucket_size));
    if((TMDY_OUTPUT->play_mode)->encoding & (PE_ULAW|PE_ALAW))
	TMDY_OUTPUT->general_output_convert(tmdy_struct, (int32 *)nullsound, (TMDY_AQ->bucket_size)/(TMDY_AQ->Bps));
    tb = (TMDY_OUTPUT->play_mode)->rate * (TMDY_AQ->Bps) * TEST_SPARE_RATE;
    ntries = 1;
    max_qbytes = (TMDY_OUTPUT->play_mode)->rate * MAX_FILLED_TIME * (TMDY_AQ->Bps);

  retry:
    chunktime = (double)(TMDY_AQ->bucket_size) / (TMDY_AQ->Bps) / (TMDY_OUTPUT->play_mode)->rate;
    qbytes = 0;

    init_time = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);	/* Start */
    for(;;)
    {
	double start, diff;

	start = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	if(start - init_time > 1.0) /* ?? */
	{
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_DEBUG,
		      "Warning: Audio test is terminated");
	    break;
	}
	(TMDY_OUTPUT->play_mode)->output_data(tmdy_struct, nullsound, (TMDY_AQ->bucket_size));
	diff = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct) - start;

	if(diff > chunktime/2 || qbytes > 1024*512 || chunktime < diff)
	    break;
	qbytes += (int32)((chunktime - diff) * tb);

	if(qbytes > max_qbytes)
	{
	    qbytes = max_qbytes;
	    break;
	}
    }
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_DISCARD, NULL);

    if((TMDY_AQ->bucket_size) * 2 > qbytes)
    {
	if(ntries == 4)
	{
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NOISY,
		      "Can't estimate audio queue length");
	    (TMDY_AQ->bucket_size) = audio_buffer_size * (TMDY_AQ->Bps);
	    free(nullsound);
	    return 2 * audio_buffer_size * (TMDY_AQ->Bps);
	}

	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_DEBUG,
		  "Retry to estimate audio queue length (%d times)",
		  ntries);
	(TMDY_AQ->bucket_size) /= 2;
	ntries++;
	goto retry;
    }

    free(nullsound);

    return qbytes;
}

/* Send audio data to play_mode->output_data(tmdy_struct, ) */
static int aq_output_data(tmdy_struct_ex_t *tmdy_struct, char *buff, int nbytes)
{
    int i;

    (TMDY_AQ->play_counter) += nbytes / (TMDY_AQ->Bps);

    while(nbytes > 0)
    {
	i = nbytes;
	if(i > (TMDY_AQ->bucket_size))
	    i = (TMDY_AQ->bucket_size);
    if((TMDY_OUTPUT->play_mode)->output_data(tmdy_struct, buff, i) == -1){
    	return -1;
    }
	nbytes -= i;
	buff += i;
    }
    return 0;
}

int aq_add(tmdy_struct_ex_t *tmdy_struct, int32 *samples, int32 count)
{
    int32 nbytes, i;
    char *buff;

	if(!((TMDY_OUTPUT->play_mode)->flag & PF_PCM_STREAM)){
		return 0;
	}

    if(!count)
    {
    if(!(TMDY_AQ->aq_fill_buffer_flag)){
	    return aq_fill_nonblocking(tmdy_struct);
    }
	return 0;
    }

    (TMDY_AQ->aq_add_count) += count;
    do_effect(tmdy_struct, samples, count);
    nbytes = TMDY_OUTPUT->general_output_convert(tmdy_struct, samples, count);
    buff = (char *)samples;

	if((TMDY_AQ->device_qsize) == 0){
      return (TMDY_OUTPUT->play_mode)->output_data(tmdy_struct, buff, nbytes);
	}

    (TMDY_AQ->aq_fill_buffer_flag) = ((TMDY_AQ->aq_add_count) <= (TMDY_AQ->aq_start_count));

    if(!(TMDY_AQ->aq_fill_buffer_flag))
	if(aq_fill_nonblocking(tmdy_struct) == -1){
		return -1;
	}

    if(!(TMDY_CONTROLS->ctl)->trace_playing)
    {
	while((i = add_play_bucket(tmdy_struct, buff, nbytes)) < nbytes)
	{
	    buff += i;
	    nbytes -= i;
	    if((TMDY_AQ->head) && (TMDY_AQ->head)->len == (TMDY_AQ->bucket_size))
	    {
	    if(aq_fill_one(tmdy_struct) == -1){
		    return -1;
	    }
	    }
	    (TMDY_AQ->aq_fill_buffer_flag) = 0;
	}
	return 0;
    }

    TMDY_MIDITRACE->trace_loop(tmdy_struct);
    while((i = add_play_bucket(tmdy_struct, buff, nbytes)) < nbytes)
    {
	/* Software buffer is full.
	 * Write one bucket to audio device.
	 */
	buff += i;
	nbytes -= i;
	aq_wait_ticks(tmdy_struct);
	TMDY_MIDITRACE->trace_loop(tmdy_struct);
    if(aq_fill_nonblocking(tmdy_struct) == -1){
	    return -1;
    }
	(TMDY_AQ->aq_fill_buffer_flag) = 0;
    }
    return 0;
}

/* alloc_soft_queue() (re-)initializes audio buckets. */
static void alloc_soft_queue(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
    char *base;

    if((TMDY_AQ->base_buckets))
    {
	free((TMDY_AQ->base_buckets)[0].data);
	free((TMDY_AQ->base_buckets));
	(TMDY_AQ->base_buckets) = NULL;
    }

    (TMDY_AQ->base_buckets) = (AudioBucket *)TMDY_COMMON->safe_malloc(tmdy_struct, (TMDY_AQ->nbuckets) * sizeof(AudioBucket));
    base = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, (TMDY_AQ->nbuckets) * (TMDY_AQ->bucket_size));
    for(i = 0; i < (TMDY_AQ->nbuckets); i++)
	(TMDY_AQ->base_buckets)[i].data = base + i * (TMDY_AQ->bucket_size);
    flush_buckets(tmdy_struct);

}

/* aq_fill_one() transfers one audio bucket to device. */
static int aq_fill_one(tmdy_struct_ex_t *tmdy_struct)
{
    AudioBucket *tmp;

	if((TMDY_AQ->head) == NULL){
		return 0;
	}
	if(aq_output_data(tmdy_struct, (TMDY_AQ->head)->data, (TMDY_AQ->bucket_size)) == -1){
		return -1;
	}
    tmp = (TMDY_AQ->head);
    (TMDY_AQ->head) = (TMDY_AQ->head)->next;
    reuse_audio_bucket(tmdy_struct,tmp);
    return 0;
}

/* aq_fill_nonblocking() transfers some audio buckets to device.
 * This function is non-blocking.  But it is possible to block because
 * of miss-estimated aq_fillable() calculation.
 */
int aq_fill_nonblocking(tmdy_struct_ex_t *tmdy_struct)
{
    int32 i, nfills;
    AudioBucket *tmp;

	if((TMDY_AQ->head) == NULL || (TMDY_AQ->head)->len != (TMDY_AQ->bucket_size) || !IS_STREAM_TRACE){
		return 0;
	}
    nfills = (aq_fillable(tmdy_struct) * (TMDY_AQ->Bps)) / (TMDY_AQ->bucket_size);
    for(i = 0; i < nfills; i++)
    {
	if((TMDY_AQ->head) == NULL || (TMDY_AQ->head)->len != (TMDY_AQ->bucket_size))
	    break;
    if(aq_output_data(tmdy_struct, (TMDY_AQ->head)->data, (TMDY_AQ->bucket_size)) == -1){
	    return RC_ERROR;
    }
	tmp = (TMDY_AQ->head);
	(TMDY_AQ->head) = (TMDY_AQ->head)->next;
	reuse_audio_bucket(tmdy_struct, tmp);
    }
    return 0;
}

int32 aq_samples(tmdy_struct_ex_t *tmdy_struct)
{
    double realtime, es;
    int s;

    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_GETSAMPLES, &s) != -1)
    {
	/* Reset counter & timer */
	if((TMDY_AQ->play_counter))
	{
	    (TMDY_AQ->play_start_time) = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	    (TMDY_AQ->play_offset_counter) = s;
	    (TMDY_AQ->play_counter) = 0;
	}
	return s;
    }

	if(!IS_STREAM_TRACE){
		return -1;
	}

    realtime = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
    if((TMDY_AQ->play_counter) == 0)
    {
	(TMDY_AQ->play_start_time) = realtime;
	return (TMDY_AQ->play_offset_counter);
    }
    es = (TMDY_OUTPUT->play_mode)->rate * (realtime - (TMDY_AQ->play_start_time));
    if(es >= (TMDY_AQ->play_counter))
    {
	/* Ouch!
	 * Audio device queue may be empty!
	 * Reset counters.
	 */

	(TMDY_AQ->play_offset_counter) += (TMDY_AQ->play_counter);
	(TMDY_AQ->play_counter) = 0;
	(TMDY_AQ->play_start_time) = realtime;
	return (TMDY_AQ->play_offset_counter);
    }

    return (int32)es + (TMDY_AQ->play_offset_counter);
}

int32 aq_filled(tmdy_struct_ex_t *tmdy_struct)
{
    double realtime, es;
    int filled;

	if(!IS_STREAM_TRACE){
		return 0;
	}
	
    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_GETFILLED, &filled) != -1)
      return filled;

    realtime = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
    if((TMDY_AQ->play_counter) == 0)
    {
	(TMDY_AQ->play_start_time) = realtime;
	return 0;
    }
    es = (TMDY_OUTPUT->play_mode)->rate * (realtime - (TMDY_AQ->play_start_time));
    if(es >= (TMDY_AQ->play_counter))
    {
	/* out of play counter */
	(TMDY_AQ->play_offset_counter) += (TMDY_AQ->play_counter);
	(TMDY_AQ->play_counter) = 0;
	(TMDY_AQ->play_start_time) = realtime;
	return 0;
    }
    return (TMDY_AQ->play_counter) - (int32)es;
}

int32 aq_soft_filled(tmdy_struct_ex_t *tmdy_struct)
{
    int32 bytes;
    AudioBucket *cur;

    bytes = 0;
    for(cur = (TMDY_AQ->head); cur != NULL; cur = cur->next)
	bytes += cur->len;
    return bytes / (TMDY_AQ->Bps);
}

int32 aq_fillable(tmdy_struct_ex_t *tmdy_struct)
{
    int fillable;
    if(!IS_STREAM_TRACE)
	return 0;
    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_GETFILLABLE, &fillable) != -1)
	return fillable;
    return (TMDY_AQ->device_qsize) / (TMDY_AQ->Bps) - aq_filled(tmdy_struct);
}

double aq_filled_ratio(tmdy_struct_ex_t *tmdy_struct)
{
    double ratio;

    if(!IS_STREAM_TRACE)
	return 1.0;
    ratio = (double)aq_filled(tmdy_struct) * (TMDY_AQ->Bps) / (TMDY_AQ->device_qsize);
    if(ratio > 1.0)
	return 1.0; /* for safety */
    return ratio;
}

int aq_get_dev_queuesize(tmdy_struct_ex_t *tmdy_struct)
{
    if(!IS_STREAM_TRACE)
	return 0;
    return (TMDY_AQ->device_qsize) / (TMDY_AQ->Bps);
}

int aq_soft_flush(tmdy_struct_ex_t *tmdy_struct)
{
    int rc;

    while((TMDY_AQ->head))
    {
	if((TMDY_AQ->head)->len < (TMDY_AQ->bucket_size))
	{
	    /* Add silence code */
	    memset ((TMDY_AQ->head)->data + (TMDY_AQ->head)->len, 0, (TMDY_AQ->bucket_size) - (TMDY_AQ->head)->len);
	    (TMDY_AQ->head)->len = (TMDY_AQ->bucket_size);
	}
    if(aq_fill_one(tmdy_struct) == -1){
	    return RC_ERROR;
    }
	TMDY_MIDITRACE->trace_loop(tmdy_struct);
	rc = TMDY_PLAYMIDI->check_apply_control(tmdy_struct);
	if(RC_IS_SKIP_FILE(rc))
	{
	    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_DISCARD, NULL);
	    flush_buckets(tmdy_struct);
	    return rc;
	}
    }
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_OUTPUT_FINISH, NULL);
    return RC_NONE;
}

int aq_flush(tmdy_struct_ex_t *tmdy_struct, int discard)
{
    int rc;
    int more_trace;

    /* to avoid infinite loop */
    double t, timeout_expect;

    (TMDY_AQ->aq_add_count) = 0;
    init_effect(tmdy_struct);

    if(discard)
    {
	TMDY_MIDITRACE->trace_flush(tmdy_struct);
	if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_DISCARD, NULL) != -1)
	{
	    flush_buckets(tmdy_struct);
	    return RC_NONE;
	}
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "ERROR: Can't discard audio buffer");
    }

    if(!IS_STREAM_TRACE)
    {
	(TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_FLUSH, NULL);
	(TMDY_AQ->play_counter) = (TMDY_AQ->play_offset_counter) = 0;
	return RC_NONE;
    }

    rc = aq_soft_flush(tmdy_struct);
	if(RC_IS_SKIP_FILE(rc)){
		return rc;
	}
    more_trace = 1;
    t = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
    timeout_expect = t + (double)aq_filled(tmdy_struct) / (TMDY_OUTPUT->play_mode)->rate;

    while(more_trace || aq_filled(tmdy_struct) > 0)
    {
	rc = TMDY_PLAYMIDI->check_apply_control(tmdy_struct);
	if(RC_IS_SKIP_FILE(rc))
	{
	    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_DISCARD, NULL);
	    flush_buckets(tmdy_struct);
	    return rc;
	}
	more_trace = TMDY_MIDITRACE->trace_loop(tmdy_struct);

	t = TMDY_UTILS->timer->get_current_calender_time(tmdy_struct);
	if(t >= timeout_expect - 0.1)
	  break;

	if(!more_trace)
	  usleep((unsigned long)((timeout_expect - t) * 1000000));
	else
	  aq_wait_ticks(tmdy_struct);
    }

    TMDY_MIDITRACE->trace_flush(tmdy_struct);
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_FLUSH, NULL);
    flush_buckets(tmdy_struct);
    return RC_NONE;
}

/* Wait a moment */
static void aq_wait_ticks(tmdy_struct_ex_t *tmdy_struct)
{
    int32 trace_wait, wait_samples;

    if((TMDY_AQ->device_qsize) == 0 ||
       (trace_wait = TMDY_MIDITRACE->trace_wait_samples(tmdy_struct)) == 0)
	return; /* No wait */
    wait_samples = ((TMDY_AQ->device_qsize) / (TMDY_AQ->Bps)) / 5; /* 20% */
    if(trace_wait != -1 && /* There are more trace events */
       trace_wait < wait_samples)
	wait_samples = trace_wait;
    usleep((unsigned int)((double)wait_samples / (TMDY_OUTPUT->play_mode)->rate * 1000000.0));
}

/* add_play_bucket() attempts to add buf to audio bucket.
 * It returns actually added bytes.
 */
static int add_play_bucket(tmdy_struct_ex_t *tmdy_struct, const char *buf, int n)
{
    int total;

    if(n == 0)
	return 0;

    if(!(TMDY_AQ->nbuckets)) {
      (TMDY_OUTPUT->play_mode)->output_data(tmdy_struct, (char *)buf, n);
      return n;
    }

    if((TMDY_AQ->head) == NULL)
	(TMDY_AQ->head) = (TMDY_AQ->tail) = next_allocated_bucket(tmdy_struct);

    total = 0;
    while(n > 0)
    {
	int i;

	if((TMDY_AQ->tail)->len == (TMDY_AQ->bucket_size))
	{
	    AudioBucket *b;
	    if((b = next_allocated_bucket(tmdy_struct)) == NULL)
		break;
	    if((TMDY_AQ->head) == NULL)
		(TMDY_AQ->head) = (TMDY_AQ->tail) = b;
	    else
		(TMDY_AQ->tail) = (TMDY_AQ->tail)->next = b;
	}

	i = (TMDY_AQ->bucket_size) - (TMDY_AQ->tail)->len;
	if(i > n)
	    i = n;
	memcpy((TMDY_AQ->tail)->data + (TMDY_AQ->tail)->len, buf + total, i);
	total += i;
	n     -= i;
	(TMDY_AQ->tail)->len += i;
    }

    return total;
}

/* Flush and clear audio bucket */
static void flush_buckets(tmdy_struct_ex_t *tmdy_struct)
{
    int i;

    (TMDY_AQ->allocated_bucket_list) = NULL;
    for(i = 0; i < (TMDY_AQ->nbuckets); i++)
	reuse_audio_bucket(tmdy_struct, &(TMDY_AQ->base_buckets)[i]);
    (TMDY_AQ->head) = (TMDY_AQ->tail) = NULL;
    (TMDY_AQ->aq_fill_buffer_flag) = ((TMDY_AQ->aq_start_count) > 0);
    (TMDY_AQ->play_counter) = (TMDY_AQ->play_offset_counter) = 0;

}

/* next_allocated_bucket() gets free bucket.  If all buckets is used, it
 * returns NULL.
 */
static AudioBucket *next_allocated_bucket(tmdy_struct_ex_t *tmdy_struct)
{
    AudioBucket *b;

    if((TMDY_AQ->allocated_bucket_list) == NULL)
	return NULL;
    b = (TMDY_AQ->allocated_bucket_list);
    (TMDY_AQ->allocated_bucket_list) = (TMDY_AQ->allocated_bucket_list)->next;
    b->len = 0;
    b->next = NULL;
    return b;
}

/* Reuse specified bucket */
static void reuse_audio_bucket(tmdy_struct_ex_t *tmdy_struct, AudioBucket *bucket)
{
    bucket->next = (TMDY_AQ->allocated_bucket_list);
    (TMDY_AQ->allocated_bucket_list) = bucket;
}




/* interfaces */

int ts_aq_calc_fragsize(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_calc_fragsize(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
void ts_aq_setup(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(TMDY_AQ->busy);
	aq_setup(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
}
void ts_aq_set_soft_queue(tmdy_struct_ex_t *tmdy_struct, double soft_buff_time, double fill_start_time){
	timidity_mutex_lock(TMDY_AQ->busy);
	aq_set_soft_queue(tmdy_struct, soft_buff_time, fill_start_time);
	timidity_mutex_unlock(TMDY_AQ->busy);
}
int ts_aq_add(tmdy_struct_ex_t *tmdy_struct, int32 *samples, int32 count){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_add(tmdy_struct, samples, count);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int32 ts_aq_samples(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_samples(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int32 ts_aq_filled(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_filled(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int32 ts_aq_soft_filled(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_soft_filled(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int32 ts_aq_fillable(tmdy_struct_ex_t *tmdy_struct){
	int32 ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_fillable(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int ts_aq_flush(tmdy_struct_ex_t *tmdy_struct, int discard){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_flush(tmdy_struct, discard);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int ts_aq_soft_flush(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_soft_flush(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int ts_aq_fill_nonblocking(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_fill_nonblocking(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
double ts_aq_filled_ratio(tmdy_struct_ex_t *tmdy_struct){
	double ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_filled_ratio(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
int ts_aq_get_dev_queuesize(tmdy_struct_ex_t *tmdy_struct){
	int ts_buf;
	timidity_mutex_lock(TMDY_AQ->busy);
	ts_buf=aq_get_dev_queuesize(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
	return ts_buf;
}
void ts_init_effect(tmdy_struct_ex_t *tmdy_struct){
	timidity_mutex_lock(TMDY_AQ->busy);
	init_effect(tmdy_struct);
	timidity_mutex_unlock(TMDY_AQ->busy);
}
void ts_do_effect(tmdy_struct_ex_t *tmdy_struct, int32* buf, int32 count){
	timidity_mutex_lock(TMDY_AQ->busy);
	do_effect(tmdy_struct, buf, count);
	timidity_mutex_unlock(TMDY_AQ->busy);
}


aq_ex_t* new_aq(tmdy_struct_ex_t *tmdy_struct){
	int i;
	aq_ex_t* aq_ex;

	aq_ex=(aq_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(aq_ex_t));
	
	timidity_mutex_init(aq_ex->busy);
	
	aq_ex->aq_calc_fragsize=ts_aq_calc_fragsize;
	aq_ex->aq_setup=ts_aq_setup;
	aq_ex->aq_set_soft_queue=ts_aq_set_soft_queue;
	aq_ex->aq_add=ts_aq_add;
	aq_ex->aq_samples=ts_aq_samples;
	aq_ex->aq_filled=ts_aq_filled;
	aq_ex->aq_soft_filled=ts_aq_soft_filled;
	aq_ex->aq_fillable=ts_aq_fillable;
	aq_ex->aq_flush=ts_aq_flush;
	aq_ex->aq_soft_flush=ts_aq_soft_flush;
	aq_ex->aq_fill_nonblocking=ts_aq_fill_nonblocking;
	aq_ex->aq_filled_ratio=ts_aq_filled_ratio;
	aq_ex->aq_get_dev_queuesize=ts_aq_get_dev_queuesize;

	/* effect.c */
	aq_ex->init_effect=ts_init_effect;
	aq_ex->do_effect=ts_do_effect;
	
	/** local variables **/
	aq_ex->nbuckets = 0;
	aq_ex->aq_fill_buffer_flag = 0;
	aq_ex->base_buckets = NULL;
	aq_ex->allocated_bucket_list = NULL;
	aq_ex->head = NULL;
	aq_ex->tail = NULL;
	
	/* effect.c */
	aq_ex->noise_sharp_type = 4;
	aq_ex->elrd_turn_counter = 0;
	return aq_ex;
}
void destroy_aq(aq_ex_t* aq){
	timidity_mutex_destroy(aq->busy);
	free(aq);
}

