/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#include <stdlib.h>
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "instrum.h"
#include "strtab.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "controls.h"
#include "utils_export.h"
#include "memb.h"
#include "zip.h"
#include "arc.h"
#include "mod.h"
#include "wrd.h"
#include "tables.h"
#include "reverb.h"
#include <math.h>

#include "readmidi_prv.h"




int32 readmidi_set_track(tmdy_struct_ex_t *tmdy_struct, int trackno, int rewindp)
{
    TMDY_READMIDI->current_read_track = trackno;
    memset(&TMDY_REVERB->chorus_status, 0, sizeof(TMDY_REVERB->chorus_status));
    if(TMDY_READMIDI->karaoke_format == 1 && TMDY_READMIDI->current_read_track == 2)
	TMDY_READMIDI->karaoke_format = 2; /* Start karaoke lyric */
    else if(TMDY_READMIDI->karaoke_format == 2 && TMDY_READMIDI->current_read_track == 3)
	TMDY_READMIDI->karaoke_format = 3; /* End karaoke lyric */
    TMDY_READMIDI->midi_port_number = 0;

    if(TMDY_READMIDI->evlist == NULL)
	return 0;
    if(rewindp)
	TMDY_READMIDI->current_midi_point = TMDY_READMIDI->evlist;
    else
    {
	/* find the last event in the list */
	while(TMDY_READMIDI->current_midi_point->next != NULL)
	    TMDY_READMIDI->current_midi_point = TMDY_READMIDI->current_midi_point->next;
    }
    return TMDY_READMIDI->current_midi_point->event.time;
}

void readmidi_add_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *a_event)
{
    MidiEventList *newev;
    int32 at;

    if(TMDY_READMIDI->event_count++ == MAX_MIDI_EVENT)
    {
	if(!TMDY_READMIDI->readmidi_error_flag)
	{
	    TMDY_READMIDI->readmidi_error_flag = 1;
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "Maxmum number of events is exceeded");
	}
	return;
    }

    at = a_event->time;
    newev = alloc_midi_event(tmdy_struct);
    newev->event = *a_event;	/* assign by value!!! */
    if(at < 0)	/* for safety */
	at = newev->event.time = 0;

    if(at >= TMDY_READMIDI->current_midi_point->event.time)
    {
	/* Forward scan */
	MidiEventList *next = TMDY_READMIDI->current_midi_point->next;
	while (next && (next->event.time <= at))
	{
	    TMDY_READMIDI->current_midi_point = next;
	    next = TMDY_READMIDI->current_midi_point->next;
	}
	newev->prev = TMDY_READMIDI->current_midi_point;
	newev->next = next;
	TMDY_READMIDI->current_midi_point->next = newev;
	if (next)
	    next->prev = newev;
    }
    else
    {
	/* Backward scan -- symmetrical to the one above */
	MidiEventList *prev = TMDY_READMIDI->current_midi_point->prev;
	while (prev && (prev->event.time > at)) {
	    TMDY_READMIDI->current_midi_point = prev;
	    prev = TMDY_READMIDI->current_midi_point->prev;
	}
	newev->prev = prev;
	newev->next = TMDY_READMIDI->current_midi_point;
	TMDY_READMIDI->current_midi_point->prev = newev;
	if (prev)
	    prev->next = newev;
    }
    TMDY_READMIDI->current_midi_point = newev;
}

void readmidi_add_ctl_event(tmdy_struct_ex_t *tmdy_struct, int32 at, int ch, int a, int b)
{
    MidiEvent ev;

    if(convert_midi_control_change(tmdy_struct, ch, a, b, &ev))
    {
	ev.time = at;
	readmidi_add_event(tmdy_struct, &ev);
    }
    else
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "(Control ch=%d %d: %d)", ch, a, b);
}

char *readmidi_make_string_event(tmdy_struct_ex_t *tmdy_struct, int type, char *string, MidiEvent *ev,
				 int cnv)
{
    char *text;
    int len;
    StringTableNode *st;
    int a, b;

    if(TMDY_READMIDI->string_event_strtab.nstring == 0)
	TMDY_UTILS->strtab->put_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab, "", 0);
    else if(TMDY_READMIDI->string_event_strtab.nstring == 0x7FFE)
    {
	SETMIDIEVENT(*ev, 0, type, 0, 0, 0);
	return NULL; /* Over flow */
    }
    a = (TMDY_READMIDI->string_event_strtab.nstring & 0xff);
    b = ((TMDY_READMIDI->string_event_strtab.nstring >> 8) & 0xff);

    len = strlen(string);
    if(cnv)
    {
	text = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), SAFE_CONVERT_LENGTH(len) + 1);
	TMDY_COMMON->code_convert(tmdy_struct, string, text + 1, SAFE_CONVERT_LENGTH(len), NULL, NULL);
    }
    else
    {
	text = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len + 1);
	memcpy(text + 1, string, len);
	text[len + 1] = '\0';
    }

    st = TMDY_UTILS->strtab->put_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab, text, strlen(text + 1) + 1);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));

    text = st->string;
    *text = type;
    SETMIDIEVENT(*ev, 0, type, 0, a, b);
    return text;
}

static char *readmidi_make_lcd_event(tmdy_struct_ex_t *tmdy_struct, int type, const uint8 *data, MidiEvent *ev)
{
    char *text;
    int len;
    StringTableNode *st;
    int a, b, i;

    if(TMDY_READMIDI->string_event_strtab.nstring == 0)
	TMDY_UTILS->strtab->put_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab, "", 0);
    else if(TMDY_READMIDI->string_event_strtab.nstring == 0x7FFE)
    {
	SETMIDIEVENT(*ev, 0, type, 0, 0, 0);
	return NULL; /* Over flow */
    }
    a = (TMDY_READMIDI->string_event_strtab.nstring & 0xff);
    b = ((TMDY_READMIDI->string_event_strtab.nstring >> 8) & 0xff);

    len = 128;
    
	text = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len + 2);

    for( i=0; i<64; i++){
	const char tbl[]= "0123456789ABCDEF";
	text[1+i*2  ]=tbl[data[i]>>4];
	text[1+i*2+1]=tbl[data[i]&0xF];
    }
    text[len + 1] = '\0';
    
    
    st = TMDY_UTILS->strtab->put_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab, text, strlen(text + 1) + 1);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));

    text = st->string;
    *text = type;
    SETMIDIEVENT(*ev, 0, type, 0, a, b);
    return text;
}

/* Computes how many (fractional) samples one MIDI delta-time unit contains */
static void compute_sample_increment(tmdy_struct_ex_t *tmdy_struct, int32 tempo, int32 divisions)
{
  double a;
  a = (double) (tempo) * (double) ((TMDY_OUTPUT->play_mode)->rate) * (65536.0/1000000.0) /
    (double)(divisions);

  TMDY_READMIDI->sample_correction = (int32)(a) & 0xFFFF;
  TMDY_READMIDI->sample_increment = (int32)(a) >> 16;

  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "Samples per delta-t: %d (correction %d)",
       TMDY_READMIDI->sample_increment, TMDY_READMIDI->sample_correction);
}

/* Read variable-length number (7 bits per byte, MSB first) */
static int32 getvl(tmdy_struct_ex_t *tmdy_struct, struct timidity_file *tf)
{
    int32 l;
    int c;

    errno = 0;
    l = 0;

    /* 1 */
    if((c = tf_getc(tf)) == EOF)
	goto eof;
    if(!(c & 0x80)) return l | c;
    l = (l | (c & 0x7f)) << 7;

    /* 2 */
    if((c = tf_getc(tf)) == EOF)
	goto eof;
    if(!(c & 0x80)) return l | c;
    l = (l | (c & 0x7f)) << 7;

    /* 3 */
    if((c = tf_getc(tf)) == EOF)
	goto eof;
    if(!(c & 0x80)) return l | c;
    l = (l | (c & 0x7f)) << 7;

    /* 4 */
    if((c = tf_getc(tf)) == EOF)
	goto eof;
    if(!(c & 0x80)) return l | c;

    /* Error */
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
	      "%s: Illigal Variable-length quantity format.",
	      (TMDY_COMMON->current_filename));
    return -2;

  eof:
    if(errno)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: read_midi_event: %s",
		  (TMDY_COMMON->current_filename), strerror(errno));
    else
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Warning: %s: Too shorten midi file.",
		  (TMDY_COMMON->current_filename));
    return -1;
}

static char *add_karaoke_title(tmdy_struct_ex_t *tmdy_struct, char *s1, char *s2)
{
    char *ks;
    int k1, k2;

    if(s1 == NULL)
	return TMDY_COMMON->safe_strdup(tmdy_struct, s2);

    k1 = strlen(s1);
    k2 = strlen(s2);
    if(k2 == 0)
	return s1;
    ks = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, k1 + k2 + 2);
    memcpy(ks, s1, k1);
    ks[k1++] = ' ';
    memcpy(ks + k1, s2, k2 + 1);
    free(s1);

    return ks;
}


/* Print a string from the file, followed by a newline. Any non-ASCII
   or unprintable characters will be converted to periods. */
static char *dumpstring(tmdy_struct_ex_t *tmdy_struct, int type, int32 len, char *label, int allocp,
			struct timidity_file *tf)
{
    char *si, *so;
    int s_maxlen = SAFE_CONVERT_LENGTH(len);
    int llen, solen;

    if(len <= 0)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_TEXT, VERB_VERBOSE, "%s", label);
	return NULL;
    }

    si = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len + 1);
    so = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), s_maxlen);

    if(len != TMDY_COMMON->tf_read(tmdy_struct, si, 1, len, tf))
    {
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	return NULL;
    }
    si[len]='\0';

    if(type == 1 &&
       TMDY_READMIDI->current_read_track == 1 &&
       TMDY_READMIDI->current_file_info->format == 1 &&
       strncmp(si, "@KMIDI", 6) == 0)
	TMDY_READMIDI->karaoke_format = 1;

    TMDY_COMMON->code_convert(tmdy_struct, si, so, s_maxlen, NULL, NULL);

    llen = strlen(label);
    solen = strlen(so);
    if(llen + solen >= MIN_MBLOCK_SIZE)
	so[MIN_MBLOCK_SIZE - llen - 1] = '\0';

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_TEXT, VERB_VERBOSE, "%s%s", label, so);

    if(allocp)
    {
	so = TMDY_COMMON->safe_strdup(tmdy_struct, so);
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	return so;
    }
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
    return NULL;
}

static uint16 gs_convert_master_vol(tmdy_struct_ex_t *tmdy_struct, int vol)
{
    double v;

    if(vol >= 0x7f)
	return 0xffff;
    v = (double)vol * (0xffff/127.0);
    if(v >= 0xffff)
	return 0xffff;
    return (uint16)v;
}

static uint16 gm_convert_master_vol(tmdy_struct_ex_t *tmdy_struct, uint16 v1, uint16 v2)
{
    return (((v1 & 0x7f) | ((v2 & 0x7f) << 7)) << 2) | 3;
}

static void check_chorus_text_start(tmdy_struct_ex_t *tmdy_struct)
{
    if(TMDY_REVERB->chorus_status.status != CHORUS_ST_OK &&
       TMDY_REVERB->chorus_status.voice_reserve[17] &&
       TMDY_REVERB->chorus_status.macro[2] &&
       TMDY_REVERB->chorus_status.pre_lpf[2] &&
       TMDY_REVERB->chorus_status.level[2] &&
       TMDY_REVERB->chorus_status.feed_back[2] &&
       TMDY_REVERB->chorus_status.delay[2] &&
       TMDY_REVERB->chorus_status.rate[2] &&
       TMDY_REVERB->chorus_status.depth[2] &&
       TMDY_REVERB->chorus_status.send_level[2])
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "Chorus text start");
	TMDY_REVERB->chorus_status.status = CHORUS_ST_OK;
    }
}

int convert_midi_control_change(tmdy_struct_ex_t *tmdy_struct, int chn, int type, int val, MidiEvent *ev_ret)
{
    switch(type)
    {
      case   0: type = ME_TONE_BANK_MSB; break;
      case   1: type = ME_MODULATION_WHEEL; break;
      case   2: type = ME_BREATH; break;
      case   4: type = ME_FOOT; break;
      case   5: type = ME_PORTAMENTO_TIME_MSB; break;
      case   6: type = ME_DATA_ENTRY_MSB; break;
      case   7: type = ME_MAINVOLUME; break;
      case   8: type = ME_BALANCE; break;
      case  10: type = ME_PAN; break;
      case  11: type = ME_EXPRESSION; break;
      case  32: type = ME_TONE_BANK_LSB; break;
      case  37: type = ME_PORTAMENTO_TIME_LSB; break;
      case  38: type = ME_DATA_ENTRY_LSB; break;
      case  64: type = ME_SUSTAIN; break;
      case  65: type = ME_PORTAMENTO; break;
      case  66: type = ME_SOSTENUTO; break;
      case  67: type = ME_SOFT_PEDAL; break;
      case  68: type = ME_LEGATO_FOOTSWITCH; break;
      case  69: type = ME_HOLD2; break;
      case  71: type = ME_HARMONIC_CONTENT; break;
      case  72: type = ME_RELEASE_TIME; break;
      case  73: type = ME_ATTACK_TIME; break;
      case  74: type = ME_BRIGHTNESS; break;
      case  84: type = ME_PORTAMENTO_CONTROL; break;
      case  91: type = ME_REVERB_EFFECT; break;
      case  92: type = ME_TREMOLO_EFFECT; break;
      case  93: type = ME_CHORUS_EFFECT; break;
      case  94: type = ME_CELESTE_EFFECT; break;
      case  95: type = ME_PHASER_EFFECT; break;
      case  96: type = ME_RPN_INC; break;
      case  97: type = ME_RPN_DEC; break;
      case  98: type = ME_NRPN_LSB; break;
      case  99: type = ME_NRPN_MSB; break;
      case 100: type = ME_RPN_LSB; break;
      case 101: type = ME_RPN_MSB; break;
      case 120: type = ME_ALL_SOUNDS_OFF; break;
      case 121: type = ME_RESET_CONTROLLERS; break;
      case 123: type = ME_ALL_NOTES_OFF; break;
      case 126: type = ME_MONO; break;
      case 127: type = ME_POLY; break;
      default: type = -1; break;
    }

    if(type != -1)
    {
	if(val > 127)
	    val = 127;
	ev_ret->type    = type;
	ev_ret->channel = chn;
	ev_ret->a       = val;
	ev_ret->b       = 0;
	return 1;
    }
    return 0;
}

static int block_to_part(tmdy_struct_ex_t *tmdy_struct, int block, int port)
{
	int p;
	p = block & 0x0F;
	if(p == 0) {p = 9;}
	else if(p <= 9) {p--;}
	return MERGE_CHANNEL_PORT2(p, port);
}

/* XG SysEx parsing function by Eric A. Welsh
 * Also handles GS patch+bank changes
 *
 * This function provides basic support for XG Multi Part Data
 * parameter change SysEx events
 *
 * NOTE - val[1] is documented as only being 0x10, but this rule is not
 * followed in real life, since I have midi that set it to 0x00 and are
 * interpreted correctly on my SW60XG ...
 */
int parse_sysex_event_multi(tmdy_struct_ex_t *tmdy_struct, uint8 *val, int32 len, MidiEvent *evm)
{
    int num_events = 0;				/* Number of events added */
	uint32 channel_tt;
	int i, j;

    if(TMDY_READMIDI->current_file_info->mid == 0 || TMDY_READMIDI->current_file_info->mid >= 0x7e)
	TMDY_READMIDI->current_file_info->mid = val[0];

    /* First, check for the various XG Bulk Dump events */
    /* val[7] is the starting value for the event type */

    /* Effect 1 */
    else if(len >= 10 &&
       val[0] == 0x43 && /* Yamaha ID */
       val[2] == 0x4C && /* XG Model ID */
       val[5] == 0x02)   /* Effect 1 */
    {
	uint8 addhigh, addmid, addlow;		/* Addresses */
	uint8 *body;				/* SysEx body */
	uint8 p;				/* Channel part number [0..15] */
	int ent;				/* Entry # of sub-event */
	uint8 *body_end;			/* End of SysEx body */

	addhigh = val[5];
	addmid = val[6];
	addlow = val[7];
	body = val + 8;
	p = val[6];
	body_end = val + len-3;

	for (ent = val[7]; body <= body_end; body++, ent++) {
	    switch(ent) {
		case 0x0C:	/* Reverb Return */
		    TMDY_REVERB->reverb_status.level = *body;
		    recompute_reverb_status(tmdy_struct);
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Level (%d)",*body);
		    break;

		case 0x2C:	/* Chorus Return */
		    TMDY_REVERB->chorus_param.chorus_level = *body;
		    recompute_chorus_status(tmdy_struct);
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Level (%d)",*body);
		    break;

		default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported XG Bulk Dump SysEx. (ADDR:%02X %02X %02X VAL:%02X)",val[5],val[6],val[7],ent);
		    continue;
		    break;
	    }
	}
    }

    /* Effect 2 (Insertion Effects) */
    if(len >= 10 &&
       val[0] == 0x43 && /* Yamaha ID */
       val[2] == 0x4C && /* XG Model ID */
       val[5] == 0x03)   /* Effect 2 (Insertion Effects) */
    {
	uint8 addhigh, addmid, addlow;		/* Addresses */
	uint8 *body;				/* SysEx body */
	uint8 p;				/* Channel part number [0..15] */
	int ent;				/* Entry # of sub-event */
	uint8 *body_end;			/* End of SysEx body */

	addhigh = val[5];
	addmid = val[6];
	addlow = val[7];
	body = val + 8;
	p = val[6];
	body_end = val + len-3;

	for (ent = val[7]; body <= body_end; body++, ent++) {
	    switch(ent) {

		/* need to add in insertion effects code */
		/* need to figure out how XG insertion effects map on to
		   the GS insertion effects */

		default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported XG Bulk Dump SysEx. (ADDR:%02X %02X %02X VAL:%02X)",val[5],val[6],val[7],ent);
		    continue;
		    break;
	    }
	}
    }

    /* XG Multi Part Data parameter change */
    /* There are two ways to do this, neither of which match the XG spec... */

    /* First method via Bulk Dump */
    else if(len >= 10 &&
       val[0] == 0x43 && /* Yamaha ID */
       val[2] == 0x4C && /* XG Model ID */
       (val[4] == 0x29 || val[4] == 0x3F) && /* Blocks 1 or 2 */
       val[5] == 0x08)   /* Multi Part */
    {
	uint8 addhigh, addmid, addlow;		/* Addresses */
	uint8 *body;				/* SysEx body */
	uint8 p;				/* Channel part number [0..15] */
	int ent;				/* Entry # of sub-event */
	uint8 *body_end;			/* End of SysEx body */

	addhigh = val[5];
	addmid = val[6];
	addlow = val[7];
	body = val + 8;
	p = val[6];
	body_end = val + len-3;

	for (ent = val[7]; body <= body_end; body++, ent++) {
	    switch(ent) {

		case 0x01:	/* bank select MSB */
		    SETMIDIEVENT(evm[num_events], 0, ME_TONE_BANK_MSB, p, *body, 0);
		    num_events++;
		    break;

		case 0x02:	/* bank select LSB */
		    SETMIDIEVENT(evm[num_events], 0, ME_TONE_BANK_LSB, p, *body, 0);
		    num_events++;
		    break;

		case 0x03:	/* program number */
		    SETMIDIEVENT(evm[num_events], 0, ME_PROGRAM, p, *body, 0);
		    num_events++;
		    break;

		case 0x04:	/* Rcv CHANNEL */
			SETMIDIEVENT(evm[num_events], 0, ME_SYSEX_XG_LSB, p, *body, 0x65);
		    break;

		case 0x05:	/* mono/poly mode */
		    if(*body == 0) {
			TMDY_PLAYMIDI->channel[p].mono = 1;
		    } else {
			TMDY_PLAYMIDI->channel[p].mono = 0;
		    }
		    break;

		case 0x08:	/* note shift ? */
		    SETMIDIEVENT(evm[num_events], 0, ME_KEYSHIFT, p, *body, 0);
		    num_events++;
		    break;

		case 0x0B:	/* volume */
		    SETMIDIEVENT(evm[num_events], 0, ME_MAINVOLUME, p, *body, 0);
		    num_events++;
		    break;

		case 0x0E:	/* pan */
		    if(*body == 0) {
		    	SETMIDIEVENT(evm[num_events], 0, ME_RANDOM_PAN, p, 0, 0);
		    }
		    else {
			SETMIDIEVENT(evm[num_events], 0, ME_PAN, p, *body, 0);
		    }
		    num_events++;
		    break;

		case 0x12:	/* chorus send */
		    SETMIDIEVENT(evm[num_events], 0, ME_CHORUS_EFFECT, p, *body, 0);
		    num_events++;
		    break;

		case 0x13:	/* reverb send */
		    SETMIDIEVENT(evm[num_events], 0, ME_REVERB_EFFECT, p, *body, 0);
		    num_events++;
		    break;

		case 0x23:	/* bend pitch control */
		    SETMIDIEVENT(evm[num_events], 0,ME_RPN_MSB,p,0,0);
		    SETMIDIEVENT(evm[num_events+1], 0,ME_RPN_LSB,p,0,0);
		    SETMIDIEVENT(evm[num_events+2], 0,ME_DATA_ENTRY_MSB,p,(*body - 0x40) & 0x7F,0);
		    num_events += 3;
		    break;

		case 0x41:	/* scale tuning */
		case 0x42:
		case 0x43:
		case 0x44:
		case 0x45:
		case 0x46:
		case 0x47:
		case 0x48:
		case 0x49:
		case 0x4a:
		case 0x4b:
		case 0x4c:
		    SETMIDIEVENT(evm[0], 0, ME_SCALE_TUNING, p, ent - 0x41, *body - 64);
		    num_events++;
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "Scale Tuning %s (CH:%d %d cent)",
			      (TMDY_COMMON->note_name)[ent - 0x41], p, *body - 64);
		    break;

		default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported XG Bulk Dump SysEx. (ADDR:%02X %02X %02X VAL:%02X)",val[5],val[6],ent,*body);
		    continue;
		    break;
	    }
	}
    }

    /* Second method: specify them one SYSEX event at a time... */
    else if(len == 8 &&
       val[0] == 0x43 && /* Yamaha ID */
       val[2] == 0x4C) /* XG Model ID */ 
    {
		uint8 p;				/* Channel part number [0..15] */
		int ent;				/* Entry # of sub-event */

		p = val[4];
		ent = val[5];

		if(val[3] == 0x01) {	/* Effect 1 */
			switch(ent) {
				case 0x0C:	/* Reverb Return */
				  SETMIDIEVENT(evm[0], 0,ME_SYSEX_XG_LSB,p,val[6],0x00);
				  num_events++;
				  break;

				case 0x2C:	/* Chorus Return */
				  SETMIDIEVENT(evm[0], 0,ME_SYSEX_XG_LSB,p,val[6],0x01);
				  num_events++;
				  break;

				default:
				  break;
			}
		} else if(val[3] == 0x08) {	/* Multi Part Data parameter change */
			switch(ent) {
				case 0x01:	/* bank select MSB */
				  SETMIDIEVENT(evm[0], 0, ME_TONE_BANK_MSB, p, val[6], 0);
				  num_events++;
				  break;

				case 0x02:	/* bank select LSB */
				  SETMIDIEVENT(evm[0], 0, ME_TONE_BANK_LSB, p, val[6], 0);
				  num_events++;
				  break;

				case 0x03:	/* program number */
				  SETMIDIEVENT(evm[0], 0, ME_PROGRAM, p, val[6], 0);
				  num_events++;
				  break;

				case 0x04:	/* Rcv CHANNEL */
				  SETMIDIEVENT(evm[num_events], 0, ME_SYSEX_XG_LSB, p, val[6], 0x65);
				  num_events++;
				  break;

				case 0x05:	/* mono/poly mode */
 				  if(val[6] == 0) {SETMIDIEVENT(evm[0], 0, ME_MONO, p, val[6], 0);}
				  else {SETMIDIEVENT(evm[0], 0, ME_POLY, p, val[6], 0);}
				  num_events++;
				  break;

				case 0x08:	/* note shift ? */
				  SETMIDIEVENT(evm[0], 0, ME_KEYSHIFT, p, val[6], 0);
				  num_events++;
				  break;

				case 0x0B:	/* volume */
				  SETMIDIEVENT(evm[0], 0, ME_MAINVOLUME, p, val[6], 0);
				  num_events++;
				  break;

				case 0x0E:	/* pan */
				  if(val[6] == 0) {
					SETMIDIEVENT(evm[0], 0, ME_RANDOM_PAN, p, 0, 0);
				  }
				  else {
					SETMIDIEVENT(evm[0], 0, ME_PAN, p, val[6], 0);
				  }
				  num_events++;
				  break;

				case 0x12:	/* chorus send */
				  SETMIDIEVENT(evm[0], 0, ME_CHORUS_EFFECT, p, val[6], 0);
				  num_events++;
				  break;

				case 0x13:	/* reverb send */
				  SETMIDIEVENT(evm[0], 0, ME_REVERB_EFFECT, p, val[6], 0);
				  num_events++;
				  break;

				case 0x23:	/* bend pitch control */
				  SETMIDIEVENT(evm[0], 0,ME_RPN_MSB,p,0,0);
				  SETMIDIEVENT(evm[1], 0,ME_RPN_LSB,p,0,0);
				  SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,(val[6] - 0x40) & 0x7F,0);
				  num_events += 3;
				  break;

				case 0x41:	/* scale tuning */
				case 0x42:
				case 0x43:
				case 0x44:
				case 0x45:
				case 0x46:
				case 0x47:
				case 0x48:
				case 0x49:
				case 0x4a:
				case 0x4b:
				case 0x4c:
					SETMIDIEVENT(evm[0],
							0, ME_SCALE_TUNING, p, ent - 0x41, val[6] - 64);
					num_events++;
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
							"Scale Tuning %s (CH:%d %d cent)",
							(TMDY_COMMON->note_name)[ent - 0x41], p, val[6] - 64);
					break;

				default:
				  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported XG SysEx. (ADDR:%02X %02X %02X VAL:%02X)",val[3],val[4],val[5],val[6]);
				  break;
			}
		}
    }
    /* parsing GS System Exclusive Message... */
	/* val[4] == Parameter Address(High)
	   val[5] == Parameter Address(Middle)
	   val[6] == Parameter Address(Low)
	   val[7]... == Data...
	   val[last] == Checksum(== 128 - (sum of addresses&data bytes % 128)) 
	*/
    else if(len >= 9 &&
	   val[0] == 0x41 && /* Roland ID */
       val[1] == 0x10 && /* Device ID */
       val[2] == 0x42 && /* GS Model ID */
       val[3] == 0x12) /* Data Set Command */
    {
		uint8 p, dp, udn, gslen, port = 0;
		int i, addr, addr_h, addr_m, addr_l, checksum;
		p = block_to_part(tmdy_struct, val[5], TMDY_READMIDI->midi_port_number);

		/* calculate checksum */
		checksum = 0;
		for(gslen = 9; gslen < len; gslen++)
			if(val[gslen] == 0xF7)
				break;
		for(i=4;i<gslen-1;i++) {
			checksum += val[i];
		}
		if(((128 - (checksum & 0x7F)) & 0x7F) != val[gslen-1]) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"GS SysEx: Checksum Error.");
			return num_events;
		}

		/* drum channel */
		dp = TMDY_READMIDI->rhythm_part[(val[5] & 0xF0) >> 4];

		/* calculate user drumset number */
		udn = (val[5] & 0xF0) >> 4;

		addr_h = val[4];
		addr_m = val[5];
		addr_l = val[6];
		if(addr_h == 0x50) {	/* for double module mode */
			port = 1;
			p = block_to_part(tmdy_struct, val[5], port);
			addr_h = 0x40;
		} else if(addr_h == 0x51) {
			port = 1;
			p = block_to_part(tmdy_struct, val[5], port);
			addr_h = 0x41;
		}
		addr = (((int32)addr_h)<<16 | ((int32)addr_m)<<8 | (int32)addr_l);

		switch(addr_h) {
		case 0x40:
			if((addr & 0xFFF000) == 0x401000) {
				switch(addr & 0xFF) {
				case 0x00:
					SETMIDIEVENT(evm[0], 0, ME_TONE_BANK_MSB,p,val[7],0);
					SETMIDIEVENT(evm[1], 0, ME_PROGRAM,p,val[8],0);
					num_events += 2;
					break;
				case 0x02:	/* Rx. Channel */
					if (val[7] == 0x10) {
						SETMIDIEVENT(evm[0], 0, ME_SYSEX_GS_LSB,
								block_to_part(tmdy_struct, val[5],
								TMDY_READMIDI->midi_port_number ^ port), 0x80, 0x45);
					} else {
						SETMIDIEVENT(evm[0], 0, ME_SYSEX_GS_LSB,
								block_to_part(tmdy_struct, val[5],
								TMDY_READMIDI->midi_port_number ^ port),
								MERGE_CHANNEL_PORT2(val[7],
								TMDY_READMIDI->midi_port_number ^ port), 0x45);
					}
					num_events++;
					break;
				case 0x13:
					if(val[7] == 0) {SETMIDIEVENT(evm[0], 0, ME_MONO,p,val[7],0);}
					else {SETMIDIEVENT(evm[0], 0, ME_POLY,p,val[7],0);}
					num_events++;
					break;
				case 0x14:	/* Assign Mode */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x24);
					num_events++;
					break;
				case 0x15:	/* Use for Rhythm Part */
					if(val[7]) {
						TMDY_READMIDI->rhythm_part[val[7] - 1] = p;
					}
					break;
				case 0x16:	/* Pitch Key Shift */
					break;
				case 0x17:	/* Pitch Offset Fine */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x26);
					num_events++;
					break;
				case 0x19:
					SETMIDIEVENT(evm[0], 0,ME_MAINVOLUME,p,val[7],0);
					num_events++;
					break;
				case 0x1A:	/* Velocity Sense Depth */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x21);
					num_events++;
					break;
				case 0x1B:	/* Velocity Sense Offset */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x22);
					num_events++;
					break;
				case 0x1C:
					if (val[7] == 0) {
						SETMIDIEVENT(evm[0], 0, ME_RANDOM_PAN, p, 0, 0);
					} else {
						SETMIDIEVENT(evm[0], 0,ME_PAN,p,val[7],0);
					}
					num_events++;
					break;
				case 0x21:
					SETMIDIEVENT(evm[0], 0,ME_CHORUS_EFFECT,p,val[7],0);
					num_events++;
					break;
				case 0x22:
					SETMIDIEVENT(evm[0], 0,ME_REVERB_EFFECT,p,val[7],0);
					num_events++;
					break;
				case 0x2C:
					SETMIDIEVENT(evm[0], 0,ME_CELESTE_EFFECT,p,val[7],0);
					num_events++;
					break;
				case 0x2A:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x00,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x01,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					SETMIDIEVENT(evm[3], 0,ME_DATA_ENTRY_LSB,p,val[8],0);
					num_events += 4;
					break;
				case 0x30:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x08,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x31:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x09,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x32:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x20,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x33:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x21,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x34:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x63,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x35:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x64,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x36:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x66,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x37:
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,p,0x01,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,p,0x0A,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,val[7],0);
					num_events += 3;
					break;
				case 0x40:	/* Scale Tuning */
					for (i = 0; i < 12; i++) {
						SETMIDIEVENT(evm[i],
								0, ME_SCALE_TUNING, p, i, val[i + 7] - 64);
						(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
								"Scale Tuning %s (CH:%d %d cent)",
								(TMDY_COMMON->note_name)[i], p, val[i + 7] - 64);
					}
					num_events += 12;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			} else if((addr & 0xFFF000) == 0x402000) {
				switch(addr & 0xFF) {
				case 0x10:	/* Bend Pitch Control */
					SETMIDIEVENT(evm[0], 0,ME_RPN_MSB,p,0,0);
					SETMIDIEVENT(evm[1], 0,ME_RPN_LSB,p,0,0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,p,(val[7] - 0x40) & 0x7F,0);
					num_events += 3;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			} else if((addr & 0xFFFF00) == 0x400100) {
				switch(addr & 0xFF) {
				case 0x30:	/* Reverb Macro */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x05);
					num_events++;
					break;
				case 0x31:	/* Reverb Character */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x06);
					num_events++;
					break;
				case 0x32:	/* Reverb Pre-LPF */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x07);
					num_events++;
					break;
				case 0x33:	/* Reverb Level */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x08);
					num_events++;
					break;
				case 0x34:	/* Reverb Time */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x09);
					num_events++;
					break;
				case 0x35:	/* Reverb Delay Feedback */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x0A);
					num_events++;
					break;
				case 0x36:	/* Unknown Reverb Parameter */
					break;
				case 0x37:	/* Reverb Predelay Time */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x0C);
					num_events++;
					break;
				case 0x38:	/* Chorus Macro */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x0D);
					num_events++;
					break;
				case 0x39:	/* Chorus Pre-LPF */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x0E);
					num_events++;
					break;
				case 0x3A:	/* Chorus Level */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x0F);
					num_events++;
					break;
				case 0x3B:	/* Chorus Feedback */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x10);
					num_events++;
					break;
				case 0x3C:	/* Chorus Delay */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x11);
					num_events++;
					break;
				case 0x3D:	/* Chorus Rate */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x12);
					num_events++;
					break;
				case 0x3E:	/* Chorus Depth */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x13);
					num_events++;
					break;
				case 0x3F:	/* Chorus Send Level to Reverb */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x14);
					num_events++;
					break;
				case 0x40:	/* Chorus Send Level to Delay */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x15);
					num_events++;
					break;
				case 0x50:	/* Delay Macro */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x16);
					num_events++;
					break;
				case 0x51:	/* Delay Pre-LPF */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x17);
					num_events++;
					break;
				case 0x52:	/* Delay Time Center */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x18);
					num_events++;
					break;
				case 0x53:	/* Delay Time Ratio Left */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x19);
					num_events++;
					break;
				case 0x54:	/* Delay Time Ratio Right */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1A);
					num_events++;
					break;
				case 0x55:	/* Delay Level Center */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1B);
					num_events++;
					break;
				case 0x56:	/* Delay Level Left */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1C);
					num_events++;
					break;
				case 0x57:	/* Delay Level Right */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1D);
					num_events++;
					break;
				case 0x58:	/* Delay Level */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1E);
					num_events++;
					break;
				case 0x59:	/* Delay Feedback */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x1F);
					num_events++;
					break;
				case 0x5A:	/* Delay Send Level to Reverb */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x20);
					num_events++;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			} else if((addr & 0xFFFF00) == 0x400200) {
				switch(addr & 0xFF) {	/* EQ Parameter */
				case 0x00:	/* EQ LOW FREQ */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x01);
					num_events++;
					break;
				case 0x01:	/* EQ LOW GAIN */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x02);
					num_events++;
					break;
				case 0x02:	/* EQ HIGH FREQ */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x03);
					num_events++;
					break;
				case 0x03:	/* EQ HIGH GAIN */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x04);
					num_events++;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			} else if((addr & 0xFFFF00) == 0x400300) {
				switch(addr & 0xFF) {	/* Insertion Effect Parameter */
				case 0x00:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x27);
					SETMIDIEVENT(evm[1], 0,ME_SYSEX_GS_LSB,p,val[8],0x28);
					num_events += 2;
					break;
				case 0x03:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x29);
					num_events++;
					break;
				case 0x04:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2A);
					num_events++;
					break;
				case 0x05:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2B);
					num_events++;
					break;
				case 0x06:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2C);
					num_events++;
					break;
				case 0x07:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2D);
					num_events++;
					break;
				case 0x08:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2E);
					num_events++;
					break;
				case 0x09:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x2F);
					num_events++;
					break;
				case 0x0A:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x30);
					num_events++;
					break;
				case 0x0B:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x31);
					num_events++;
					break;
				case 0x0C:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x32);
					num_events++;
					break;
				case 0x0D:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x33);
					num_events++;
					break;
				case 0x0E:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x34);
					num_events++;
					break;
				case 0x0F:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x35);
					num_events++;
					break;
				case 0x10:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x36);
					num_events++;
					break;
				case 0x11:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x37);
					num_events++;
					break;
				case 0x12:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x38);
					num_events++;
					break;
				case 0x13:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x39);
					num_events++;
					break;
				case 0x14:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3A);
					num_events++;
					break;
				case 0x15:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3B);
					num_events++;
					break;
				case 0x16:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3C);
					num_events++;
					break;
				case 0x17:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3D);
					num_events++;
					break;
				case 0x18:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3E);
					num_events++;
					break;
				case 0x19:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x3F);
					num_events++;
					break;
				case 0x1B:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x40);
					num_events++;
					break;
				case 0x1C:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x41);
					num_events++;
					break;
				case 0x1D:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x42);
					num_events++;
					break;
				case 0x1E:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x43);
					num_events++;
					break;
				case 0x1F:
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x44);
					num_events++;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			} else if((addr & 0xFFF000) == 0x404000) {
				switch(addr & 0xFF) {
				case 0x00:	/* TONE MAP NUMBER */
					if(val[7] == 0) {
						SETMIDIEVENT(evm[0], 0, ME_TONE_BANK_LSB,p,TMDY_PLAYMIDI->channel[p].tone_map0_number,0);
					} else {
						SETMIDIEVENT(evm[0], 0, ME_TONE_BANK_LSB,p,val[7],0);
					}
					num_events++;
					break;
				case 0x01:	/* TONE MAP-0 NUMBER */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x25);
					num_events++;
					break;
				case 0x20:	/* EQ ON/OFF */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x00);
					num_events++;
					break;
				case 0x22:	/* EFX ON/OFF */
					SETMIDIEVENT(evm[0], 0,ME_SYSEX_GS_LSB,p,val[7],0x23);
					num_events++;
					break;
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
				}
			}
			break;
		case 0x41:
			switch(addr & 0xF00) {
			case 0x100:	/* Play Note */
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x18,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
				num_events += 3;
				break;
			case 0x200:
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1A,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
				num_events += 3;
				break;
			case 0x400:
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1C,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0x1);
				num_events += 3;
				break;
			case 0x500:
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1D,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
				num_events += 3;
				break;
			case 0x600:
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1E,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
				num_events += 3;
				break;
			case 0x700:
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Rx. Note Off (CH:%d NOTE:%d VAL:%d)",dp,val[6],val[7]);
				if(TMDY_PLAYMIDI->channel[dp].drums[val[6]] == NULL) {TMDY_PLAYMIDI->play_midi_setup_drums(tmdy_struct, dp, val[6]);}
				TMDY_PLAYMIDI->channel[dp].drums[val[6]]->rx_note_off = val[7];
				break;
			case 0x900:
				SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1F,0);
				SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
				SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
				num_events += 3;
				break;
			default:
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
				break;
			}
			break;
#if 0
		case 0x20:	/* User Instrument */
			switch(addr & 0xF00) {
				case 0x000:	/* Source Map */
					get_userinst(tmdy_struct, 64+udn, val[6])->source_map = val[7];
					break;
				case 0x100:	/* Source Bank */
					get_userinst(tmdy_struct, 64+udn, val[6])->source_bank = val[7];
					break;
#if !defined(TIMIDITY_TOOLS)
				case 0x200:	/* Source Prog */
					get_userinst(tmdy_struct, 64+udn, val[6])->source_prog = val[7];
					break;
#endif
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
			}
			break;
#endif
		case 0x21:	/* User Drumset */
			switch(addr & 0xF00) {
				case 0x100:	/* Play Note */
					get_userdrum(tmdy_struct, 64+udn, val[6])->play_note = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x18,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0x1);
					num_events += 3;
					break;
				case 0x200:	/* Level */
					get_userdrum(tmdy_struct, 64+udn, val[6])->level = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1A,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
					num_events += 3;
					break;
				case 0x300:	/* Assign Group */
					get_userdrum(tmdy_struct, 64+udn, val[6])->assign_group = val[7];
					if(val[7] != 0) {recompute_userdrum_altassign(tmdy_struct, udn+64, val[7]);}
					break;
				case 0x400:	/* Panpot */
					get_userdrum(tmdy_struct, 64+udn, val[6])->pan = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1C,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
					num_events += 3;
					break;
				case 0x500:	/* Reverb Send Level */
					get_userdrum(tmdy_struct, 64+udn, val[6])->reverb_send_level = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1D,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
					num_events += 3;
					break;
				case 0x600:	/* Chorus Send Level */
					get_userdrum(tmdy_struct, 64+udn, val[6])->chorus_send_level = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1E,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
					num_events += 3;
					break;
				case 0x700:	/* Rx. Note Off */
					get_userdrum(tmdy_struct, 64+udn, val[6])->rx_note_off = val[7];
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Rx. Note Off (CH:%d NOTE:%d VAL:%d)",dp,val[6],val[7]);
					if(TMDY_PLAYMIDI->channel[dp].drums[val[6]] == NULL) {TMDY_PLAYMIDI->play_midi_setup_drums(tmdy_struct, dp, val[6]);}
					TMDY_PLAYMIDI->channel[dp].drums[val[6]]->rx_note_off = val[7];
					break;
				case 0x800:	/* Rx. Note On */
					get_userdrum(tmdy_struct, 64+udn, val[6])->rx_note_on = val[7];
					break;
				case 0x900:	/* Delay Send Level */
					get_userdrum(tmdy_struct, 64+udn, val[6])->delay_send_level = val[7];
					SETMIDIEVENT(evm[0], 0,ME_NRPN_MSB,dp,0x1F,0);
					SETMIDIEVENT(evm[1], 0,ME_NRPN_LSB,dp,val[6],0);
					SETMIDIEVENT(evm[2], 0,ME_DATA_ENTRY_MSB,dp,val[7],0);
					num_events += 3;
					break;
				case 0xA00:	/* Source Map */
					get_userdrum(tmdy_struct, 64+udn, val[6])->source_map = val[7];
					break;
				case 0xB00:	/* Source Prog */
					get_userdrum(tmdy_struct, 64+udn, val[6])->source_prog = val[7];
					break;
#if !defined(TIMIDITY_TOOLS)
				case 0xC00:	/* Source Note */
					get_userdrum(tmdy_struct, 64+udn, val[6])->source_note = val[7];
					break;
#endif
				default:
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Unsupported GS SysEx. (ADDR:%02X %02X %02X VAL:%02X %02X)",addr_h,addr_m,addr_l,val[7],val[8]);
					break;
			}
			break;
		case 0x00:	/* System */
			switch (addr & 0xfff0) {
			case 0x0100:	/* Channel Msg Rx Port (A) */
				SETMIDIEVENT(evm[0], 0, ME_SYSEX_GS_LSB,
						block_to_part(tmdy_struct, addr & 0xf, 0), val[7], 0x46);
				num_events++;
				break;
			case 0x0110:	/* Channel Msg Rx Port (B) */
				SETMIDIEVENT(evm[0], 0, ME_SYSEX_GS_LSB,
						block_to_part(tmdy_struct, addr & 0xf, 1), val[7], 0x46);
				num_events++;
				break;
			default:
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY, "Unsupported GS SysEx. "
						"(ADDR:%02X %02X %02X VAL:%02X %02X)",
						addr_h, addr_m, addr_l, val[7], val[8]);
				break;
			}
			break;
		}
    }

	/* Non-RealTime / RealTime Universal SysEx messages
	 * 0 0x7e(Non-RealTime) / 0x7f(RealTime)
	 * 1 SysEx device ID.  Could be from 0x00 to 0x7f.
	 *   0x7f means disregard device.
	 * 2 Sub ID
	 * ...
	 * E 0xf7
	 */
	if (len > 4 && val[0] >= 0x7e)
		switch (val[2]) {
		case 0x01:	/* Sample Dump header */
		case 0x02:	/* Sample Dump packet */
		case 0x03:	/* Dump Request */
		case 0x05:	/* Sample Dump extensions */
		case 0x06:	/* Inquiry Message */
		case 0x07:	/* File Dump */
			break;
		case 0x08:	/* MIDI Tuning Standard */
			switch (val[3]) {
			case 0x01:
				SETMIDIEVENT(evm[0], 0, ME_BULK_TUNING_DUMP, 0, val[4], 0);
				for (i = 0; i < 128; i++) {
					SETMIDIEVENT(evm[i * 2 + 1], 0, ME_BULK_TUNING_DUMP,
							1, i, val[i * 3 + 21]);
					SETMIDIEVENT(evm[i * 2 + 2], 0, ME_BULK_TUNING_DUMP,
							2, val[i * 3 + 22], val[i * 3 + 23]);
				}
				num_events += 257;
				break;
			case 0x02:
				SETMIDIEVENT(evm[0], 0, ME_SINGLE_NOTE_TUNING,
						0, val[4], 0);
				for (i = 0; i < val[5]; i++) {
					SETMIDIEVENT(evm[i * 2 + 1], 0, ME_SINGLE_NOTE_TUNING,
							1, val[i * 4 + 6], val[i * 4 + 7]);
					SETMIDIEVENT(evm[i * 2 + 2], 0, ME_SINGLE_NOTE_TUNING,
							2, val[i * 4 + 8], val[i * 4 + 9]);
				}
				num_events += val[5] * 2 + 1;
				break;
			case 0x0b:
				channel_tt = ((val[4] & 0x03) << 14 | val[5] << 7 | val[6])
						<< ((val[4] >> 2) * 16);
				if (val[1] == 0x7f) {
					SETMIDIEVENT(evm[0], 0, ME_MASTER_TEMPER_TYPE,
							0, val[7], (val[0] == 0x7f));
					num_events++;
				} else {
					for (i = j = 0; i < MAX_CHANNELS; i++)
						if (channel_tt & 1 << i) {
							SETMIDIEVENT(evm[j], 0, ME_TEMPER_TYPE,
									MERGE_CHANNEL_PORT(i),
									val[7], (val[0] == 0x7f));
							j++;
						}
					num_events += j;
				}
				break;
			}
			break;
		case 0x09:	/* General MIDI Message */
		case 0x7b:	/* End of File */
		case 0x7c:	/* Handshaking Message: Wait */
		case 0x7d:	/* Handshaking Message: Cancel */
		case 0x7e:	/* Handshaking Message: NAK */
		case 0x7f:	/* Handshaking Message: ACK */
			break;
		}

    return(num_events);
}

int parse_sysex_event(tmdy_struct_ex_t *tmdy_struct, uint8 *val, int32 len, MidiEvent *ev)
{
	uint16 vol;
	
    if(TMDY_READMIDI->current_file_info->mid == 0 || TMDY_READMIDI->current_file_info->mid >= 0x7e)
	TMDY_READMIDI->current_file_info->mid = val[0];

    if(len >= 10 &&
       val[0] == 0x41 && /* Roland ID */
       val[1] == 0x10 && /* Device ID */
       val[2] == 0x42 && /* GS Model ID */
       val[3] == 0x12)   /* Data Set Command */
    {
	/* Roland GS-Based Synthesizers.
	 * val[4..6] is address, val[7..len-2] is body.
	 *
	 * GS     Channel part number
	 * 0      10
	 * 1-9    1-9
	 * 10-15  11-16
	 */

	int32 addr,checksum,i;		/* SysEx address */
	uint8 *body;		/* SysEx body */
	uint8 p,gslen;		/* Channel part number [0..15] */

	/* check Checksum */
	checksum = 0;
	for(gslen = 9; gslen < len; gslen++)
		if(val[gslen] == 0xF7)
			break;
	for(i=4;i<gslen-1;i++) {
		checksum += val[i];
	}
	if(((128 - (checksum & 0x7F)) & 0x7F) != val[gslen-1]) {
		return 0;
	}

	addr = (((int32)val[4])<<16 |
		((int32)val[5])<<8 |
		(int32)val[6]);
	body = val + 7;
	p = (uint8)((addr >> 8) & 0xF);
	if(p == 0)
	    p = 9;
	else if(p <= 9)
	    p--;
	p = MERGE_CHANNEL_PORT(p);

	if(val[4] == 0x50) {	/* for double module mode */
		p += 16;
		addr = (((int32)0x40)<<16 |
			((int32)val[5])<<8 |
			(int32)val[6]);
	} else {	/* single module mode */
		addr = (((int32)val[4])<<16 |
			((int32)val[5])<<8 |
			(int32)val[6]);
	}

	if((addr & 0xFFF0FF) == 0x401015) /* Rhythm Parts */
	{
#ifdef GS_DRUMPART
/* GS drum part check from Masaaki Koyanagi's patch (GS_Drum_Part_Check()) */
/* Modified by Masanao Izumo */
	    SETMIDIEVENT(*ev, 0, ME_DRUMPART, p, *body, 0);
	    return 1;
#else
	    return 0;
#endif /* GS_DRUMPART */
	}

	if((addr & 0xFFF0FF) == 0x401016) /* Key Shift */
	{
	    SETMIDIEVENT(*ev, 0, ME_KEYSHIFT, p, *body, 0);
	    return 1;
	}

	if(addr == 0x400004) /* Master Volume */
	{
	    vol = gs_convert_master_vol(tmdy_struct, *body);
	    SETMIDIEVENT(*ev, 0, ME_MASTER_VOLUME,
			 0, vol & 0xFF, (vol >> 8) & 0xFF);
	    return 1;
	}

	if((addr & 0xFFF0FF) == 0x401019) /* Volume on/off */
	{
#if 0
	    SETMIDIEVENT(*ev, 0, ME_VOLUME_ONOFF, p, *body >= 64, 0);
#endif
	    return 0;
	}

	if((addr & 0xFFF0FF) == 0x401002) /* Receive channel on/off */
	{
#if 0
	    SETMIDIEVENT(*ev, 0, ME_RECEIVE_CHANNEL, (uint8)p, *body >= 64, 0);
#endif
	    return 0;
	}

	if(0x402000 <= addr && addr <= 0x402F5A) /* Controller Routing */
	    return 0;

	if((addr & 0xFFF0FF) == 0x401040) /* Alternate Scale Tunings */
	    return 0;

	if((addr & 0xFFFFF0) == 0x400130) /* Changing Effects */
	{
	    switch(addr & 0xF)
	    {
	      case 0x8: /* macro */
		memcpy(TMDY_REVERB->chorus_status.macro, body, 3);
		break;
	      case 0x9: /* PRE-LPF */
		memcpy(TMDY_REVERB->chorus_status.pre_lpf, body, 3);
		break;
	      case 0xa: /* level */
		memcpy(TMDY_REVERB->chorus_status.level, body, 3);
		break;
	      case 0xb: /* feed back */
		memcpy(TMDY_REVERB->chorus_status.feed_back, body, 3);
		break;
	      case 0xc: /* delay */
		memcpy(TMDY_REVERB->chorus_status.delay, body, 3);
		break;
	      case 0xd: /* rate */
		memcpy(TMDY_REVERB->chorus_status.rate, body, 3);
		break;
	      case 0xe: /* depth */
		memcpy(TMDY_REVERB->chorus_status.depth, body, 3);
		break;
	      case 0xf: /* send level */
		memcpy(TMDY_REVERB->chorus_status.send_level, body, 3);
		break;
		  default: break;
	    }

	    check_chorus_text_start(tmdy_struct);
	    return 0;
	}

	if((addr & 0xFFF0FF) == 0x401003) /* Rx Pitch-Bend */
	    return 0;

	if(addr == 0x400110) /* Voice Reserve */
	{
	    if(len >= 25)
		memcpy(TMDY_REVERB->chorus_status.voice_reserve, body, 18);
	    check_chorus_text_start(tmdy_struct);
	    return 0;
	}

	if(addr == 0x40007F ||	/* GS Reset */
	   addr == 0x00007F)	/* SC-88 Single Module */
	{
	    SETMIDIEVENT(*ev, 0, ME_RESET, 0, GS_SYSTEM_MODE, 0);
	    return 1;
	}
	return 0;
    }

    if(len > 9 &&
       val[0] == 0x41 && /* Roland ID */
       val[1] == 0x10 && /* Device ID */
       val[2] == 0x45 && 
       val[3] == 0x12 && 
       val[4] == 0x10 && 
       val[5] == 0x00 && 
       val[6] == 0x00)
    {
	/* Text Insert for SC */
	uint8 save;

	len -= 2;
	save = val[len];
	val[len] = '\0';
	if(readmidi_make_string_event(tmdy_struct, ME_INSERT_TEXT, (char *)val + 7, ev, 1))
	{
	    val[len] = save;
	    return 1;
	}
	val[len] = save;
	return 0;
    }

    if(len > 9 &&                     /* GS lcd event. by T.Nogami*/
       val[0] == 0x41 && /* Roland ID */
       val[1] == 0x10 && /* Device ID */
       val[2] == 0x45 && 
       val[3] == 0x12 && 
       val[4] == 0x10 && 
       val[5] == 0x01 && 
       val[6] == 0x00)
    {
	/* Text Insert for SC */
	uint8 save;

	len -= 2;
	save = val[len];
	val[len] = '\0';
	if(readmidi_make_lcd_event(tmdy_struct, ME_GSLCD, (uint8 *)val + 7, ev))
	{
	    val[len] = save;
	    return 1;
	}
	val[len] = save;
	return 0;
    }
    
    if(len >= 8 &&
       val[0] == 0x43 &&
       val[1] == 0x10 &&
       val[2] == 0x4C &&
       val[3] == 0x00 &&
       val[4] == 0x00 &&
       val[5] == 0x7E)
    {
	/* XG SYSTEM ON */
	SETMIDIEVENT(*ev, 0, ME_RESET, 0, XG_SYSTEM_MODE, 0);
	return 1;
    }

	/* Non-RealTime / RealTime Universal SysEx messages
	 * 0 0x7e(Non-RealTime) / 0x7f(RealTime)
	 * 1 SysEx device ID.  Could be from 0x00 to 0x7f.
	 *   0x7f means disregard device.
	 * 2 Sub ID
	 * ...
	 * E 0xf7
	 */
	if (len > 4 && val[0] >= 0x7e)
		switch (val[2]) {
		case 0x01:	/* Sample Dump header */
		case 0x02:	/* Sample Dump packet */
		case 0x03:	/* Dump Request */
			break;
		case 0x04:	/* MIDI Time Code Setup/Device Control */
			switch (val[3]) {
			case 0x01:	/* Master Volume */
				vol = gm_convert_master_vol(tmdy_struct, val[4], val[5]);
				if (val[1] == 0x7f) {
					SETMIDIEVENT(*ev, 0, ME_MASTER_VOLUME, 0,
							vol & 0xff, vol >> 8 & 0xff);
				} else {
					SETMIDIEVENT(*ev, 0, ME_MAINVOLUME,
							MERGE_CHANNEL_PORT(val[1]),
							vol >> 8 & 0xff, 0);
				}
				return 1;
			}
			break;
		case 0x05:	/* Sample Dump extensions */
		case 0x06:	/* Inquiry Message */
		case 0x07:	/* File Dump */
			break;
		case 0x08:	/* MIDI Tuning Standard */
			switch (val[3]) {
			case 0x0a:
				SETMIDIEVENT(*ev, 0, ME_TEMPER_KEYSIG, 0,
						val[4] - 0x40 + val[5] * 16, (val[0] == 0x7f));
				return 1;
			}
			break;
		case 0x09:	/* General MIDI Message */
			/* GM System Enable/Disable */
			if(val[3]) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "SysEx: GM System Enable");
				SETMIDIEVENT(*ev, 0, ME_RESET, 0, GM_SYSTEM_MODE, 0);
			} else {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "SysEx: GM System Disable");
				SETMIDIEVENT(*ev, 0, ME_RESET, 0, DEFAULT_SYSTEM_MODE, 0);
			}
			return 1;
		case 0x7b:	/* End of File */
		case 0x7c:	/* Handshaking Message: Wait */
		case 0x7d:	/* Handshaking Message: Cancel */
		case 0x7e:	/* Handshaking Message: NAK */
		case 0x7f:	/* Handshaking Message: ACK */
			break;
		}

    return 0;
}

static int read_sysex_event(tmdy_struct_ex_t *tmdy_struct, int32 at, int me, int32 len,
			    struct timidity_file *tf)
{
    uint8 *val;
    MidiEvent ev, evm[260]; /* maximum number of XG bulk dump events */
    int ne, i;

    if(len == 0)
	return 0;
    if(me != 0xF0)
    {
	TMDY_COMMON->skip(tmdy_struct, tf, len);
	return 0;
    }

    val = (uint8 *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len);
    if(TMDY_COMMON->tf_read(tmdy_struct, val, 1, len, tf) != len)
    {
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	return -1;
    }
    if(parse_sysex_event(tmdy_struct, val, len, &ev))
    {
	ev.time = at;
	readmidi_add_event(tmdy_struct, &ev);
    }
    if ((ne = parse_sysex_event_multi(tmdy_struct, val, len, evm)))
    {
	for (i = 0; i < ne; i++) {
	    evm[i].time = at;
	    readmidi_add_event(tmdy_struct, &evm[i]);
	}
    }
    
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));

    return 0;
}

static char *fix_string(tmdy_struct_ex_t *tmdy_struct, char *s)
{
    int i, j, w;
    char c;

    if(s == NULL)
	return NULL;
    while(*s == ' ' || *s == '\t' || *s == '\r' || *s == '\n')
	s++;

    /* s =~ tr/ \t\r\n/ /s; */
    w = 0;
    for(i = j = 0; (c = s[i]) != '\0'; i++)
    {
	if(c == '\t' || c == '\r' || c == '\n')
	    c = ' ';
	if(w)
	    w = (c == ' ');
	if(!w)
	{
	    s[j++] = c;
	    w = (c == ' ');
	}
    }

    /* s =~ s/ $//; */
    if(j > 0 && s[j - 1] == ' ')
	j--;

    s[j] = '\0';
    return s;
}

static void smf_time_signature(tmdy_struct_ex_t *tmdy_struct, int32 at, struct timidity_file *tf, int len)
{
    int n, d, c, b;

    /* Time Signature (nn dd cc bb)
     * [0]: numerator
     * [1]: denominator
     * [2]: number of MIDI clocks in a metronome click
     * [3]: number of notated 32nd-notes in a MIDI
     *      quarter-note (24 MIDI Clocks).
     */

    if(len != 4)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE, "Invalid time signature");
	TMDY_COMMON->skip(tmdy_struct, tf, len);
	return;
    }

    n = tf_getc(tf);
    d = (1<<tf_getc(tf));
    c = tf_getc(tf);
    b = tf_getc(tf);

    if(n == 0 || d == 0)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE, "Invalid time signature");
	return;
    }

    MIDIEVENT(at, ME_TIMESIG, 0, n, d);
    MIDIEVENT(at, ME_TIMESIG, 1, c, b);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
	      "Time signature: %d/%d %d clock %d q.n.", n, d, c, b);
    if(TMDY_READMIDI->current_file_info->time_sig_n == -1)
    {
	TMDY_READMIDI->current_file_info->time_sig_n = n;
	TMDY_READMIDI->current_file_info->time_sig_d = d;
	TMDY_READMIDI->current_file_info->time_sig_c = c;
	TMDY_READMIDI->current_file_info->time_sig_b = b;
    }
}

static void smf_key_signature(tmdy_struct_ex_t *tmdy_struct, int32 at, struct timidity_file *tf, int len)
{
	int8 sf, mi;
	/* Key Signature (sf mi)
	 * sf = -7:  7 flats
	 * sf = -1:  1 flat
	 * sf = 0:   key of C
	 * sf = 1:   1 sharp
	 * sf = 7:   7 sharps
	 * mi = 0:  major key
	 * mi = 1:  minor key
	 */
	
	if (len != 2) {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE, "Invalid key signature");
		TMDY_COMMON->skip(tmdy_struct, tf, len);
		return;
	}
	sf = tf_getc(tf);
	mi = tf_getc(tf);
	if (sf < -7 || sf > 7) {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE, "Invalid key signature");
		return;
	}
	if (mi != 0 && mi != 1) {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE, "Invalid key signature");
		return;
	}
	MIDIEVENT(at, ME_KEYSIG, 0, sf, mi);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Key signature: %d %s %s", abs(sf),
			(sf < 0) ? "flat(s)" : "sharp(s)", (mi) ? "minor" : "major");
}

/* Used for WRD reader */
int dump_current_timesig(tmdy_struct_ex_t *tmdy_struct, MidiEvent *codes, int maxlen)
{
    int i, n;
    MidiEventList *e;

    if(maxlen <= 0 || TMDY_READMIDI->evlist == NULL)
	return 0;
    n = 0;
    for(i = 0, e = TMDY_READMIDI->evlist; i < TMDY_READMIDI->event_count; i++, e = e->next)
	if(e->event.type == ME_TIMESIG && e->event.channel == 0)
	{
	    if(n == 0 && e->event.time > 0)
	    {
		/* 4/4 is default */
		SETMIDIEVENT(codes[0], 0, ME_TIMESIG, 0, 4, 4);
		n++;
		if(maxlen == 1)
		    return 1;
	    }

	    if(n > 0)
	    {
		if(e->event.a == codes[n - 1].a &&
		   e->event.b == codes[n - 1].b)
		    continue; /* Unchanged */
		if(e->event.time == codes[n - 1].time)
		    n--; /* overwrite previous code */
	    }
	    codes[n++] = e->event;
	    if(n == maxlen)
		return n;
	}
    return n;
}

/* Read a SMF track */
static int read_smf_track(tmdy_struct_ex_t *tmdy_struct, struct timidity_file *tf, int trackno, int rewindp)
{
    int32 len, next_pos, pos;
    char tmp[4];
    int lastchan, laststatus;
    int me, type, a, b, c;
    int i;
    int32 smf_at_time;

    smf_at_time = readmidi_set_track(tmdy_struct, trackno, rewindp);

    /* Check the formalities */
    if((TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4) || (TMDY_COMMON->tf_read(tmdy_struct, &len, 4, 1, tf) != 1))
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: Can't read track header.", (TMDY_COMMON->current_filename));
	return -1;
    }
    len = BE_LONG(len);
    next_pos = TMDY_COMMON->tf_tell(tmdy_struct, tf) + len;
    if(strncmp(tmp, "MTrk", 4))
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: Corrupt MIDI file.", (TMDY_COMMON->current_filename));
	return -2;
    }

    lastchan = laststatus = 0;

    for(;;)
    {
	if(TMDY_READMIDI->readmidi_error_flag)
	    return -1;
	if((len = getvl(tmdy_struct, tf)) < 0)
	    return -1;
	smf_at_time += len;
	errno = 0;
	if((i = tf_getc(tf)) == EOF)
	{
	    if(errno)
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: read_midi_event: %s",
			  (TMDY_COMMON->current_filename), strerror(errno));
	    else
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "Warning: %s: Too shorten midi file.",
			  (TMDY_COMMON->current_filename));
	    return -1;
	}

	me = (uint8)i;
	if(me == 0xF0 || me == 0xF7) /* SysEx event */
	{
	    if((len = getvl(tmdy_struct, tf)) < 0)
		return -1;
	    if((i = read_sysex_event(tmdy_struct, smf_at_time, me, len, tf)) != 0)
		return i;
	}
	else if(me == 0xFF) /* Meta event */
	{
	    type = tf_getc(tf);
	    if((len = getvl(tmdy_struct, tf)) < 0)
		return -1;
	    if(type > 0 && type < 16)
	    {
		static char *label[] =
		{
		    "Text event: ", "Text: ", "Copyright: ", "Track name: ",
		    "Instrument: ", "Lyric: ", "Marker: ", "Cue point: "
		};

		if(type == 5 || /* Lyric */
		   (type == 1 && (TMDY_READMIDI->opt_trace_text_meta_event ||
				  TMDY_READMIDI->karaoke_format == 2 ||
				  TMDY_REVERB->chorus_status.status == CHORUS_ST_OK)) ||
		   (type == 6 &&  (TMDY_READMIDI->current_file_info->format == 0 ||
				   (TMDY_READMIDI->current_file_info->format == 1 &&
				    TMDY_READMIDI->current_read_track == 0))))
		{
		    char *str, *text;
		    MidiEvent ev;

		    str = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len + 3);
		    if(type != 6)
		    {
			i = TMDY_COMMON->tf_read(tmdy_struct, str, 1, len, tf);
			str[len] = '\0';
		    }
		    else
		    {
			i = TMDY_COMMON->tf_read(tmdy_struct, str + 1, 1, len, tf);
			str[0] = MARKER_START_CHAR;
			str[len + 1] = MARKER_END_CHAR;
			str[len + 2] = '\0';
		    }

		    if(i != len)
		    {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				  "Warning: %s: Too shorten midi file.",
				  (TMDY_COMMON->current_filename));
			TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
			return -1;
		    }

		    if((text = readmidi_make_string_event(tmdy_struct, 1, str, &ev, 1))
		       == NULL)
		    {
			TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
			continue;
		    }
		    ev.time = smf_at_time;

		    if(type == 6)
		    {
			if(strlen(fix_string(tmdy_struct, text + 1)) == 2)
			{
			    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
			    continue; /* Empty Marker */
			}
		    }

		    switch(type)
		    {
		      case 1:
			if(TMDY_READMIDI->karaoke_format == 2)
			{
			    *text = ME_KARAOKE_LYRIC;
			    if(TMDY_READMIDI->karaoke_title_flag == 0 &&
			       strncmp(str, "@T", 2) == 0)
				TMDY_READMIDI->current_file_info->karaoke_title =
				    add_karaoke_title(tmdy_struct, TMDY_READMIDI->current_file_info->
						      karaoke_title, str + 2);
			    ev.type = ME_KARAOKE_LYRIC;
			    readmidi_add_event(tmdy_struct, &ev);
			    continue;
			}
			if(TMDY_REVERB->chorus_status.status == CHORUS_ST_OK)
			{
			    *text = ME_CHORUS_TEXT;
			    ev.type = ME_CHORUS_TEXT;
			    readmidi_add_event(tmdy_struct, &ev);
			    continue;
			}
			*text = ME_TEXT;
			ev.type = ME_TEXT;
			readmidi_add_event(tmdy_struct, &ev);
			continue;
		      case 5:
			*text = ME_LYRIC;
			ev.type = ME_LYRIC;
			readmidi_add_event(tmdy_struct, &ev);
			continue;
		      case 6:
			*text = ME_MARKER;
			ev.type = ME_MARKER;
			readmidi_add_event(tmdy_struct, &ev);
			continue;
		    }
		}

		if(type == 3 && /* Sequence or Track Name */
		   (TMDY_READMIDI->current_file_info->format == 0 ||
		    (TMDY_READMIDI->current_file_info->format == 1 &&
		     TMDY_READMIDI->current_read_track == 0)))
		{
		  if(TMDY_READMIDI->current_file_info->seq_name == NULL) {
		    char *name = dumpstring(tmdy_struct, 3, len, "Sequence: ", 1, tf);
		    TMDY_READMIDI->current_file_info->seq_name = TMDY_COMMON->safe_strdup(tmdy_struct, fix_string(tmdy_struct, name));
		    free(name);
		  }
		    else
			dumpstring(tmdy_struct, 3, len, "Sequence: ", 0, tf);
		}
		else if(type == 1 &&
			TMDY_READMIDI->current_file_info->first_text == NULL &&
			(TMDY_READMIDI->current_file_info->format == 0 ||
			 (TMDY_READMIDI->current_file_info->format == 1 &&
			  TMDY_READMIDI->current_read_track == 0))) {
		  char *name = dumpstring(tmdy_struct, 1, len, "Text: ", 1, tf);
		  TMDY_READMIDI->current_file_info->first_text = TMDY_COMMON->safe_strdup(tmdy_struct, fix_string(tmdy_struct, name));
		  free(name);
		}
		else
		    dumpstring(tmdy_struct, type, len, label[(type>7) ? 0 : type], 0, tf);
	    }
	    else
	    {
		switch(type)
		{
		  case 0x00:
		    if(len == 2)
		    {
			a = tf_getc(tf);
			b = tf_getc(tf);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				  "(Sequence Number %02x %02x)", a, b);
		    }
		    else
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				  "(Sequence Number len=%d)", len);
		    break;

		  case 0x2F: /* End of Track */
		    pos = TMDY_COMMON->tf_tell(tmdy_struct, tf);
		    if(pos < next_pos)
			TMDY_COMMON->tf_seek(tmdy_struct, tf, next_pos - pos, SEEK_CUR);
		    return 0;

		  case 0x51: /* Tempo */
		    a = tf_getc(tf);
		    b = tf_getc(tf);
		    c = tf_getc(tf);
		    MIDIEVENT(smf_at_time, ME_TEMPO, c, a, b);
		    break;

		  case 0x54:
		    /* SMPTE Offset (hr mn se fr ff)
		     * hr: hours&type
		     *     0     1     2     3    4    5    6    7   bits
		     *     0  |<--type -->|<---- hours [0..23]---->|
		     * type: 00: 24 frames/second
		     *       01: 25 frames/second
		     *       10: 30 frames/second (drop frame)
		     *       11: 30 frames/second (non-drop frame)
		     * mn: minis [0..59]
		     * se: seconds [0..59]
		     * fr: frames [0..29]
		     * ff: fractional frames [0..99]
		     */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(SMPTE Offset meta event)");
		    TMDY_COMMON->skip(tmdy_struct, tf, len);
		    break;

		  case 0x58: /* Time Signature */
		    smf_time_signature(tmdy_struct, smf_at_time, tf, len);
		    break;

		  case 0x59: /* Key Signature */
		    smf_key_signature(tmdy_struct, smf_at_time, tf, len);
		    break;

		  case 0x7f: /* Sequencer-Specific Meta-Event */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sequencer-Specific meta event, length %ld)",
			      len);
		    TMDY_COMMON->skip(tmdy_struct, tf, len);
		    break;

		  case 0x20: /* MIDI channel prefix (SMF v1.0) */
		    if(len == 1)
		    {
			int midi_channel_prefix = tf_getc(tf);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				  "(MIDI channel prefix %d)",
				  midi_channel_prefix);
		    }
		    else
			TMDY_COMMON->skip(tmdy_struct, tf, len);
		    break;

		  case 0x21: /* MIDI port number */
		    if(len == 1)
		    {
			if((TMDY_READMIDI->midi_port_number = tf_getc(tf))
			   == EOF)
			{
			    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				      "Warning: %s: Too shorten midi file.",
				      (TMDY_COMMON->current_filename));
			    return -1;
			}
			TMDY_READMIDI->midi_port_number &= 0xF;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				  "(MIDI port number %d)", TMDY_READMIDI->midi_port_number);
		    }
		    else
			TMDY_COMMON->skip(tmdy_struct, tf, len);
		    break;

		  default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Meta event type 0x%02x, length %ld)",
			      type, len);
		    TMDY_COMMON->skip(tmdy_struct, tf, len);
		    break;
		}
	    }
	}
	else /* MIDI event */
	{
	    a = me;
	    if(a & 0x80) /* status byte */
	    {
		lastchan = MERGE_CHANNEL_PORT(a & 0x0F);
		laststatus = (a >> 4) & 0x07;
		if(laststatus != 7)
		    a = tf_getc(tf) & 0x7F;
	    }
	    switch(laststatus)
	    {
	      case 0: /* Note off */
		b = tf_getc(tf) & 0x7F;
		MIDIEVENT(smf_at_time, ME_NOTEOFF, lastchan, a,b);
		break;

	      case 1: /* Note on */
		b = tf_getc(tf) & 0x7F;
		if(b)
		{
		    MIDIEVENT(smf_at_time, ME_NOTEON, lastchan, a,b);
		}
		else /* b == 0 means Note Off */
		{
		    MIDIEVENT(smf_at_time, ME_NOTEOFF, lastchan, a, 0);
		}
		break;

	      case 2: /* Key Pressure */
		b = tf_getc(tf) & 0x7F;
		MIDIEVENT(smf_at_time, ME_KEYPRESSURE, lastchan, a, b);
		break;

	      case 3: /* Control change */
		b = tf_getc(tf);
		readmidi_add_ctl_event(tmdy_struct, smf_at_time, lastchan, a, b);
		break;

	      case 4: /* Program change */
		MIDIEVENT(smf_at_time, ME_PROGRAM, lastchan, a, 0);
		break;

	      case 5: /* Channel pressure */
		MIDIEVENT(smf_at_time, ME_CHANNEL_PRESSURE, lastchan, a, 0);
		break;

	      case 6: /* Pitch wheel */
		b = tf_getc(tf) & 0x7F;
		MIDIEVENT(smf_at_time, ME_PITCHWHEEL, lastchan, a, b);
		break;

	      default: /* case 7: */
		/* Ignore this event */
		switch(lastchan & 0xF)
		{
		  case 2: /* Sys Com Song Position Pntr */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys Com Song Position Pntr)");
		    tf_getc(tf);
		    tf_getc(tf);
		    break;

		  case 3: /* Sys Com Song Select(Song #) */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys Com Song Select(Song #))");
		    tf_getc(tf);
		    break;

		  case 6: /* Sys Com tune request */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys Com tune request)");
		    break;
		  case 8: /* Sys real time timing clock */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys real time timing clock)");
		    break;
		  case 10: /* Sys real time start */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys real time start)");
		    break;
		  case 11: /* Sys real time continue */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys real time continue)");
		    break;
		  case 12: /* Sys real time stop */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys real time stop)");
		    break;
		  case 14: /* Sys real time active sensing */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(Sys real time active sensing)");
		    break;
#if 0
		  case 15: /* Meta */
		  case 0: /* SysEx */
		  case 7: /* SysEx */
#endif
		  default: /* 1, 4, 5, 9, 13 */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "*** Can't happen: status 0x%02X channel 0x%02X",
			      laststatus, lastchan & 0xF);
		    break;
		}
		}
	}
    }
    /*NOTREACHED*/
}

/* Free the linked event list from memory. */
static void free_midi_list(tmdy_struct_ex_t *tmdy_struct)
{
    if(TMDY_READMIDI->evlist != NULL)
    {
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_READMIDI->mempool);
	TMDY_READMIDI->evlist = NULL;
    }
}

static void move_channels(tmdy_struct_ex_t *tmdy_struct, int *chidx)
{
	int i, ch, maxch, newch;
	MidiEventList *e;
	
	for (i = 0; i < 256; i++)
		chidx[i] = -1;
	/* check channels */
	for (i = maxch = 0, e = TMDY_READMIDI->evlist; i < TMDY_READMIDI->event_count; i++, e = e->next)
		if (! GLOBAL_CHANNEL_EVENT_TYPE(e->event.type)) {
			if ((ch = e->event.channel) < REDUCE_CHANNELS)
				chidx[ch] = ch;
			if (maxch < ch)
				maxch = ch;
		}
	if (maxch >= REDUCE_CHANNELS)
		/* Move channel if enable */
		for (i = maxch = 0, e = TMDY_READMIDI->evlist; i < TMDY_READMIDI->event_count; i++, e = e->next)
			if (! GLOBAL_CHANNEL_EVENT_TYPE(e->event.type)) {
				if (chidx[ch = e->event.channel] != -1)
					ch = e->event.channel = chidx[ch];
				else {	/* -1 */
					newch = ch % REDUCE_CHANNELS;
					while (newch < ch && newch < MAX_CHANNELS) {
						if (chidx[newch] == -1) {
							(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
									"channel %d => %d", ch, newch);
							ch = e->event.channel = chidx[ch] = newch;
							break;
						}
						newch += REDUCE_CHANNELS;
					}
					if (chidx[ch] == -1) {
						if (ch < MAX_CHANNELS)
							chidx[ch] = ch;
						else {
							newch = ch % MAX_CHANNELS;
							(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
									"channel %d => %d (mixed)", ch, newch);
							ch = e->event.channel = chidx[ch] = newch;
						}
					}
				}
				if (maxch < ch)
					maxch = ch;
			}
	for (i = 0, e = TMDY_READMIDI->evlist; i < TMDY_READMIDI->event_count; i++, e = e->next)
		if (e->event.type == ME_SYSEX_GS_LSB) {
			if (e->event.b == 0x45 || e->event.b == 0x46)
				if (maxch < e->event.channel)
					maxch = e->event.channel;
		} else if (e->event.type == ME_SYSEX_XG_LSB) {
			if (e->event.b == 0x65)
				if (maxch < e->event.channel)
					maxch = e->event.channel;
		}
	TMDY_READMIDI->current_file_info->max_channel = maxch;
}

void change_system_mode(tmdy_struct_ex_t *tmdy_struct, int mode)
{
    int mid;

    if(TMDY_READMIDI->opt_system_mid)
    {
	mid = TMDY_READMIDI->opt_system_mid;
	mode = -1; /* Always use opt_system_mid */
    }
    else
	mid = TMDY_READMIDI->current_file_info->mid;
    switch(mode)
    {
      case GM_SYSTEM_MODE:
	if(TMDY_READMIDI->play_system_mode == DEFAULT_SYSTEM_MODE)
	{
	    TMDY_READMIDI->play_system_mode = GM_SYSTEM_MODE;
	    vol_table = def_vol_table;
	}
	break;
      case GS_SYSTEM_MODE:
	TMDY_READMIDI->play_system_mode = GS_SYSTEM_MODE;
	vol_table = gs_vol_table;
	break;
      case XG_SYSTEM_MODE:
	TMDY_READMIDI->play_system_mode = XG_SYSTEM_MODE;
	vol_table = xg_vol_table;
	break;
      default:
	switch(mid)
	{
	  case 0x41:
	    TMDY_READMIDI->play_system_mode = GS_SYSTEM_MODE;
	    vol_table = gs_vol_table;
	    break;
	  case 0x43:
	    TMDY_READMIDI->play_system_mode = XG_SYSTEM_MODE;
	    vol_table = xg_vol_table;
	    break;
	  case 0x7e:
	    TMDY_READMIDI->play_system_mode = GM_SYSTEM_MODE;
	    vol_table = def_vol_table;
	    break;
	  default:
	    TMDY_READMIDI->play_system_mode = DEFAULT_SYSTEM_MODE;
		vol_table = def_vol_table;
	    break;
	}
	break;
    }
}

int get_default_mapID(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE)
	return ISDRUMCHANNEL(ch) ? XG_DRUM_MAP : XG_NORMAL_MAP;
    return INST_NO_MAP;
}

/* Allocate an array of MidiEvents and fill it from the linked list of
   events, marking used instruments for loading. Convert event times to
   samples: handle tempo changes. Strip unnecessary events from the list.
   Free the linked list. */
static MidiEvent *groom_list(tmdy_struct_ex_t *tmdy_struct, int32 divisions, int32 *eventsp, int32 *samplesp)
{
    MidiEvent *groomed_list, *lp;
    MidiEventList *meep;
    int32 i, j, our_event_count, tempo, skip_this_event;
    int32 sample_cum, samples_to_do, at, st, dt, counting_time;
    int ch, gch;
    uint8 current_set[MAX_CHANNELS],
	warn_tonebank[128], warn_drumset[128];
    int8 bank_lsb[MAX_CHANNELS], bank_msb[MAX_CHANNELS], mapID[MAX_CHANNELS];
    int current_program[MAX_CHANNELS];
    int wrd_args[WRD_MAXPARAM];
    int wrd_argc;
    int chidx[256];
    int newbank, newprog;

    move_channels(tmdy_struct, chidx);

    COPY_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, TMDY_READMIDI->current_file_info->drumchannels);
    COPY_CHANNELMASK(TMDY_PLAYMIDI->drumchannel_mask, TMDY_READMIDI->current_file_info->drumchannel_mask);

    /* Move drumchannels */
    for(ch = REDUCE_CHANNELS; ch < MAX_CHANNELS; ch++)
    {
	i = chidx[ch];
	if(i != -1 && i != ch && !IS_SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannel_mask, i))
	{
	    if(IS_SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, ch))
		SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, i);
	    else
		UNSET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, i);
	}
    }

    memset(warn_tonebank, 0, sizeof(warn_tonebank));
    if (TMDY_PLAYMIDI->special_tonebank >= 0)
	newbank = TMDY_PLAYMIDI->special_tonebank;
    else
	newbank = TMDY_PLAYMIDI->default_tonebank;
    for(j = 0; j < MAX_CHANNELS; j++)
    {
	if(ISDRUMCHANNEL(j))
	    current_set[j] = 0;
	else
	{
	    if (TMDY_INSTRUM->tonebank[newbank] == NULL)
	    {
		if (warn_tonebank[newbank] == 0)
		{
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
			      "Tone bank %d is undefined", newbank);
		    warn_tonebank[newbank] = 1;
		}
		newbank = 0;
	    }
	    current_set[j] = newbank;
	}
	bank_lsb[j] = bank_msb[j] = 0;
	if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE && j % 16 == 9)
	    bank_msb[j] = 127; /* Use MSB=127 for XG */
	current_program[j] = TMDY_INSTRUM->default_program[j];
    }

    memset(warn_drumset, 0, sizeof(warn_drumset));
    tempo = 500000;
    compute_sample_increment(tmdy_struct, tempo, divisions);

    /* This may allocate a bit more than we need */
    groomed_list = lp =
	(MidiEvent *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(MidiEvent) * (TMDY_READMIDI->event_count + 1));
    meep = TMDY_READMIDI->evlist;

    our_event_count = 0;
    st = at = sample_cum = 0;
    counting_time = 2; /* We strip any silence before the first NOTE ON. */
    wrd_argc = 0;
    change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);

    for(j = 0; j < MAX_CHANNELS; j++)
	mapID[j] = get_default_mapID(tmdy_struct, j);

    for(i = 0; i < TMDY_READMIDI->event_count; i++)
    {
	skip_this_event = 0;
	ch = meep->event.channel;
	gch = GLOBAL_CHANNEL_EVENT_TYPE(meep->event.type);
	if(!gch && ch >= MAX_CHANNELS) /* For safety */
	    meep->event.channel = ch = ch % MAX_CHANNELS;

	if(!gch && IS_SET_CHANNELMASK(TMDY_READMIDI->quietchannels, ch))
	    skip_this_event = 1;
	else switch(meep->event.type)
	{
	  case ME_NONE:
	    skip_this_event = 1;
	    break;
	  case ME_RESET:
	    change_system_mode(tmdy_struct, meep->event.a);
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "MIDI reset at %d sec",
		      (int)((double)st / (TMDY_OUTPUT->play_mode)->rate + 0.5));
	    for(j = 0; j < MAX_CHANNELS; j++)
	    {
		if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE && j % 16 == 9)
		    mapID[j] = XG_DRUM_MAP;
		else
		    mapID[j] = get_default_mapID(tmdy_struct, j);
		if(ISDRUMCHANNEL(j))
		    current_set[j] = 0;
		else
		{
		    if(TMDY_PLAYMIDI->special_tonebank >= 0)
			current_set[j] = TMDY_PLAYMIDI->special_tonebank;
		    else
			current_set[j] = TMDY_PLAYMIDI->default_tonebank;
		    if(TMDY_INSTRUM->tonebank[current_set[j]] == NULL)
			current_set[j] = 0;
		}
		bank_lsb[j] = bank_msb[j] = 0;
		if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE && j % 16 == 9)
		{
		    bank_msb[j] = 127; /* Use MSB=127 for XG */
		}
		current_program[j] = TMDY_INSTRUM->default_program[j];
	    }
	    break;

	  case ME_PROGRAM:
	    if(ISDRUMCHANNEL(ch))
		newbank = current_program[ch];
	    else
		newbank = current_set[ch];
	    newprog = meep->event.a;
	    switch(TMDY_READMIDI->play_system_mode)
	    {
	      case GS_SYSTEM_MODE:
		switch(bank_lsb[ch])
		{
		  case 0:	/* No change */
		    break;
		  case 1:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(GS ch=%d SC-55 MAP)", ch);
		    mapID[ch] = (!ISDRUMCHANNEL(ch) ? SC_55_TONE_MAP
				 : SC_55_DRUM_MAP);
		    break;
		  case 2:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(GS ch=%d SC-88 MAP)", ch);
		    mapID[ch] = (!ISDRUMCHANNEL(ch) ? SC_88_TONE_MAP
				 : SC_88_DRUM_MAP);
		    break;
		  case 3:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(GS ch=%d SC-88Pro MAP)", ch);
		    mapID[ch] = (!ISDRUMCHANNEL(ch) ? SC_88PRO_TONE_MAP
				 : SC_88PRO_DRUM_MAP);
		    break;
		  default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(GS: ch=%d Strange bank LSB %d)",
			      ch, bank_lsb[ch]);
		    break;
		}
		newbank = bank_msb[ch];
		break;

	      case XG_SYSTEM_MODE: /* XG */
		switch(bank_msb[ch])
		{
		  case 0: /* Normal */
		    if(ch == 9  && bank_lsb[ch] == 127 && mapID[ch] == XG_DRUM_MAP) {
		      /* FIXME: Why this part is drum?  Is this correct? */
		      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
				"Warning: XG bank 0/127 is found. It may be not correctly played.");
		      ;
		    } else {
		      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "(XG ch=%d Normal voice)",
				ch);
		      TMDY_PLAYMIDI->midi_drumpart_change(tmdy_struct, ch, 0);
		      mapID[ch] = XG_NORMAL_MAP;
		    }
		    break;
		  case 64: /* SFX voice */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "(XG ch=%d SFX voice)",
			      ch);
		    TMDY_PLAYMIDI->midi_drumpart_change(tmdy_struct, ch, 0);
		    mapID[ch] = XG_SFX64_MAP;
		    break;
		  case 126: /* SFX kit */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "(XG ch=%d SFX kit)", ch);
		    TMDY_PLAYMIDI->midi_drumpart_change(tmdy_struct, ch, 1);
		    mapID[ch] = XG_SFX126_MAP;
		    break;
		  case 127: /* Drum kit */
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(XG ch=%d Drum kit)", ch);
		    TMDY_PLAYMIDI->midi_drumpart_change(tmdy_struct, ch, 1);
		    mapID[ch] = XG_DRUM_MAP;
		    break;
		  default:
		    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
			      "(XG: ch=%d Strange bank MSB %d)",
			      ch, bank_msb[ch]);
		    break;
		}
		newbank = bank_lsb[ch];
		break;

	      default:
		newbank = bank_msb[ch];
		break;
	    }

	    if(ISDRUMCHANNEL(ch))
		current_set[ch] = newprog;
	    else
	    {
		if(TMDY_PLAYMIDI->special_tonebank >= 0)
		    newbank = TMDY_PLAYMIDI->special_tonebank;
		if(current_program[ch] == SPECIAL_PROGRAM)
		    skip_this_event = 1;
		current_set[ch] = newbank;
	    }
	    current_program[ch] = newprog;
	    break;

	  case ME_NOTEON:
	    if(counting_time)
		counting_time = 1;
	    if(ISDRUMCHANNEL(ch))
	    {
		newbank = current_set[ch];
		newprog = meep->event.a;
		TMDY_INSTRUM->instrument_map(tmdy_struct, mapID[ch], &newbank, &newprog);

		if(!TMDY_INSTRUM->drumset[newbank]) /* Is this a defined drumset? */
		{
		    if(warn_drumset[newbank] == 0)
		    {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
				  "Drum set %d is undefined", newbank);
			warn_drumset[newbank] = 1;
		    }
		    newbank = 0;
		}

		/* Mark this instrument to be loaded */
		if(!(TMDY_INSTRUM->drumset[newbank]->tone[newprog].instrument))
		    TMDY_INSTRUM->drumset[newbank]->tone[newprog].instrument =
			MAGIC_LOAD_INSTRUMENT;
	    }
	    else
	    {
		if(current_program[ch] == SPECIAL_PROGRAM)
		    break;
		newbank = current_set[ch];
		newprog = current_program[ch];
		TMDY_INSTRUM->instrument_map(tmdy_struct, mapID[ch], &newbank, &newprog);
		if(TMDY_INSTRUM->tonebank[newbank] == NULL)
		{
		    if(warn_tonebank[newbank] == 0)
		    {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
				  "Tone bank %d is undefined", newbank);
			warn_tonebank[newbank] = 1;
		    }
		    newbank = 0;
		}

		/* Mark this instrument to be loaded */
		if(!(TMDY_INSTRUM->tonebank[newbank]->tone[newprog].instrument))
		    TMDY_INSTRUM->tonebank[newbank]->tone[newprog].instrument =
			MAGIC_LOAD_INSTRUMENT;
	    }
	    break;

	  case ME_TONE_BANK_MSB:
	    bank_msb[ch] = meep->event.a;
	    break;

	  case ME_TONE_BANK_LSB:
	    bank_lsb[ch] = meep->event.a;
	    break;

	  case ME_CHORUS_TEXT:
	  case ME_LYRIC:
	  case ME_MARKER:
	  case ME_INSERT_TEXT:
	  case ME_TEXT:
	  case ME_KARAOKE_LYRIC:
	    if((meep->event.a | meep->event.b) == 0)
		skip_this_event = 1;
	    else if(counting_time && (TMDY_CONTROLS->ctl)->trace_playing)
		counting_time = 1;
	    break;

	  case ME_GSLCD:
	    if (counting_time && (TMDY_CONTROLS->ctl)->trace_playing)
		counting_time = 1;
	    skip_this_event = !(TMDY_CONTROLS->ctl)->trace_playing;
	    break;

	  case ME_DRUMPART:
	    TMDY_PLAYMIDI->midi_drumpart_change(tmdy_struct, ch, meep->event.a);
	    break;

	  case ME_WRD:
	    if(TMDY_READMIDI->readmidi_wrd_mode == WRD_TRACE_MIMPI)
	    {
		wrd_args[wrd_argc++] = meep->event.a | 256 * meep->event.b;
		if(ch != WRD_ARG)
		{
		    if(ch == WRD_MAG) {
			TMDY_WRD->wrdt->apply(tmdy_struct, WRD_MAGPRELOAD, wrd_argc, wrd_args);
		    }
		    else if(ch == WRD_PLOAD)
			TMDY_WRD->wrdt->apply(tmdy_struct, WRD_PHOPRELOAD, wrd_argc, wrd_args);
		    else if(ch == WRD_PATH)
			TMDY_WRD->wrdt->apply(tmdy_struct, WRD_PATH, wrd_argc, wrd_args);
		    wrd_argc = 0;
		}
	    }
	    if(counting_time == 2 && TMDY_READMIDI->readmidi_wrd_mode != WRD_TRACE_NOTHING)
		counting_time = 1;
	    break;

	  case ME_SHERRY:
	    if(counting_time == 2)
		counting_time = 1;
	    break;

	  case ME_NOTE_STEP:
	    if(counting_time == 2)
		skip_this_event = 1;
	    break;
        }

	/* Recompute time in samples*/
	if((dt = meep->event.time - at) && !counting_time)
	{
	    samples_to_do = TMDY_READMIDI->sample_increment * dt;
	    sample_cum += TMDY_READMIDI->sample_correction * dt;
	    if(sample_cum & 0xFFFF0000)
	    {
		samples_to_do += ((sample_cum >> 16) & 0xFFFF);
		sample_cum &= 0x0000FFFF;
	    }
	    st += samples_to_do;
	    if(st < 0)
	    {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "Overflow the sample counter");
		free(groomed_list);
		return NULL;
	    }
	}
	else if(counting_time == 1)
	    counting_time = 0;

	if(meep->event.type == ME_TEMPO)
	{
	    tempo = ch + meep->event.b * 256 + meep->event.a * 65536;
	    tempo *= TMDY_READMIDI->tempo_adjust;
	    compute_sample_increment(tmdy_struct, tempo, divisions);
	}

	if(!skip_this_event)
	{
	    /* Add the event to the list */
	    *lp = meep->event;
	    lp->time = st;
	    lp++;
	    our_event_count++;
	}
	at = meep->event.time;
	meep = meep->next;
    }
    /* Add an End-of-Track event */
    lp->time = st;
    lp->type = ME_EOT;
    our_event_count++;
    free_midi_list(tmdy_struct);
    
    *eventsp = our_event_count;
    *samplesp = st;
    return groomed_list;
}

static int read_smf_file(tmdy_struct_ex_t *tmdy_struct, struct timidity_file *tf)
{
    int32 len, divisions;
    int16 format, tracks, divisions_tmp;
    int i;

    if(TMDY_READMIDI->current_file_info->file_type == IS_OTHER_FILE)
	TMDY_READMIDI->current_file_info->file_type = IS_SMF_FILE;

    if(TMDY_READMIDI->current_file_info->karaoke_title == NULL)
	TMDY_READMIDI->karaoke_title_flag = 0;
    else
	TMDY_READMIDI->karaoke_title_flag = 1;

    errno = 0;
    if(TMDY_COMMON->tf_read(tmdy_struct, &len, 4, 1, tf) != 1)
    {
	if(errno)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: %s", (TMDY_COMMON->current_filename),
		      strerror(errno));
	else
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		      "%s: Not a MIDI file!", (TMDY_COMMON->current_filename));
	return 1;
    }
    len = BE_LONG(len);

    TMDY_COMMON->tf_read(tmdy_struct, &format, 2, 1, tf);
    TMDY_COMMON->tf_read(tmdy_struct, &tracks, 2, 1, tf);
    TMDY_COMMON->tf_read(tmdy_struct, &divisions_tmp, 2, 1, tf);
    format = BE_SHORT(format);
    tracks = BE_SHORT(tracks);
    divisions_tmp = BE_SHORT(divisions_tmp);

    if(divisions_tmp < 0)
    {
	/* SMPTE time -- totally untested. Got a MIDI file that uses this? */
	divisions=
	    (int32)(-(divisions_tmp / 256)) * (int32)(divisions_tmp & 0xFF);
    }
    else
	divisions = (int32)divisions_tmp;

    if(len > 6)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		  "%s: MIDI file header size %ld bytes",
		  (TMDY_COMMON->current_filename), len);
	TMDY_COMMON->skip(tmdy_struct, tf, len - 6); /* skip the excess */
    }
    if(format < 0 || format > 2)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: Unknown MIDI file format %d", (TMDY_COMMON->current_filename), format);
	return 1;
    }
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Format: %d  Tracks: %d  Divisions: %d",
	      format, tracks, divisions);

    TMDY_READMIDI->current_file_info->format = format;
    TMDY_READMIDI->current_file_info->tracks = tracks;
    TMDY_READMIDI->current_file_info->divisions = divisions;
    if(tf->url->url_tell != NULL)
	TMDY_READMIDI->current_file_info->hdrsiz = (int16)TMDY_COMMON->tf_tell(tmdy_struct, tf);
    else
	TMDY_READMIDI->current_file_info->hdrsiz = -1;

    switch(format)
    {
      case 0:
	if(read_smf_track(tmdy_struct,tf, 0, 1))
	{
	    if(TMDY_READMIDI->ignore_midi_error)
		break;
	    return 1;
	}
	break;

      case 1:
	for(i = 0; i < tracks; i++)
	{
	    if(read_smf_track(tmdy_struct,tf, i, 1))
	    {
		if(TMDY_READMIDI->ignore_midi_error)
		    break;
		return 1;
	    }
	}
	break;

      case 2: /* We simply play the tracks sequentially */
	for(i = 0; i < tracks; i++)
	{
	    if(read_smf_track(tmdy_struct,tf, i, 0))
	    {
		if(TMDY_READMIDI->ignore_midi_error)
		    break;
		return 1;
	    }
	}
	break;
    }
    return 0;
}

void readmidi_read_init(tmdy_struct_ex_t *tmdy_struct)
{
    int i;

	/* initialize effect status */
	for (i = 0; i < MAX_CHANNELS; i++)
		init_channel_layer(tmdy_struct, i);
	init_reverb_status(tmdy_struct);
	init_delay_status(tmdy_struct);
	init_chorus_status(tmdy_struct);
	init_eq_status(tmdy_struct);
	init_insertion_effect_status(tmdy_struct);
	init_userdrum(tmdy_struct);
	init_userinst(tmdy_struct);
	TMDY_READMIDI->rhythm_part[0] = 9;
	TMDY_READMIDI->rhythm_part[1] = 9;

    /* Put a do-nothing event first in the list for easier processing */
    TMDY_READMIDI->evlist = TMDY_READMIDI->current_midi_point = alloc_midi_event(tmdy_struct);
    TMDY_READMIDI->evlist->event.time = 0;
    TMDY_READMIDI->evlist->event.type = ME_NONE;
    TMDY_READMIDI->evlist->event.channel = 0;
    TMDY_READMIDI->evlist->event.a = 0;
    TMDY_READMIDI->evlist->event.b = 0;
    TMDY_READMIDI->evlist->prev = NULL;
    TMDY_READMIDI->evlist->next = NULL;
    TMDY_READMIDI->readmidi_error_flag = 0;
    TMDY_READMIDI->event_count = 1;

    if(TMDY_READMIDI->string_event_table != NULL)
    {
	free(TMDY_READMIDI->string_event_table[0]);
	free(TMDY_READMIDI->string_event_table);
	TMDY_READMIDI->string_event_table = NULL;
	TMDY_READMIDI->string_event_table_size = 0;
    }
    TMDY_UTILS->strtab->init_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab);
    TMDY_READMIDI->karaoke_format = 0;

    for(i = 0; i < 256; i++)
	TMDY_READMIDI->default_channel_program[i] = -1;
    TMDY_READMIDI->readmidi_wrd_mode = WRD_TRACE_NOTHING;
}

static void insert_note_steps(tmdy_struct_ex_t*  tmdy_struct)
{
	MidiEventList *e;
	int32 i, n, at, lasttime, meas, beat;
	uint8 num, denom, a, b;
	
	e = TMDY_READMIDI->evlist;
	for (i = n = 0; i < TMDY_READMIDI->event_count - 1 && n < 256 - 1; i++, e = e->next)
		if (e->event.type == ME_TIMESIG && e->event.channel == 0) {
			if (n == 0 && e->event.time > 0) {	/* 4/4 is default */
				SETMIDIEVENT(TMDY_READMIDI->timesig[n], 0, ME_TIMESIG, 0, 4, 4);
				n++;
			}
			if (n > 0 && e->event.a == TMDY_READMIDI->timesig[n - 1].a
					&& e->event.b == TMDY_READMIDI->timesig[n - 1].b)
				continue;	/* unchanged */
			if (n > 0 && e->event.time == TMDY_READMIDI->timesig[n - 1].time)
				n--;	/* overwrite previous timesig */
			TMDY_READMIDI->timesig[n++] = e->event;
		}
	if (n == 0) {
		SETMIDIEVENT(TMDY_READMIDI->timesig[n], 0, ME_TIMESIG, 0, 4, 4);
		n++;
	}
	TMDY_READMIDI->timesig[n] = TMDY_READMIDI->timesig[n - 1];
	TMDY_READMIDI->timesig[n].time = 0x7fffffff;	/* stopper */
	lasttime = e->event.time;
	readmidi_set_track(tmdy_struct, 0, 1);
	at = n = meas = beat = 0;
	while (at < lasttime && ! TMDY_READMIDI->readmidi_error_flag) {
		if (at >= TMDY_READMIDI->timesig[n].time) {
			if (beat != 0)
				meas++, beat = 0;
			num = TMDY_READMIDI->timesig[n].a, denom = TMDY_READMIDI->timesig[n].b, n++;
		}
		a = meas & 0xff;
		b = (meas >> 8 & 0x0f) + (beat + 1 << 4);
		MIDIEVENT(at, ME_NOTE_STEP, 0, a, b);
		if (++beat == num)
			meas++, beat = 0;
		at += TMDY_READMIDI->current_file_info->divisions * 4 / denom;
	}
}

MidiEvent *read_midi_file(tmdy_struct_ex_t *tmdy_struct, struct timidity_file *tf, int32 *count, int32 *sp,
			  char *fn)
{
    char magic[4];
    MidiEvent *ev;
    int err, macbin_check, mtype, i;

    macbin_check = 1;
    TMDY_READMIDI->current_file_info = get_midi_file_info(tmdy_struct, (TMDY_COMMON->current_filename), 1);
    COPY_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, TMDY_READMIDI->current_file_info->drumchannels);
    COPY_CHANNELMASK(TMDY_PLAYMIDI->drumchannel_mask, TMDY_READMIDI->current_file_info->drumchannel_mask);

    errno = 0;

    if((mtype = TMDY_MOD->get_module_type(tmdy_struct, fn)) > 0)
    {
	readmidi_read_init(tmdy_struct);
    	if(!IS_URL_SEEK_SAFE(tf->url))
    	    tf->url = TMDY_ARC->url->url_cache_open(tmdy_struct, tf->url, 1);
  	err = TMDY_MOD->load_module_file(tmdy_struct, tf, mtype);
	if(!err)
	{
	    TMDY_READMIDI->current_file_info->format = 0;
	    memset(&TMDY_PLAYMIDI->drumchannels, 0, sizeof(TMDY_PLAYMIDI->drumchannels));
	    goto grooming;
	}
	free_midi_list(tmdy_struct);

	if(err == 2)
	    return NULL;
	TMDY_ARC->url->url_rewind(tmdy_struct, tf->url);
	TMDY_ARC->url->url_cache_disable(tmdy_struct, tf->url);
    }

#if MAX_CHANNELS > 16
    for(i = 16; i < MAX_CHANNELS; i++)
    {
	if(!IS_SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannel_mask, i))
	{
	    if(IS_SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, i & 0xF))
		SET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, i);
	    else
		UNSET_CHANNELMASK(TMDY_PLAYMIDI->drumchannels, i);
	}
    }
#endif

    if(TMDY_READMIDI->opt_default_mid &&
       (TMDY_READMIDI->current_file_info->mid == 0 || TMDY_READMIDI->current_file_info->mid >= 0x7e))
	TMDY_READMIDI->current_file_info->mid = TMDY_READMIDI->opt_default_mid;

  retry_read:
    if(TMDY_COMMON->tf_read(tmdy_struct, magic, 1, 4, tf) != 4)
    {
	if(errno)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: %s", (TMDY_COMMON->current_filename),
		      strerror(errno));
	else
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		      "%s: Not a MIDI file!", (TMDY_COMMON->current_filename));
	return NULL;
    }

    if(memcmp(magic, "MThd", 4) == 0)
    {
	readmidi_read_init(tmdy_struct);
	err = read_smf_file(tmdy_struct, tf);
    }
    else if(memcmp(magic, "RCM-", 4) == 0 || memcmp(magic, "COME", 4) == 0)
    {
	readmidi_read_init(tmdy_struct);
	err = read_rcp_file(tmdy_struct, tf, magic, fn);
    }
    else if (strncmp(magic, "RIFF", 4) == 0) {
       if (TMDY_COMMON->tf_read(tmdy_struct, magic, 1, 4, tf) == 4 &&
           TMDY_COMMON->tf_read(tmdy_struct, magic, 1, 4, tf) == 4 &&
           strncmp(magic, "RMID", 4) == 0 &&
           TMDY_COMMON->tf_read(tmdy_struct, magic, 1, 4, tf) == 4 &&
           strncmp(magic, "data", 4) == 0 &&
           TMDY_COMMON->tf_read(tmdy_struct, magic, 1, 4, tf) == 4) {
           goto retry_read;
       } else {
           err = 1;
           (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
                     "%s: Not a MIDI file!", (TMDY_COMMON->current_filename));
       }
    }
    else if(memcmp(magic, "melo", 4) == 0)
    {
	readmidi_read_init(tmdy_struct);
	err = read_mfi_file(tmdy_struct, tf);
    }
    else
    {
	if(macbin_check && magic[0] == 0)
	{
	    /* Mac Binary */
	    macbin_check = 0;
	    TMDY_COMMON->skip(tmdy_struct, tf, 128 - 4);
	    goto retry_read;
	}
	else if(memcmp(magic, "RIFF", 4) == 0)
	{
	    /* RIFF MIDI file */
	    TMDY_COMMON->skip(tmdy_struct, tf, 20 - 4);
	    goto retry_read;
	}
	err = 1;
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		  "%s: Not a MIDI file!", (TMDY_COMMON->current_filename));
    }

    if(err)
    {
	free_midi_list(tmdy_struct);
	if(TMDY_READMIDI->string_event_strtab.nstring > 0)
	    TMDY_UTILS->strtab->delete_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab);
	return NULL;
    }

    /* Read WRD file */
    if(!((TMDY_OUTPUT->play_mode)->flag&PF_CAN_TRACE))
    {
	if(TMDY_WRD->wrdt->start != NULL)
	    TMDY_WRD->wrdt->start(tmdy_struct, WRD_TRACE_NOTHING);
	TMDY_READMIDI->readmidi_wrd_mode = WRD_TRACE_NOTHING;
    }
    else if(TMDY_WRD->wrdt->id != '-' && TMDY_WRD->wrdt->opened)
    {
	TMDY_READMIDI->readmidi_wrd_mode = TMDY_WRD->import_wrd_file(tmdy_struct, fn);
	if(TMDY_WRD->wrdt->start != NULL)
	    if(TMDY_WRD->wrdt->start(tmdy_struct, TMDY_READMIDI->readmidi_wrd_mode) == -1)
	    {
		/* strip all WRD events */
		MidiEventList *e;
		int32 i;
		for(i = 0, e = TMDY_READMIDI->evlist; i < TMDY_READMIDI->event_count; i++, e = e->next)
		    if (e->event.type == ME_WRD || e->event.type == ME_SHERRY)
			e->event.type = ME_NONE;
	    }
    }
    else
	TMDY_READMIDI->readmidi_wrd_mode = WRD_TRACE_NOTHING;

    /* make lyric table */
    if(TMDY_READMIDI->string_event_strtab.nstring > 0)
    {
	TMDY_READMIDI->string_event_table_size = TMDY_READMIDI->string_event_strtab.nstring;
	TMDY_READMIDI->string_event_table = TMDY_UTILS->strtab->make_string_array(tmdy_struct, &TMDY_READMIDI->string_event_strtab);
	if(TMDY_READMIDI->string_event_table == NULL)
	{
	    TMDY_UTILS->strtab->delete_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab);
	    TMDY_READMIDI->string_event_table_size = 0;
	}
    }

  grooming:
    insert_note_steps(tmdy_struct);
    ev = groom_list(tmdy_struct, TMDY_READMIDI->current_file_info->divisions, count, sp);
    if(ev == NULL)
    {
	free_midi_list(tmdy_struct);
	if(TMDY_READMIDI->string_event_strtab.nstring > 0)
	    TMDY_UTILS->strtab->delete_string_table(tmdy_struct, &TMDY_READMIDI->string_event_strtab);
	return NULL;
    }
    TMDY_READMIDI->current_file_info->samples = *sp;
    if(TMDY_READMIDI->current_file_info->first_text == NULL)
	TMDY_READMIDI->current_file_info->first_text = TMDY_COMMON->safe_strdup(tmdy_struct, "");
    TMDY_READMIDI->current_file_info->readflag = 1;
    return ev;
}


struct midi_file_info *new_midi_file_info(tmdy_struct_ex_t *tmdy_struct, const char *filename)
{
    struct midi_file_info *p;
    p = (struct midi_file_info *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(struct midi_file_info));

    /* Initialize default members */
    memset(p, 0, sizeof(struct midi_file_info));
    p->hdrsiz = -1;
    p->format = -1;
    p->tracks = -1;
    p->divisions = -1;
    p->time_sig_n = p->time_sig_d = -1;
    p->samples = -1;
    p->max_channel = -1;
    p->file_type = IS_OTHER_FILE;
    if(filename != NULL)
	p->filename = TMDY_COMMON->safe_strdup(tmdy_struct, filename);
    COPY_CHANNELMASK(p->drumchannels, TMDY_PLAYMIDI->default_drumchannels);
    COPY_CHANNELMASK(p->drumchannel_mask, TMDY_PLAYMIDI->default_drumchannel_mask);

    /* Append to midi_file_info */
    p->next = TMDY_READMIDI->midi_file_info;
    TMDY_READMIDI->midi_file_info = p;

    return p;
}

void free_all_midi_file_info(tmdy_struct_ex_t *tmdy_struct)
{
  struct midi_file_info *info, *next;

  info = TMDY_READMIDI->midi_file_info;
  while (info) {
    next = info->next;
    free(info->filename);
    if (info->seq_name)
      free(info->seq_name);
    if (info->karaoke_title != NULL && info->karaoke_title == info->first_text)
      free(info->karaoke_title);
    else {
      if (info->karaoke_title)
	free(info->karaoke_title);
      if (info->first_text)
	free(info->first_text);
      if (info->midi_data)
	free(info->midi_data);
      if (info->pcm_filename)
	free(info->pcm_filename); /* Note: this memory is freed in playmidi.c*/
    }
    free(info);
    info = next;
  }
  TMDY_READMIDI->midi_file_info = NULL;
  TMDY_READMIDI->current_file_info = NULL;
}

struct midi_file_info *get_midi_file_info(tmdy_struct_ex_t *tmdy_struct, char *filename, int newp)
{
    struct midi_file_info *p;

    filename = TMDY_ARC->url->url_expand_home_dir(tmdy_struct, filename);
    /* Linear search */
    for(p = TMDY_READMIDI->midi_file_info; p; p = p->next)
	if(!strcmp(filename, p->filename))
	    return p;
    if(newp)
	return new_midi_file_info(tmdy_struct, filename);
    return NULL;
}

struct timidity_file *open_midi_file(tmdy_struct_ex_t *tmdy_struct, char *fn,
				     int decompress, int noise_mode)
{
    struct midi_file_info *infop;
    struct timidity_file *tf;
#if defined(SMFCONV) && defined(__W32__)
    extern int opt_rcpcv_dll;
#endif

    infop = get_midi_file_info(tmdy_struct, fn, 0);
    if(infop == NULL || infop->midi_data == NULL)
	tf = TMDY_COMMON->open_file(tmdy_struct, fn, decompress, noise_mode);
    else
    {
	tf = TMDY_COMMON->open_with_mem(tmdy_struct, infop->midi_data, infop->midi_data_size,
			   noise_mode);
	if(infop->compressed)
	{
	    if((tf->url = TMDY_ARC->url->url_inflate_open(tmdy_struct, tf->url, infop->midi_data_size, 1))
	       == NULL)
	    {
		TMDY_COMMON->close_file(tmdy_struct, tf);
		return NULL;
	    }
	}
    }

#if defined(SMFCONV) && defined(__W32__)
    /* smf convert */
    if(tf != NULL && opt_rcpcv_dll)
    {
    	if(smfconv_w32(tmdy_struct, tf, fn))
	{
	    TMDY_COMMON->close_file(tmdy_struct, tf);
	    return NULL;
	}
    }
#endif

    return tf;
}

#ifndef NO_MIDI_CACHE
static long deflate_url_reader(tmdy_struct_ex_t *tmdy_struct, char *buf, long size, void *user_val)
{
    return TMDY_ARC->url->url_nread(tmdy_struct, (URL)user_val, buf, size);
}

/*
 * URL data into deflated buffer.
 */
static void url_make_file_data(tmdy_struct_ex_t *tmdy_struct, URL url, struct midi_file_info *infop)
{
    char buff[BUFSIZ];
    MemBuffer b;
    long n;
    DeflateHandler compressor;

    TMDY_UTILS->memb->init_memb(tmdy_struct, &b);

    /* url => b */
    if((compressor = TMDY_ARC->zip->open_deflate_handler(tmdy_struct, deflate_url_reader, url,
					  ARC_DEFLATE_LEVEL)) == NULL)
	return;
    while((n = TMDY_ARC->zip->zip_deflate(tmdy_struct, compressor, buff, sizeof(buff))) > 0)
	TMDY_UTILS->memb->push_memb(tmdy_struct, &b, buff, n);
    TMDY_ARC->zip->close_deflate_handler(tmdy_struct, compressor);
    infop->compressed = 1;

    /* b => mem */
    infop->midi_data_size = b.total_size;
    TMDY_UTILS->memb->rewind_memb(tmdy_struct, &b);
    infop->midi_data = (void *)TMDY_COMMON->safe_malloc(tmdy_struct, infop->midi_data_size);
    TMDY_UTILS->memb->read_memb(tmdy_struct, &b, infop->midi_data, infop->midi_data_size);
    TMDY_UTILS->memb->delete_memb(tmdy_struct, &b);
}

static int check_need_cache(tmdy_struct_ex_t *tmdy_struct, URL url, char *filename)
{
    int t1, t2;
    t1 = url_check_type(tmdy_struct, filename);
    t2 = url->type;
    return (t1 == URL_http_t || t1 == URL_ftp_t || t1 == URL_news_t)
	 && t2 != URL_arc_t;
}
#else
/*ARGSUSED*/
static void url_make_file_data(tmdy_struct_ex_t *tmdy_struct, URL url, struct midi_file_info *infop)
{
}
/*ARGSUSED*/
static int check_need_cache(tmdy_struct_ex_t *tmdy_struct, URL url, char *filename)
{
    return 0;
}
#endif /* NO_MIDI_CACHE */

int check_midi_file(tmdy_struct_ex_t *tmdy_struct, char *filename)
{
    struct midi_file_info *p;
    struct timidity_file *tf;
    char tmp[4];
    int32 len;
    int16 format;
    int check_cache;

    if(filename == NULL)
    {
	if(TMDY_READMIDI->current_file_info == NULL)
	    return -1;
	filename = TMDY_READMIDI->current_file_info->filename;
    }

    p = get_midi_file_info(tmdy_struct, filename, 0);
    if(p != NULL)
	return p->format;
    p = get_midi_file_info(tmdy_struct, filename, 1);

    if(TMDY_MOD->get_module_type(tmdy_struct, filename) > 0)
    {
	p->format = 0;
	return 0;
    }

    tf = TMDY_COMMON->open_file(tmdy_struct, filename, 1, OF_SILENT);
    if(tf == NULL)
	return -1;

    check_cache = check_need_cache(tmdy_struct, tf->url, filename);
    if(check_cache)
    {
	if(!IS_URL_SEEK_SAFE(tf->url))
	{
	    if((tf->url = TMDY_ARC->url->url_cache_open(tmdy_struct, tf->url, 1)) == NULL)
	    {
		TMDY_COMMON->close_file(tmdy_struct, tf);
		return -1;
	    }
	}
    }

    /* Parse MIDI header */
    if(TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return -1;
    }

    if(tmp[0] == 0)
    {
	TMDY_COMMON->skip(tmdy_struct, tf, 128 - 4);
	if(TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4)
	{
	    TMDY_COMMON->close_file(tmdy_struct, tf);
	    return -1;
	}
    }

    if(strncmp(tmp, "RCM-", 4) == 0 ||
       strncmp(tmp, "COME", 4) == 0 ||
       strncmp(tmp, "M1", 2) == 0)
    {
	p->format = 1;
	goto end_of_header;
    }

    if(strncmp(tmp, "MThd", 4) != 0)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return -1;
    }

    if(TMDY_COMMON->tf_read(tmdy_struct, &len, 4, 1, tf) != 1)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return -1;
    }
    len = BE_LONG(len);

    TMDY_COMMON->tf_read(tmdy_struct, &format, 2, 1, tf);
    format = BE_SHORT(format);
    if(format < 0 || format > 2)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return -1;
    }
    TMDY_COMMON->skip(tmdy_struct, tf, len - 2);

    p->format = format;
    p->hdrsiz = (int16)TMDY_COMMON->tf_tell(tmdy_struct, tf);

  end_of_header:
    if(check_cache)
    {
	TMDY_ARC->url->url_rewind(tmdy_struct, tf->url);
	TMDY_ARC->url->url_cache_disable(tmdy_struct, tf->url);
	url_make_file_data(tmdy_struct, tf->url, p);
    }
    TMDY_COMMON->close_file(tmdy_struct, tf);
    return format;
}

static char *get_midi_title1(tmdy_struct_ex_t *tmdy_struct, struct midi_file_info *p)
{
    char *s;

    if(p->format != 0 && p->format != 1)
	return NULL;

    if((s = p->seq_name) == NULL)
	if((s = p->karaoke_title) == NULL)
	    s = p->first_text;
    if(s != NULL)
    {
	int all_space, i;

	all_space = 1;
	for(i = 0; s[i]; i++)
	    if(s[i] != ' ')
	    {
		all_space = 0;
		break;
	    }
	if(all_space)
	    s = NULL;
    }
    return s;
}

char *get_midi_title(tmdy_struct_ex_t *tmdy_struct, char *filename)
{
    struct midi_file_info *p;
    struct timidity_file *tf;
    char tmp[4];
    int32 len;
    int16 format, tracks, trk;
    int laststatus, check_cache;
    int mtype;

    if(filename == NULL)
    {
	if(TMDY_READMIDI->current_file_info == NULL)
	    return NULL;
	filename = TMDY_READMIDI->current_file_info->filename;
    }

    p = get_midi_file_info(tmdy_struct, filename, 0);
    if(p == NULL)
	p = get_midi_file_info(tmdy_struct, filename, 1);
    else 
    {
	if(p->seq_name != NULL || p->first_text != NULL || p->format < 0)
	    return get_midi_title1(tmdy_struct, p);
    }

    tf = TMDY_COMMON->open_file(tmdy_struct, filename, 1, OF_SILENT);
    if(tf == NULL)
	return NULL;

    mtype = TMDY_MOD->get_module_type(tmdy_struct, filename);
    check_cache = check_need_cache(tmdy_struct, tf->url, filename);
    if(check_cache || mtype > 0)
    {
	if(!IS_URL_SEEK_SAFE(tf->url))
	{
	    if((tf->url = TMDY_ARC->url->url_cache_open(tmdy_struct, tf->url, 1)) == NULL)
	    {
		TMDY_COMMON->close_file(tmdy_struct, tf);
		return NULL;
	    }
	}
    }

    if(mtype > 0)
    {
	char *title, *str;

	title = TMDY_MOD->get_module_title(tmdy_struct, tf, mtype);
	if(title == NULL)
	{
	    /* No title */
	    p->seq_name = NULL;
	    p->format = 0;
	    goto end_of_parse;
	}

	len = (int32)strlen(title);
	len = SAFE_CONVERT_LENGTH(len);
	str = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len);
	TMDY_COMMON->code_convert(tmdy_struct, title, str, len, NULL, NULL);
	p->seq_name = (char *)TMDY_COMMON->safe_strdup(tmdy_struct, str);
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	p->format = 0;
	free (title);
	goto end_of_parse;
    }

    /* Parse MIDI header */
    if(TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return NULL;
    }

    if(tmp[0] == 0)
    {
	TMDY_COMMON->skip(tmdy_struct, tf, 128 - 4);
	if(TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4)
	{
	    TMDY_COMMON->close_file(tmdy_struct, tf);
	    return NULL;
	}
    }

    if(memcmp(tmp, "RCM-", 4) == 0 || memcmp(tmp, "COME", 4) == 0)
    {
    	int i;
	char local[0x40 + 1];
	char *str;

	p->format = 1;
	TMDY_COMMON->skip(tmdy_struct, tf, 0x20 - 4);
	TMDY_COMMON->tf_read(tmdy_struct, local, 1, 0x40, tf);
	local[0x40]='\0';

	for(i = 0x40 - 1; i >= 0; i--)
	{
	    if(local[i] == 0x20)
		local[i] = '\0';
	    else if(local[i] != '\0')
		break;
	}

	i = SAFE_CONVERT_LENGTH(i + 1);
	str = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), i);
	TMDY_COMMON->code_convert(tmdy_struct, local, str, i, NULL, NULL);
	p->seq_name = (char *)TMDY_COMMON->safe_strdup(tmdy_struct, str);
	TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	p->format = 1;
	goto end_of_parse;
    }
    if(memcmp(tmp, "melo", 4) == 0)
    {
	int i;
	char *master, *converted;
	
	master = get_mfi_file_title(tmdy_struct, tf);
	if (master != NULL)
	{
	    i = SAFE_CONVERT_LENGTH(strlen(master) + 1);
	    converted = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), i);
	    TMDY_COMMON->code_convert(tmdy_struct, master, converted, i, NULL, NULL);
	    p->seq_name = (char *)TMDY_COMMON->safe_strdup(tmdy_struct, converted);
	    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
	}
	else
	{
	    p->seq_name = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, 1);
	    p->seq_name[0] = '\0';
	}
	p->format = 0;
	goto end_of_parse;
    }

    if(strncmp(tmp, "M1", 2) == 0)
    {
	/* I don't know MPC file format */
	p->format = 1;
	goto end_of_parse;
    }

	  if(strncmp(tmp, "RIFF", 4) == 0)
	  {
	/* RIFF MIDI file */
	TMDY_COMMON->skip(tmdy_struct, tf, 20 - 4);
  if(TMDY_COMMON->tf_read(tmdy_struct, tmp, 1, 4, tf) != 4)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return NULL;
    }
	  }

    if(strncmp(tmp, "MThd", 4) != 0)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return NULL;
    }

    if(TMDY_COMMON->tf_read(tmdy_struct, &len, 4, 1, tf) != 1)
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return NULL;
    }

    len = BE_LONG(len);

    TMDY_COMMON->tf_read(tmdy_struct, &format, 2, 1, tf);
    TMDY_COMMON->tf_read(tmdy_struct, &tracks, 2, 1, tf);
    format = BE_SHORT(format);
    tracks = BE_SHORT(tracks);
    p->format = format;
    p->tracks = tracks;
    if(format < 0 || format > 2)
    {
	p->format = -1;
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return NULL;
    }

    TMDY_COMMON->skip(tmdy_struct, tf, len - 4);
    p->hdrsiz = (int16)TMDY_COMMON->tf_tell(tmdy_struct, tf);

    if(format == 2)
	goto end_of_parse;

    if(tracks >= 3)
    {
	tracks = 3;
	TMDY_READMIDI->karaoke_format = 0;
    }
    else
    {
	tracks = 1;
	TMDY_READMIDI->karaoke_format = -1;
    }

    for(trk = 0; trk < tracks; trk++)
    {
	int32 next_pos, pos;

	if(trk >= 1 && TMDY_READMIDI->karaoke_format == -1)
	    break;

	if((TMDY_COMMON->tf_read(tmdy_struct, tmp,1,4,tf) != 4) || (TMDY_COMMON->tf_read(tmdy_struct, &len,4,1,tf) != 1))
	    break;

	if(memcmp(tmp, "MTrk", 4))
	    break;

	next_pos = TMDY_COMMON->tf_tell(tmdy_struct, tf) + len;
	laststatus = -1;
	for(;;)
	{
	    int i, me, type;

	    /* skip Variable-length quantity */
	    do
	    {
		if((i = tf_getc(tf)) == EOF)
		    goto end_of_parse;
	    } while (i & 0x80);

	    if((me = tf_getc(tf)) == EOF)
		goto end_of_parse;

	    if(me == 0xF0 || me == 0xF7) /* SysEx */
	    {
		if((len = getvl(tmdy_struct, tf)) < 0)
		    goto end_of_parse;
		if((p->mid == 0 || p->mid >= 0x7e) && len > 0 && me == 0xF0)
		{
		    p->mid = tf_getc(tf);
		    len--;
		}
		TMDY_COMMON->skip(tmdy_struct, tf, len);
	    }
	    else if(me == 0xFF) /* Meta */
	    {
		type = tf_getc(tf);
		if((len = getvl(tmdy_struct, tf)) < 0)
		    goto end_of_parse;
		if((type == 1 || type == 3) && len > 0 &&
		   (trk == 0 || TMDY_READMIDI->karaoke_format != -1))
		{
		    char *si, *so;
		    int s_maxlen = SAFE_CONVERT_LENGTH(len);

		    si = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), len + 1);
		    so = (char *)TMDY_UTILS->mblock->new_segment(tmdy_struct, &(TMDY_COMMON->tmpbuffer), s_maxlen);

		    if(len != TMDY_COMMON->tf_read(tmdy_struct, si, 1, len, tf))
		    {
			TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
			goto end_of_parse;
		    }

		    si[len]='\0';
		    TMDY_COMMON->code_convert(tmdy_struct, si, so, s_maxlen, NULL, NULL);
		    if(trk == 0 && type == 3)
		    {
		      if(p->seq_name == NULL) {
			char *name = TMDY_COMMON->safe_strdup(tmdy_struct, so);
			p->seq_name = TMDY_COMMON->safe_strdup(tmdy_struct, fix_string(tmdy_struct, name));
			free(name);
		      }
		      TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
		      if(TMDY_READMIDI->karaoke_format == -1)
			goto end_of_parse;
		    }
		    if(p->first_text == NULL) {
		      char *name;
		      name = TMDY_COMMON->safe_strdup(tmdy_struct, so);
		      p->first_text = TMDY_COMMON->safe_strdup(tmdy_struct, fix_string(tmdy_struct, name));
		      free(name);
		    }
		    if(TMDY_READMIDI->karaoke_format != -1)
		    {
			if(trk == 1 && strncmp(si, "@KMIDI", 6) == 0)
			    TMDY_READMIDI->karaoke_format = 1;
			else if(TMDY_READMIDI->karaoke_format == 1 && trk == 2)
			    TMDY_READMIDI->karaoke_format = 2;
		    }
		    if(type == 1 && TMDY_READMIDI->karaoke_format == 2)
		    {
			if(strncmp(si, "@T", 2) == 0)
			    p->karaoke_title =
				add_karaoke_title(tmdy_struct, p->karaoke_title, si + 2);
			else if(si[0] == '\\')
			    goto end_of_parse;
		    }
		    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &(TMDY_COMMON->tmpbuffer));
		}
		else if(type == 0x2F)
		{
		    pos = TMDY_COMMON->tf_tell(tmdy_struct, tf);
		    if(pos < next_pos)
			TMDY_COMMON->tf_seek(tmdy_struct, tf, next_pos - pos, SEEK_CUR);
		    break; /* End of track */
		}
		else
		    TMDY_COMMON->skip(tmdy_struct, tf, len);
	    }
	    else /* MIDI event */
	    {
		/* skip MIDI event */
		TMDY_READMIDI->karaoke_format = -1;
		if(trk != 0)
		    goto end_of_parse;

		if(me & 0x80) /* status byte */
		{
		    laststatus = (me >> 4) & 0x07;
		    if(laststatus != 7)
			tf_getc(tf);
		}

		switch(laststatus)
		{
		  case 0: case 1: case 2: case 3: case 6:
		    tf_getc(tf);
		    break;
		  case 7:
		    if(!(me & 0x80))
			break;
		    switch(me & 0x0F)
		    {
		      case 2:
			tf_getc(tf);
			tf_getc(tf);
			break;
		      case 3:
			tf_getc(tf);
			break;
		    }
		    break;
		}
	    }
	}
    }

  end_of_parse:
    if(check_cache)
    {
	TMDY_ARC->url->url_rewind(tmdy_struct, tf->url);
	TMDY_ARC->url->url_cache_disable(tmdy_struct, tf->url);
	url_make_file_data(tmdy_struct, tf->url, p);
    }
    TMDY_COMMON->close_file(tmdy_struct, tf);
    if(p->first_text == NULL)
	p->first_text = TMDY_COMMON->safe_strdup(tmdy_struct, "");
    return get_midi_title1(tmdy_struct, p);
}

int midi_file_save_as(tmdy_struct_ex_t *tmdy_struct, char *in_name, char *out_name)
{
    struct timidity_file *tf;
    FILE* ofp;
    char buff[BUFSIZ];
    long n;

    if(in_name == NULL)
    {
	if(TMDY_READMIDI->current_file_info == NULL)
	    return 0;
	in_name = TMDY_READMIDI->current_file_info->filename;
    }
    out_name = (char *)TMDY_ARC->url->url_expand_home_dir(tmdy_struct, out_name);

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Save as %s...", out_name);

    errno = 0;
    if((tf = open_midi_file(tmdy_struct, in_name, 1, 0)) == NULL)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: %s", out_name,
		  errno ? strerror(errno) : "Can't save file");
	return -1;
    }

    errno = 0;
    if((ofp = fopen(out_name, "wb")) == NULL)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: %s", out_name,
		  errno ? strerror(errno) : "Can't save file");
	TMDY_COMMON->close_file(tmdy_struct, tf);
	return -1;
    }

    while((n = TMDY_COMMON->tf_read(tmdy_struct, buff, 1, sizeof(buff), tf)) > 0)
	fwrite(buff, 1, n, ofp);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NORMAL, "Save as %s...Done", out_name);

    fclose(ofp);
    TMDY_COMMON->close_file(tmdy_struct, tf);
    return 0;
}

char *event2string(tmdy_struct_ex_t *tmdy_struct, int id)
{
    if(id == 0)
	return "";
#ifdef ABORT_AT_FATAL
    if(id >= TMDY_READMIDI->string_event_table_size)
	abort();
#endif /* ABORT_AT_FATAL */
    if(TMDY_READMIDI->string_event_table == NULL || id < 0 || id >= TMDY_READMIDI->string_event_table_size)
	return NULL;
    return TMDY_READMIDI->string_event_table[id];
}

/*! fixed filter cutoff frequency for GS system effects in Hz. */
#define SYSTEM_EFFECT_LPF_FC 110

/*! highpass shelving filter gain table for GS system effects in decibels.
    every value is negative or 0, so that this filter works as a sort of lowpass-filter. */
FLOAT_T gs_system_effect_hsf_gain_table[8] = {
	0, -1.9, -4.0, -6.5, -9.0, -12.0, -16.5, -24.0
};

void init_delay_status(tmdy_struct_ex_t *tmdy_struct)
{
	TMDY_REVERB->delay_status.type = 0;
	TMDY_REVERB->delay_status.level = 0x40;
	TMDY_REVERB->delay_status.level_center = 0x7F;
	TMDY_REVERB->delay_status.level_left = 0;
	TMDY_REVERB->delay_status.level_right = 0;
	TMDY_REVERB->delay_status.time_center = 340.0;
	TMDY_REVERB->delay_status.time_ratio_left = 1.0 / 24.0;
	TMDY_REVERB->delay_status.time_ratio_right = 1.0 / 24.0;
	TMDY_REVERB->delay_status.feedback = 0x50;
	TMDY_REVERB->delay_status.pre_lpf = 0;
	recompute_delay_status(tmdy_struct);
}

void recompute_delay_status(tmdy_struct_ex_t *tmdy_struct)
{
	FLOAT_T dBGain;

	TMDY_REVERB->delay_status.sample_c = TMDY_REVERB->delay_status.time_center * (TMDY_OUTPUT->play_mode)->rate / 1000;
	TMDY_REVERB->delay_status.sample_l = TMDY_REVERB->delay_status.sample_c * TMDY_REVERB->delay_status.time_ratio_left;
	TMDY_REVERB->delay_status.sample_r = TMDY_REVERB->delay_status.sample_c * TMDY_REVERB->delay_status.time_ratio_right;
	if(TMDY_REVERB->delay_status.sample_c > (TMDY_OUTPUT->play_mode)->rate) {TMDY_REVERB->delay_status.sample_c = (TMDY_OUTPUT->play_mode)->rate;}
	if(TMDY_REVERB->delay_status.sample_l > (TMDY_OUTPUT->play_mode)->rate) {TMDY_REVERB->delay_status.sample_l = (TMDY_OUTPUT->play_mode)->rate;}
	if(TMDY_REVERB->delay_status.sample_r > (TMDY_OUTPUT->play_mode)->rate) {TMDY_REVERB->delay_status.sample_r = (TMDY_OUTPUT->play_mode)->rate;}
	TMDY_REVERB->delay_status.level_ratio_c = (double)TMDY_REVERB->delay_status.level * (double)TMDY_REVERB->delay_status.level_center / 16129.0;
	TMDY_REVERB->delay_status.level_ratio_l = (double)TMDY_REVERB->delay_status.level * (double)TMDY_REVERB->delay_status.level_left / 16129.0;
	TMDY_REVERB->delay_status.level_ratio_r = (double)TMDY_REVERB->delay_status.level * (double)TMDY_REVERB->delay_status.level_right / 16129.0;
	TMDY_REVERB->delay_status.feedback_ratio = (double)(TMDY_REVERB->delay_status.feedback - 64) * 0.0153125;
	TMDY_REVERB->delay_status.send_reverb_ratio = (double)TMDY_REVERB->delay_status.send_reverb / 127.0;

	if(TMDY_REVERB->delay_status.level_left || TMDY_REVERB->delay_status.level_right && TMDY_REVERB->delay_status.type == 0) {
		TMDY_REVERB->delay_status.type = 1;
	}

	if(TMDY_REVERB->delay_status.pre_lpf) {
		dBGain = gs_system_effect_hsf_gain_table[TMDY_REVERB->delay_status.pre_lpf];
		/* calculate highpass shelving filter's coefficients */
		TMDY_REVERB->calc_highshelf_coefs(tmdy_struct, TMDY_REVERB->delay_status.high_coef, SYSTEM_EFFECT_LPF_FC, dBGain, (TMDY_OUTPUT->play_mode)->rate);
	}
}

void set_delay_macro(tmdy_struct_ex_t *tmdy_struct, int macro)
{
	if(macro > 3) {TMDY_REVERB->delay_status.type = 2;}
	macro *= 10;
	TMDY_REVERB->delay_status.time_center = delay_time_center_table[delay_macro_presets[macro+1]];
	TMDY_REVERB->delay_status.time_ratio_left = (double)delay_macro_presets[macro+2] / 24;
	TMDY_REVERB->delay_status.time_ratio_right = (double)delay_macro_presets[macro+3] / 24;
	TMDY_REVERB->delay_status.level_center = delay_macro_presets[macro+4];
	TMDY_REVERB->delay_status.level_left = delay_macro_presets[macro+5];
	TMDY_REVERB->delay_status.level_right = delay_macro_presets[macro+6];
	TMDY_REVERB->delay_status.level = delay_macro_presets[macro+7];
	TMDY_REVERB->delay_status.feedback = delay_macro_presets[macro+8];
}

void init_reverb_status(tmdy_struct_ex_t *tmdy_struct)
{
	TMDY_REVERB->reverb_status.character = 0x04;
	TMDY_REVERB->reverb_status.pre_lpf = 0;
	TMDY_REVERB->reverb_status.level = 0x40;
	TMDY_REVERB->reverb_status.time = 0x40;
	TMDY_REVERB->reverb_status.delay_feedback = 0;
	TMDY_REVERB->reverb_status.pre_delay_time = 0;
	recompute_reverb_status(tmdy_struct);
	TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
}

void recompute_reverb_status(tmdy_struct_ex_t *tmdy_struct)
{
	FLOAT_T dBGain;

	TMDY_REVERB->reverb_status.level_ratio = (double)TMDY_REVERB->reverb_status.level / 127.0f;
	TMDY_REVERB->reverb_status.time_ratio = (double)TMDY_REVERB->reverb_status.time / 128.0f + 0.5f;

	if(TMDY_REVERB->reverb_status.pre_lpf) {
		dBGain = gs_system_effect_hsf_gain_table[TMDY_REVERB->reverb_status.pre_lpf];
		/* calculate highpass shelving filter's coefficients */
		TMDY_REVERB->calc_highshelf_coefs(tmdy_struct, TMDY_REVERB->reverb_status.high_coef, SYSTEM_EFFECT_LPF_FC, dBGain, (TMDY_OUTPUT->play_mode)->rate);
	}
}

void set_reverb_macro(tmdy_struct_ex_t *tmdy_struct, int macro)
{
	macro *= 6;
	TMDY_REVERB->reverb_status.character = reverb_macro_presets[macro];
	TMDY_REVERB->reverb_status.pre_lpf = reverb_macro_presets[macro+1];
	TMDY_REVERB->reverb_status.level = reverb_macro_presets[macro+2];
	TMDY_REVERB->reverb_status.time = reverb_macro_presets[macro+3];
	TMDY_REVERB->reverb_status.delay_feedback = reverb_macro_presets[macro+4];
	TMDY_REVERB->reverb_status.pre_delay_time = reverb_macro_presets[macro+5];
}

void init_chorus_status(tmdy_struct_ex_t *tmdy_struct)
{
	TMDY_REVERB->chorus_param.chorus_macro = 0;
	TMDY_REVERB->chorus_param.chorus_pre_lpf = 0;
	TMDY_REVERB->chorus_param.chorus_level = 0x40;
	TMDY_REVERB->chorus_param.chorus_feedback = 0x08;
	TMDY_REVERB->chorus_param.chorus_delay = 0x50;
	TMDY_REVERB->chorus_param.chorus_rate = 0x03;
	TMDY_REVERB->chorus_param.chorus_depth = 0x13;
	TMDY_REVERB->chorus_param.chorus_send_level_to_reverb = 0;
	TMDY_REVERB->chorus_param.chorus_send_level_to_delay = 0;
	recompute_chorus_status(tmdy_struct);
	TMDY_REVERB->init_chorus_lfo(tmdy_struct);
}

void recompute_chorus_status(tmdy_struct_ex_t *tmdy_struct)
{
	FLOAT_T dBGain;

	TMDY_REVERB->chorus_param.delay_in_sample = pre_delay_time_table[TMDY_REVERB->chorus_param.chorus_delay] * (double)(TMDY_OUTPUT->play_mode)->rate / 1000.0;
	TMDY_REVERB->chorus_param.depth_in_sample = TMDY_REVERB->chorus_param.delay_in_sample * TMDY_REVERB->chorus_param.chorus_depth / 127;
	TMDY_REVERB->chorus_param.cycle_in_sample = (TMDY_OUTPUT->play_mode)->rate / rate1_table[TMDY_REVERB->chorus_param.chorus_rate];
	TMDY_REVERB->chorus_param.feedback_ratio = (double)TMDY_REVERB->chorus_param.chorus_feedback * 0.0077165;
	TMDY_REVERB->chorus_param.level_ratio = (double)TMDY_REVERB->chorus_param.chorus_level / 127.0;
	TMDY_REVERB->chorus_param.send_reverb_ratio = (double)TMDY_REVERB->chorus_param.chorus_send_level_to_reverb / 127.0;
	TMDY_REVERB->chorus_param.send_delay_ratio = (double)TMDY_REVERB->chorus_param.chorus_send_level_to_delay / 127.0;

	if(TMDY_REVERB->chorus_param.chorus_pre_lpf) {
		dBGain = gs_system_effect_hsf_gain_table[TMDY_REVERB->chorus_param.chorus_pre_lpf];
		/* calculate highpass shelving filter's coefficients */
		TMDY_REVERB->calc_highshelf_coefs(tmdy_struct, TMDY_REVERB->chorus_param.high_coef, SYSTEM_EFFECT_LPF_FC, dBGain, (TMDY_OUTPUT->play_mode)->rate);
	}
}

void set_chorus_macro(tmdy_struct_ex_t *tmdy_struct, int macro)
{
	macro *= 8;
	TMDY_REVERB->chorus_param.chorus_pre_lpf = chorus_macro_presets[macro];
	TMDY_REVERB->chorus_param.chorus_level = chorus_macro_presets[macro+1];
	TMDY_REVERB->chorus_param.chorus_feedback = chorus_macro_presets[macro+2];
	TMDY_REVERB->chorus_param.chorus_delay = chorus_macro_presets[macro+3];
	TMDY_REVERB->chorus_param.chorus_rate = chorus_macro_presets[macro+4];
	TMDY_REVERB->chorus_param.chorus_depth = chorus_macro_presets[macro+5];
	TMDY_REVERB->chorus_param.chorus_send_level_to_reverb = chorus_macro_presets[macro+6];
	TMDY_REVERB->chorus_param.chorus_send_level_to_delay = chorus_macro_presets[macro+7];
}

void init_eq_status(tmdy_struct_ex_t *tmdy_struct)
{
	TMDY_REVERB->eq_status.low_freq = 0;
	TMDY_REVERB->eq_status.low_gain = 0x40;
	TMDY_REVERB->eq_status.high_freq = 0;
	TMDY_REVERB->eq_status.high_gain = 0x40;
	recompute_eq_status(tmdy_struct);
}

void recompute_eq_status(tmdy_struct_ex_t *tmdy_struct)
{
	int32 freq;
	FLOAT_T dbGain;

	/* Lowpass Shelving Filter */
	if(TMDY_REVERB->eq_status.low_freq == 0) {freq = 200;}
	else {freq = 400;}
	dbGain = TMDY_REVERB->eq_status.low_gain - 0x40;
	if(freq < (TMDY_OUTPUT->play_mode)->rate / 2) {
		TMDY_REVERB->calc_lowshelf_coefs(tmdy_struct, TMDY_REVERB->eq_status.low_coef,freq,dbGain,(TMDY_OUTPUT->play_mode)->rate);
	}

	/* Highpass Shelving Filter */
	if(TMDY_REVERB->eq_status.high_freq == 0) {freq = 3000;}
	else {freq = 6000;}
	dbGain = TMDY_REVERB->eq_status.high_gain - 0x40;
	if(freq < (TMDY_OUTPUT->play_mode)->rate / 2) {
		TMDY_REVERB->calc_highshelf_coefs(tmdy_struct, TMDY_REVERB->eq_status.high_coef,freq,dbGain,(TMDY_OUTPUT->play_mode)->rate);
	}
}

/*! convert GS user drumset assign groups to internal "alternate assign". */
void recompute_userdrum_altassign(tmdy_struct_ex_t *tmdy_struct, int bank, int group)
{
	int number = 0;
	char *params[131], param[10];
	ToneBank *bk;
	UserDrumset *p;
	
	for(p = TMDY_READMIDI->userdrum_first; p != NULL; p = p->next) {
		if(p->assign_group == group) {
			sprintf(param, "%d", p->prog);
			params[number] = TMDY_COMMON->safe_strdup(tmdy_struct, param);
			number++;
		}
	}
	params[number] = NULL;

	TMDY_INSTRUM->alloc_instrument_bank(tmdy_struct, 1, bank);
	bk = TMDY_INSTRUM->drumset[bank];
	bk->alt =TMDY_INSTRUM->add_altassign_string(tmdy_struct, bk->alt, params, number);
}

/*! initialize GS user drumset. */
void init_userdrum(tmdy_struct_ex_t *tmdy_struct)
{
	int i;
	AlternateAssign *alt;

	free_userdrum(tmdy_struct);

	for(i=0;i<2;i++) {	/* allocate alternative assign */
		alt = (AlternateAssign *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(AlternateAssign));
		memset(alt, 0, sizeof(AlternateAssign));
		TMDY_INSTRUM->alloc_instrument_bank(tmdy_struct, 1, 64 + i);
		TMDY_INSTRUM->drumset[64 + i]->alt = alt;
	}
}

/*! recompute GS user drumset. */
void recompute_userdrum(tmdy_struct_ex_t *tmdy_struct, int bank, int prog)
{
	UserDrumset *p;

	p = get_userdrum(tmdy_struct, bank, prog);

	if(TMDY_INSTRUM->drumset[bank]->tone[prog].name) {TMDY_PLAYMIDI->free_tone_bank_element(tmdy_struct, 1, bank, prog);}
	if(TMDY_INSTRUM->drumset[p->source_prog]) {
		if(TMDY_INSTRUM->drumset[p->source_prog]->tone[p->source_note].name) {
			memcpy(&(TMDY_INSTRUM->drumset)[bank]->tone[prog], &(TMDY_INSTRUM->drumset)[p->source_prog]->tone[p->source_note], sizeof(ToneBankElement));
			TMDY_PLAYMIDI->dup_tone_bank_element(tmdy_struct, 1, bank, prog);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"User Drumset (%d %d -> %d %d)", p->source_prog, p->source_note, bank, prog);
		} else if(TMDY_INSTRUM->drumset[0]->tone[p->source_note].name) {
			memcpy(&(TMDY_INSTRUM->drumset)[bank]->tone[prog], &(TMDY_INSTRUM->drumset)[0]->tone[p->source_note], sizeof(ToneBankElement));
			TMDY_PLAYMIDI->dup_tone_bank_element(tmdy_struct, 1, bank, prog);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"User Drumset (%d %d -> %d %d)", 0, p->source_note, bank, prog);
		}
	}
}

/*! get pointer to requested GS user drumset.
   if it's not found, allocate a new item first. */
UserDrumset *get_userdrum(tmdy_struct_ex_t *tmdy_struct, int bank, int prog)
{
	UserDrumset *p;

	for(p = TMDY_READMIDI->userdrum_first; p != NULL; p = p->next) {
		if(p->bank == bank && p->prog == prog) {return p;}
	}

	p = (UserDrumset *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(UserDrumset));
	memset(p, 0, sizeof(UserDrumset));
	p->next = NULL;
	if(TMDY_READMIDI->userdrum_first == NULL) {
		TMDY_READMIDI->userdrum_first = p;
		TMDY_READMIDI->userdrum_last = p;
	} else {
		TMDY_READMIDI->userdrum_last->next = p;
		TMDY_READMIDI->userdrum_last = p;
	}
	p->bank = bank;
	p->prog = prog;

	return p;
}

/*! free GS user drumset. */
void free_userdrum(tmdy_struct_ex_t *tmdy_struct)
{
	UserDrumset *p, *next;
	
	for(p = TMDY_READMIDI->userdrum_first; p != NULL; p = next){
		next = p->next;
		free(p);
    }
	TMDY_READMIDI->userdrum_first = TMDY_READMIDI->userdrum_last = NULL;
}

/*! initialize GS user instrument. */
void init_userinst(tmdy_struct_ex_t *tmdy_struct)
{
	free_userinst(tmdy_struct);
}

/*! recompute GS user instrument. */
void recompute_userinst(tmdy_struct_ex_t *tmdy_struct, int bank, int prog)
{
	UserInstrument *p;

	p = get_userinst(tmdy_struct, bank, prog);

	if(TMDY_INSTRUM->tonebank[bank]->tone[prog].name) {TMDY_PLAYMIDI->free_tone_bank_element(tmdy_struct, 0, bank, prog);}
	if(TMDY_INSTRUM->tonebank[p->source_bank]) {
		if(TMDY_INSTRUM->tonebank[p->source_bank]->tone[p->source_prog].name) {
			memcpy(&(TMDY_INSTRUM->tonebank)[bank]->tone[prog], &(TMDY_INSTRUM->tonebank)[p->source_bank]->tone[p->source_prog], sizeof(ToneBankElement));
			TMDY_PLAYMIDI->dup_tone_bank_element(tmdy_struct, 0, bank, prog);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"User Instrument (%d %d -> %d %d)", p->source_bank, p->source_prog, bank, prog);
		} else if(TMDY_INSTRUM->tonebank[0]->tone[p->source_prog].name) {
			memcpy(&(TMDY_INSTRUM->tonebank)[bank]->tone[prog], &(TMDY_INSTRUM->tonebank)[0]->tone[p->source_prog], sizeof(ToneBankElement));
			TMDY_PLAYMIDI->dup_tone_bank_element(tmdy_struct, 0, bank, prog);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"User Instrument (%d %d -> %d %d)", 0, p->source_prog, bank, prog);
		}
	}
}

/*! get pointer to requested GS user instrument.
   if it's not found, allocate a new item first. */
UserInstrument *get_userinst(tmdy_struct_ex_t *tmdy_struct, int bank, int prog)
{
	UserInstrument *p;

	for(p = TMDY_READMIDI->userinst_first; p != NULL; p = p->next) {
		if(p->bank == bank && p->prog == prog) {return p;}
	}

	p = (UserInstrument *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(UserInstrument));
	memset(p, 0, sizeof(UserInstrument));
	p->next = NULL;
	if(TMDY_READMIDI->userinst_first == NULL) {
		TMDY_READMIDI->userinst_first = p;
		TMDY_READMIDI->userinst_last = p;
	} else {
		TMDY_READMIDI->userinst_last->next = p;
		TMDY_READMIDI->userinst_last = p;
	}
	p->bank = bank;
	p->prog = prog;

	return p;
}

/*! free GS user instrument. */
void free_userinst(tmdy_struct_ex_t *tmdy_struct)
{
	UserInstrument *p, *next;

	for(p = TMDY_READMIDI->userinst_first; p != NULL; p = next){
		next = p->next;
		free(p);
    }
	TMDY_READMIDI->userinst_first = TMDY_READMIDI->userinst_last = NULL;
}

/*! initialize GS insertion effect parameters */
void init_insertion_effect_status(tmdy_struct_ex_t *tmdy_struct)
{
	int i;
	struct GSInsertionEffect *st = &gs_ieffect;

	TMDY_REVERB->free_effect_list(tmdy_struct, st->ef);
	st->ef = NULL;

	for(i = 0; i < 20; i++) {st->parameter[i] = 0;}

	st->type = 0;
	st->type_lsb = 0;
	st->type_msb = 0;
	st->send_reverb = 0x28;
	st->send_chorus = 0;
	st->send_delay = 0;
	st->control_source1 = 0;
	st->control_depth1 = 0x40;
	st->control_source2 = 0;
	st->control_depth2 = 0x40;
	st->send_eq_switch = 0x01;
}

/*! set GS default insertion effect parameters according to effect type. */
void set_insertion_effect_default_parameter(tmdy_struct_ex_t *tmdy_struct)
{
	struct GSInsertionEffect *st = &gs_ieffect;
	int8 *param = st->parameter;

	switch(st->type) {
	case 0x0110: /* Overdrive */
		param[0] = 48;
		param[1] = 1;
		param[2] = 1;
		param[16] = 0x40;
		param[17] = 0x40;
		param[18] = 0x40;
		param[19] = 96;
		break;
	case 0x0111: /* Distortion */
		param[0] = 76;
		param[1] = 3;
		param[2] = 1;
		param[16] = 0x40;
		param[17] = 0x38;
		param[18] = 0x40;
		param[19] = 84;
		break;
	case 0x1103: /* OD1 / OD2 */
		param[0] = 0;
		param[1] = 48;
		param[2] = 1;
		param[3] = 1;
		param[15] = 0x40;
		param[16] = 96;
		param[5] = 1;
		param[6] = 76;
		param[7] = 3;
		param[8] = 1;
		param[17] = 0x40;
		param[18] = 84;
		param[19] = 127;
		break;
	default: break;
	}
}

/*! convert GS insertion effect parameters for internal 2-band equalizer. */
static void *conv_gs_ie_to_eq2(tmdy_struct_ex_t *tmdy_struct, struct GSInsertionEffect *ieffect)
{
	struct InfoEQ2 *eq = (struct InfoEQ2 *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(struct InfoEQ2));

	eq->high_freq = 4000;
	eq->high_gain = ieffect->parameter[16] - 0x40;
	eq->low_freq = 400;
	eq->low_gain = ieffect->parameter[17] - 0x40;

	return eq;
}

/*! recompute GS insertion effect parameters. */
void recompute_insertion_effect(tmdy_struct_ex_t *tmdy_struct)
{
	struct GSInsertionEffect *st = &gs_ieffect;
	void *info;

	TMDY_REVERB->free_effect_list(tmdy_struct, st->ef);
	st->ef = NULL;

	switch(st->type) {
	case 0x0110: /* Overdrive */
		info = conv_gs_ie_to_eq2(tmdy_struct, st);
		st->ef = TMDY_REVERB->push_effect(tmdy_struct, st->ef, EFFECT_EQ2, info);
		break;
	case 0x0111: /* Distortion */
		info = conv_gs_ie_to_eq2(tmdy_struct, st);
		st->ef = TMDY_REVERB->push_effect(tmdy_struct, st->ef, EFFECT_EQ2, info);
		break;
	case 0x1103: /* OD1 / OD2 */
		break;
	default: break;
	}
}

/*! initialize channel layers. */
void init_channel_layer(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	if (ch >= MAX_CHANNELS)
		return;
	CLEAR_CHANNELMASK(TMDY_PLAYMIDI->channel[ch].channel_layer);
	SET_CHANNELMASK(TMDY_PLAYMIDI->channel[ch].channel_layer, ch);
	TMDY_PLAYMIDI->channel[ch].port_select = (ch < REDUCE_CHANNELS) ? 0 : 1;
}

/*! add a new layer. */
void add_channel_layer(tmdy_struct_ex_t *tmdy_struct, int to_ch, int from_ch)
{
	if (to_ch >= MAX_CHANNELS || from_ch >= MAX_CHANNELS)
		return;
	/* add a channel layer */
	UNSET_CHANNELMASK(TMDY_PLAYMIDI->channel[to_ch].channel_layer, to_ch);
	SET_CHANNELMASK(TMDY_PLAYMIDI->channel[to_ch].channel_layer, from_ch);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Channel Layer (CH:%d -> CH:%d)", from_ch, to_ch);
}

/*! remove all layers for this channel. */
void remove_channel_layer(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	int i, offset;
	
	if (ch >= MAX_CHANNELS)
		return;
	/* remove channel layers */
	offset = (ch < REDUCE_CHANNELS) ? 0 : REDUCE_CHANNELS;
	for (i = offset; i < offset + REDUCE_CHANNELS; i++)
		UNSET_CHANNELMASK(TMDY_PLAYMIDI->channel[i].channel_layer, ch);
	SET_CHANNELMASK(TMDY_PLAYMIDI->channel[ch].channel_layer, ch);
}







readmidi_ex_t* new_readmidi(tmdy_struct_ex_t *tmdy_struct){
	readmidi_ex_t* readmidi_ex;
	
	readmidi_ex=(readmidi_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(readmidi_ex_t));

	readmidi_ex->readmidi_read_init=readmidi_read_init;
	readmidi_ex->readmidi_set_track=readmidi_set_track;
	readmidi_ex->readmidi_add_event=readmidi_add_event;
	readmidi_ex->readmidi_add_ctl_event=readmidi_add_ctl_event;
	readmidi_ex->parse_sysex_event=parse_sysex_event;
	readmidi_ex->parse_sysex_event_multi=parse_sysex_event_multi;
	readmidi_ex->convert_midi_control_change=convert_midi_control_change;
	readmidi_ex->readmidi_make_string_event=readmidi_make_string_event;
	readmidi_ex->read_midi_file=read_midi_file;
	readmidi_ex->get_midi_file_info=get_midi_file_info;
	readmidi_ex->new_midi_file_info=new_midi_file_info;
	readmidi_ex->free_all_midi_file_info=free_all_midi_file_info;
	readmidi_ex->check_midi_file=check_midi_file;
	readmidi_ex->get_midi_title=get_midi_title;
	readmidi_ex->open_midi_file=open_midi_file;
	readmidi_ex->midi_file_save_as=midi_file_save_as;
	readmidi_ex->event2string=event2string;
	readmidi_ex->change_system_mode=change_system_mode;
	readmidi_ex->get_default_mapID=get_default_mapID;
	readmidi_ex->dump_current_timesig=dump_current_timesig;



	readmidi_ex->current_file_info=NULL;
#ifdef ALWAYS_TRACE_TEXT_META_EVENT
	readmidi_ex->opt_trace_text_meta_event=1;
#else
	readmidi_ex->opt_trace_text_meta_event=1;
#endif /* ALWAYS_TRACE_TEXT_META_EVENT */
	readmidi_ex->opt_default_mid=0;
	readmidi_ex->opt_system_mid=0;
	readmidi_ex->ignore_midi_error=1;
	readmidi_ex->readmidi_error_flag=0;
	readmidi_ex->readmidi_wrd_mode=0;
	readmidi_ex->play_system_mode=DEFAULT_SYSTEM_MODE;
	readmidi_ex->tempo_adjust=1.0;
	readmidi_ex->midi_file_info = NULL;


	readmidi_ex->recompute_delay_status=recompute_delay_status;
	readmidi_ex->set_delay_macro=set_delay_macro;
	readmidi_ex->recompute_chorus_status=recompute_chorus_status;
	readmidi_ex->set_chorus_macro=set_chorus_macro;
	readmidi_ex->recompute_reverb_status=recompute_reverb_status;
	readmidi_ex->set_reverb_macro=set_reverb_macro;
	readmidi_ex->recompute_eq_status=recompute_eq_status;
	readmidi_ex->set_insertion_effect_default_parameter=set_insertion_effect_default_parameter;
	
	readmidi_ex->recompute_insertion_effect=recompute_insertion_effect;

	readmidi_ex->recompute_userdrum=recompute_userdrum;
	readmidi_ex->free_userdrum=free_userdrum;

	readmidi_ex->recompute_userinst=recompute_userinst;
	readmidi_ex->free_userinst=free_userinst;

	readmidi_ex->init_channel_layer=init_channel_layer;
	readmidi_ex->add_channel_layer=add_channel_layer;
	readmidi_ex->remove_channel_layer=remove_channel_layer;

	/**** private values ****/
	readmidi_ex->string_event_table = NULL;
	readmidi_ex->string_event_table_size = 0;
	readmidi_ex->userdrum_first = (UserDrumset *)NULL;
	readmidi_ex->userdrum_last = (UserDrumset *)NULL;
	readmidi_ex->userinst_first = (UserInstrument *)NULL;
	readmidi_ex->userinst_last = (UserInstrument *)NULL;
	readmidi_ex->midi_file_info =NULL;
	TMDY_UTILS->mblock->init_mblock(tmdy_struct, &(readmidi_ex->mempool));
	
	/* rcp.c */
	readmidi_ex->read_rcp_file=read_rcp_file;
	/* smfconv.c */
#ifdef SMFCONV
#ifdef __W32__
	readmidi_ex->is_midifile_filename=is_midifile_filename;
	readmidi_ex->smfconv_w32=smfconv_w32;
#endif /* __W32__ */
#endif /* SMFCONV */
	readmidi_ex->is_midifile_check[0].type=IS_RCP_FILE;
	readmidi_ex->is_midifile_check[0].ext=(char *)safe_strdup(tmdy_struct,".RCP");
	readmidi_ex->is_midifile_check[0].header=(char *)safe_strdup(tmdy_struct,"RCM-PC98V2.0(C)COME ON MUSIC");
	readmidi_ex->is_midifile_check[0].len=28;

	readmidi_ex->is_midifile_check[1].type=IS_R36_FILE;
	readmidi_ex->is_midifile_check[1].ext=(char *)safe_strdup(tmdy_struct,".R36");
	readmidi_ex->is_midifile_check[1].header=(char *)safe_strdup(tmdy_struct,"RCM-PC98V2.0(C)COME ON MUSIC");
	readmidi_ex->is_midifile_check[1].len=28;

	readmidi_ex->is_midifile_check[2].type=IS_G18_FILE;
	readmidi_ex->is_midifile_check[2].ext=(char *)safe_strdup(tmdy_struct,".G18");
	readmidi_ex->is_midifile_check[2].header=(char *)safe_strdup(tmdy_struct,"COME ON MUSIC RECOMPOSER RCP3.0");
	readmidi_ex->is_midifile_check[2].len=31;

	readmidi_ex->is_midifile_check[3].type=IS_G36_FILE;
	readmidi_ex->is_midifile_check[3].ext=(char *)safe_strdup(tmdy_struct,".G36");
	readmidi_ex->is_midifile_check[3].header=(char *)safe_strdup(tmdy_struct,"COME ON MUSIC RECOMPOSER RCP3.0");
	readmidi_ex->is_midifile_check[3].len=31;

	readmidi_ex->is_midifile_check[4].type=IS_MCP_FILE;
	readmidi_ex->is_midifile_check[4].ext=(char *)safe_strdup(tmdy_struct,".MCP");
	readmidi_ex->is_midifile_check[4].header=(char *)safe_strdup(tmdy_struct,"M1");
	readmidi_ex->is_midifile_check[4].len=2;
	
	readmidi_ex->is_midifile_check[5].type=IS_OTHER_FILE;
	readmidi_ex->is_midifile_check[5].ext=NULL;
	readmidi_ex->is_midifile_check[5].header=NULL;
	readmidi_ex->is_midifile_check[5].len=0;

	readmidi_ex->read_mfi_file=read_mfi_file;
	readmidi_ex->get_mfi_file_title=get_mfi_file_title;
	
	
	return readmidi_ex;
}
void destroy_readmidi(readmidi_ex_t* readmidi){
	int i;
	for(i=0;i<5;i++){
		free(readmidi->is_midifile_check[i].ext);
		free(readmidi->is_midifile_check[i].header);
	}
	free(readmidi);
}

