/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    playmidi.c -- random stuff in need of rearrangement

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef __W32__
#include "interface.h"
#endif
#include <stdio.h>
#include <stdlib.h>

#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#ifdef __W32__
#include <windows.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "tables.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "mix.h"
#include "controls.h"
#include "miditrace.h"
#include "recache.h"
#include "arc.h"
#include "reverb.h"
#include "wrd.h"
#include "aq.h"
#include "freq.h"
#include "quantity.h"
#include "timidity_main.h"

#include "playmidi_prv.h" //guha

/* m2m.c */
extern void convert_mod_to_midi_file(tmdy_struct_ex_t *tmdy_struct, MidiEvent * ev);

#define ABORT_AT_FATAL 1 /*#################*/
#define MYCHECK(s) do { if(s == 0) { printf("## L %d\n", __LINE__); abort(); } } while(0)

//extern VOLATILE int intr;

/* #define SUPPRESS_CHANNEL_LAYER */

#ifdef SOLARIS
/* shut gcc warning up */
int usleep(unsigned int useconds);
#endif

#ifdef SUPPORT_SOUNDSPEC
#include "soundspec.h"
#endif /* SUPPORT_SOUNDSPEC */

#include "tables.h"

#define PLAY_INTERLEAVE_SEC		1.0
#define PORTAMENTO_TIME_TUNING		(1.0 / 5000.0)
#define PORTAMENTO_CONTROL_RATIO	256	/* controls per sec */
#define DEFAULT_CHORUS_DELAY1		0.02
#define DEFAULT_CHORUS_DELAY2		0.003
#define CHORUS_OPPOSITE_THRESHOLD	32
#define CHORUS_VELOCITY_TUNING1		0.7
#define CHORUS_VELOCITY_TUNING2		0.6
#define EOT_PRESEARCH_LEN		32
#define SPEED_CHANGE_RATE		1.0594630943592953  /* 2^(1/12) */


static void set_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch, int level);

/* for auto amplitude compensation */


static void update_portamento_controls(tmdy_struct_ex_t *tmdy_struct, int ch);
static void update_rpn_map(tmdy_struct_ex_t *tmdy_struct, int ch, int addr, int update_now);
static void ctl_prog_event(tmdy_struct_ex_t *tmdy_struct, int ch);
static void ctl_timestamp(tmdy_struct_ex_t *tmdy_struct);
static void ctl_updatetime(tmdy_struct_ex_t *tmdy_struct, int32 samples);
static void ctl_pause_event(tmdy_struct_ex_t *tmdy_struct, int pause, int32 samples);
static void update_legato_controls(tmdy_struct_ex_t *tmdy_struct, int ch);
static void update_channel_freq(tmdy_struct_ex_t *tmdy_struct, int ch);
static void set_single_note_tuning(tmdy_struct_ex_t *tmdy_struct, int, int, int, int);

#define IS_SYSEX_EVENT_TYPE(type) ((type) == ME_NONE || (type) >= ME_RANDOM_PAN)

static char *event_name(tmdy_struct_ex_t *tmdy_struct, int type)
{
#define EVENT_NAME(X) case X: return #X
	switch (type) {
	EVENT_NAME(ME_NONE);
	EVENT_NAME(ME_NOTEOFF);
	EVENT_NAME(ME_NOTEON);
	EVENT_NAME(ME_KEYPRESSURE);
	EVENT_NAME(ME_PROGRAM);
	EVENT_NAME(ME_CHANNEL_PRESSURE);
	EVENT_NAME(ME_PITCHWHEEL);
	EVENT_NAME(ME_TONE_BANK_MSB);
	EVENT_NAME(ME_TONE_BANK_LSB);
	EVENT_NAME(ME_MODULATION_WHEEL);
	EVENT_NAME(ME_BREATH);
	EVENT_NAME(ME_FOOT);
	EVENT_NAME(ME_MAINVOLUME);
	EVENT_NAME(ME_BALANCE);
	EVENT_NAME(ME_PAN);
	EVENT_NAME(ME_EXPRESSION);
	EVENT_NAME(ME_SUSTAIN);
	EVENT_NAME(ME_PORTAMENTO_TIME_MSB);
	EVENT_NAME(ME_PORTAMENTO_TIME_LSB);
	EVENT_NAME(ME_PORTAMENTO);
	EVENT_NAME(ME_PORTAMENTO_CONTROL);
	EVENT_NAME(ME_DATA_ENTRY_MSB);
	EVENT_NAME(ME_DATA_ENTRY_LSB);
	EVENT_NAME(ME_SOSTENUTO);
	EVENT_NAME(ME_SOFT_PEDAL);
	EVENT_NAME(ME_LEGATO_FOOTSWITCH);
	EVENT_NAME(ME_HOLD2);
	EVENT_NAME(ME_HARMONIC_CONTENT);
	EVENT_NAME(ME_RELEASE_TIME);
	EVENT_NAME(ME_ATTACK_TIME);
	EVENT_NAME(ME_BRIGHTNESS);
	EVENT_NAME(ME_REVERB_EFFECT);
	EVENT_NAME(ME_TREMOLO_EFFECT);
	EVENT_NAME(ME_CHORUS_EFFECT);
	EVENT_NAME(ME_CELESTE_EFFECT);
	EVENT_NAME(ME_PHASER_EFFECT);
	EVENT_NAME(ME_RPN_INC);
	EVENT_NAME(ME_RPN_DEC);
	EVENT_NAME(ME_NRPN_LSB);
	EVENT_NAME(ME_NRPN_MSB);
	EVENT_NAME(ME_RPN_LSB);
	EVENT_NAME(ME_RPN_MSB);
	EVENT_NAME(ME_ALL_SOUNDS_OFF);
	EVENT_NAME(ME_RESET_CONTROLLERS);
	EVENT_NAME(ME_ALL_NOTES_OFF);
	EVENT_NAME(ME_MONO);
	EVENT_NAME(ME_POLY);
#if 0
	EVENT_NAME(ME_VOLUME_ONOFF);		/* Not supported */
#endif
	EVENT_NAME(ME_SCALE_TUNING);
	EVENT_NAME(ME_BULK_TUNING_DUMP);
	EVENT_NAME(ME_SINGLE_NOTE_TUNING);
	EVENT_NAME(ME_RANDOM_PAN);
	EVENT_NAME(ME_SET_PATCH);
	EVENT_NAME(ME_DRUMPART);
	EVENT_NAME(ME_KEYSHIFT);
	EVENT_NAME(ME_PATCH_OFFS);
	EVENT_NAME(ME_TEMPO);
	EVENT_NAME(ME_CHORUS_TEXT);
	EVENT_NAME(ME_LYRIC);
	EVENT_NAME(ME_GSLCD);
	EVENT_NAME(ME_MARKER);
	EVENT_NAME(ME_INSERT_TEXT);
	EVENT_NAME(ME_TEXT);
	EVENT_NAME(ME_KARAOKE_LYRIC);
	EVENT_NAME(ME_MASTER_VOLUME);
	EVENT_NAME(ME_RESET);
	EVENT_NAME(ME_NOTE_STEP);
	EVENT_NAME(ME_TIMESIG);
	EVENT_NAME(ME_KEYSIG);
	EVENT_NAME(ME_TEMPER_KEYSIG);
	EVENT_NAME(ME_TEMPER_TYPE);
	EVENT_NAME(ME_MASTER_TEMPER_TYPE);
	EVENT_NAME(ME_SYSEX_LSB);
	EVENT_NAME(ME_SYSEX_MSB);
	EVENT_NAME(ME_SYSEX_GS_LSB);
	EVENT_NAME(ME_SYSEX_GS_MSB);
	EVENT_NAME(ME_SYSEX_XG_LSB);
	EVENT_NAME(ME_SYSEX_XG_MSB);
	EVENT_NAME(ME_WRD);
	EVENT_NAME(ME_SHERRY);
	EVENT_NAME(ME_BARMARKER);
	EVENT_NAME(ME_STEP);
	EVENT_NAME(ME_LAST);
	EVENT_NAME(ME_EOT);
	}
	return "Unknown";
#undef EVENT_NAME
}

static void adjust_amplification(tmdy_struct_ex_t *tmdy_struct)
{
    /* compensate master volume */
    (TMDY_PLAYMIDI->master_volume) = (double)((TMDY_PLAYMIDI->amplification)) / 100.0 *
	((double)(TMDY_PLAYMIDI->master_volume_ratio) * ((TMDY_PLAYMIDI->compensation_ratio)/0xFFFF));
}

static int new_vidq(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i;

    if((TMDY_PLAYMIDI->opt_overlap_voice_allow))
    {
	i = ch * 128 + note;
	return (TMDY_PLAYMIDI->vidq_head)[i]++;
    }
    return 0;
}

static int last_vidq(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i;

    if((TMDY_PLAYMIDI->opt_overlap_voice_allow))
    {
	i = ch * 128 + note;
	if((TMDY_PLAYMIDI->vidq_head)[i] == (TMDY_PLAYMIDI->vidq_tail)[i])
	{
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_DEBUG_SILLY,
		      "channel=%d, note=%d: Voice is already OFF", ch, note);
	    return -1;
	}
	return (TMDY_PLAYMIDI->vidq_tail)[i]++;
    }
    return 0;
}

static void reset_voices(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
    for(i = 0; i < MAX_VOICES; i++)
    {
	(TMDY_PLAYMIDI->voice)[i].status = VOICE_FREE;
	(TMDY_PLAYMIDI->voice)[i].temper_instant = 0;
	(TMDY_PLAYMIDI->voice)[i].chorus_link = i;
    }
    (TMDY_PLAYMIDI->upper_voices) = 0;
    memset(TMDY_PLAYMIDI->vidq_head, 0, sizeof(TMDY_PLAYMIDI->vidq_head));
    memset(TMDY_PLAYMIDI->vidq_tail, 0, sizeof(TMDY_PLAYMIDI->vidq_tail));
}

static void kill_note(tmdy_struct_ex_t *tmdy_struct, int i)
{
    (TMDY_PLAYMIDI->voice)[i].status = VOICE_DIE;
    if(!TMDY_PLAYMIDI->prescanning_flag)
	ctl_note_event(tmdy_struct, i);
}

static void kill_all_voices(tmdy_struct_ex_t *tmdy_struct)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices);

    for(i = 0; i < uv; i++)
	if((TMDY_PLAYMIDI->voice)[i].status & ~(VOICE_FREE | VOICE_DIE))
	    kill_note(tmdy_struct, i);
    memset(TMDY_PLAYMIDI->vidq_head, 0, sizeof(TMDY_PLAYMIDI->vidq_head));
    memset(TMDY_PLAYMIDI->vidq_tail, 0, sizeof(TMDY_PLAYMIDI->vidq_tail));
}

static void reset_drum_controllers(tmdy_struct_ex_t *tmdy_struct, struct DrumParts *d[], int note)
{
    int i,j;

    if(note == -1)
    {
	for(i = 0; i < 128; i++)
	    if(d[i] != NULL)
	    {
		d[i]->drum_panning = NO_PANNING;
		for(j=0;j<6;j++) {d[i]->drum_envelope_rate[j] = -1;}
		d[i]->pan_random = 0;
		d[i]->drum_level = 1.0f;
		d[i]->coarse = 0;
		d[i]->fine = 0;
		d[i]->delay_level = 0;
		d[i]->chorus_level = 0;
		d[i]->reverb_level = 0;
		d[i]->play_note = -1;
		d[i]->rx_note_off = 0;
		d[i]->drum_cutoff_freq = 0;
		d[i]->drum_resonance = 0;
	    }
    }
    else
    {
	d[note]->drum_panning = NO_PANNING;
	for(j=0;j<6;j++) {d[note]->drum_envelope_rate[j] = -1;}
	d[note]->pan_random = 0;
	d[note]->drum_level = 1.0f;
	d[note]->coarse = 0;
	d[note]->fine = 0;
	d[note]->delay_level = 0;
	d[note]->chorus_level = 0;
	d[note]->reverb_level = 0;
	d[note]->play_note = -1;
	d[note]->rx_note_off = 0;
	d[note]->drum_cutoff_freq = 0;
	d[note]->drum_resonance = 0;
    }
}

static void reset_nrpn_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i,j;

  /* NRPN */
  reset_drum_controllers(tmdy_struct, (TMDY_PLAYMIDI->channel)[c].drums, -1);
  (TMDY_PLAYMIDI->channel)[c].vibrato_ratio = 0;
  (TMDY_PLAYMIDI->channel)[c].vibrato_depth = 0;
  (TMDY_PLAYMIDI->channel)[c].vibrato_delay = 0;
  (TMDY_PLAYMIDI->channel)[c].param_cutoff_freq = 0;
  (TMDY_PLAYMIDI->channel)[c].param_resonance = 0;
  (TMDY_PLAYMIDI->channel)[c].cutoff_freq_coef = 1.0;
  (TMDY_PLAYMIDI->channel)[c].resonance_dB = 0;

  /* GS & XG System Exclusive */
  (TMDY_PLAYMIDI->channel)[c].eq_on = 1;
  (TMDY_PLAYMIDI->channel)[c].insertion_effect = 0;
  (TMDY_PLAYMIDI->channel)[c].velocity_sense_depth = 0x40;
  (TMDY_PLAYMIDI->channel)[c].velocity_sense_offset = 0x40;
  (TMDY_PLAYMIDI->channel)[c].pitch_offset_fine = 0;
  (TMDY_PLAYMIDI->channel)[c].legato = 0;
  (TMDY_PLAYMIDI->channel)[c].assign_mode = 1;
  if(TMDY_READMIDI->play_system_mode == GS_SYSTEM_MODE) {
	  (TMDY_PLAYMIDI->channel)[c].bank_lsb = (TMDY_PLAYMIDI->channel)[c].tone_map0_number;
  }
	for (i = 0; i < 12; i++)
		(TMDY_PLAYMIDI->channel)[c].scale_tuning[i] = 0;
	(TMDY_PLAYMIDI->channel)[c].prev_scale_tuning = 0;
	(TMDY_PLAYMIDI->channel)[c].temper_type = 0;

  TMDY_READMIDI->init_channel_layer(tmdy_struct, c);

  /* channel pressure & polyphonic key pressure control */
  (TMDY_PLAYMIDI->channel)[c].caf_lfo1_rate_ctl = 0;

  (TMDY_PLAYMIDI->channel)[c].sysex_gs_msb_addr = (TMDY_PLAYMIDI->channel)[c].sysex_gs_msb_val =
	(TMDY_PLAYMIDI->channel)[c].sysex_xg_msb_addr = (TMDY_PLAYMIDI->channel)[c].sysex_xg_msb_val =
	(TMDY_PLAYMIDI->channel)[c].sysex_msb_addr = (TMDY_PLAYMIDI->channel)[c].sysex_msb_val = 0;
}

/* Process the Reset All Controllers event */
static void reset_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i,j;
    /* Some standard says, although the SCC docs say 0. */
    
  if(TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE)
      (TMDY_PLAYMIDI->channel)[c].volume = 100;
  else
      (TMDY_PLAYMIDI->channel)[c].volume = 90;
  if (TMDY_PLAYMIDI->prescanning_flag) {
    if ((TMDY_PLAYMIDI->channel)[c].volume > (TMDY_PLAYMIDI->mainvolume_max)) {	/* pick maximum value of mainvolume */
      (TMDY_PLAYMIDI->mainvolume_max) = (TMDY_PLAYMIDI->channel)[c].volume;
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"ME_MAINVOLUME/max (CH:%d VAL:%#x)",c,(TMDY_PLAYMIDI->mainvolume_max));
    }
  }

  (TMDY_PLAYMIDI->channel)[c].expression=127; /* SCC-1 does this. */
  (TMDY_PLAYMIDI->channel)[c].sustain=0;
  (TMDY_PLAYMIDI->channel)[c].pitchbend=0x2000;
  (TMDY_PLAYMIDI->channel)[c].pitchfactor=0; /* to be computed */
  (TMDY_PLAYMIDI->channel)[c].modulation_wheel = 0;
  (TMDY_PLAYMIDI->channel)[c].portamento_time_lsb = 0;
  (TMDY_PLAYMIDI->channel)[c].portamento_time_msb = 0;
  (TMDY_PLAYMIDI->channel)[c].porta_control_ratio = 0;
  (TMDY_PLAYMIDI->channel)[c].portamento = 0;
  (TMDY_PLAYMIDI->channel)[c].last_note_fine = -1;
  for(j=0;j<6;j++) {(TMDY_PLAYMIDI->channel)[c].envelope_rate[j] = -1;}
  update_portamento_controls(tmdy_struct, c);
  set_reverb_level(tmdy_struct, c, -1);
  if((TMDY_PLAYMIDI->opt_chorus_control) == 1)
      (TMDY_PLAYMIDI->channel)[c].chorus_level = 0;
  else
      (TMDY_PLAYMIDI->channel)[c].chorus_level = -(TMDY_PLAYMIDI->opt_chorus_control);
  (TMDY_PLAYMIDI->channel)[c].mono = 0;
  (TMDY_PLAYMIDI->channel)[c].delay_level = 0;
}

static void redraw_controllers(tmdy_struct_ex_t *tmdy_struct, int c)
{
    ctl_mode_event(tmdy_struct, CTLE_VOLUME, 1, c, (TMDY_PLAYMIDI->channel)[c].volume);
    ctl_mode_event(tmdy_struct, CTLE_EXPRESSION, 1, c, (TMDY_PLAYMIDI->channel)[c].expression);
    ctl_mode_event(tmdy_struct, CTLE_SUSTAIN, 1, c, (TMDY_PLAYMIDI->channel)[c].sustain);
    ctl_mode_event(tmdy_struct, CTLE_MOD_WHEEL, 1, c, (TMDY_PLAYMIDI->channel)[c].modulation_wheel);
    ctl_mode_event(tmdy_struct, CTLE_PITCH_BEND, 1, c, (TMDY_PLAYMIDI->channel)[c].pitchbend);
    ctl_prog_event(tmdy_struct, c);
    ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, c, (TMDY_PLAYMIDI->channel)[c].temper_type);
    ctl_mode_event(tmdy_struct, CTLE_MUTE, 1,
    		c, (IS_SET_CHANNELMASK(TMDY_PLAYMIDI->channel_mute, c)) ? 1 : 0);
    ctl_mode_event(tmdy_struct, CTLE_CHORUS_EFFECT, 1, c, get_chorus_level(tmdy_struct, c));
    ctl_mode_event(tmdy_struct, CTLE_REVERB_EFFECT, 1, c, get_reverb_level(tmdy_struct, c));
}

static void reset_midi(tmdy_struct_ex_t *tmdy_struct, int playing)
{
	int i, cnt;
	
	for (i = 0; i < MAX_CHANNELS; i++) {
		reset_controllers(tmdy_struct, i);
		reset_nrpn_controllers(tmdy_struct, i);
		/* The rest of these are unaffected
		 * by the Reset All Controllers event
		 */
		(TMDY_PLAYMIDI->channel)[i].program = TMDY_INSTRUM->default_program[i];
		(TMDY_PLAYMIDI->channel)[i].panning = NO_PANNING;
		(TMDY_PLAYMIDI->channel)[i].pan_random = 0;
		/* tone bank or drum set */
		if (ISDRUMCHANNEL(i)) {
			(TMDY_PLAYMIDI->channel)[i].bank = 0;
			(TMDY_PLAYMIDI->channel)[i].altassign = TMDY_INSTRUM->drumset[0]->alt;
		} else {
			if ((TMDY_PLAYMIDI->special_tonebank) >= 0)
				(TMDY_PLAYMIDI->channel)[i].bank = (TMDY_PLAYMIDI->special_tonebank);
			else
				(TMDY_PLAYMIDI->channel)[i].bank = (TMDY_PLAYMIDI->default_tonebank);
		}
		(TMDY_PLAYMIDI->channel)[i].bank_lsb = (TMDY_PLAYMIDI->channel)[i].bank_msb =
				(TMDY_PLAYMIDI->channel)[i].tone_map0_number = 0;
		if (TMDY_READMIDI->play_system_mode == XG_SYSTEM_MODE && i % 16 == 9)
			(TMDY_PLAYMIDI->channel)[i].bank_msb = 127;	/* Use MSB=127 for XG */
		update_rpn_map(tmdy_struct,i, RPN_ADDR_FFFF, 0);
		(TMDY_PLAYMIDI->channel)[i].special_sample = 0;
		(TMDY_PLAYMIDI->channel)[i].key_shift = 0;
		(TMDY_PLAYMIDI->channel)[i].mapID = TMDY_READMIDI->get_default_mapID(tmdy_struct, i);
		(TMDY_PLAYMIDI->channel)[i].lasttime = 0;
	}
	if (playing) {
		kill_all_voices(tmdy_struct);
		if ((TMDY_PLAYMIDI->temper_type_mute)) {
			if ((TMDY_PLAYMIDI->temper_type_mute) & 1)
				FILL_CHANNELMASK(TMDY_PLAYMIDI->channel_mute);
			else
				CLEAR_CHANNELMASK(TMDY_PLAYMIDI->channel_mute);
		}
		for (i = 0; i < MAX_CHANNELS; i++)
			redraw_controllers(tmdy_struct, i);
		if (TMDY_PLAYMIDI->midi_streaming && (TMDY_MAIN->free_instruments_afterwards)) {
			TMDY_INSTRUM->free_instruments(tmdy_struct, 0);
			/* free unused memory */
			cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);
			if (cnt > 0)
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
						"%d memory blocks are free", cnt);
		}
	} else
		reset_voices(tmdy_struct);
	(TMDY_PLAYMIDI->master_volume_ratio) = 0xffff;
	adjust_amplification(tmdy_struct);
	init_freq_table_tuning();
	if (TMDY_READMIDI->current_file_info) {
		COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), TMDY_READMIDI->current_file_info->drumchannels);
		COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannel_mask),
				TMDY_READMIDI->current_file_info->drumchannel_mask);
	} else {
		COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), (TMDY_PLAYMIDI->default_drumchannels));
		COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannel_mask), (TMDY_PLAYMIDI->default_drumchannel_mask));
	}
	ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (TMDY_PLAYMIDI->amplification), 0);
	ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (TMDY_PLAYMIDI->note_key_offset), 0);
	ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (TMDY_PLAYMIDI->midi_time_ratio) + 0.5, 0);
}

void recompute_freq(tmdy_struct_ex_t *tmdy_struct, int v)
{
	int i;
	int ch = (TMDY_PLAYMIDI->voice)[v].channel;
	int note = (TMDY_PLAYMIDI->voice)[v].note;
	int32 tuning = 0;
	int8 st = (TMDY_PLAYMIDI->channel)[ch].scale_tuning[note % 12];
	int8 tt = (TMDY_PLAYMIDI->channel)[ch].temper_type;
	uint8 tp = (TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0003];
	int32 f;
	int pb = (TMDY_PLAYMIDI->channel)[ch].pitchbend;
	int32 tmp;
	FLOAT_T pf, root_freq;
	int32 a;
	
	if (! (TMDY_PLAYMIDI->voice)[v].sample->sample_rate)
		return;
	if (! (TMDY_PLAYMIDI->opt_modulation_wheel))
		(TMDY_PLAYMIDI->voice)[v].modulation_wheel = 0;
	if (! TMDY_PLAYMIDI->opt_portamento)
		(TMDY_PLAYMIDI->voice)[v].porta_control_ratio = 0;
	(TMDY_PLAYMIDI->voice)[v].vibrato_control_ratio = (TMDY_PLAYMIDI->voice)[v].orig_vibrato_control_ratio;
	if ((TMDY_PLAYMIDI->voice)[v].vibrato_control_ratio || (TMDY_PLAYMIDI->voice)[v].modulation_wheel > 0) {
		/* This instrument has vibrato. Invalidate any precomputed
		 * sample_increments.
		 */
		if ((TMDY_PLAYMIDI->voice)[v].modulation_wheel > 0) {
			(TMDY_PLAYMIDI->voice)[v].vibrato_control_ratio = (TMDY_OUTPUT->play_mode)->rate / 2.0
					* MODULATION_WHEEL_RATE / VIBRATO_SAMPLE_INCREMENTS;
			(TMDY_PLAYMIDI->voice)[v].vibrato_delay = 0;
		}
		for (i = 0; i < VIBRATO_SAMPLE_INCREMENTS; i++)
			(TMDY_PLAYMIDI->voice)[v].vibrato_sample_increment[i] = 0;
		(TMDY_PLAYMIDI->voice)[v].cache = NULL;
	}
	/* fine: [0..128] => [-256..256]
	 * 1 coarse = 256 fine (= 1 note)
	 * 1 fine = 2^5 tuning
	 */
	tuning = ((TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0001] - 0x40
			+ ((TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0002] - 0x40) * 64) << 7;
	/* for NRPN Coarse Pitch of Drum (GS) & Fine Pitch of Drum (XG) */
	if (ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL
			&& ((TMDY_PLAYMIDI->channel)[ch].drums[note]->fine
			|| (TMDY_PLAYMIDI->channel)[ch].drums[note]->coarse)) {
		tuning += ((TMDY_PLAYMIDI->channel)[ch].drums[note]->fine
				+ (TMDY_PLAYMIDI->channel)[ch].drums[note]->coarse * 64) << 7;
	}
	if ((TMDY_PLAYMIDI->opt_modulation_envelope)) {
		if ((TMDY_PLAYMIDI->voice)[v].sample->tremolo_to_pitch)
			tuning += lookup_triangular((TMDY_PLAYMIDI->voice)[v].tremolo_phase >> RATE_SHIFT)
					* ((TMDY_PLAYMIDI->voice)[v].sample->tremolo_to_pitch << 13) / 100.0 + 0.5;
		if ((TMDY_PLAYMIDI->voice)[v].sample->modenv_to_pitch)
			tuning += (TMDY_PLAYMIDI->voice)[v].last_modenv_volume
					* ((TMDY_PLAYMIDI->voice)[v].sample->modenv_to_pitch << 13) / 100.0 + 0.5;
	}
	/* GS/XG - Scale Tuning */
	if (! ISDRUMCHANNEL(ch)) {
		tuning += (st << 13) / 100.0 + 0.5;
		if (st != (TMDY_PLAYMIDI->channel)[ch].prev_scale_tuning) {
			(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
			(TMDY_PLAYMIDI->channel)[ch].prev_scale_tuning = st;
		}
	}
	if (! (TMDY_PLAYMIDI->opt_pure_intonation)
			&& (TMDY_PLAYMIDI->opt_temper_control) && (TMDY_PLAYMIDI->voice)[v].temper_instant) {
		switch (tt) {
		case 0:
			f = freq_table_tuning[tp][note];
			break;
		case 1:
			f = freq_table_pytha[(TMDY_PLAYMIDI->current_freq_table)][note];
			break;
		case 2:
			if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
				f = freq_table_meantone[(TMDY_PLAYMIDI->current_freq_table)][note];
			else
				f = freq_table_meantone[(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			break;
		case 3:
			if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
				f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table)][note];
			else
				f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			break;
		default:	/* user-defined temperaments */
			if ((tt -= 0x40) >= 0 && tt < 4) {
				if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
					f = freq_table_user[tt][(TMDY_PLAYMIDI->current_freq_table)][note];
				else
					f = freq_table_user[tt][(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			} else
				f = freq_table[note];
			break;
		}
		(TMDY_PLAYMIDI->voice)[v].orig_frequency = f;
	}
	if (! (TMDY_PLAYMIDI->voice)[v].porta_control_ratio) {
		if (tuning == 0 && pb == 0x2000)
			(TMDY_PLAYMIDI->voice)[v].frequency = (TMDY_PLAYMIDI->voice)[v].orig_frequency;
		else {
			pb -= 0x2000;
			if (! (TMDY_PLAYMIDI->channel)[ch].pitchfactor) {
				/* Damn.  Somebody bent the pitch. */
				tmp = pb * (TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0000] + tuning;
				if (tmp >= 0)
					(TMDY_PLAYMIDI->channel)[ch].pitchfactor = bend_fine[tmp >> 5 & 0xff]
							* bend_coarse[tmp >> 13 & 0x7f];
				else
					(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 1.0 /
							(bend_fine[-tmp >> 5 & 0xff]
							* bend_coarse[-tmp >> 13 & 0x7f]);
			}
			(TMDY_PLAYMIDI->voice)[v].frequency =
					(TMDY_PLAYMIDI->voice)[v].orig_frequency * (TMDY_PLAYMIDI->channel)[ch].pitchfactor;
			if ((TMDY_PLAYMIDI->voice)[v].frequency != (TMDY_PLAYMIDI->voice)[v].orig_frequency)
				(TMDY_PLAYMIDI->voice)[v].cache = NULL;
		}
	} else {	/* Portamento */
		pb -= 0x2000;
		tmp = pb * (TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0000]
				+ ((TMDY_PLAYMIDI->voice)[v].porta_pb << 5) + tuning;
		if (tmp >= 0)
			pf = bend_fine[tmp >> 5 & 0xff]
					* bend_coarse[tmp >> 13 & 0x7f];
		else
			pf = 1.0 / (bend_fine[-tmp >> 5 & 0xff]
					* bend_coarse[-tmp >> 13 & 0x7f]);
		(TMDY_PLAYMIDI->voice)[v].frequency = (TMDY_PLAYMIDI->voice)[v].orig_frequency * pf;
		(TMDY_PLAYMIDI->voice)[v].cache = NULL;
	}
	if (ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL
			&& (TMDY_PLAYMIDI->channel)[ch].drums[note]->play_note != -1)
		root_freq = (TMDY_PLAYMIDI->voice)[v].sample->root_freq
				* (double) freq_table[(TMDY_PLAYMIDI->channel)[ch].drums[note]->play_note]
				/ (TMDY_PLAYMIDI->voice)[v].orig_frequency;
	else
		root_freq = (TMDY_PLAYMIDI->voice)[v].sample->root_freq;
	a = TIM_FSCALE(((double) (TMDY_PLAYMIDI->voice)[v].sample->sample_rate
			* (TMDY_PLAYMIDI->voice)[v].frequency + (TMDY_PLAYMIDI->channel)[ch].pitch_offset_fine)
			/ (root_freq * (TMDY_OUTPUT->play_mode)->rate), FRACTION_BITS) + 0.5;
	/* need to preserve the loop direction */
	(TMDY_PLAYMIDI->voice)[v].sample_increment = ((TMDY_PLAYMIDI->voice)[v].sample_increment >= 0) ? a : -a;
#ifdef ABORT_AT_FATAL
	if ((TMDY_PLAYMIDI->voice)[v].sample_increment == 0) {
		fprintf(stderr, "Invalid sample increment a=%e %ld %ld %ld %ld%s\n",
				(double)a, (long) (TMDY_PLAYMIDI->voice)[v].sample->sample_rate,
				(long) (TMDY_PLAYMIDI->voice)[v].frequency, (long) (TMDY_PLAYMIDI->voice)[v].sample->root_freq,
				(long) (TMDY_OUTPUT->play_mode)->rate, ((TMDY_PLAYMIDI->voice)[v].cache) ? " (Cached)" : "");
		abort();
	}
#endif	/* ABORT_AT_FATAL */
}

static int32 calc_velocity(tmdy_struct_ex_t *tmdy_struct, int32 ch,int32 vel)
{
	int32 velocity;
	velocity = (TMDY_PLAYMIDI->channel)[ch].velocity_sense_depth * vel / 64 + ((TMDY_PLAYMIDI->channel)[ch].velocity_sense_offset - 64) * 2;
	if(velocity > 127) {velocity = 127;}
	return velocity;
}

static void recompute_amp(tmdy_struct_ex_t *tmdy_struct, int v)
{
	FLOAT_T tempamp;

	/* master_volume and sample->volume are percentages, used to scale
	 *  amplitude directly, NOT perceived volume
	 *
	 * all other MIDI volumes are linear in perceived volume, 0-127
	 * use a lookup table for the non-linear scalings
	 */
	if(TMDY_READMIDI->play_system_mode == GS_SYSTEM_MODE) {	/* use measured curve */ 
	tempamp = (TMDY_PLAYMIDI->master_volume) *
		   (TMDY_PLAYMIDI->voice)[v].sample->volume *
		   sc_vel_table[calc_velocity(tmdy_struct, (TMDY_PLAYMIDI->voice)[v].channel,(TMDY_PLAYMIDI->voice)[v].velocity)] *
		   sc_vol_table[(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].volume] *
		   sc_vol_table[(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].expression]; /* 21 bits */
	} else if (IS_CURRENT_MOD_FILE) {	/* use linear curve */
	tempamp = (TMDY_PLAYMIDI->master_volume) *
		  (TMDY_PLAYMIDI->voice)[v].sample->volume *
		  calc_velocity(tmdy_struct, (TMDY_PLAYMIDI->voice)[v].channel,(TMDY_PLAYMIDI->voice)[v].velocity) *
		  (TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].volume *
		  (TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].expression; /* 21 bits */
	} else {	/* use generic exponential curve */
	tempamp = (TMDY_PLAYMIDI->master_volume) *
		  (TMDY_PLAYMIDI->voice)[v].sample->volume *
		  perceived_vol_table[calc_velocity(tmdy_struct, (TMDY_PLAYMIDI->voice)[v].channel,(TMDY_PLAYMIDI->voice)[v].velocity)] *
		  perceived_vol_table[(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].volume] *
		  perceived_vol_table[(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].expression]; /* 21 bits */
	}

	/* every digital effect increases amplitude, so that it must be reduced in advance. */
	if(((TMDY_PLAYMIDI->opt_reverb_control) || (TMDY_PLAYMIDI->opt_chorus_control)
			|| (TMDY_PLAYMIDI->opt_delay_control) || ((TMDY_PLAYMIDI->opt_eq_control) && (TMDY_REVERB->eq_status.low_gain != 0x40 || TMDY_REVERB->eq_status.high_gain != 0x40)) || (TMDY_PLAYMIDI->opt_insertion_effect))
			&& !((TMDY_OUTPUT->play_mode)->encoding & PE_MONO)) {
		tempamp *= 0.55f * 1.35f;
	} else {
		tempamp *= 1.35f;
	}

	/* NRPN - drum instrument tva level */
	if(ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[v].channel)) {
		if((TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].drums[(TMDY_PLAYMIDI->voice)[v].note] != NULL) {
			tempamp *= (TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[v].channel].drums[(TMDY_PLAYMIDI->voice)[v].note]->drum_level;
		}
		tempamp *= (double)(TMDY_PLAYMIDI->opt_drum_power) * 0.01f;
	}

	if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO))
    	{
		if((TMDY_PLAYMIDI->voice)[v].panning == 64)
		{
			(TMDY_PLAYMIDI->voice)[v].panned = PANNED_CENTER;
			(TMDY_PLAYMIDI->voice)[v].left_amp = (TMDY_PLAYMIDI->voice)[v].right_amp = TIM_FSCALENEG(tempamp * sc_pan_table[64], 27);
		}
		else if ((TMDY_PLAYMIDI->voice)[v].panning < 2)
		{
			(TMDY_PLAYMIDI->voice)[v].panned = PANNED_LEFT;
			(TMDY_PLAYMIDI->voice)[v].left_amp = TIM_FSCALENEG(tempamp, 20);
			(TMDY_PLAYMIDI->voice)[v].right_amp = 0;
		}
		else if((TMDY_PLAYMIDI->voice)[v].panning == 127)
		{
#ifdef SMOOTH_MIXING
			if((TMDY_PLAYMIDI->voice)[v].panned == PANNED_MYSTERY) {
				(TMDY_PLAYMIDI->voice)[v].old_left_mix = (TMDY_PLAYMIDI->voice)[v].old_right_mix;
				(TMDY_PLAYMIDI->voice)[v].old_right_mix = 0;
			}
#endif
			(TMDY_PLAYMIDI->voice)[v].panned = PANNED_RIGHT;
			(TMDY_PLAYMIDI->voice)[v].left_amp =  TIM_FSCALENEG(tempamp, 20);
			(TMDY_PLAYMIDI->voice)[v].right_amp = 0;
		}
		else
		{
#ifdef SMOOTH_MIXING
			if((TMDY_PLAYMIDI->voice)[v].panned == PANNED_RIGHT) {
				(TMDY_PLAYMIDI->voice)[v].old_right_mix = (TMDY_PLAYMIDI->voice)[v].old_left_mix;
				(TMDY_PLAYMIDI->voice)[v].old_left_mix = 0;
			}
#endif
			(TMDY_PLAYMIDI->voice)[v].panned = PANNED_MYSTERY;
			(TMDY_PLAYMIDI->voice)[v].left_amp = TIM_FSCALENEG(tempamp * sc_pan_table[127 - (TMDY_PLAYMIDI->voice)[v].panning], 27);
			(TMDY_PLAYMIDI->voice)[v].right_amp = TIM_FSCALENEG(tempamp * sc_pan_table[(TMDY_PLAYMIDI->voice)[v].panning], 27);
		}
    	}
    	else
    	{
		(TMDY_PLAYMIDI->voice)[v].panned = PANNED_CENTER;
		(TMDY_PLAYMIDI->voice)[v].left_amp = TIM_FSCALENEG(tempamp, 21);
    	}
}

void recompute_channel_filter(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
	int ch = e->channel, note, prog, bk;
	double coef = 1.0f, reso = 0;
	ToneBankElement *elm;

	if((TMDY_PLAYMIDI->channel)[ch].special_sample > 0) {return;}

	note = MIDI_EVENT_NOTE(e);
	prog = (TMDY_PLAYMIDI->channel)[ch].program;
	bk = (TMDY_PLAYMIDI->channel)[ch].bank;
	elm = &(TMDY_INSTRUM->tonebank)[bk]->tone[prog];

	/* Soft Pedal */
	if((TMDY_PLAYMIDI->channel)[ch].soft_pedal > 63) {
		if(note > 49) {	/* tre corde */
			coef *= 1.0 - 0.20 * ((double)(TMDY_PLAYMIDI->channel)[ch].soft_pedal - 64) / 63.0f;
		} else {	/* una corda (due corde) */
			coef *= 1.0 - 0.25 * ((double)(TMDY_PLAYMIDI->channel)[ch].soft_pedal - 64) / 63.0f;
		}
	}

	if(!ISDRUMCHANNEL(ch)) {
		/* NRPN Filter Cutoff */
		coef *= pow(1.26, (double)((TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq) / 8.0f);
		/* NRPN Resonance */
		reso = (double)(TMDY_PLAYMIDI->channel)[ch].param_resonance * 0.5f;
	}

	(TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef = coef;
	(TMDY_PLAYMIDI->channel)[ch].resonance_dB = reso;

/*	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Cutoff Frequency (CH:%d VAL:%f)",ch,coef);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Resonance (CH:%d VAL:%f)",ch,reso);*/
}

void recompute_voice_filter(tmdy_struct_ex_t *tmdy_struct, int v)
{
	int ch = (TMDY_PLAYMIDI->voice)[v].channel, note = (TMDY_PLAYMIDI->voice)[v].note;
	double coef = 1.0, reso = 0, cent = 0;
	int32 freq;
	FilterCoefficients *fc = &((TMDY_PLAYMIDI->voice)[v].fc);
	Sample *sp = (Sample *) &(TMDY_PLAYMIDI->voice)[v].sample;

	if(fc->freq == -1) {return;}

	coef = (TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef;

	if(ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL) {
		/* NRPN Drum Instrument Filter Cutoff */
		coef *= pow(1.26, (double)((TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_cutoff_freq) / 8.0f);
		/* NRPN Drum Instrument Filter Resonance */
		reso += (double)(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_resonance * 0.5f;
	}

	if(sp->vel_to_fc) {	/* velocity to filter cutoff frequency */
		if((TMDY_PLAYMIDI->voice)[v].velocity > sp->vel_to_fc_threshold)
			cent += sp->vel_to_fc * (double)(127 - (TMDY_PLAYMIDI->voice)[v].velocity) / 127.0f;
		else
			coef += -1200;
	}
	if(sp->vel_to_resonance) {	/* velocity to filter resonance */
		reso += (double)(TMDY_PLAYMIDI->voice)[v].velocity * sp->vel_to_resonance / 127.0f / 10.0f;
	}
	if(sp->key_to_fc) {	/* filter cutoff key-follow */
		cent += sp->vel_to_fc * (double)((TMDY_PLAYMIDI->voice)[v].note - sp->key_to_fc_bpo);
	}

	if((TMDY_PLAYMIDI->opt_modulation_envelope)) {
		if((TMDY_PLAYMIDI->voice)[v].sample->tremolo_to_fc) {
			cent += (double)(TMDY_PLAYMIDI->voice)[v].sample->tremolo_to_fc * lookup_triangular((TMDY_PLAYMIDI->voice)[v].tremolo_phase >> RATE_SHIFT);
		}
		if((TMDY_PLAYMIDI->voice)[v].sample->modenv_to_fc) {
			cent += (double)(TMDY_PLAYMIDI->voice)[v].sample->modenv_to_fc * (TMDY_PLAYMIDI->voice)[v].last_modenv_volume;
		}
	}

	if(cent != 0) {coef *= pow(2.0, cent / 1200.0f);}

	freq = (double)fc->orig_freq * coef;

	if (fc->filter_calculated == 0 && freq > (TMDY_OUTPUT->play_mode)->rate / 2) {
		fc->freq = -1;
		return;
	}
	else if(freq < 5) {freq = 5;}
	else if(freq > 20000) {freq = 20000;}
	fc->freq = freq;

	fc->reso_dB = fc->orig_reso_dB + (TMDY_PLAYMIDI->channel)[ch].resonance_dB + reso;
	if(fc->reso_dB < 0.0f) {fc->reso_dB = 0.0f;}
	else if(fc->reso_dB > 96.0f) {fc->reso_dB = 96.0f;}
	fc->reso_dB -= 3.01f;
}

FLOAT_T calc_drum_tva_level(tmdy_struct_ex_t *tmdy_struct, int ch,int note,int level)
{
	int def_level,nbank,nprog;
	ToneBank *bank;

	if((TMDY_PLAYMIDI->channel)[ch].special_sample > 0) {return 1.0;}

	nbank = (TMDY_PLAYMIDI->channel)[ch].bank;
	nprog = note;
	TMDY_INSTRUM->instrument_map(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].mapID, &nbank, &nprog);

	if(ISDRUMCHANNEL(ch)) {
		bank = TMDY_INSTRUM->drumset[nbank];
		if(bank == NULL) {bank = TMDY_INSTRUM->drumset[0];}
	} else {
		return 1.0;
	}

	def_level = bank->tone[nprog].tva_level;

	if(def_level == -1 || def_level == 0) {def_level = 127;}
	else if(def_level > 127) {def_level = 127;}

	return (sc_drum_level_table[level] / sc_drum_level_table[def_level]);
}

void recompute_bank_parameter(tmdy_struct_ex_t *tmdy_struct, int ch,int note)
{
	int prog;
	ToneBank *bank;

	if((TMDY_PLAYMIDI->channel)[ch].special_sample > 0) {return;}

	prog = (TMDY_PLAYMIDI->channel)[ch].program;

	if(ISDRUMCHANNEL(ch)) {
		return;
	} else {
		if(prog == SPECIAL_PROGRAM) {return;}
		bank = TMDY_INSTRUM->tonebank[(int)(TMDY_PLAYMIDI->channel)[ch].bank];
		if(bank == NULL) {bank = TMDY_INSTRUM->tonebank[0];}
	}

	(TMDY_PLAYMIDI->channel)[ch].legato = bank->tone[prog].legato;
}

static void *memdup(tmdy_struct_ex_t *tmdy_struct, void *s,size_t size)
{
	void *p;

	p = TMDY_COMMON->safe_malloc(tmdy_struct, size);
	if(p != NULL) {memcpy(p,s,size);}

	return p;
}

void dup_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = ((dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank);
	ToneBankElement *elm = &bank[bk]->tone[prog];
	int i;
	
	if (elm->name)
		elm->name = TMDY_COMMON->safe_strdup(tmdy_struct, elm->name);
	if (elm->comment)
		elm->comment = TMDY_COMMON->safe_strdup(tmdy_struct, elm->comment);
	if (elm->tunenum)
		elm->tune = (float *) memdup(tmdy_struct, 
				elm->tune, elm->tunenum * sizeof(float));
	if (elm->envratenum) {
		elm->envrate = (int **) memdup(tmdy_struct, 
				elm->envrate, elm->envratenum * sizeof(int *));
		for (i = 0; i < elm->envratenum; i++)
			elm->envrate[i] = (int *) memdup(tmdy_struct, elm->envrate[i], 6 * sizeof(int));
	}
	if (elm->envofsnum) {
		elm->envofs = (int **) memdup(tmdy_struct, 
				elm->envofs, elm->envofsnum * sizeof(int *));
		for (i = 0; i < elm->envofsnum; i++)
			elm->envofs[i] = (int *) memdup(tmdy_struct, elm->envofs[i], 6 * sizeof(int));
	}
	if (elm->tremnum) {
		elm->trem = (Quantity **) memdup(tmdy_struct, elm->trem, elm->tremnum * sizeof(Quantity *));
		for (i = 0; i < elm->tremnum; i++)
			elm->trem[i] = (Quantity *) memdup(tmdy_struct, elm->trem[i], 3 * sizeof(Quantity));
	}
	if (elm->vibnum) {
		elm->vib = (Quantity **) memdup(tmdy_struct, elm->vib, elm->vibnum * sizeof(Quantity *));
		for (i = 0; i < elm->vibnum; i++)
			elm->vib[i] = (Quantity *) memdup(tmdy_struct, elm->vib[i], 3 * sizeof(Quantity));
	}
}

void free_tone_bank_element(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = (dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank;
	ToneBankElement *elm = &bank[bk]->tone[prog];
	int i;
	
	if (elm->name) {
		free(elm->name);
		elm->name = NULL;
	}
	if (elm->comment) {
		free(elm->comment);
		elm->comment = NULL;
	}
	if (elm->tune) {
		free(elm->tune);
		elm->tune = NULL;
	}
	if (elm->envratenum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->envrate, elm->envratenum);
		elm->envrate = NULL;
		elm->envratenum = 0;
	}
	if (elm->envofsnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->envofs, elm->envofsnum);
		elm->envofs = NULL;
		elm->envofsnum = 0;
	}
	if (elm->tremnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->trem, elm->tremnum);
		elm->trem = NULL;
		elm->tremnum = 0;
	}
	if (elm->vibnum) {
		TMDY_COMMON->free_ptr_list(tmdy_struct, elm->vib, elm->vibnum);
		elm->vib = NULL;
		elm->vibnum = 0;
	}
	elm->instype = 0;
}

Instrument *play_midi_load_instrument(tmdy_struct_ex_t *tmdy_struct, int dr, int bk, int prog)
{
	ToneBank **bank = (dr) ? TMDY_INSTRUM->drumset : TMDY_INSTRUM->tonebank;
	Instrument *ip;
	ToneBankElement *elm;
	int load_success = 0;
	
	if (bank[bk] == NULL)
		bk = 0;
	if ((ip = bank[bk]->tone[prog].instrument) == MAGIC_LOAD_INSTRUMENT) {
		if (ip = bank[bk]->tone[prog].instrument =
				TMDY_INSTRUM->load_instrument(tmdy_struct, dr, bk, prog))
			load_success = 1;
#ifndef SUPPRESS_CHANNEL_LAYER
	} else if (ip == NULL) {
		if (ip = bank[bk]->tone[prog].instrument =
				TMDY_INSTRUM->load_instrument(tmdy_struct, dr, bk, prog))
			load_success = 1;
#endif
	}
	if (ip == NULL && bk) {
		/* Instrument is not found.
		 * Retry to load the instrument from bank 0
		 */
		if ((ip = bank[0]->tone[prog].instrument) == NULL
				|| ip == MAGIC_LOAD_INSTRUMENT)
			ip = bank[0]->tone[prog].instrument =
					TMDY_INSTRUM->load_instrument(tmdy_struct, dr, 0, prog);
		if (ip) {
			/* duplicate tone bank parameter */
			elm = &bank[bk]->tone[prog];
			memcpy(elm, &bank[0]->tone[prog], sizeof(ToneBankElement));
			elm->instrument = ip;
			dup_tone_bank_element(tmdy_struct, dr, bk, prog);
			load_success = 1;
		}
	}
	if (load_success)
		(TMDY_AQ->aq_add)(tmdy_struct,NULL, 0);	/* Update software buffer */
	if (ip == MAGIC_ERROR_INSTRUMENT)
		return NULL;
	if (ip == NULL)
		bank[bk]->tone[prog].instrument = MAGIC_ERROR_INSTRUMENT;
	return ip;
}

#if 0
/* reduce_voice_CPU() may not have any speed advantage over reduce_voice().
 * So this function is not used, now.
 */

/* The goal of this routine is to free as much CPU as possible without
   loosing too much sound quality.  We would like to know how long a note
   has been playing, but since we usually can't calculate this, we guess at
   the value instead.  A bad guess is better than nothing.  Notes which
   have been playing a short amount of time are killed first.  This causes
   decays and notes to be cut earlier, saving more CPU time.  It also causes
   notes which are closer to ending not to be cut as often, so it cuts
   a different note instead and saves more CPU in the long run.  ON voices
   are treated a little differently, since sound quality is more important
   than saving CPU at this point.  Duration guesses for loop regions are very
   crude, but are still better than nothing, they DO help.  Non-looping ON
   notes are cut before looping ON notes.  Since a looping ON note is more
   likely to have been playing for a long time, we want to keep it because it
   sounds better to keep long notes.
*/
static int reduce_voice_CPU(tmdy_struct_ex_t *tmdy_struct)
{
    int32 lv, v, vr;
    int i, j, lowest=-0x7FFFFFFF;
    int32 duration;

    i = (TMDY_PLAYMIDI->upper_voices);
    lv = 0x7FFFFFFF;
    
    /* Look for the decaying note with the longest remaining decay time */
    /* Protect drum decays.  They do not take as much CPU (?) and truncating
       them early sounds bad, especially on snares and cymbals */
    for(j = 0; j < i; j++)
    {
	if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
	/* skip notes that don't need resampling (most drums) */
	if ((TMDY_PLAYMIDI->voice)[j].sample->note_to_use)
	    continue;
	if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_DIE | VOICE_SUSTAINED))
	{
	    /* Choose note with longest decay time remaining */
	    /* This frees more CPU than choosing lowest volume */
	    if (!(TMDY_PLAYMIDI->voice)[j].envelope_increment) duration = 0;
	    else duration =
	    	((TMDY_PLAYMIDI->voice)[j].envelope_target - (TMDY_PLAYMIDI->voice)[j].envelope_volume) /
	    	(TMDY_PLAYMIDI->voice)[j].envelope_increment;
	    v = -duration;
	    if(v < lv)
	    {
		lv = v;
		lowest = j;
	    }
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	/* This can still cause a click, but if we had a free voice to
	   spare for ramping down this note, we wouldn't need to kill it
	   in the first place... Still, this needs to be fixed. Perhaps
	   we could use a reserve of voices to play dying notes only. */

	(TMDY_PLAYMIDI->cut_notes)++;
	return lowest;
    }

    /* try to remove VOICE_DIE before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_SUSTAINED))
      {
	/* continue protecting non-resample decays */
	if ((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_DIE) && (TMDY_PLAYMIDI->voice)[j].sample->note_to_use)
		continue;

	/* choose note which has been on the shortest amount of time */
	/* this is a VERY crude estimate... */
	if ((TMDY_PLAYMIDI->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset - (TMDY_PLAYMIDI->voice)[j].sample->loop_start;
	else
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset;
	if ((TMDY_PLAYMIDI->voice)[j].sample_increment > 0)
	    duration /= (TMDY_PLAYMIDI->voice)[j].sample_increment;
	v = duration;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -1)
    {
	(TMDY_PLAYMIDI->cut_notes)++;
	return lowest;
    }

    /* try to remove VOICE_SUSTAINED before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_SUSTAINED)
      {
	/* choose note which has been on the shortest amount of time */
	/* this is a VERY crude estimate... */
	if ((TMDY_PLAYMIDI->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset - (TMDY_PLAYMIDI->voice)[j].sample->loop_start;
	else
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset;
	if ((TMDY_PLAYMIDI->voice)[j].sample_increment > 0)
	    duration /= (TMDY_PLAYMIDI->voice)[j].sample_increment;
	v = duration;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(TMDY_PLAYMIDI->cut_notes)++;
	return lowest;
    }

    /* try to remove chorus before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].chorus_link < j)
      {
	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	if ((TMDY_PLAYMIDI->voice)[j].sample->modes & MODES_LOOPING)
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset - (TMDY_PLAYMIDI->voice)[j].sample->loop_start;
	else
	    duration = (TMDY_PLAYMIDI->voice)[j].sample_offset;
	if ((TMDY_PLAYMIDI->voice)[j].sample_increment > 0)
	    duration /= (TMDY_PLAYMIDI->voice)[j].sample_increment;
	v = (TMDY_PLAYMIDI->voice)[j].left_mix * duration;
	vr = (TMDY_PLAYMIDI->voice)[j].right_mix * duration;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(TMDY_PLAYMIDI->cut_notes)++;

	/* hack - double volume of chorus partner, fix pan */
	j = (TMDY_PLAYMIDI->voice)[lowest].chorus_link;
	(TMDY_PLAYMIDI->voice)[j].velocity <<= 1;
    	(TMDY_PLAYMIDI->voice)[j].panning = (TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[lowest].channel].panning;
    	recompute_amp(tmdy_struct, j);
    	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, j);

	return lowest;
    }

    (TMDY_PLAYMIDI->lost_notes)++;

    /* try to remove non-looping voices first */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
      if(!((TMDY_PLAYMIDI->voice)[j].sample->modes & MODES_LOOPING))
      {
	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	duration = (TMDY_PLAYMIDI->voice)[j].sample_offset;
	if ((TMDY_PLAYMIDI->voice)[j].sample_increment > 0)
	    duration /= (TMDY_PLAYMIDI->voice)[j].sample_increment;
	v = (TMDY_PLAYMIDI->voice)[j].left_mix * duration;
	vr = (TMDY_PLAYMIDI->voice)[j].right_mix * duration;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	return lowest;
    }

    lv = 0x7FFFFFFF;
    lowest = 0;
    for(j = 0; j < i; j++)
    {
	if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE || (TMDY_PLAYMIDI->voice)[j].cache != NULL)
	    continue;
	if (!((TMDY_PLAYMIDI->voice)[j].sample->modes & MODES_LOOPING)) continue;

	/* score notes based on both volume AND duration */
	/* this scoring function needs some more tweaking... */
	duration = (TMDY_PLAYMIDI->voice)[j].sample_offset - (TMDY_PLAYMIDI->voice)[j].sample->loop_start;
	if ((TMDY_PLAYMIDI->voice)[j].sample_increment > 0)
	    duration /= (TMDY_PLAYMIDI->voice)[j].sample_increment;
	v = (TMDY_PLAYMIDI->voice)[j].left_mix * duration;
	vr = (TMDY_PLAYMIDI->voice)[j].right_mix * duration;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && vr > v)
	    v = vr;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }

    return lowest;
}
#endif

/* this reduces voices while maintaining sound quality */
static int reduce_voice(tmdy_struct_ex_t *tmdy_struct)
{
    int32 lv, v;
    int i, j, lowest=-0x7FFFFFFF;

    i = (TMDY_PLAYMIDI->upper_voices);
    lv = 0x7FFFFFFF;
    
    /* Look for the decaying note with the smallest volume */
    /* Protect drum decays.  Truncating them early sounds bad, especially on
       snares and cymbals */
    for(j = 0; j < i; j++)
    {
	if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE ||
	   ((TMDY_PLAYMIDI->voice)[j].sample->note_to_use && ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[j].channel)))
	    continue;
	
	if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_DIE | VOICE_SUSTAINED))
	{
	    /* find lowest volume */
	    v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	    if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    	v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	    if(v < lv)
	    {
		lv = v;
		lowest = j;
	    }
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	/* This can still cause a click, but if we had a free voice to
	   spare for ramping down this note, we wouldn't need to kill it
	   in the first place... Still, this needs to be fixed. Perhaps
	   we could use a reserve of voices to play dying notes only. */

	(TMDY_PLAYMIDI->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!TMDY_PLAYMIDI->prescanning_flag)
	    ctl_note_event(tmdy_struct, lowest);
	return lowest;
    }

    /* try to remove VOICE_DIE before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_SUSTAINED))
      {
	/* continue protecting drum decays */
	if ((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_DIE) &&
	    ((TMDY_PLAYMIDI->voice)[j].sample->note_to_use && ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[j].channel)))
		continue;
	/* find lowest volume */
	v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -1)
    {
	(TMDY_PLAYMIDI->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!TMDY_PLAYMIDI->prescanning_flag)
	    ctl_note_event(tmdy_struct, lowest);
	return lowest;
    }

    /* try to remove VOICE_SUSTAINED before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_SUSTAINED)
      {
	/* find lowest volume */
	v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(TMDY_PLAYMIDI->cut_notes)++;
	free_voice(tmdy_struct, lowest);
	if(!TMDY_PLAYMIDI->prescanning_flag)
	    ctl_note_event(tmdy_struct, lowest);
	return lowest;
    }

    /* try to remove chorus before VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
      if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE)
	    continue;
      if((TMDY_PLAYMIDI->voice)[j].chorus_link < j)
      {
	/* find lowest volume */
	v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
      }
    }
    if(lowest != -0x7FFFFFFF)
    {
	(TMDY_PLAYMIDI->cut_notes)++;

	/* hack - double volume of chorus partner, fix pan */
	j = (TMDY_PLAYMIDI->voice)[lowest].chorus_link;
	(TMDY_PLAYMIDI->voice)[j].velocity <<= 1;
    	(TMDY_PLAYMIDI->voice)[j].panning = (TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[lowest].channel].panning;
    	recompute_amp(tmdy_struct, j);
    	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, j);

	free_voice(tmdy_struct, lowest);
	if(!TMDY_PLAYMIDI->prescanning_flag)
	    ctl_note_event(tmdy_struct, lowest);
	return lowest;
    }

    (TMDY_PLAYMIDI->lost_notes)++;

    /* remove non-drum VOICE_ON */
    lv = 0x7FFFFFFF;
    lowest = -0x7FFFFFFF;
    for(j = 0; j < i; j++)
    {
        if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE ||
	   ((TMDY_PLAYMIDI->voice)[j].sample->note_to_use && ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[j].channel)))
	   	continue;

	/* find lowest volume */
	v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }
    if(lowest != -0x7FFFFFFF)
    {
	free_voice(tmdy_struct, lowest);
	if(!TMDY_PLAYMIDI->prescanning_flag)
	    ctl_note_event(tmdy_struct, lowest);
	return lowest;
    }

    /* remove all other types of notes */
    lv = 0x7FFFFFFF;
    lowest = 0;
    for(j = 0; j < i; j++)
    {
	if((TMDY_PLAYMIDI->voice)[j].status & VOICE_FREE)
	    continue;
	/* find lowest volume */
	v = (TMDY_PLAYMIDI->voice)[j].left_mix;
	if((TMDY_PLAYMIDI->voice)[j].panned == PANNED_MYSTERY && (TMDY_PLAYMIDI->voice)[j].right_mix > v)
	    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
	if(v < lv)
	{
	    lv = v;
	    lowest = j;
	}
    }

    free_voice(tmdy_struct, lowest);
    if(!TMDY_PLAYMIDI->prescanning_flag)
	ctl_note_event(tmdy_struct, lowest);
    return lowest;
}



/* Only one instance of a note can be playing on a single channel. */
static int find_voice(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
  int i, j, lowest=-1, note, ch, status_check, mono_check;
  AlternateAssign *altassign;

  note = MIDI_EVENT_NOTE(e);
  ch = e->channel;

  if((TMDY_PLAYMIDI->opt_overlap_voice_allow))
      status_check = (VOICE_OFF | VOICE_SUSTAINED);
  else
      status_check = 0xFF;
  mono_check = (TMDY_PLAYMIDI->channel)[ch].mono;
  altassign = TMDY_INSTRUM->find_altassign(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].altassign, note);

  i = (TMDY_PLAYMIDI->upper_voices);
  for(j = 0; j < i; j++)
      if((TMDY_PLAYMIDI->voice)[j].status == VOICE_FREE)
      {
	  lowest = j; /* lower volume */
	  break;
      }

	for (j = 0; j < i; j++)
		if ((TMDY_PLAYMIDI->voice)[j].status != VOICE_FREE && (TMDY_PLAYMIDI->voice)[j].channel == ch
				&& (((TMDY_PLAYMIDI->voice)[j].status & status_check) && (TMDY_PLAYMIDI->voice)[j].note == note
				|| mono_check
				|| (altassign && TMDY_INSTRUM->find_altassign(tmdy_struct, altassign, (TMDY_PLAYMIDI->voice)[j].note))))
			kill_note(tmdy_struct, j);
		else if ((TMDY_PLAYMIDI->voice)[j].status != VOICE_FREE && (TMDY_PLAYMIDI->voice)[j].channel == ch
				&& (TMDY_PLAYMIDI->voice)[j].note == note && (((TMDY_PLAYMIDI->channel)[ch].assign_mode == 1
				&& (TMDY_PLAYMIDI->voice)[j].proximate_flag == 0)
				|| (TMDY_PLAYMIDI->channel)[ch].assign_mode == 0))
			kill_note(tmdy_struct, j);
	for (j = 0; j < i; j++)
		if ((TMDY_PLAYMIDI->voice)[j].channel == ch && (TMDY_PLAYMIDI->voice)[j].note == note)
			(TMDY_PLAYMIDI->voice)[j].proximate_flag = 0;

  if(lowest != -1)
  {
      /* Found a free voice. */
      if((TMDY_PLAYMIDI->upper_voices) <= lowest)
	  (TMDY_PLAYMIDI->upper_voices) = lowest + 1;
      return lowest;
  }

  if(i < (TMDY_PLAYMIDI->voices))
      return (TMDY_PLAYMIDI->upper_voices)++;
  return reduce_voice(tmdy_struct);
}

void free_voice(tmdy_struct_ex_t *tmdy_struct, int v1)
{
    int v2;

    v2 = (TMDY_PLAYMIDI->voice)[v1].chorus_link;
    if(v1 != v2)
    {
	/* Unlink chorus link */
	(TMDY_PLAYMIDI->voice)[v1].chorus_link = v1;
	(TMDY_PLAYMIDI->voice)[v2].chorus_link = v2;
    }
    (TMDY_PLAYMIDI->voice)[v1].status = VOICE_FREE;
    (TMDY_PLAYMIDI->voice)[v1].temper_instant = 0;
}

static int find_free_voice(tmdy_struct_ex_t *tmdy_struct)
{
    int i, nv = (TMDY_PLAYMIDI->voices), lowest;
    int32 lv, v;

    for(i = 0; i < nv; i++)
	if((TMDY_PLAYMIDI->voice)[i].status == VOICE_FREE)
	{
	    if((TMDY_PLAYMIDI->upper_voices) <= i)
		(TMDY_PLAYMIDI->upper_voices) = i + 1;
	    return i;
	}

    (TMDY_PLAYMIDI->upper_voices) = (TMDY_PLAYMIDI->voices);

    /* Look for the decaying note with the lowest volume */
    lv = 0x7FFFFFFF;
    lowest = -1;
    for(i = 0; i < nv; i++)
    {
	if((TMDY_PLAYMIDI->voice)[i].status & ~(VOICE_ON | VOICE_DIE) &&
	   !((TMDY_PLAYMIDI->voice)[i].sample->note_to_use && ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[i].channel)))
	{
	    v = (TMDY_PLAYMIDI->voice)[i].left_mix;
	    if(((TMDY_PLAYMIDI->voice)[i].panned==PANNED_MYSTERY) && ((TMDY_PLAYMIDI->voice)[i].right_mix>v))
		v = (TMDY_PLAYMIDI->voice)[i].right_mix;
	    if(v<lv)
	    {
		lv = v;
		lowest = i;
	    }
	}
    }
    if(lowest != -1 && !TMDY_PLAYMIDI->prescanning_flag)
    {
	free_voice(tmdy_struct, lowest);
	ctl_note_event(tmdy_struct, lowest);
    }
    return lowest;
}

static int select_play_sample(tmdy_struct_ex_t *tmdy_struct, Sample *splist,
		int nsp, int note, int *vlist, MidiEvent *e)
{
	int32 f, fs, ft, fst, fc, cdiff, diff;
	int8 tt = (TMDY_PLAYMIDI->channel)[e->channel].temper_type;
	uint8 tp = (TMDY_PLAYMIDI->channel)[e->channel].rpnmap[RPN_ADDR_0003];
	Sample *sp, *spc;
	int16 st;
	double ratio;
	int i, j, nv, vel;
	
	if ((TMDY_PLAYMIDI->opt_pure_intonation)) {
		if ((TMDY_PLAYMIDI->current_keysig) < 8)
			f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table)][note];
		else
			f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table) + 12][note];
	} else if ((TMDY_PLAYMIDI->opt_temper_control))
		switch (tt) {
		case 0:
			f = freq_table_tuning[tp][note];
			break;
		case 1:
			f = freq_table_pytha[(TMDY_PLAYMIDI->current_freq_table)][note];
			break;
		case 2:
			if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
				f = freq_table_meantone[(TMDY_PLAYMIDI->current_freq_table)][note];
			else
				f = freq_table_meantone[(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			break;
		case 3:
			if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
				f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table)][note];
			else
				f = freq_table_pureint[(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			break;
		default:	/* user-defined temperaments */
			if ((tt -= 0x40) >= 0 && tt < 4) {
				if ((TMDY_PLAYMIDI->current_temper_keysig) < 8)
					f = freq_table_user[tt][(TMDY_PLAYMIDI->current_freq_table)][note];
				else
					f = freq_table_user[tt][(TMDY_PLAYMIDI->current_freq_table) + 12][note];
			} else
				f = freq_table[note];
			break;
		}
	else
		f = freq_table[note];
	if ((TMDY_PLAYMIDI->opt_temper_control))
		fs = (tt) ? freq_table[note] : freq_table_tuning[tp][note];
	else
		fs = freq_table[note];
	vel = e->b;
	nv = 0;
	for (i = 0, sp = splist; i < nsp; i++, sp++) {
		/* SF2 - Scale Tuning */
		if ((st = sp->scale_tuning) != 100) {
			ratio = pow(2.0, (note - 60) * (st - 100) / 1200.0);
			ft = f * ratio + 0.5, fst = fs * ratio + 0.5;
		} else
			ft = f, fst = fs;
		if (sp->low_freq <= fst && sp->high_freq >= fst
				&& sp->low_vel <= vel && sp->high_vel >= vel) {
			j = vlist[nv] = find_voice(tmdy_struct, e);
			(TMDY_PLAYMIDI->voice)[j].orig_frequency = ft;
			MYCHECK((TMDY_PLAYMIDI->voice)[j].orig_frequency);
			(TMDY_PLAYMIDI->voice)[j].sample = sp;
			(TMDY_PLAYMIDI->voice)[j].status = VOICE_ON;
			nv++;
		}
	}
	if (nv == 0) {
		cdiff = 0x7fffffff;
		for (i = 0, sp = splist; i < nsp; i++, sp++) {
			/* SF2 - Scale Tuning */
			if ((st = sp->scale_tuning) != 100) {
				ratio = pow(2.0, (note - 60) * (st - 100) / 1200.0);
				ft = f * ratio + 0.5, fst = fs * ratio + 0.5;
			} else
				ft = f, fst = fs;
			diff = abs(sp->root_freq - fst);
			if (diff < cdiff) {
				fc = ft;
				spc = sp;
				cdiff = diff;
			}
		}
		j = vlist[nv] = find_voice(tmdy_struct, e);
		(TMDY_PLAYMIDI->voice)[j].orig_frequency = fc;
		MYCHECK((TMDY_PLAYMIDI->voice)[j].orig_frequency);
		(TMDY_PLAYMIDI->voice)[j].sample = spc;
		(TMDY_PLAYMIDI->voice)[j].status = VOICE_ON;
		nv++;
	}
	return nv;
}

static int find_samples(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int *vlist)
{
	Instrument *ip;
	int i, nv, note, ch, prog, bk;

	ch = e->channel;
	if((TMDY_PLAYMIDI->channel)[ch].special_sample > 0)
	{
	    SpecialPatch *s;

	    s = TMDY_INSTRUM->special_patch[(TMDY_PLAYMIDI->channel)[ch].special_sample];
	    if(s == NULL)
	    {
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
			  "Strange: Special patch %d is not installed",
			  (TMDY_PLAYMIDI->channel)[ch].special_sample);
		return 0;
	    }
	    note = e->a + (TMDY_PLAYMIDI->channel)[ch].key_shift + (TMDY_PLAYMIDI->note_key_offset) + (TMDY_PLAYMIDI->key_adjust);

	    if(note < 0)
		note = 0;
	    else if(note > 127)
		note = 127;
	    return select_play_sample(tmdy_struct, s->sample, s->samples, note, vlist, e);
	}

	bk = (TMDY_PLAYMIDI->channel)[ch].bank;
	if(ISDRUMCHANNEL(ch))
	{
	    note = e->a & 0x7F;
	    TMDY_INSTRUM->instrument_map(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].mapID, &bk, &note);
	    if(!(ip = play_midi_load_instrument(tmdy_struct, 1, bk, note)))
		return 0;	/* No instrument? Then we can't play. */

		if(ip->type == INST_GUS && ip->samples != 1)
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_VERBOSE,
			  "Strange: percussion instrument with %d samples!",
			  ip->samples);
		if(ip->sample->note_to_use)
		note = ip->sample->note_to_use;
		if(ip->type == INST_SF2) {
			nv = select_play_sample(tmdy_struct, ip->sample, ip->samples, note, vlist, e);
			/* Replace the sample if the sample is cached. */
			if(!TMDY_PLAYMIDI->prescanning_flag)
			{
				if(ip->sample->note_to_use)
				note = MIDI_EVENT_NOTE(e);

				for(i = 0; i < nv; i++)
				{
				int j;

				j = vlist[i];
				if(!(TMDY_PLAYMIDI->opt_realtime_playing) && TMDY_RECACHE->allocate_cache_size > 0 &&
				   !(TMDY_PLAYMIDI->channel)[ch].portamento)
				{
					(TMDY_PLAYMIDI->voice)[j].cache = TMDY_RECACHE->resamp_cache_fetch(tmdy_struct, (TMDY_PLAYMIDI->voice)[j].sample, note);
					if((TMDY_PLAYMIDI->voice)[j].cache) /* cache hit */
					(TMDY_PLAYMIDI->voice)[j].sample = (TMDY_PLAYMIDI->voice)[j].cache->resampled;
				}
				else
					(TMDY_PLAYMIDI->voice)[j].cache = NULL;
				}
			}
			return nv;
		} else {
			i = vlist[0] = find_voice(tmdy_struct, e);
			(TMDY_PLAYMIDI->voice)[i].orig_frequency = freq_table[note];
			(TMDY_PLAYMIDI->voice)[i].sample = ip->sample;
			(TMDY_PLAYMIDI->voice)[i].status = VOICE_ON;
			return 1;
		}
	}

	prog = (TMDY_PLAYMIDI->channel)[ch].program;
	if(prog == SPECIAL_PROGRAM)
	    ip = TMDY_INSTRUM->default_instrument;
	else
	{
	    TMDY_INSTRUM->instrument_map(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].mapID, &bk, &prog);
	    if(!(ip = play_midi_load_instrument(tmdy_struct, 0, bk, prog)))
		return 0;	/* No instrument? Then we can't play. */
	}

	if(ip->sample->note_to_use)
	    note = ip->sample->note_to_use + (TMDY_PLAYMIDI->channel)[ch].key_shift + (TMDY_PLAYMIDI->note_key_offset) + (TMDY_PLAYMIDI->key_adjust);
	else
	    note = e->a + (TMDY_PLAYMIDI->channel)[ch].key_shift + (TMDY_PLAYMIDI->note_key_offset) + (TMDY_PLAYMIDI->key_adjust);
	if(note < 0)
	    note = 0;
	else if(note > 127)
	    note = 127;

	nv = select_play_sample(tmdy_struct, ip->sample, ip->samples, note, vlist, e);

	/* Replace the sample if the sample is cached. */
	if(!TMDY_PLAYMIDI->prescanning_flag)
	{
	    if(ip->sample->note_to_use)
		note = MIDI_EVENT_NOTE(e);

	    for(i = 0; i < nv; i++)
	    {
		int j;

		j = vlist[i];
		if(!(TMDY_PLAYMIDI->opt_realtime_playing) && TMDY_RECACHE->allocate_cache_size > 0 &&
		   !(TMDY_PLAYMIDI->channel)[ch].portamento)
		{
		    (TMDY_PLAYMIDI->voice)[j].cache = TMDY_RECACHE->resamp_cache_fetch(tmdy_struct, (TMDY_PLAYMIDI->voice)[j].sample, note);
		    if((TMDY_PLAYMIDI->voice)[j].cache) /* cache hit */
			(TMDY_PLAYMIDI->voice)[j].sample = (TMDY_PLAYMIDI->voice)[j].cache->resampled;
		}
		else
		    (TMDY_PLAYMIDI->voice)[j].cache = NULL;
	    }
	}
	return nv;
}

static int get_panning(tmdy_struct_ex_t *tmdy_struct, int ch, int note,int v)
{
    int i, pan;

	if((TMDY_PLAYMIDI->voice)[v].sample_panning_average == -1) {	/* mono sample */
		if((TMDY_PLAYMIDI->channel)[ch].panning != NO_PANNING) {pan = (int)(TMDY_PLAYMIDI->channel)[ch].panning - 64;}
		else {pan = 0;}
		if(ISDRUMCHANNEL(ch) &&
		 (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL &&
		 (TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning != NO_PANNING) {
			pan += (TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning;
		} else {
			pan += (TMDY_PLAYMIDI->voice)[v].sample->panning;
		}
	} else {	/* stereo sample */
		if((TMDY_PLAYMIDI->channel)[ch].panning != NO_PANNING) {pan = (int)(TMDY_PLAYMIDI->channel)[ch].panning - 64;}
		else {pan = 0;}
		if(ISDRUMCHANNEL(ch) &&
		 (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL &&
		 (TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning != NO_PANNING) {
			pan += (TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning - 64;
		}
		pan += (TMDY_PLAYMIDI->voice)[v].sample->panning - (TMDY_PLAYMIDI->voice)[v].sample_panning_average + 64;
	}

	if (pan > 127) pan = 127;
	else if (pan < 0) pan = 0;

	return pan;
}

static void calc_sample_panning_average(tmdy_struct_ex_t *tmdy_struct, int nv,int *vlist)
{
	int i, v, average = 0;

	if(!nv) {return;}	/* error! */
	else if(nv == 1) {	/* mono sample */
		v = vlist[0];
		(TMDY_PLAYMIDI->voice)[v].sample_panning_average = -1;
		return;
	}

	for(i=0;i<nv;i++) {
		v = vlist[i];
		average += (TMDY_PLAYMIDI->voice)[v].sample->panning;
	}
	average /= nv;

	for(i=0;i<nv;i++) {
		v = vlist[i];
		(TMDY_PLAYMIDI->voice)[v].sample_panning_average = average;
	}
}

static void start_note(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int i, int vid, int cnt)
{
  int j, ch, note, pan;

  ch = e->channel;

  note = MIDI_EVENT_NOTE(e);
  (TMDY_PLAYMIDI->voice)[i].status = VOICE_ON;
  (TMDY_PLAYMIDI->voice)[i].channel = ch;
  (TMDY_PLAYMIDI->voice)[i].note = note;
  (TMDY_PLAYMIDI->voice)[i].velocity = e->b;
  (TMDY_PLAYMIDI->voice)[i].chorus_link = i;	/* No link */
  (TMDY_PLAYMIDI->voice)[i].proximate_flag = 1;

  j = (TMDY_PLAYMIDI->channel)[ch].special_sample;
  if(j == 0 || TMDY_INSTRUM->special_patch[j] == NULL)
      (TMDY_PLAYMIDI->voice)[i].sample_offset = 0;
  else
  {
      (TMDY_PLAYMIDI->voice)[i].sample_offset =
	  TMDY_INSTRUM->special_patch[j]->sample_offset << FRACTION_BITS;
      if((TMDY_PLAYMIDI->voice)[i].sample->modes & MODES_LOOPING)
      {
	  if((TMDY_PLAYMIDI->voice)[i].sample_offset > (TMDY_PLAYMIDI->voice)[i].sample->loop_end)
	      (TMDY_PLAYMIDI->voice)[i].sample_offset = (TMDY_PLAYMIDI->voice)[i].sample->loop_start;
      }
      else if((TMDY_PLAYMIDI->voice)[i].sample_offset > (TMDY_PLAYMIDI->voice)[i].sample->data_length)
      {
	  free_voice(tmdy_struct, i);
	  return;
      }
  }
  (TMDY_PLAYMIDI->voice)[i].sample_increment=0; /* make sure it isn't negative */
  (TMDY_PLAYMIDI->voice)[i].modulation_wheel=(TMDY_PLAYMIDI->channel)[ch].modulation_wheel;
  (TMDY_PLAYMIDI->voice)[i].delay = (TMDY_PLAYMIDI->voice)[i].sample->envelope_delay;
  (TMDY_PLAYMIDI->voice)[i].modenv_delay = (TMDY_PLAYMIDI->voice)[i].sample->modenv_delay;
  (TMDY_PLAYMIDI->voice)[i].tremolo_delay = (TMDY_PLAYMIDI->voice)[i].sample->tremolo_delay;
  (TMDY_PLAYMIDI->voice)[i].vid=vid;

  (TMDY_PLAYMIDI->voice)[i].tremolo_phase=0;
  (TMDY_PLAYMIDI->voice)[i].tremolo_phase_increment=(TMDY_PLAYMIDI->voice)[i].sample->tremolo_phase_increment;
  (TMDY_PLAYMIDI->voice)[i].tremolo_sweep=(TMDY_PLAYMIDI->voice)[i].sample->tremolo_sweep_increment;
  (TMDY_PLAYMIDI->voice)[i].tremolo_sweep_position=0;
  (TMDY_PLAYMIDI->voice)[i].tremolo_depth = (TMDY_PLAYMIDI->voice)[i].sample->tremolo_depth;

  (TMDY_PLAYMIDI->voice)[i].vibrato_sweep=(TMDY_PLAYMIDI->voice)[i].sample->vibrato_sweep_increment;
  (TMDY_PLAYMIDI->voice)[i].vibrato_sweep_position=0;

  (TMDY_PLAYMIDI->voice)[i].delay_counter = 0;

  memset(&((TMDY_PLAYMIDI->voice)[i].fc), 0, sizeof(FilterCoefficients));
  if((TMDY_PLAYMIDI->opt_lpf_def) && (TMDY_PLAYMIDI->voice)[i].sample->cutoff_freq) {
	  (TMDY_PLAYMIDI->voice)[i].fc.orig_freq = (TMDY_PLAYMIDI->voice)[i].sample->cutoff_freq;
	  (TMDY_PLAYMIDI->voice)[i].fc.orig_reso_dB = (double)(TMDY_PLAYMIDI->voice)[i].sample->resonance / 10.0f;
  } else {
	  (TMDY_PLAYMIDI->voice)[i].fc.freq = -1;
  }

  if((TMDY_PLAYMIDI->opt_nrpn_vibrato))
  {
	  if((TMDY_PLAYMIDI->channel)[ch].vibrato_ratio) {
	      (TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_control_ratio + (TMDY_PLAYMIDI->channel)[ch].vibrato_ratio;
		  if((TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio < 0) {(TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio = 0;}
	  }
	  if((TMDY_PLAYMIDI->channel)[ch].vibrato_depth) {
	      (TMDY_PLAYMIDI->voice)[i].vibrato_depth = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_depth + (TMDY_PLAYMIDI->channel)[ch].vibrato_depth;
		  if((TMDY_PLAYMIDI->voice)[i].vibrato_depth > 255) {(TMDY_PLAYMIDI->voice)[i].vibrato_depth = 255;}
		  else if((TMDY_PLAYMIDI->voice)[i].vibrato_depth < -255) {(TMDY_PLAYMIDI->voice)[i].vibrato_depth = -255;}
	  }
      (TMDY_PLAYMIDI->voice)[i].vibrato_delay = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_delay + (TMDY_PLAYMIDI->channel)[ch].vibrato_delay;
  }
  else
  {
      (TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_control_ratio;
      (TMDY_PLAYMIDI->voice)[i].vibrato_depth = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_depth;
      (TMDY_PLAYMIDI->voice)[i].vibrato_delay = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_delay;
  }
  (TMDY_PLAYMIDI->voice)[i].orig_vibrato_control_ratio = (TMDY_PLAYMIDI->voice)[i].sample->vibrato_control_ratio;

  (TMDY_PLAYMIDI->voice)[i].vibrato_control_counter=(TMDY_PLAYMIDI->voice)[i].vibrato_phase=0;
  for (j=0; j<VIBRATO_SAMPLE_INCREMENTS; j++)
    (TMDY_PLAYMIDI->voice)[i].vibrato_sample_increment[j]=0;

  /* Pan */
  (TMDY_PLAYMIDI->voice)[i].panning = get_panning(tmdy_struct, ch, note, i);

  (TMDY_PLAYMIDI->voice)[i].porta_control_counter = 0;
  if((TMDY_PLAYMIDI->channel)[ch].legato && (TMDY_PLAYMIDI->channel)[ch].legato_flag) {
	  update_legato_controls(tmdy_struct, ch);
  }
  else if((TMDY_PLAYMIDI->channel)[ch].portamento && !(TMDY_PLAYMIDI->channel)[ch].porta_control_ratio)
      update_portamento_controls(tmdy_struct, ch);
  if((TMDY_PLAYMIDI->channel)[ch].porta_control_ratio)
  {
      if((TMDY_PLAYMIDI->channel)[ch].last_note_fine == -1)
      {
	  /* first on */
	  (TMDY_PLAYMIDI->channel)[ch].last_note_fine = (TMDY_PLAYMIDI->voice)[i].note * 256;
	  (TMDY_PLAYMIDI->channel)[ch].porta_control_ratio = 0;
      }
      else
      {
	  (TMDY_PLAYMIDI->voice)[i].porta_control_ratio = (TMDY_PLAYMIDI->channel)[ch].porta_control_ratio;
	  (TMDY_PLAYMIDI->voice)[i].porta_dpb = (TMDY_PLAYMIDI->channel)[ch].porta_dpb;
	  (TMDY_PLAYMIDI->voice)[i].porta_pb = (TMDY_PLAYMIDI->channel)[ch].last_note_fine -
	      (TMDY_PLAYMIDI->voice)[i].note * 256;
	  if((TMDY_PLAYMIDI->voice)[i].porta_pb == 0)
	      (TMDY_PLAYMIDI->voice)[i].porta_control_ratio = 0;
      }
  }

  if(cnt == 0)
      (TMDY_PLAYMIDI->channel)[ch].last_note_fine = (TMDY_PLAYMIDI->voice)[i].note * 256;

  recompute_amp(tmdy_struct, i);
  if ((TMDY_PLAYMIDI->voice)[i].sample->modes & MODES_ENVELOPE)
    {
      /* Ramp up from 0 */
	  (TMDY_PLAYMIDI->voice)[i].envelope_stage=0;
      (TMDY_PLAYMIDI->voice)[i].envelope_volume=0;
      (TMDY_PLAYMIDI->voice)[i].control_counter=0;
      TMDY_MIX->recompute_envelope(tmdy_struct, i);
	  TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	  (TMDY_PLAYMIDI->voice)[i].modenv_stage = 0;
      (TMDY_PLAYMIDI->voice)[i].modenv_volume = 0;
      TMDY_MIX->recompute_modulation_envelope(tmdy_struct, i);
	  TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);

    }
  else
    {
      (TMDY_PLAYMIDI->voice)[i].envelope_increment=0;
	  (TMDY_PLAYMIDI->voice)[i].modenv_increment=0;
      TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	  TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);
    }
  recompute_freq(tmdy_struct, i);
  recompute_voice_filter(tmdy_struct, i);

  (TMDY_PLAYMIDI->voice)[i].timeout = -1;
  if(!TMDY_PLAYMIDI->prescanning_flag)
      ctl_note_event(tmdy_struct, i);
}

static void finish_note(tmdy_struct_ex_t *tmdy_struct, int i)
{
    if ((TMDY_PLAYMIDI->voice)[i].sample->modes & MODES_ENVELOPE)
    {
	/* We need to get the envelope out of Sustain stage. */
	/* Note that (TMDY_PLAYMIDI->voice)[i].envelope_stage < 3 */
	(TMDY_PLAYMIDI->voice)[i].status=VOICE_OFF;
	(TMDY_PLAYMIDI->voice)[i].envelope_stage=3;
	TMDY_MIX->recompute_envelope(tmdy_struct, i);
	(TMDY_PLAYMIDI->voice)[i].modenv_stage=3;
	TMDY_MIX->recompute_modulation_envelope(tmdy_struct, i);
	TMDY_MIX->apply_modulation_envelope(tmdy_struct, i);
	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	ctl_note_event(tmdy_struct, i);
	}
    else
    {
	if(TMDY_READMIDI->current_file_info->pcm_mode != PCM_MODE_NON)
	{
	    free_voice(tmdy_struct, i);
	    ctl_note_event(tmdy_struct, i);
	}
	else
	{
	    /* Set status to OFF so resample_voice() will let this voice out
		of its loop, if any. In any case, this voice dies when it
		    hits the end of its data (ofs>=data_length). */
	    if((TMDY_PLAYMIDI->voice)[i].status != VOICE_OFF)
	    {
		(TMDY_PLAYMIDI->voice)[i].status = VOICE_OFF;
		ctl_note_event(tmdy_struct, i);
	    }
	}
    }
}

static void set_envelope_time(tmdy_struct_ex_t *tmdy_struct, int ch,int val,int stage)
{
	val = val & 0x7F;
	switch(stage) {
	case 0:	/* Attack */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Attack Time (CH:%d VALUE:%d)",ch,val);
		break;
	case 2: /* Decay */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Decay Time (CH:%d VALUE:%d)",ch,val);
		break;
	case 3:	/* Release */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Release Time (CH:%d VALUE:%d)",ch,val);
		break;
	default:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"? Time (CH:%d VALUE:%d)",ch,val);
	}
	(TMDY_PLAYMIDI->channel)[ch].envelope_rate[stage] = val;
}

static void new_delay_voice(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2,ch=(TMDY_PLAYMIDI->voice)[v1].channel;
	FLOAT_T delay,vol;
	FLOAT_T threshold = 1.0;

	/* NRPN Delay Send Level of Drum */
	if(ISDRUMCHANNEL(ch) &&	(TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note]->delay_level / 127.0;
	}

	vol = (TMDY_PLAYMIDI->voice)[v1].velocity * level / 127.0 * TMDY_REVERB->delay_status.level_ratio_c;

	if (vol > threshold) {
		delay = 0;
		if((v2 = find_free_voice(tmdy_struct)) == -1) {return;}
		(TMDY_PLAYMIDI->voice)[v2].cache = NULL;
		delay += TMDY_REVERB->delay_status.time_center;
		(TMDY_PLAYMIDI->voice)[v2] = (TMDY_PLAYMIDI->voice)[v1];	/* copy all parameters */
		(TMDY_PLAYMIDI->voice)[v2].velocity = (uint8)vol;
		(TMDY_PLAYMIDI->voice)[v2].delay += (int32)((TMDY_OUTPUT->play_mode)->rate * delay / 1000);
		recompute_amp(tmdy_struct, v2);
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
		recompute_freq(tmdy_struct, v2);
	}
}


static void new_chorus_voice(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2, ch;
    uint8 vol;

    if((v2 = find_free_voice(tmdy_struct)) == -1)
	return;
    ch = (TMDY_PLAYMIDI->voice)[v1].channel;
    vol = (TMDY_PLAYMIDI->voice)[v1].velocity;
    (TMDY_PLAYMIDI->voice)[v2] = (TMDY_PLAYMIDI->voice)[v1];	/* copy all parameters */

	/* NRPN Chorus Send Level of Drum */
	if(ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note]->chorus_level / 127.0;
	}

    /* Choose lower voice index for base voice (v1) */
    if(v1 > v2)
    {
	int tmp;
	tmp = v1;
	v1 = v2;
	v2 = v1;
    }

    /* v1: Base churos voice
     * v2: Sub chorus voice (detuned)
     */

    (TMDY_PLAYMIDI->voice)[v1].velocity = (uint8)(vol * CHORUS_VELOCITY_TUNING1);
    (TMDY_PLAYMIDI->voice)[v2].velocity = (uint8)(vol * CHORUS_VELOCITY_TUNING2);

    /* Make doubled link v1 and v2 */
    (TMDY_PLAYMIDI->voice)[v1].chorus_link = v2;
    (TMDY_PLAYMIDI->voice)[v2].chorus_link = v1;

    level >>= 2;		     /* scale level to a "better" value */
    if((TMDY_PLAYMIDI->channel)[ch].pitchbend + level < 0x2000)
        (TMDY_PLAYMIDI->voice)[v2].orig_frequency *= bend_fine[level];
    else
	(TMDY_PLAYMIDI->voice)[v2].orig_frequency /= bend_fine[level];

    MYCHECK((TMDY_PLAYMIDI->voice)[v2].orig_frequency);

    (TMDY_PLAYMIDI->voice)[v2].cache = NULL;

    /* set panning & delay */
    if(!((TMDY_OUTPUT->play_mode)->encoding & PE_MONO))
    {
	double delay;

	if((TMDY_PLAYMIDI->voice)[v2].panned == PANNED_CENTER)
	{
	    (TMDY_PLAYMIDI->voice)[v2].panning = 64 + TMDY_COMMON->int_rand(tmdy_struct, 40) - 20; /* 64 +- rand(20) */
	    delay = 0;
	}
	else
	{
	    int panning = (TMDY_PLAYMIDI->voice)[v2].panning;

	    if(panning < CHORUS_OPPOSITE_THRESHOLD)
	    {
		(TMDY_PLAYMIDI->voice)[v2].panning = 127;
		delay = DEFAULT_CHORUS_DELAY1;
	    }
	    else if(panning > 127 - CHORUS_OPPOSITE_THRESHOLD)
	    {
		(TMDY_PLAYMIDI->voice)[v2].panning = 0;
		delay = DEFAULT_CHORUS_DELAY1;
	    }
	    else
	    {
		(TMDY_PLAYMIDI->voice)[v2].panning = (panning < 64 ? 0 : 127);
		delay = DEFAULT_CHORUS_DELAY2;
	    }
	}
	(TMDY_PLAYMIDI->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
    }

    recompute_amp(tmdy_struct, v1);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v1);
    recompute_amp(tmdy_struct, v2);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);

    /* (TMDY_PLAYMIDI->voice)[v2].orig_frequency is changed.
     * Update the depened parameters.
     */
    recompute_freq(tmdy_struct, v2);
}

/* Yet another chorus implementation
 *	by Eric A. Welsh <ewelsh@gpc.wustl.edu>.
 */
static void new_chorus_voice_alternate(tmdy_struct_ex_t *tmdy_struct, int v1, int level)
{
    int v2, ch, panlevel;
    uint8 vol, pan;
    double delay;

    if((v2 = find_free_voice(tmdy_struct)) == -1)
	return;
    ch = (TMDY_PLAYMIDI->voice)[v1].channel;
    (TMDY_PLAYMIDI->voice)[v2] = (TMDY_PLAYMIDI->voice)[v1];

	/* NRPN Chorus Send Level of Drum */
	if(ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note] != NULL) {
		level *= (FLOAT_T)(TMDY_PLAYMIDI->channel)[ch].drums[(TMDY_PLAYMIDI->voice)[v1].note]->chorus_level / 127.0;
	}

    /* for our purposes, hard left will be equal to 1 instead of 0 */
    pan = (TMDY_PLAYMIDI->voice)[v1].panning;
    if (!pan) pan = 1;

    /* Choose lower voice index for base voice (v1) */
    if(v1 > v2)
    {
	int tmp;
	tmp = v1;
	v1 = v2;
	v2 = v1;
    }

    /* lower the volumes so that the two notes add to roughly the orig. vol */
    vol = (TMDY_PLAYMIDI->voice)[v1].velocity;
    (TMDY_PLAYMIDI->voice)[v1].velocity  = (uint8)(vol * CHORUS_VELOCITY_TUNING1);
    (TMDY_PLAYMIDI->voice)[v2].velocity  = (uint8)(vol * CHORUS_VELOCITY_TUNING1);

    /* Make doubled link v1 and v2 */
    (TMDY_PLAYMIDI->voice)[v1].chorus_link = v2;
    (TMDY_PLAYMIDI->voice)[v2].chorus_link = v1;

    /* detune notes for chorus effect */
    level >>= 2;		/* scale to a "better" value */
    if (level)
    {
        if((TMDY_PLAYMIDI->channel)[ch].pitchbend + level < 0x2000)
            (TMDY_PLAYMIDI->voice)[v2].orig_frequency *= bend_fine[level];
        else
	    (TMDY_PLAYMIDI->voice)[v2].orig_frequency /= bend_fine[level];
        (TMDY_PLAYMIDI->voice)[v2].cache = NULL;
    }

    MYCHECK((TMDY_PLAYMIDI->voice)[v2].orig_frequency);

    delay = 0.003;

    /* Try to keep the delayed voice from cancelling out the other voice */
    /* Don't bother with trying to figure out drum pitches... */
    /* Don't bother with mod files for the same reason... */
    /* Drums and mods could be fixed, but pitch detection is too expensive */
    if (!ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[v1].channel) && !(IS_CURRENT_MOD_FILE))
    {
    	double freq, frac;
    
    	freq = TMDY_FREQ->pitch_freq_table[(TMDY_PLAYMIDI->voice)[v1].note];
    	delay *= freq;
    	frac = delay - floor(delay);

	/* force the delay away from 0.5 period */
    	if (frac < 0.5 && frac > 0.40)
    	{
    	    delay = (floor(delay) + 0.40) / freq;
    	    if ((TMDY_OUTPUT->play_mode)->encoding & ~PE_MONO)
    	    	delay += (0.5 - frac) * (1.0 - labs(64 - pan) / 63.0) / freq;
    	}
    	else if (frac >= 0.5 && frac < 0.60)
    	{
    	    delay = (floor(delay) + 0.60) / freq;
    	    if ((TMDY_OUTPUT->play_mode)->encoding & ~PE_MONO)
    	    	delay += (0.5 - frac) * (1.0 - labs(64 - pan) / 63.0) / freq;
    	}
    	else
	    delay = 0.003;
    }

    /* set panning & delay for pseudo-surround effect */
    if((TMDY_OUTPUT->play_mode)->encoding & PE_MONO)    /* delay sounds good */
        (TMDY_PLAYMIDI->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
    else
    {
        panlevel = 63;
        if (pan - panlevel < 1) panlevel = pan - 1;
        if (pan + panlevel > 127) panlevel = 127 - pan;
        (TMDY_PLAYMIDI->voice)[v1].panning -= panlevel;
        (TMDY_PLAYMIDI->voice)[v2].panning += panlevel;

        /* choose which voice is delayed based on panning */
        if ((TMDY_PLAYMIDI->voice)[v1].panned == PANNED_CENTER) {
            /* randomly choose which voice is delayed */
            if (TMDY_COMMON->int_rand(tmdy_struct, 2))
                (TMDY_PLAYMIDI->voice)[v1].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
            else
                (TMDY_PLAYMIDI->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
        else if (pan - 64 < 0) {
            (TMDY_PLAYMIDI->voice)[v2].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
        else {
            (TMDY_PLAYMIDI->voice)[v1].delay += (int)((TMDY_OUTPUT->play_mode)->rate * delay);
        }
    }

    recompute_amp(tmdy_struct, v1);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v1);
    recompute_amp(tmdy_struct, v2);
    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
    if (level) recompute_freq(tmdy_struct, v2);
}

static void note_on(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
    int i, nv, v, ch, note;
    int vlist[32];
    int vid;

    if((nv = find_samples(tmdy_struct, e, vlist)) == 0)
	return;

    note = MIDI_EVENT_NOTE(e);
    vid = new_vidq(tmdy_struct, e->channel, note);
    ch = e->channel;

	recompute_bank_parameter(tmdy_struct, ch,note);
	recompute_channel_filter(tmdy_struct, e);
	calc_sample_panning_average(tmdy_struct, nv, vlist);

    for(i = 0; i < nv; i++)
    {
	v = vlist[i];
	if(ISDRUMCHANNEL(ch) &&
	   (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL &&
	   (TMDY_PLAYMIDI->channel)[ch].drums[note]->pan_random)
	    (TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	else if((TMDY_PLAYMIDI->channel)[ch].pan_random)
	{
	    (TMDY_PLAYMIDI->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	    ctl_mode_event(tmdy_struct, CTLE_PANNING, 1, ch, (TMDY_PLAYMIDI->channel)[ch].panning);
	}
	start_note(tmdy_struct, e, v, vid, nv - i - 1);
#ifdef SMOOTH_MIXING
	(TMDY_PLAYMIDI->voice)[v].old_left_mix = (TMDY_PLAYMIDI->voice)[v].old_right_mix =
	(TMDY_PLAYMIDI->voice)[v].left_mix_inc = (TMDY_PLAYMIDI->voice)[v].left_mix_offset =
	(TMDY_PLAYMIDI->voice)[v].right_mix_inc = (TMDY_PLAYMIDI->voice)[v].right_mix_offset = 0;
#endif
#ifdef USE_DSP_EFFECT
	if((TMDY_PLAYMIDI->opt_surround_chorus))
	    new_chorus_voice_alternate(tmdy_struct, v, 0);
#else
	if(((TMDY_PLAYMIDI->channel)[ch].chorus_level || (TMDY_PLAYMIDI->opt_surround_chorus)))
	{
	    if((TMDY_PLAYMIDI->opt_surround_chorus))
		new_chorus_voice_alternate(tmdy_struct, v, (TMDY_PLAYMIDI->channel)[ch].chorus_level);
	    else
		new_chorus_voice(tmdy_struct, v, (TMDY_PLAYMIDI->channel)[ch].chorus_level);
	}
	if((TMDY_PLAYMIDI->channel)[ch].delay_level)
	{
	    new_delay_voice(tmdy_struct, v, (TMDY_PLAYMIDI->channel)[ch].delay_level);
	}
#endif
    }

    (TMDY_PLAYMIDI->channel)[ch].legato_flag = 1;
}

static void set_voice_timeout(tmdy_struct_ex_t *tmdy_struct, Voice *vp, int ch, int note)
{
    int prog;
    ToneBank *bank;

    if((TMDY_PLAYMIDI->channel)[ch].special_sample > 0)
	return;

    if(ISDRUMCHANNEL(ch))
    {
	prog = note;
	bank = TMDY_INSTRUM->drumset[(int)(TMDY_PLAYMIDI->channel)[ch].bank];
	if(bank == NULL)
	    bank = TMDY_INSTRUM->drumset[0];
    }
    else
    {
	if((prog = (TMDY_PLAYMIDI->channel)[ch].program) == SPECIAL_PROGRAM)
	    return;
	bank = TMDY_INSTRUM->tonebank[(int)(TMDY_PLAYMIDI->channel)[ch].bank];
	if(bank == NULL)
	    bank = TMDY_INSTRUM->tonebank[0];
    }

    if(bank->tone[prog].loop_timeout > 0)
	vp->timeout = (int32)(bank->tone[prog].loop_timeout
			      * (TMDY_OUTPUT->play_mode)->rate * (TMDY_PLAYMIDI->midi_time_ratio)
			      + (TMDY_PLAYMIDI->current_sample));
}

static void note_off(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
  int uv = (TMDY_PLAYMIDI->upper_voices), i;
  int ch, note, vid, sustain;

  ch = e->channel;
  note = MIDI_EVENT_NOTE(e);

  if(ISDRUMCHANNEL(ch) && (TMDY_PLAYMIDI->channel)[ch].drums[note] != NULL &&
     !(TMDY_PLAYMIDI->channel)[ch].drums[note]->rx_note_off)
  {
      ToneBank *bank;

      bank = TMDY_INSTRUM->drumset[(TMDY_PLAYMIDI->channel)[ch].bank];
      if(bank == NULL) bank = TMDY_INSTRUM->drumset[0];
      
      /* uh oh, this drum doesn't have an instrument loaded yet */
      if (bank->tone[note].instrument == NULL)
          return;

      /* this drum is not loaded for some reason (error occured?) */
      if (IS_MAGIC_INSTRUMENT(bank->tone[note].instrument))
          return;

      /* only disallow Note Off if the drum sample is not looped */
      if (!(bank->tone[note].instrument->sample->modes & MODES_LOOPING))
	  return;	/* Note Off is not allowed. */
  }

  if((vid = last_vidq(tmdy_struct, ch, note)) == -1)
      return;
  sustain = (TMDY_PLAYMIDI->channel)[ch].sustain;
  for(i = 0; i < uv; i++)
  {
      if((TMDY_PLAYMIDI->voice)[i].status==VOICE_ON &&
	 (TMDY_PLAYMIDI->voice)[i].channel==ch &&
	 (TMDY_PLAYMIDI->voice)[i].note==note &&
	 (TMDY_PLAYMIDI->voice)[i].vid==vid)
      {
	  if(sustain)
	  {
	      (TMDY_PLAYMIDI->voice)[i].status=VOICE_SUSTAINED;
	      set_voice_timeout(tmdy_struct, TMDY_PLAYMIDI->voice + i, ch, note);
	      ctl_note_event(tmdy_struct, i);
	  }
	  else
	      finish_note(tmdy_struct, i);
      }
  }

  (TMDY_PLAYMIDI->channel)[ch].legato_flag = 0;
}

/* Process the All Notes Off event */
static void all_notes_off(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG, "All notes off on channel %d", c);
  for(i = 0; i < uv; i++)
    if ((TMDY_PLAYMIDI->voice)[i].status==VOICE_ON &&
	(TMDY_PLAYMIDI->voice)[i].channel==c)
      {
	if ((TMDY_PLAYMIDI->channel)[c].sustain)
	  {
	    (TMDY_PLAYMIDI->voice)[i].status=VOICE_SUSTAINED;
	    set_voice_timeout(tmdy_struct, TMDY_PLAYMIDI->voice + i, c, (TMDY_PLAYMIDI->voice)[i].note);
	    ctl_note_event(tmdy_struct, i);
	  }
	else
	  finish_note(tmdy_struct, i);
      }
  for(i = 0; i < 128; i++)
      (TMDY_PLAYMIDI->vidq_head)[c * 128 + i] = (TMDY_PLAYMIDI->vidq_tail)[c * 128 + i] = 0;
}

/* Process the All Sounds Off event */
static void all_sounds_off(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  for(i = 0; i < uv; i++)
    if ((TMDY_PLAYMIDI->voice)[i].channel==c &&
	((TMDY_PLAYMIDI->voice)[i].status & ~(VOICE_FREE | VOICE_DIE)))
      {
	kill_note(tmdy_struct, i);
      }
  for(i = 0; i < 128; i++)
      (TMDY_PLAYMIDI->vidq_head)[c * 128 + i] = (TMDY_PLAYMIDI->vidq_tail)[c * 128 + i] = 0;
}

static void adjust_pressure(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
 /*   int i, uv = (TMDY_PLAYMIDI->upper_voices);
    int note, ch;
	FLOAT_T pressure, amp_ctl, rate_ctl1, pitch_depth1, cutoff_ctl, coef;

	ch = e->channel;
	pressure = (FLOAT_T)e->b / 127.0f;
	amp_ctl = (TMDY_PLAYMIDI->channel)[ch].caf_amp_ctl * pressure;
	rate_ctl1 = (TMDY_PLAYMIDI->channel)[ch].caf_rate_ctl1 * pressure + 1.0f;
	pitch_depth1 = (TMDY_PLAYMIDI->channel)[ch].caf_pitch_depth1 * pressure + 1.0f;
	cutoff_ctl = (TMDY_PLAYMIDI->channel)[ch].caf_cutoff_ctl * pressure + 1.0f;
    note = MIDI_EVENT_NOTE(e);

    for(i = 0; i < uv; i++)
    if((TMDY_PLAYMIDI->voice)[i].status == VOICE_ON &&
       (TMDY_PLAYMIDI->voice)[i].channel == ch &&
       (TMDY_PLAYMIDI->voice)[i].note == note)
    {
		recompute_amp(tmdy_struct, i);
		(TMDY_PLAYMIDI->voice)[i].tremolo_depth = amp_ctl * 255;
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		(TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio *= rate_ctl1;
		(TMDY_PLAYMIDI->voice)[i].vibrato_depth *= pitch_depth1;
		recompute_freq(tmdy_struct, i);
		if((TMDY_PLAYMIDI->opt_lpf_def) && (TMDY_PLAYMIDI->voice)[i].sample->cutoff_freq) {
			coef = (TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef;
			(TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef *= cutoff_ctl;
			recompute_voice_filter(tmdy_struct, i);
			(TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef = coef;
		}
    }*/
}

static void adjust_channel_pressure(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e)
{
 /*   if((TMDY_PLAYMIDI->opt_channel_pressure))
    {
	int i, uv = (TMDY_PLAYMIDI->upper_voices);
	int ch;
	FLOAT_T pressure, amp_ctl, rate_ctl1, pitch_depth1, cutoff_ctl, coef;

	ch = e->channel;
	pressure = (FLOAT_T)e->a / 127.0f;
	amp_ctl = (TMDY_PLAYMIDI->channel)[ch].caf_amp_ctl * pressure;
	rate_ctl1 = (TMDY_PLAYMIDI->channel)[ch].caf_rate_ctl1 * pressure + 1.0f;
	pitch_depth1 = (TMDY_PLAYMIDI->channel)[ch].caf_pitch_depth1 * pressure + 1.0f;
	cutoff_ctl = (TMDY_PLAYMIDI->channel)[ch].caf_cutoff_ctl * pressure + 1.0f;
	  
	for(i = 0; i < uv; i++)
	{
	    if((TMDY_PLAYMIDI->voice)[i].status == VOICE_ON && (TMDY_PLAYMIDI->voice)[i].channel == ch)
	    {
		recompute_amp(tmdy_struct, i);
		(TMDY_PLAYMIDI->voice)[i].tremolo_depth = amp_ctl * 255;
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		(TMDY_PLAYMIDI->voice)[i].vibrato_control_ratio *= rate_ctl1;
		(TMDY_PLAYMIDI->voice)[i].vibrato_depth *= pitch_depth1;
		recompute_freq(tmdy_struct, i);
		if((TMDY_PLAYMIDI->opt_lpf_def) && (TMDY_PLAYMIDI->voice)[i].sample->cutoff_freq) {
			coef = (TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef;
			(TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef *= cutoff_ctl;
			recompute_voice_filter(tmdy_struct, i);
			(TMDY_PLAYMIDI->channel)[ch].cutoff_freq_coef = coef;
		}
	    }
	}
    }*/
}

static void adjust_panning(tmdy_struct_ex_t *tmdy_struct, int c)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices), pan = (TMDY_PLAYMIDI->channel)[c].panning;
    for(i = 0; i < uv; i++)
    {
	if (((TMDY_PLAYMIDI->voice)[i].channel==c) &&
	    ((TMDY_PLAYMIDI->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
	{
            /* adjust pan to include drum/sample pan offsets */
			pan = get_panning(tmdy_struct, c,i,i);

	    /* Hack to handle -EFchorus=2 in a "reasonable" way */
#ifdef USE_DSP_EFFECT
	    if((TMDY_PLAYMIDI->opt_surround_chorus) && (TMDY_PLAYMIDI->voice)[i].chorus_link != i)
#else
	    if(((TMDY_PLAYMIDI->channel)[c].chorus_level || (TMDY_PLAYMIDI->opt_surround_chorus)) &&
	       (TMDY_PLAYMIDI->voice)[i].chorus_link != i)
#endif
	    {
		int v1, v2;

		if(i >= (TMDY_PLAYMIDI->voice)[i].chorus_link)
		    /* `i' is not base chorus voice.
		     *  This sub voice is already updated.
		     */
		    continue;

		v1 = i;				/* base voice */
		v2 = (TMDY_PLAYMIDI->voice)[i].chorus_link;	/* sub voice (detuned) */

		if((TMDY_PLAYMIDI->opt_surround_chorus)) /* Surround chorus mode by Eric. */
		{
		    int panlevel;

		    if (!pan) pan = 1;	/* make hard left be 1 instead of 0 */
		    panlevel = 63;
		    if (pan - panlevel < 1) panlevel = pan - 1;
		    if (pan + panlevel > 127) panlevel = 127 - pan;
		    (TMDY_PLAYMIDI->voice)[v1].panning = pan - panlevel;
		    (TMDY_PLAYMIDI->voice)[v2].panning = pan + panlevel;
		}
		else
		{
		    (TMDY_PLAYMIDI->voice)[v1].panning = pan;
		    if(pan > 60 && pan < 68) /* PANNED_CENTER */
			(TMDY_PLAYMIDI->voice)[v2].panning =
			    64 + TMDY_COMMON->int_rand(tmdy_struct, 40) - 20; /* 64 +- rand(20) */
		    else if(pan < CHORUS_OPPOSITE_THRESHOLD)
			(TMDY_PLAYMIDI->voice)[v2].panning = 127;
		    else if(pan > 127 - CHORUS_OPPOSITE_THRESHOLD)
			(TMDY_PLAYMIDI->voice)[v2].panning = 0;
		    else
			(TMDY_PLAYMIDI->voice)[v2].panning = (pan < 64 ? 0 : 127);
		}
		recompute_amp(tmdy_struct, v2);
		TMDY_MIX->apply_envelope_to_amp(tmdy_struct, v2);
		/* v1 == i, so v1 will be updated next */
	    }
	    else
		(TMDY_PLAYMIDI->voice)[i].panning = pan;

	    recompute_amp(tmdy_struct, i);
	    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
	}
    }
}

void play_midi_setup_drums(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    (TMDY_PLAYMIDI->channel)[ch].drums[note] = (struct DrumParts *)
	TMDY_UTILS->mblock->new_segment(tmdy_struct, &TMDY_PLAYMIDI->playmidi_pool, sizeof(struct DrumParts));
    reset_drum_controllers(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].drums, note);
}

static void adjust_drum_panning(tmdy_struct_ex_t *tmdy_struct, int ch, int note)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices), pan;

    for(i = 0; i < uv; i++) {
		if((TMDY_PLAYMIDI->voice)[i].channel == ch &&
		   (TMDY_PLAYMIDI->voice)[i].note == note &&
		   ((TMDY_PLAYMIDI->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
		{
			(TMDY_PLAYMIDI->voice)[i].panning = get_panning(tmdy_struct, ch,note,i);
			recompute_amp(tmdy_struct, i);
			TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		}
	}
}

static void drop_sustain(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  for(i = 0; i < uv; i++)
    if ((TMDY_PLAYMIDI->voice)[i].status==VOICE_SUSTAINED && (TMDY_PLAYMIDI->voice)[i].channel==c)
      finish_note(tmdy_struct, i);
}

static void adjust_pitch(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  for(i = 0; i < uv; i++)
    if ((TMDY_PLAYMIDI->voice)[i].status!=VOICE_FREE && (TMDY_PLAYMIDI->voice)[i].channel==c)
	recompute_freq(tmdy_struct, i);
}

static void adjust_volume(tmdy_struct_ex_t *tmdy_struct, int c)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  for(i = 0; i < uv; i++)
    if ((TMDY_PLAYMIDI->voice)[i].channel==c &&
	((TMDY_PLAYMIDI->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED)))
      {
	recompute_amp(tmdy_struct, i);
	TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
      }
}

static void set_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch, int level)
{
    if((TMDY_PLAYMIDI->opt_reverb_control) <= 0)
    {
	(TMDY_PLAYMIDI->channel)[ch].reverb_level = (TMDY_PLAYMIDI->channel)[ch].reverb_id =
	    -(TMDY_PLAYMIDI->opt_reverb_control);
	(TMDY_PLAYMIDI->make_rvid_flag) = 1;
	return;
    }
    (TMDY_PLAYMIDI->channel)[ch].reverb_level = level;
    (TMDY_PLAYMIDI->make_rvid_flag) = 0;	/* to update reverb_id */
}

int get_reverb_level(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    if((TMDY_PLAYMIDI->opt_reverb_control) <= 0)
	return -(TMDY_PLAYMIDI->opt_reverb_control);

    if((TMDY_PLAYMIDI->channel)[ch].reverb_level == -1)
	return DEFAULT_REVERB_SEND_LEVEL;
    return (TMDY_PLAYMIDI->channel)[ch].reverb_level;
}

int get_chorus_level(tmdy_struct_ex_t *tmdy_struct, int ch)
{
#ifdef DISALLOW_DRUM_BENDS
    if(ISDRUMCHANNEL(ch))
	return 0; /* Not supported drum channel chorus */
#endif
    if((TMDY_PLAYMIDI->opt_chorus_control) == 1)
	return (TMDY_PLAYMIDI->channel)[ch].chorus_level;
    return -(TMDY_PLAYMIDI->opt_chorus_control);
}

static void make_rvid(tmdy_struct_ex_t *tmdy_struct)
{
    int i, j, lv, maxrv;

    for(maxrv = MAX_CHANNELS - 1; maxrv >= 0; maxrv--)
    {
	if((TMDY_PLAYMIDI->channel)[maxrv].reverb_level == -1)
	    (TMDY_PLAYMIDI->channel)[maxrv].reverb_id = -1;
	else if((TMDY_PLAYMIDI->channel)[maxrv].reverb_level >= 0)
	    break;
    }

    /* collect same reverb level. */
    for(i = 0; i <= maxrv; i++)
    {
	if((lv = (TMDY_PLAYMIDI->channel)[i].reverb_level) == -1)
	{
	    (TMDY_PLAYMIDI->channel)[i].reverb_id = -1;
	    continue;
	}
	(TMDY_PLAYMIDI->channel)[i].reverb_id = i;
	for(j = 0; j < i; j++)
	{
	    if((TMDY_PLAYMIDI->channel)[j].reverb_level == lv)
	    {
		(TMDY_PLAYMIDI->channel)[i].reverb_id = j;
		break;
	    }
	}
    }
}

static void adjust_master_volume(tmdy_struct_ex_t *tmdy_struct)
{
  int i, uv = (TMDY_PLAYMIDI->upper_voices);
  adjust_amplification(tmdy_struct);
  for(i = 0; i < uv; i++)
      if((TMDY_PLAYMIDI->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED))
      {
	  recompute_amp(tmdy_struct, i);
	  TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
      }
}

int midi_drumpart_change(tmdy_struct_ex_t *tmdy_struct, int ch, int isdrum)
{
    if(IS_SET_CHANNELMASK((TMDY_PLAYMIDI->drumchannel_mask), ch))
	return 0;
    if(isdrum)
    {
	SET_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), ch);
	SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, ch);
    }
    else
    {
	UNSET_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), ch);
	UNSET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, ch);
    }

    return 1;
}

void midi_program_change(tmdy_struct_ex_t *tmdy_struct, int ch, int prog)
{
	int dr = ISDRUMCHANNEL(ch);
	int newbank, b, p;
	
	switch (TMDY_READMIDI->play_system_mode) {
	case GS_SYSTEM_MODE:	/* GS */
		switch ((TMDY_PLAYMIDI->channel)[ch].bank_lsb) {
		case 0:		/* No change */
			break;
		case 1:
			(TMDY_PLAYMIDI->channel)[ch].mapID = (dr) ? SC_55_DRUM_MAP : SC_55_TONE_MAP;
			break;
		case 2:
			(TMDY_PLAYMIDI->channel)[ch].mapID = (dr) ? SC_88_DRUM_MAP : SC_88_TONE_MAP;
			break;
		case 3:
			(TMDY_PLAYMIDI->channel)[ch].mapID = (dr) ? SC_88PRO_DRUM_MAP : SC_88PRO_TONE_MAP;
			break;
		default:
			break;
		}
		newbank = (TMDY_PLAYMIDI->channel)[ch].bank_msb;
		break;
	case XG_SYSTEM_MODE:	/* XG */
		switch ((TMDY_PLAYMIDI->channel)[ch].bank_msb) {
		case 0:		/* Normal */
			if (ch == 9 && (TMDY_PLAYMIDI->channel)[ch].bank_lsb == 127
					&& (TMDY_PLAYMIDI->channel)[ch].mapID == XG_DRUM_MAP)
				/* FIXME: Why this part is drum?  Is this correct? */
				break;
			midi_drumpart_change(tmdy_struct, ch, 0);
			(TMDY_PLAYMIDI->channel)[ch].mapID = XG_NORMAL_MAP;
			break;
		case 64:	/* SFX voice */
			midi_drumpart_change(tmdy_struct, ch, 0);
			(TMDY_PLAYMIDI->channel)[ch].mapID = XG_SFX64_MAP;
			break;
		case 126:	/* SFX kit */
			midi_drumpart_change(tmdy_struct, ch, 1);
			(TMDY_PLAYMIDI->channel)[ch].mapID = XG_SFX126_MAP;
			break;
		case 127:	/* Drumset */
			midi_drumpart_change(tmdy_struct, ch, 1);
			(TMDY_PLAYMIDI->channel)[ch].mapID = XG_DRUM_MAP;
			break;
		default:
			break;
		}
		newbank = (TMDY_PLAYMIDI->channel)[ch].bank_lsb;
		break;
	default:
		newbank = (TMDY_PLAYMIDI->channel)[ch].bank_msb;
		break;
	}
	if (dr) {
		(TMDY_PLAYMIDI->channel)[ch].bank = prog;	/* newbank is ignored */
		(TMDY_PLAYMIDI->channel)[ch].program = prog;
		if (TMDY_INSTRUM->drumset[prog] == NULL || TMDY_INSTRUM->drumset[prog]->alt == NULL)
			(TMDY_PLAYMIDI->channel)[ch].altassign = TMDY_INSTRUM->drumset[0]->alt;
		else
			(TMDY_PLAYMIDI->channel)[ch].altassign = TMDY_INSTRUM->drumset[prog]->alt;
		ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, 1);
	} else {
		(TMDY_PLAYMIDI->channel)[ch].bank = ((TMDY_PLAYMIDI->special_tonebank) >= 0)
				? (TMDY_PLAYMIDI->special_tonebank) : newbank;
		(TMDY_PLAYMIDI->channel)[ch].program = (TMDY_INSTRUM->default_program[ch] == SPECIAL_PROGRAM)
				? SPECIAL_PROGRAM : prog;
		(TMDY_PLAYMIDI->channel)[ch].altassign = NULL;
		ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, 0);
		if ((TMDY_PLAYMIDI->opt_realtime_playing) && ((TMDY_OUTPUT->play_mode)->flag & PF_PCM_STREAM)) {
			b = (TMDY_PLAYMIDI->channel)[ch].bank, p = prog;
			TMDY_INSTRUM->instrument_map(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].mapID, &b, &p);
			play_midi_load_instrument(tmdy_struct, 0, b, p);
		}
	}
}

static void process_sysex_event(tmdy_struct_ex_t *tmdy_struct, int ev,int ch,int val,int b)
{
	if(ev == ME_SYSEX_GS_LSB)
	{
		switch(b)
		{
		case 0x00:	/* EQ ON/OFF */
			if(!(TMDY_PLAYMIDI->opt_eq_control)) {break;}
			if((TMDY_PLAYMIDI->channel)[ch].eq_on != val) {
				if(val) {
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ ON (CH:%d)",ch);
				} else {
					(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ OFF (CH:%d)",ch);
				}
			}
			(TMDY_PLAYMIDI->channel)[ch].eq_on = val;
			break;
		case 0x01:	/* EQ LOW FREQ */
			if(!(TMDY_PLAYMIDI->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.low_freq = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ LOW FREQ (%d)",val);
			break;
		case 0x02:	/* EQ LOW GAIN */
			if(!(TMDY_PLAYMIDI->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.low_gain = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ LOW GAIN (%d dB)",val - 0x40);
			break;
		case 0x03:	/* EQ HIGH FREQ */
			if(!(TMDY_PLAYMIDI->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.high_freq = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ HIGH FREQ (%d)",val);
			break;
		case 0x04:	/* EQ HIGH GAIN */
			if(!(TMDY_PLAYMIDI->opt_eq_control)) {break;}
			TMDY_REVERB->eq_status.high_gain = val;
			TMDY_READMIDI->recompute_eq_status(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EQ HIGH GAIN (%d dB)",val - 0x40);
			break;
		case 0x05:	/* Reverb Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Macro (%d)",val);
			TMDY_READMIDI->set_reverb_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			break;
		case 0x06:	/* Reverb Character */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Character (%d)",val);
			if(TMDY_REVERB->reverb_status.character != val) {
				TMDY_REVERB->reverb_status.character = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x07:	/* Reverb Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Pre-LPF (%d)",val);
			if(TMDY_REVERB->reverb_status.pre_lpf != val) {
				TMDY_REVERB->reverb_status.pre_lpf = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x08:	/* Reverb Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Level (%d)",val);
			if(TMDY_REVERB->reverb_status.level != val) {
				TMDY_REVERB->reverb_status.level = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x09:	/* Reverb Time */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Time (%d)",val);
			if(TMDY_REVERB->reverb_status.time != val) {
				TMDY_REVERB->reverb_status.time = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
				TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			}
			break;
		case 0x0A:	/* Reverb Delay Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Delay Feedback (%d)",val);
			if(TMDY_REVERB->reverb_status.delay_feedback != val) {
				TMDY_REVERB->reverb_status.delay_feedback = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x0C:	/* Reverb Predelay Time */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Predelay Time (%d)",val);
			if(TMDY_REVERB->reverb_status.pre_delay_time != val) {
				TMDY_REVERB->reverb_status.pre_delay_time = val;
				TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			}
			break;
		case 0x0D:	/* Chorus Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Macro (%d)",val);
			TMDY_REVERB->init_ch_chorus(tmdy_struct);
			TMDY_READMIDI->set_chorus_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x0E:	/* Chorus Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Pre-LPF (%d)",val);
			TMDY_REVERB->chorus_param.chorus_pre_lpf = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x0F:	/* Chorus Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Level (%d)",val);
			TMDY_REVERB->chorus_param.chorus_level = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x10:	/* Chorus Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Feedback (%d)",val);
			TMDY_REVERB->chorus_param.chorus_feedback = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x11:	/* Chorus Delay */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Delay (%d)",val);
			TMDY_REVERB->chorus_param.chorus_delay = val;
			TMDY_REVERB->init_ch_chorus(tmdy_struct);
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x12:	/* Chorus Rate */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Rate (%d)",val);
			TMDY_REVERB->chorus_param.chorus_rate = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x13:	/* Chorus Depth */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Depth (%d)",val);
			TMDY_REVERB->chorus_param.chorus_depth = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			TMDY_REVERB->init_chorus_lfo(tmdy_struct);
			break;
		case 0x14:	/* Chorus Send Level to Reverb */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send Level to Reverb (%d)",val);
			TMDY_REVERB->chorus_param.chorus_send_level_to_reverb = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x15:	/* Chorus Send Level to Delay */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send Level to Delay (%d)",val);
			TMDY_REVERB->chorus_param.chorus_send_level_to_delay = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x16:	/* Delay Macro */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Macro (%d)",val);
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->set_delay_macro(tmdy_struct, val);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x17:	/* Delay Pre-LPF */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Pre-LPF (%d)",val);
			TMDY_REVERB->delay_status.pre_lpf = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x18:	/* Delay Time Center */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Center (%d)",val);
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_REVERB->delay_status.time_center = delay_time_center_table[val > 0x73 ? 0x73 : val];
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x19:	/* Delay Time Ratio Left */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Ratio Left (%d)",val);
			if(val == 0) {val = 1;}
			TMDY_REVERB->delay_status.time_ratio_left = (double)val / 24;
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1A:	/* Delay Time Ratio Right */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Time Ratio Right (%d)",val);
			if(val == 0) {val = 1;}
			TMDY_REVERB->delay_status.time_ratio_right = (double)val / 24;
			TMDY_REVERB->init_ch_delay(tmdy_struct);
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1B:	/* Delay Level Center */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Center (%d)",val);
			TMDY_REVERB->delay_status.level_center = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1C:	/* Delay Level Left */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Left (%d)",val);
			TMDY_REVERB->delay_status.level_left = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1D:	/* Delay Level Right */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level Right (%d)",val);
			TMDY_REVERB->delay_status.level_right = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1E:	/* Delay Level */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Level (%d)",val);
			TMDY_REVERB->delay_status.level = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x1F:	/* Delay Feedback */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Feedback (%d)",val);
			TMDY_REVERB->delay_status.feedback = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x20:	/* Delay Send Level to Reverb */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send Level to Reverb (%d)",val);
			TMDY_REVERB->delay_status.send_reverb = val;
			TMDY_READMIDI->recompute_delay_status(tmdy_struct);
			break;
		case 0x21:	/* Velocity Sense Depth */
			(TMDY_PLAYMIDI->channel)[ch].velocity_sense_depth = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Velocity Sense Depth (CH:%d VAL:%d)",ch,val);
			break;
		case 0x22:	/* Velocity Sense Offset */
			(TMDY_PLAYMIDI->channel)[ch].velocity_sense_offset = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Velocity Sense Offset (CH:%d VAL:%d)",ch,val);
			break;
		case 0x23:	/* Insertion Effect ON/OFF */
			if(!(TMDY_PLAYMIDI->opt_insertion_effect)) {break;}
			if((TMDY_PLAYMIDI->channel)[ch].insertion_effect != val) {
				if(val) {(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX ON (CH:%d)",ch);}
				else {(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX OFF (CH:%d)",ch);}
			}
			(TMDY_PLAYMIDI->channel)[ch].insertion_effect = val;
			break;
		case 0x24:	/* Assign Mode */
			(TMDY_PLAYMIDI->channel)[ch].assign_mode = val;
			if(val == 0) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Single (CH:%d)",ch);
			} else if(val == 1) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Limited-Multi (CH:%d)",ch);
			} else if(val == 2) {
				(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Assign Mode: Full-Multi (CH:%d)",ch);
			}
			break;
		case 0x25:	/* TONE MAP-0 NUMBER */
			(TMDY_PLAYMIDI->channel)[ch].tone_map0_number = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tone Map-0 Number (CH:%d VAL:%d)",ch,val);
			break;
		case 0x26:	/* Pitch Offset Fine */
			(TMDY_PLAYMIDI->channel)[ch].pitch_offset_fine = (FLOAT_T)((((int32)val << 4) | (int32)val) - 0x80) / 10.0;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Pitch Offset Fine (CH:%d %3fHz)",ch,(TMDY_PLAYMIDI->channel)[ch].pitch_offset_fine);
			break;
		case 0x27:	/* Insertion Effect Parameter */
			if(!(TMDY_PLAYMIDI->opt_insertion_effect)) {break;}
			gs_ieffect.type_msb = val;
			break;
		case 0x28:	/* Insertion Effect Parameter */
			if(!(TMDY_PLAYMIDI->opt_insertion_effect)) {break;}
			gs_ieffect.type_lsb = val;
			gs_ieffect.type = ((int32)gs_ieffect.type_msb << 8) | (int32)gs_ieffect.type_lsb;
			TMDY_READMIDI->set_insertion_effect_default_parameter(tmdy_struct);
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"EFX TYPE (%02X %02X)",gs_ieffect.type_msb,gs_ieffect.type_lsb);
			break;
		case 0x29:
			gs_ieffect.parameter[0] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2A:
			gs_ieffect.parameter[1] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2B:
			gs_ieffect.parameter[2] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2C:
			gs_ieffect.parameter[3] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2D:
			gs_ieffect.parameter[4] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2E:
			gs_ieffect.parameter[5] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x2F:
			gs_ieffect.parameter[6] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x30:
			gs_ieffect.parameter[7] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x31:
			gs_ieffect.parameter[8] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x32:
			gs_ieffect.parameter[9] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x33:
			gs_ieffect.parameter[10] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x34:
			gs_ieffect.parameter[11] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x35:
			gs_ieffect.parameter[12] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x36:
			gs_ieffect.parameter[13] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x37:
			gs_ieffect.parameter[14] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x38:
			gs_ieffect.parameter[15] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x39:
			gs_ieffect.parameter[16] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3A:
			gs_ieffect.parameter[17] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3B:
			gs_ieffect.parameter[18] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3C:
			gs_ieffect.parameter[19] = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3D:
			gs_ieffect.send_reverb = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3E:
			gs_ieffect.send_chorus = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x3F:
			gs_ieffect.send_delay = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x40:
			gs_ieffect.control_source1 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x41:
			gs_ieffect.control_depth1 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x42:
			gs_ieffect.control_source2 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x43:
			gs_ieffect.control_depth2 = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x44:
			gs_ieffect.send_eq_switch = val;
			TMDY_READMIDI->recompute_insertion_effect(tmdy_struct);
			break;
		case 0x45:	/* Rx. Channel */
			reset_controllers(tmdy_struct, ch);
			redraw_controllers(tmdy_struct, ch);
			all_notes_off(tmdy_struct, ch);
			if (val == 0x80)
				TMDY_READMIDI->remove_channel_layer(tmdy_struct, ch);
			else
				TMDY_READMIDI->add_channel_layer(tmdy_struct, ch, val);
			break;
		case 0x46:	/* Channel Msg Rx Port */
			reset_controllers(tmdy_struct, ch);
			redraw_controllers(tmdy_struct, ch);
			all_notes_off(tmdy_struct, ch);
			(TMDY_PLAYMIDI->channel)[ch].port_select = val;
			break;
			
			/* MOD PITCH CONTROL */
			/* 0x45~ MOD, CAF, PAF */
		default:
			break;
		}
	} else if(ev == ME_SYSEX_XG_LSB) {
		switch(b)
		{
		case 0x00:	/* Reverb Return */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Reverb Return (%d)",val);
			TMDY_REVERB->reverb_status.level = val;
			TMDY_READMIDI->recompute_reverb_status(tmdy_struct);
			TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
			break;
		case 0x01:	/* Chorus Return */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Return (%d)",val);
			TMDY_REVERB->chorus_param.chorus_level = val;
			TMDY_READMIDI->recompute_chorus_status(tmdy_struct);
			break;
		case 0x65:	/* Rcv CHANNEL */
			if (val == 0x7f)
				TMDY_READMIDI->remove_channel_layer(tmdy_struct, ch);
			else
				TMDY_READMIDI->add_channel_layer(tmdy_struct, ch, val);
			break;
		default:
			break;
		}
	}
}

static void play_midi_prescan(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
    int i, j, k, ch, orig_ch, port_ch, offset, layered;
    
    if((TMDY_PLAYMIDI->opt_amp_compensation)) {(TMDY_PLAYMIDI->mainvolume_max) = 0;}
    else {(TMDY_PLAYMIDI->mainvolume_max) = 0x7f;}
    (TMDY_PLAYMIDI->compensation_ratio) = 1.0;

    TMDY_PLAYMIDI->prescanning_flag = 1;
    TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
    reset_midi(tmdy_struct, 0);
    TMDY_RECACHE->resamp_cache_reset(tmdy_struct);

	while (ev->type != ME_EOT) {
#ifndef SUPPRESS_CHANNEL_LAYER
		orig_ch = ev->channel;
		layered = ! IS_SYSEX_EVENT_TYPE(ev->type);
		for (j = 0; j < MAX_CHANNELS; j += 16) {
			port_ch = (orig_ch + j) % MAX_CHANNELS;
			offset = (port_ch < 16) ? 0 : 16;
			for (k = offset; k < offset + 16; k++) {
				if (! layered && (j || k != offset))
					continue;
				if (layered) {
					if (! IS_SET_CHANNELMASK(
							(TMDY_PLAYMIDI->channel)[k].channel_layer, port_ch)
							|| (TMDY_PLAYMIDI->channel)[k].port_select ^ (orig_ch >= 16))
						continue;
					ev->channel = k;
				}
#endif
	ch = ev->channel;

	switch(ev->type)
	{
	  case ME_NOTEON:
	    if(((TMDY_PLAYMIDI->channel)[ch].portamento_time_msb |
		(TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb) == 0 ||
	       (TMDY_PLAYMIDI->channel)[ch].portamento == 0)
	    {
		int nv;
		int vlist[32];
		Voice *vp;

		nv = find_samples(tmdy_struct, ev, vlist);
		for(i = 0; i < nv; i++)
		{
		    vp = TMDY_PLAYMIDI->voice + vlist[i];
		    start_note(tmdy_struct, ev, vlist[i], 0, nv - i - 1);
		    TMDY_RECACHE->resamp_cache_refer_on(tmdy_struct, vp, ev->time);
		    vp->status = VOICE_FREE;
		    vp->temper_instant = 0;
		}
	    }
	    break;

	  case ME_NOTEOFF:
	    TMDY_RECACHE->resamp_cache_refer_off(tmdy_struct, ch, MIDI_EVENT_NOTE(ev), ev->time);
	    break;

	  case ME_PORTAMENTO_TIME_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].portamento_time_msb = ev->a;
	    break;

	  case ME_PORTAMENTO_TIME_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb = ev->a;
	    break;

	  case ME_PORTAMENTO:
	    (TMDY_PLAYMIDI->channel)[ch].portamento = (ev->a >= 64);

	  case ME_RESET_CONTROLLERS:
	    reset_controllers(tmdy_struct, ch);
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    break;

	  case ME_PROGRAM:
	    midi_program_change(tmdy_struct, ch, ev->a);
	    break;

	  case ME_TONE_BANK_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].bank_msb = ev->a;
	    break;

	  case ME_TONE_BANK_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].bank_lsb = ev->a;
	    break;

	  case ME_RESET:
	    TMDY_READMIDI->change_system_mode(tmdy_struct, ev->a);
	    reset_midi(tmdy_struct, 0);
	    break;

	  case ME_PITCHWHEEL:
	  case ME_ALL_NOTES_OFF:
	  case ME_ALL_SOUNDS_OFF:
	  case ME_MONO:
	  case ME_POLY:
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    break;

	  case ME_DRUMPART:
	    if(midi_drumpart_change(tmdy_struct, ch, ev->a))
		midi_program_change(tmdy_struct, ch, (TMDY_PLAYMIDI->channel)[ch].program);
	    break;

	  case ME_KEYSHIFT:
	    TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
	    (TMDY_PLAYMIDI->channel)[ch].key_shift = (int)ev->a - 0x40;
	    break;

	  case ME_SCALE_TUNING:
		TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, ev->time);
		(TMDY_PLAYMIDI->channel)[ch].scale_tuning[ev->a] = ev->b;
		break;

	  case ME_MAINVOLUME:
	    if (ev->a > (TMDY_PLAYMIDI->mainvolume_max)) {
	      (TMDY_PLAYMIDI->mainvolume_max) = ev->a;
	      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"ME_MAINVOLUME/max (CH:%d VAL:%#x)",ev->channel,ev->a);
	    }
	    break;
	}
#ifndef SUPPRESS_CHANNEL_LAYER
			}
		}
		ev->channel = orig_ch;
#endif
	ev++;
    }

    /* calculate compensation ratio */
    if (0 < (TMDY_PLAYMIDI->mainvolume_max) && (TMDY_PLAYMIDI->mainvolume_max) < 0x7f) {
      (TMDY_PLAYMIDI->compensation_ratio) = pow((double)0x7f/(double)(TMDY_PLAYMIDI->mainvolume_max), 4);
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_DEBUG,"Compensation ratio:%lf",(TMDY_PLAYMIDI->compensation_ratio));
    }

    for(i = 0; i < MAX_CHANNELS; i++)
	TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, i, ev->time);
    TMDY_RECACHE->resamp_cache_create(tmdy_struct);
    TMDY_PLAYMIDI->prescanning_flag = 0;
}

static int32 midi_cnv_vib_rate(tmdy_struct_ex_t *tmdy_struct, int rate)
{
    return (int32)((double)(TMDY_OUTPUT->play_mode)->rate * MODULATION_WHEEL_RATE
		   / (midi_time_table[rate] *
		      2.0 * VIBRATO_SAMPLE_INCREMENTS));
}

static int midi_cnv_vib_depth(tmdy_struct_ex_t *tmdy_struct, int depth)
{
    return (int)(depth * VIBRATO_DEPTH_TUNING);
}

static int32 midi_cnv_vib_delay(tmdy_struct_ex_t *tmdy_struct, int delay)
{
    return (int32)(midi_time_table[delay]);
}

static int last_rpn_addr(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	int lsb, msb, addr, i;
	struct rpn_tag_map_t *addrmap;
	struct rpn_tag_map_t {
		int addr, mask, tag;
	};
	static struct rpn_tag_map_t nrpn_addr_map[] = {
		{0x0108, 0xffff, NRPN_ADDR_0108},
		{0x0109, 0xffff, NRPN_ADDR_0109},
		{0x010a, 0xffff, NRPN_ADDR_010A},
		{0x0120, 0xffff, NRPN_ADDR_0120},
		{0x0121, 0xffff, NRPN_ADDR_0121},
		{0x0163, 0xffff, NRPN_ADDR_0163},
		{0x0164, 0xffff, NRPN_ADDR_0164},
		{0x0166, 0xffff, NRPN_ADDR_0166},
		{0x1400, 0xff00, NRPN_ADDR_1400},
		{0x1500, 0xff00, NRPN_ADDR_1500},
		{0x1600, 0xff00, NRPN_ADDR_1600},
		{0x1700, 0xff00, NRPN_ADDR_1700},
		{0x1800, 0xff00, NRPN_ADDR_1800},
		{0x1900, 0xff00, NRPN_ADDR_1900},
		{0x1a00, 0xff00, NRPN_ADDR_1A00},
		{0x1c00, 0xff00, NRPN_ADDR_1C00},
		{0x1d00, 0xff00, NRPN_ADDR_1D00},
		{0x1e00, 0xff00, NRPN_ADDR_1E00},
		{0x1f00, 0xff00, NRPN_ADDR_1F00},
		{-1, -1, 0}
	};
	static struct rpn_tag_map_t rpn_addr_map[] = {
		{0x0000, 0xffff, RPN_ADDR_0000},
		{0x0001, 0xffff, RPN_ADDR_0001},
		{0x0002, 0xffff, RPN_ADDR_0002},
		{0x0003, 0xffff, RPN_ADDR_0003},
		{0x0004, 0xffff, RPN_ADDR_0004},
		{0x7f7f, 0xffff, RPN_ADDR_7F7F},
		{0xffff, 0xffff, RPN_ADDR_FFFF},
		{-1, -1}
	};
	
	if ((TMDY_PLAYMIDI->channel)[ch].nrpn == -1)
		return -1;
	lsb = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
	msb = (TMDY_PLAYMIDI->channel)[ch].lastmrpn;
	if (lsb == 0xff || msb == 0xff)
		return -1;
	addr = (msb << 8 | lsb);
	if ((TMDY_PLAYMIDI->channel)[ch].nrpn)
		addrmap = nrpn_addr_map;
	else
		addrmap = rpn_addr_map;
	for (i = 0; addrmap[i].addr != -1; i++)
		if (addrmap[i].addr == (addr & addrmap[i].mask))
			return addrmap[i].tag;
	return -1;
}

static void update_channel_freq(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	int i, uv = (TMDY_PLAYMIDI->upper_voices);
	for (i = 0; i < uv; i++)
		if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE && (TMDY_PLAYMIDI->voice)[i].channel == ch)
	recompute_freq(tmdy_struct, i);
}

static void update_rpn_map(tmdy_struct_ex_t *tmdy_struct, int ch, int addr, int update_now)
{
	int val, drumflag, i, note;
	
	val = (TMDY_PLAYMIDI->channel)[ch].rpnmap[addr];
	drumflag = 0;
	switch (addr) {
	case NRPN_ADDR_0108:	/* Vibrato Rate */
		if ((TMDY_PLAYMIDI->opt_nrpn_vibrato)) {
			/* from -10.72 Hz to +10.72 Hz. */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Rate (CH:%d VAL:%d)", ch, val - 64);
			(TMDY_PLAYMIDI->channel)[ch].vibrato_ratio = 168 * (val - 64)
					* (VIBRATO_RATE_TUNING * (TMDY_OUTPUT->play_mode)->rate)
					/ (2 * VIBRATO_SAMPLE_INCREMENTS);
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_0109:	/* Vibrato Depth */
		if ((TMDY_PLAYMIDI->opt_nrpn_vibrato)) {
			/* from -10cents to +10cents. */
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Depth (CH:%d VAL:%d)", ch, val - 64);
			(TMDY_PLAYMIDI->channel)[ch].vibrato_depth =
					(double) (val - 64) * 0.15625 * 256 / 400;
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_010A:	/* Vibrato Delay */
		if ((TMDY_PLAYMIDI->opt_nrpn_vibrato)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Vibrato Delay (CH:%d VAL:%d)", ch, val);
			(TMDY_PLAYMIDI->channel)[ch].vibrato_delay =
					(TMDY_OUTPUT->play_mode)->rate * delay_time_center_table[val] * 0.001;
		}
		if (update_now)
			update_channel_freq(tmdy_struct, ch);
		break;
	case NRPN_ADDR_0120:	/* Filter Cutoff Frequency */
		if ((TMDY_PLAYMIDI->opt_lpf_def)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Filter Cutoff (CH:%d VAL:%d)", ch, val - 64);
			(TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq = val - 64;
		}
		break;
	case NRPN_ADDR_0121:	/* Filter Resonance */
		if ((TMDY_PLAYMIDI->opt_lpf_def)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Filter Resonance (CH:%d VAL:%d)", ch, val - 64);
			(TMDY_PLAYMIDI->channel)[ch].param_resonance = val - 64;
		}
		break;
	case NRPN_ADDR_0163:	/* Attack Time */
		if ((TMDY_PLAYMIDI->opt_tva_attack)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"Attack Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 0);
		}
		break;
	case NRPN_ADDR_0164:	/* EG Decay Time */
		if ((TMDY_PLAYMIDI->opt_tva_decay)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"EG Decay Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 2);
		}
		break;
	case NRPN_ADDR_0166:	/* EG Release Time */
		if ((TMDY_PLAYMIDI->opt_tva_release)) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"EG Release Time (CH:%d VAL:%d)", ch, val);
			set_envelope_time(tmdy_struct, ch, val, 3);
		}
		break;
	case NRPN_ADDR_1400:	/* Drum Filter Cutoff (XG) */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Filter Cutoff (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_cutoff_freq = val - 64;
		break;
	case NRPN_ADDR_1500:	/* Drum Filter Resonance (XG) */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Filter Resonance (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_resonance = val - 64;
		break;
	case NRPN_ADDR_1600:	/* Drum EG Attack Time (XG) */
		drumflag = 1;
		if ((TMDY_PLAYMIDI->opt_tva_attack)) {
			val = val & 0x7f;
			note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
			if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
				play_midi_setup_drums(tmdy_struct, ch, note);
			val	-= 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"XG Drum Attack Time (CH:%d NOTE:%d VALUE:%d)",
					ch, note, val);
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_envelope_rate[0] = val;
		}
		break;
	case NRPN_ADDR_1700:	/* Drum EG Decay Time (XG) */
		drumflag = 1;
		if ((TMDY_PLAYMIDI->opt_tva_decay)) {
			val = val & 0x7f;
			note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
			if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
				play_midi_setup_drums(tmdy_struct, ch, note);
			val	-= 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
					"XG Drum Decay Time (CH:%d NOTE:%d VALUE:%d)",
					ch, note, val);
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_envelope_rate[2] = val;
		}
		break;
	case NRPN_ADDR_1800:	/* Coarse Pitch of Drum (GS) */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		if ((TMDY_PLAYMIDI->current_event)->b == 0x01) {
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->play_note = val;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Drum Instrument Play Note (CH:%d NOTE:%d VALUE:%d)",
			ch, note, (TMDY_PLAYMIDI->channel)[ch].drums[note]->play_note);
		} else {
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->coarse = val - 64;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
			"Drum Instrument Pitch Coarse (CH:%d NOTE:%d VALUE:%d)",
			ch, note, (TMDY_PLAYMIDI->channel)[ch].drums[note]->coarse);
		}
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	case NRPN_ADDR_1900:	/* Fine Pitch of Drum (XG) */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->fine = val - 64;
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument Pitch Fine (CH:%d NOTE:%d VALUE:%d)",
				ch, note, (TMDY_PLAYMIDI->channel)[ch].drums[note]->fine);
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	case NRPN_ADDR_1A00:	/* Level of Drum */	 
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Drum Instrument TVA Level (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_level =
				calc_drum_tva_level(tmdy_struct, ch, note, val);
		break;
	case NRPN_ADDR_1C00:	/* Panpot of Drum */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		if(val == 0) {
			val = TMDY_COMMON->int_rand(tmdy_struct, 128);
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->pan_random = 1;
		} else
			(TMDY_PLAYMIDI->channel)[ch].drums[note]->pan_random = 0;
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->drum_panning = val;
		if (update_now && (TMDY_PLAYMIDI->adjust_panning_immediately)
				&& ! (TMDY_PLAYMIDI->channel)[ch].pan_random)
			adjust_drum_panning(tmdy_struct, ch, note);
		break;
	case NRPN_ADDR_1D00:	/* Reverb Send Level of Drum */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Reverb Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->reverb_level = val;
		break;
	case NRPN_ADDR_1E00:	/* Chorus Send Level of Drum */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Chorus Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->chorus_level = val;
		break;
	case NRPN_ADDR_1F00:	/* Variation Send Level of Drum */
		drumflag = 1;
		note = (TMDY_PLAYMIDI->channel)[ch].lastlrpn;
		if ((TMDY_PLAYMIDI->channel)[ch].drums[note] == NULL)
			play_midi_setup_drums(tmdy_struct, ch, note);
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
				"Delay Send Level of Drum (CH:%d NOTE:%d VALUE:%d)",
				ch, note, val);
		(TMDY_PLAYMIDI->channel)[ch].drums[note]->delay_level = val;
		break;
	case RPN_ADDR_0000:		/* Pitch bend sensitivity */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Pitch Bend Sensitivity (CH:%d VALUE:%d)", ch, val);
		/* for mod2mid.c, arpeggio */
		if (! IS_CURRENT_MOD_FILE && (TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0000] > 24)
			(TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0000] = 24;
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0001:		/* Master Fine Tuning */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Master Fine Tuning (CH:%d VALUE:%d)", ch, val);
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0002:		/* Master Coarse Tuning */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Master Coarse Tuning (CH:%d VALUE:%d)", ch, val);
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	case RPN_ADDR_0003:		/* Tuning Program Select */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Tuning Program Select (CH:%d VALUE:%d)", ch, val);
		for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
			if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
				(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
				recompute_freq(tmdy_struct, i);
			}
		break;
	case RPN_ADDR_0004:		/* Tuning Bank Select */
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG,
				"Tuning Bank Select (CH:%d VALUE:%d)", ch, val);
		for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
			if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
				(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
				recompute_freq(tmdy_struct, i);
			}
		break;
	case RPN_ADDR_7F7F:		/* RPN reset */
		(TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag = 1;
		break;
	case RPN_ADDR_FFFF:		/* RPN initialize */
		/* All reset to defaults */
		(TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag = 0;
		memset((TMDY_PLAYMIDI->channel)[ch].rpnmap, 0, sizeof((TMDY_PLAYMIDI->channel)[ch].rpnmap));
		(TMDY_PLAYMIDI->channel)[ch].lastlrpn = (TMDY_PLAYMIDI->channel)[ch].lastmrpn = 0;
		(TMDY_PLAYMIDI->channel)[ch].nrpn = 0;
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0000] = 2;
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0001] = 0x40;
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[RPN_ADDR_0002] = 0x40;
		(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
		break;
	}
	drumflag = 0;
	if (drumflag && midi_drumpart_change(tmdy_struct, ch, 1)) {
		midi_program_change(tmdy_struct, ch, (TMDY_PLAYMIDI->channel)[ch].program);
		if (update_now)
			ctl_prog_event(tmdy_struct, ch);
	}
}

static void seek_forward(tmdy_struct_ex_t *tmdy_struct, int32 until_time)
{
    int32 i;
    int j, k, ch, orig_ch, port_ch, offset, layered;

    (TMDY_PLAYMIDI->playmidi_seek_flag) = 1;
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_START_SKIP, WRD_NOARG);
	while (MIDI_EVENT_TIME((TMDY_PLAYMIDI->current_event)) < until_time) {
#ifndef SUPPRESS_CHANNEL_LAYER
		orig_ch = (TMDY_PLAYMIDI->current_event)->channel;
		layered = ! IS_SYSEX_EVENT_TYPE((TMDY_PLAYMIDI->current_event)->type);
		for (j = 0; j < MAX_CHANNELS; j += 16) {
			port_ch = (orig_ch + j) % MAX_CHANNELS;
			offset = (port_ch < 16) ? 0 : 16;
			for (k = offset; k < offset + 16; k++) {
				if (! layered && (j || k != offset))
					continue;
				if (layered) {
					if (! IS_SET_CHANNELMASK(
							(TMDY_PLAYMIDI->channel)[k].channel_layer, port_ch)
							|| (TMDY_PLAYMIDI->channel)[k].port_select ^ (orig_ch >=16))
						continue;
					(TMDY_PLAYMIDI->current_event)->channel = k;
				}
#endif
	ch = (TMDY_PLAYMIDI->current_event)->channel;
	
	switch((TMDY_PLAYMIDI->current_event)->type)
	{
	  case ME_PITCHWHEEL:
	    (TMDY_PLAYMIDI->channel)[ch].pitchbend = (TMDY_PLAYMIDI->current_event)->a + (TMDY_PLAYMIDI->current_event)->b * 128;
	    (TMDY_PLAYMIDI->channel)[ch].pitchfactor=0;
	    break;

	  case ME_MAINVOLUME:
	    (TMDY_PLAYMIDI->channel)[ch].volume = (TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_MASTER_VOLUME:
	    (TMDY_PLAYMIDI->master_volume_ratio) =
		(int32)(TMDY_PLAYMIDI->current_event)->a + 256 * (int32)(TMDY_PLAYMIDI->current_event)->b;
	    break;

	  case ME_PAN:
	    (TMDY_PLAYMIDI->channel)[ch].panning = (TMDY_PLAYMIDI->current_event)->a;
	    (TMDY_PLAYMIDI->channel)[ch].pan_random = 0;
	    break;

	  case ME_EXPRESSION:
	    (TMDY_PLAYMIDI->channel)[ch].expression=(TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_PROGRAM:
	    midi_program_change(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a);
	    break;

	  case ME_SUSTAIN:
	    (TMDY_PLAYMIDI->channel)[ch].sustain = ((TMDY_PLAYMIDI->current_event)->a >= 64);
	    break;

	  case ME_SOSTENUTO:
	    break;

	  case ME_LEGATO_FOOTSWITCH:
        (TMDY_PLAYMIDI->channel)[ch].legato = ((TMDY_PLAYMIDI->current_event)->a >= 64);
	    break;

      case ME_HOLD2:
        break;

	  case ME_FOOT:
	    break;

	  case ME_BREATH:
	    break;

	  case ME_BALANCE:
	    break;

	  case ME_RESET_CONTROLLERS:
	    reset_controllers(tmdy_struct, ch);
	    break;

	  case ME_TONE_BANK_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].bank_msb = (TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_TONE_BANK_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].bank_lsb = (TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_MODULATION_WHEEL:
	    (TMDY_PLAYMIDI->channel)[ch].modulation_wheel =
		midi_cnv_vib_depth(tmdy_struct, (TMDY_PLAYMIDI->current_event)->a);
	    break;

	  case ME_PORTAMENTO_TIME_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].portamento_time_msb = (TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_PORTAMENTO_TIME_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb = (TMDY_PLAYMIDI->current_event)->a;
	    break;

	  case ME_PORTAMENTO:
	    (TMDY_PLAYMIDI->channel)[ch].portamento = ((TMDY_PLAYMIDI->current_event)->a >= 64);
	    break;

	  case ME_MONO:
	    (TMDY_PLAYMIDI->channel)[ch].mono = 1;
	    break;

	  case ME_POLY:
	    (TMDY_PLAYMIDI->channel)[ch].mono = 0;
	    break;

	  case ME_SOFT_PEDAL:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].soft_pedal = (TMDY_PLAYMIDI->current_event)->a;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Soft Pedal (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].soft_pedal);
		  }
		  break;

	  case ME_HARMONIC_CONTENT:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].param_resonance = (TMDY_PLAYMIDI->current_event)->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Harmonic Content (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].param_resonance);
		  }
		  break;

	  case ME_BRIGHTNESS:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq = (TMDY_PLAYMIDI->current_event)->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Brightness (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq);
		  }
		  break;

	    /* RPNs */
	  case ME_NRPN_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].lastlrpn = (TMDY_PLAYMIDI->current_event)->a;
	    (TMDY_PLAYMIDI->channel)[ch].nrpn = 1;
	    break;
	  case ME_NRPN_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].lastmrpn = (TMDY_PLAYMIDI->current_event)->a;
	    (TMDY_PLAYMIDI->channel)[ch].nrpn = 1;
	    break;
	  case ME_RPN_LSB:
	    (TMDY_PLAYMIDI->channel)[ch].lastlrpn = (TMDY_PLAYMIDI->current_event)->a;
	    (TMDY_PLAYMIDI->channel)[ch].nrpn = 0;
	    break;
	  case ME_RPN_MSB:
	    (TMDY_PLAYMIDI->channel)[ch].lastmrpn = (TMDY_PLAYMIDI->current_event)->a;
	    (TMDY_PLAYMIDI->channel)[ch].nrpn = 0;
	    break;
	  case ME_RPN_INC:
	    if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		if((TMDY_PLAYMIDI->channel)[ch].rpnmap[i] < 127)
		    (TMDY_PLAYMIDI->channel)[ch].rpnmap[i]++;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	case ME_RPN_DEC:
	    if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		if((TMDY_PLAYMIDI->channel)[ch].rpnmap[i] > 0)
		    (TMDY_PLAYMIDI->channel)[ch].rpnmap[i]--;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	  case ME_DATA_ENTRY_MSB:
	    if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	    {
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[i] = (TMDY_PLAYMIDI->current_event)->a;
		update_rpn_map(tmdy_struct, ch, i, 0);
	    }
	    break;
	  case ME_DATA_ENTRY_LSB:
	    if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
		break;
	    /* Ignore */
	    (TMDY_PLAYMIDI->channel)[ch].nrpn = -1;
	    break;

	  case ME_REVERB_EFFECT:
	    set_reverb_level(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a);
	    break;

	  case ME_CHORUS_EFFECT:
	    if((TMDY_PLAYMIDI->opt_chorus_control) == 1)
		(TMDY_PLAYMIDI->channel)[ch].chorus_level = (TMDY_PLAYMIDI->current_event)->a;
	    else
		(TMDY_PLAYMIDI->channel)[ch].chorus_level = -(TMDY_PLAYMIDI->opt_chorus_control);

		if((TMDY_PLAYMIDI->current_event)->a) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send (CH:%d LEVEL:%d)",ch,(TMDY_PLAYMIDI->current_event)->a);
		}
		break;

	  case ME_TREMOLO_EFFECT:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tremolo Send (CH:%d LEVEL:%d)",ch,(TMDY_PLAYMIDI->current_event)->a);
		break;

	  case ME_CELESTE_EFFECT:
		if((TMDY_PLAYMIDI->opt_delay_control)) {
			(TMDY_PLAYMIDI->channel)[ch].delay_level = (TMDY_PLAYMIDI->current_event)->a;
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send (CH:%d LEVEL:%d)",ch,(TMDY_PLAYMIDI->current_event)->a);
		}
	    break;

	  case ME_ATTACK_TIME:
	  	if(!(TMDY_PLAYMIDI->opt_tva_attack)) { break; }
		set_envelope_time(tmdy_struct, ch,(TMDY_PLAYMIDI->current_event)->a,0);
		break;

	  case ME_RELEASE_TIME:
	  	if(!(TMDY_PLAYMIDI->opt_tva_release)) { break; }
		set_envelope_time(tmdy_struct, ch,(TMDY_PLAYMIDI->current_event)->a,3);
		break;

	  case ME_PHASER_EFFECT:
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Phaser Send (CH:%d LEVEL:%d)",ch,(TMDY_PLAYMIDI->current_event)->a);
		break;

	  case ME_RANDOM_PAN:
	    (TMDY_PLAYMIDI->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	    (TMDY_PLAYMIDI->channel)[ch].pan_random = 1;
	    break;

	  case ME_SET_PATCH:
	    i = (TMDY_PLAYMIDI->channel)[ch].special_sample = (TMDY_PLAYMIDI->current_event)->a;
	    if(TMDY_INSTRUM->special_patch[i] != NULL)
		TMDY_INSTRUM->special_patch[i]->sample_offset = 0;
	    break;

	  case ME_TEMPO:
	    (TMDY_PLAYMIDI->current_play_tempo) = ch +
		(TMDY_PLAYMIDI->current_event)->b * 256 + (TMDY_PLAYMIDI->current_event)->a * 65536;
	    break;

	  case ME_RESET:
	    TMDY_READMIDI->change_system_mode(tmdy_struct, (TMDY_PLAYMIDI->current_event)->a);
	    reset_midi(tmdy_struct, 0);
	    break;

	  case ME_PATCH_OFFS:
	    i = (TMDY_PLAYMIDI->channel)[ch].special_sample;
	    if(TMDY_INSTRUM->special_patch[i] != NULL)
		TMDY_INSTRUM->special_patch[i]->sample_offset =
		    ((TMDY_PLAYMIDI->current_event)->a | 256 * (TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_WRD:
	    TMDY_WRD->wrd_midi_event(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a | 256 * (TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SHERRY:
	    TMDY_WRD->wrd_sherry_event(tmdy_struct, ch |
			     ((TMDY_PLAYMIDI->current_event)->a<<8) |
			     ((TMDY_PLAYMIDI->current_event)->b<<16));
	    break;

	  case ME_DRUMPART:
	    if(midi_drumpart_change(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a))
		midi_program_change(tmdy_struct, ch, (TMDY_PLAYMIDI->channel)[ch].program);
	    break;

	  case ME_KEYSHIFT:
	    (TMDY_PLAYMIDI->channel)[ch].key_shift = (int)(TMDY_PLAYMIDI->current_event)->a - 0x40;
	    break;

	case ME_KEYSIG:
		(TMDY_PLAYMIDI->current_keysig) = (TMDY_PLAYMIDI->current_event)->a + (TMDY_PLAYMIDI->current_event)->b * 16;
		(TMDY_PLAYMIDI->current_temper_keysig) = (TMDY_PLAYMIDI->current_keysig);
		break;

	case ME_SCALE_TUNING:
		(TMDY_PLAYMIDI->channel)[ch].scale_tuning[(TMDY_PLAYMIDI->current_event)->a] = (TMDY_PLAYMIDI->current_event)->b;
		break;

	case ME_BULK_TUNING_DUMP:
		set_single_note_tuning(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a, (TMDY_PLAYMIDI->current_event)->b, 0);
		break;

	case ME_SINGLE_NOTE_TUNING:
		set_single_note_tuning(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a, (TMDY_PLAYMIDI->current_event)->b, 0);
		break;

	case ME_TEMPER_KEYSIG:
		(TMDY_PLAYMIDI->current_temper_keysig) = (TMDY_PLAYMIDI->current_event)->a;
		break;

	case ME_TEMPER_TYPE:
		(TMDY_PLAYMIDI->channel)[ch].temper_type = (TMDY_PLAYMIDI->current_event)->a;
		break;

	case ME_MASTER_TEMPER_TYPE:
		for (i = 0; i < MAX_CHANNELS; i++)
			(TMDY_PLAYMIDI->channel)[i].temper_type = (TMDY_PLAYMIDI->current_event)->a;
		break;

	  case ME_SYSEX_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SYSEX_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SYSEX_GS_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_GS_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SYSEX_GS_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_GS_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SYSEX_XG_LSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_XG_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_SYSEX_XG_MSB:
	    process_sysex_event(tmdy_struct, ME_SYSEX_XG_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	  case ME_EOT:
	    (TMDY_PLAYMIDI->current_sample) = (TMDY_PLAYMIDI->current_event)->time;
	    (TMDY_PLAYMIDI->playmidi_seek_flag) = 0;
	    return;
	}
#ifndef SUPPRESS_CHANNEL_LAYER
			}
		}
		(TMDY_PLAYMIDI->current_event)->channel = orig_ch;
#endif
	(TMDY_PLAYMIDI->current_event)++;
    }
    TMDY_WRD->wrd_midi_event(tmdy_struct, WRD_END_SKIP, WRD_NOARG);

    (TMDY_PLAYMIDI->playmidi_seek_flag) = 0;
    if((TMDY_PLAYMIDI->current_event) != (TMDY_PLAYMIDI->event_list))
	(TMDY_PLAYMIDI->current_event)--;
    (TMDY_PLAYMIDI->current_sample) = until_time;
}

static void skip_to(tmdy_struct_ex_t *tmdy_struct, int32 until_time)
{
  int ch;

  TMDY_MIDITRACE->trace_flush(tmdy_struct);
  (TMDY_PLAYMIDI->current_event) = NULL;

  if ((TMDY_PLAYMIDI->current_sample) > until_time)
    (TMDY_PLAYMIDI->current_sample)=0;

  TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
  reset_midi(tmdy_struct, 0);

  (TMDY_PLAYMIDI->buffered_count)=0;
  (TMDY_PLAYMIDI->buffer_pointer)=(TMDY_PLAYMIDI->common_buffer);
  (TMDY_PLAYMIDI->current_event)=(TMDY_PLAYMIDI->event_list);
  (TMDY_PLAYMIDI->current_play_tempo) = 500000; /* 120 BPM */

  if (until_time)
    seek_forward(tmdy_struct, until_time);
  for(ch = 0; ch < MAX_CHANNELS; ch++)
      (TMDY_PLAYMIDI->channel)[ch].lasttime = (TMDY_PLAYMIDI->current_sample);

  ctl_mode_event(tmdy_struct, CTLE_RESET, 0, 0, 0);
  TMDY_MIDITRACE->trace_offset(tmdy_struct, until_time);

#ifdef SUPPORT_SOUNDSPEC
  soundspec_update_wave(NULL, 0);
#endif /* SUPPORT_SOUNDSPEC */
}

static int32 sync_restart(tmdy_struct_ex_t *tmdy_struct, int only_trace_ok)
{
    int32 cur;

    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
    if(cur == -1)
    {
	if(only_trace_ok)
	    return -1;
	cur = (TMDY_PLAYMIDI->current_sample);
    }
    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
    skip_to(tmdy_struct, cur);
    return cur;
}

static int playmidi_change_rate(tmdy_struct_ex_t *tmdy_struct, int32 rate, int restart)
{
    int arg;

    if(rate == (TMDY_OUTPUT->play_mode)->rate)
	return 1; /* Not need to change */

    if(rate < MIN_OUTPUT_RATE || rate > MAX_OUTPUT_RATE)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Out of sample rate: %d", rate);
	return -1;
    }

    if(restart)
    {
	if(((TMDY_PLAYMIDI->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct)) == -1)
	    (TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->current_sample);
    }
    else
	(TMDY_PLAYMIDI->midi_restart_time) = 0;

    arg = (int)rate;
    if((TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_RATE, &arg) == -1)
    {
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Can't change sample rate to %d", rate);
	return -1;
    }

    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
    (TMDY_AQ->aq_setup)(tmdy_struct);
    (TMDY_AQ->aq_set_soft_queue)(tmdy_struct, -1.0, -1.0);
    TMDY_INSTRUM->free_instruments(tmdy_struct,1);
#ifdef SUPPORT_SOUNDSPEC
    soundspec_reinit();
#endif /* SUPPORT_SOUNDSPEC */
    return 0;
}

void playmidi_output_changed(tmdy_struct_ex_t *tmdy_struct, int play_state)
{
    if((TMDY_OUTPUT->target_play_mode) == NULL)
	return;
    (TMDY_OUTPUT->play_mode) = (TMDY_OUTPUT->target_play_mode);

    if(play_state == 0)
    {
	/* Playing */
	if(((TMDY_PLAYMIDI->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct)) == -1)
	    (TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->current_sample);
    }
    else /* Not playing */
	(TMDY_PLAYMIDI->midi_restart_time) = 0;

    if(play_state != 2)
    {
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);
	(TMDY_AQ->aq_setup)(tmdy_struct);
	(TMDY_AQ->aq_set_soft_queue)(tmdy_struct, -1.0, -1.0);
	TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);
    }
    TMDY_INSTRUM->free_instruments(tmdy_struct,1);
#ifdef SUPPORT_SOUNDSPEC
    soundspec_reinit();
#endif /* SUPPORT_SOUNDSPEC */
    (TMDY_OUTPUT->target_play_mode) = NULL;
}

int check_apply_control(tmdy_struct_ex_t *tmdy_struct)
{
    int rc;
    int32 val;

    if((TMDY_PLAYMIDI->file_from_stdin))
	return RC_NONE;
    rc = (TMDY_CONTROLS->ctl)->read(tmdy_struct, &val);
    switch(rc)
    {
      case RC_CHANGE_VOLUME:
	if (val>0 || (TMDY_PLAYMIDI->amplification) > -val)
	    (TMDY_PLAYMIDI->amplification) += val;
	else
	    (TMDY_PLAYMIDI->amplification)=0;
	if ((TMDY_PLAYMIDI->amplification) > MAX_AMPLIFICATION)
	    (TMDY_PLAYMIDI->amplification)=MAX_AMPLIFICATION;
	adjust_amplification(tmdy_struct);
	ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (TMDY_PLAYMIDI->amplification), 0);
	break;
      case RC_SYNC_RESTART:
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);
	break;
      case RC_TOGGLE_PAUSE:
	(TMDY_PLAYMIDI->play_pause_flag) = !(TMDY_PLAYMIDI->play_pause_flag);
	ctl_pause_event(tmdy_struct, (TMDY_PLAYMIDI->play_pause_flag), 0);
	return RC_NONE;
      case RC_TOGGLE_SNDSPEC:
#ifdef SUPPORT_SOUNDSPEC
	if(view_soundspec_flag)
	    close_soundspec();
	else
	    open_soundspec();
	if(view_soundspec_flag || ctl_speana_flag)
	    soundspec_update_wave(NULL, -1);
	return RC_NONE;
      case RC_TOGGLE_CTL_SPEANA:
	ctl_speana_flag = !ctl_speana_flag;
	if(view_soundspec_flag || ctl_speana_flag)
	    soundspec_update_wave(NULL, -1);
	return RC_NONE;
#endif /* SUPPORT_SOUNDSPEC */
      case RC_CHANGE_RATE:
	if(playmidi_change_rate(tmdy_struct, val, 0))
	    return RC_NONE;
	return RC_RELOAD;
      case RC_OUTPUT_CHANGED:
	playmidi_output_changed(tmdy_struct, 1);
	return RC_RELOAD;
    }
    return rc;
}

static void voice_increment(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i;
    for(i = 0; i < n; i++)
    {
	if((TMDY_PLAYMIDI->voices) == MAX_VOICES)
	    break;
	(TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)].status = VOICE_FREE;
	(TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)].temper_instant = 0;
	(TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)].chorus_link = (TMDY_PLAYMIDI->voices);
	(TMDY_PLAYMIDI->voices)++;
    }
    if(n > 0)
	ctl_mode_event(tmdy_struct, CTLE_MAXVOICES, 1, (TMDY_PLAYMIDI->voices), 0);
}

static void voice_decrement(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i, j, lowest;
    int32 lv, v;

    /* decrease voice */
    for(i = 0; i < n && (TMDY_PLAYMIDI->voices) > 0; i++)
    {
	(TMDY_PLAYMIDI->voices)--;
	if((TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)].status == VOICE_FREE)
	    continue;	/* found */

	for(j = 0; j < (TMDY_PLAYMIDI->voices); j++)
	    if((TMDY_PLAYMIDI->voice)[j].status == VOICE_FREE)
		break;
	if(j != (TMDY_PLAYMIDI->voices))
	{
	    (TMDY_PLAYMIDI->voice)[j] = (TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)];
	    continue;	/* found */
	}

	/* Look for the decaying note with the lowest volume */
	lv = 0x7FFFFFFF;
	lowest = -1;
	for(j = 0; j <= (TMDY_PLAYMIDI->voices); j++)
	{
	    if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_DIE))
	    {
		v = (TMDY_PLAYMIDI->voice)[j].left_mix;
		if(((TMDY_PLAYMIDI->voice)[j].panned==PANNED_MYSTERY) &&
		   ((TMDY_PLAYMIDI->voice)[j].right_mix > v))
		    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
		if(v < lv)
		{
		    lv = v;
		    lowest = j;
		}
	    }
	}

	if(lowest != -1)
	{
	    (TMDY_PLAYMIDI->cut_notes)++;
	    free_voice(tmdy_struct, lowest);
	    ctl_note_event(tmdy_struct, lowest);
	    (TMDY_PLAYMIDI->voice)[lowest] = (TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)];
	}
	else
	    (TMDY_PLAYMIDI->lost_notes)++;
    }
    if((TMDY_PLAYMIDI->upper_voices) > (TMDY_PLAYMIDI->voices))
	(TMDY_PLAYMIDI->upper_voices) = (TMDY_PLAYMIDI->voices);
    if(n > 0)
	ctl_mode_event(tmdy_struct, CTLE_MAXVOICES, 1, (TMDY_PLAYMIDI->voices), 0);
}

/* EAW -- do not throw away good notes, stop decrementing */
static void voice_decrement_conservative(tmdy_struct_ex_t *tmdy_struct, int n)
{
    int i, j, lowest, finalnv;
    int32 lv, v;

    /* decrease voice */
    finalnv = (TMDY_PLAYMIDI->voices) - n;
    for(i = 1; i <= n && (TMDY_PLAYMIDI->voices) > 0; i++)
    {
	if((TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)-1].status == VOICE_FREE) {
	    (TMDY_PLAYMIDI->voices)--;
	    continue;	/* found */
	}

	for(j = 0; j < finalnv; j++)
	    if((TMDY_PLAYMIDI->voice)[j].status == VOICE_FREE)
		break;
	if(j != finalnv)
	{
	    (TMDY_PLAYMIDI->voice)[j] = (TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)-1];
	    (TMDY_PLAYMIDI->voices)--;
	    continue;	/* found */
	}

	/* Look for the decaying note with the lowest volume */
	lv = 0x7FFFFFFF;
	lowest = -1;
	for(j = 0; j < (TMDY_PLAYMIDI->voices); j++)
	{
	    if((TMDY_PLAYMIDI->voice)[j].status & ~(VOICE_ON | VOICE_DIE) &&
	       !((TMDY_PLAYMIDI->voice)[j].sample->note_to_use &&
	         ISDRUMCHANNEL((TMDY_PLAYMIDI->voice)[j].channel)))
	    {
		v = (TMDY_PLAYMIDI->voice)[j].left_mix;
		if(((TMDY_PLAYMIDI->voice)[j].panned==PANNED_MYSTERY) &&
		   ((TMDY_PLAYMIDI->voice)[j].right_mix > v))
		    v = (TMDY_PLAYMIDI->voice)[j].right_mix;
		if(v < lv)
		{
		    lv = v;
		    lowest = j;
		}
	    }
	}

	if(lowest != -1)
	{
	    (TMDY_PLAYMIDI->voices)--;
	    (TMDY_PLAYMIDI->cut_notes)++;
	    free_voice(tmdy_struct, lowest);
	    ctl_note_event(tmdy_struct, lowest);
	    (TMDY_PLAYMIDI->voice)[lowest] = (TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->voices)];
	}
	else break;
    }
    if((TMDY_PLAYMIDI->upper_voices) > (TMDY_PLAYMIDI->voices))
	(TMDY_PLAYMIDI->upper_voices) = (TMDY_PLAYMIDI->voices);
}

void restore_voices(tmdy_struct_ex_t *tmdy_struct, int save_voices)
{
#ifdef REDUCE_VOICE_TIME_TUNING
//    static int old_voices = -1;
    if(TMDY_PLAYMIDI->restore_voices_old_voices == -1 || save_voices)
	TMDY_PLAYMIDI->restore_voices_old_voices = (TMDY_PLAYMIDI->voices);
    else if ((TMDY_PLAYMIDI->voices) < TMDY_PLAYMIDI->restore_voices_old_voices)
	voice_increment(tmdy_struct, TMDY_PLAYMIDI->restore_voices_old_voices - (TMDY_PLAYMIDI->voices));
    else
	voice_decrement(tmdy_struct, (TMDY_PLAYMIDI->voices) - TMDY_PLAYMIDI->restore_voices_old_voices);
#endif /* REDUCE_VOICE_TIME_TUNING */
}
	

static int apply_controls(tmdy_struct_ex_t *tmdy_struct)
{
    int rc, i, jump_flag = 0;
    int32 val, cur;
    FLOAT_T r;
    ChannelBitMask tmp_chbitmask;

    /* ASCII renditions of CD player pictograms indicate approximate effect */
    do
    {
	switch(rc=(TMDY_CONTROLS->ctl)->read(tmdy_struct, &val))
	{
	  case RC_STOP:
	  case RC_QUIT:		/* [] */
	  case RC_LOAD_FILE:
	  case RC_NEXT:		/* >>| */
	  case RC_REALLY_PREVIOUS: /* |<< */
	  case RC_TUNE_END:	/* skip */
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    return rc;

	  case RC_CHANGE_VOLUME:
	    if (val>0 || (TMDY_PLAYMIDI->amplification) > -val)
		(TMDY_PLAYMIDI->amplification) += val;
	    else
		(TMDY_PLAYMIDI->amplification)=0;
	    if ((TMDY_PLAYMIDI->amplification) > MAX_AMPLIFICATION)
		(TMDY_PLAYMIDI->amplification)=MAX_AMPLIFICATION;
	    adjust_amplification(tmdy_struct);
	    for (i=0; i<(TMDY_PLAYMIDI->upper_voices); i++)
		if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE)
		{
		    recompute_amp(tmdy_struct, i);
		    TMDY_MIX->apply_envelope_to_amp(tmdy_struct, i);
		}
	    ctl_mode_event(tmdy_struct, CTLE_MASTER_VOLUME, 0, (TMDY_PLAYMIDI->amplification), 0);
	    continue;

	  case RC_CHANGE_REV_EFFB:
	  case RC_CHANGE_REV_TIME:
	    TMDY_REVERB->reverb_rc_event(tmdy_struct, rc, val);
	    sync_restart(tmdy_struct, 0);
	    continue;

	  case RC_PREVIOUS:	/* |<< */
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if ((TMDY_PLAYMIDI->current_sample) < 2*(TMDY_OUTPUT->play_mode->rate))
		return RC_REALLY_PREVIOUS;
	    return RC_RESTART;

	  case RC_RESTART:	/* |<< */
	    if((TMDY_PLAYMIDI->play_pause_flag))
	    {
		(TMDY_PLAYMIDI->midi_restart_time) = 0;
		ctl_pause_event(tmdy_struct, 1, 0);
		continue;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    skip_to(tmdy_struct, 0);
	    ctl_updatetime(tmdy_struct, 0);
	    jump_flag = 1;
		(TMDY_PLAYMIDI->midi_restart_time) = 0;
	    continue;

	  case RC_JUMP:
	    if((TMDY_PLAYMIDI->play_pause_flag))
	    {
		(TMDY_PLAYMIDI->midi_restart_time) = val;
		ctl_pause_event(tmdy_struct, 1, val);
		continue;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if (val >= (TMDY_PLAYMIDI->sample_count))
		return RC_TUNE_END;
	    skip_to(tmdy_struct, val);
	    ctl_updatetime(tmdy_struct, val);
	    return rc;

	  case RC_FORWARD:	/* >> */
	    if((TMDY_PLAYMIDI->play_pause_flag))
	    {
		(TMDY_PLAYMIDI->midi_restart_time) += val;
		if((TMDY_PLAYMIDI->midi_restart_time) > (TMDY_PLAYMIDI->sample_count))
		    (TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->sample_count);
		ctl_pause_event(tmdy_struct, 1, (TMDY_PLAYMIDI->midi_restart_time));
		continue;
	    }
	    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if(cur == -1)
		cur = (TMDY_PLAYMIDI->current_sample);
	    if(val + cur >= (TMDY_PLAYMIDI->sample_count))
		return RC_TUNE_END;
	    skip_to(tmdy_struct, val + cur);
	    ctl_updatetime(tmdy_struct, val + cur);
	    return RC_JUMP;

	  case RC_BACK:		/* << */
	    if((TMDY_PLAYMIDI->play_pause_flag))
	    {
		(TMDY_PLAYMIDI->midi_restart_time) -= val;
		if((TMDY_PLAYMIDI->midi_restart_time) < 0)
		    (TMDY_PLAYMIDI->midi_restart_time) = 0;
		ctl_pause_event(tmdy_struct, 1, (TMDY_PLAYMIDI->midi_restart_time));
		continue;
	    }
	    cur = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    if(cur == -1)
		cur = (TMDY_PLAYMIDI->current_sample);
	    if(cur > val)
	    {
		skip_to(tmdy_struct, cur - val);
		ctl_updatetime(tmdy_struct, cur - val);
	    }
	    else
	    {
		skip_to(tmdy_struct, 0);
		ctl_updatetime(tmdy_struct, 0);
		(TMDY_PLAYMIDI->midi_restart_time) = 0;
	    }
	    return RC_JUMP;

	  case RC_TOGGLE_PAUSE:
	    if((TMDY_PLAYMIDI->play_pause_flag))
	    {
		(TMDY_PLAYMIDI->play_pause_flag) = 0;
		skip_to(tmdy_struct, (TMDY_PLAYMIDI->midi_restart_time));
	    }
	    else
	    {
		(TMDY_PLAYMIDI->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
		if((TMDY_PLAYMIDI->midi_restart_time) == -1)
		    (TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->current_sample);
		(TMDY_AQ->aq_flush)(tmdy_struct, 1);
		(TMDY_PLAYMIDI->play_pause_flag) = 1;
	    }
	    ctl_pause_event(tmdy_struct, (TMDY_PLAYMIDI->play_pause_flag), (TMDY_PLAYMIDI->midi_restart_time));
	    jump_flag = 1;
	    continue;

	  case RC_KEYUP:
	  case RC_KEYDOWN:
	    (TMDY_PLAYMIDI->note_key_offset) += val;
	    (TMDY_PLAYMIDI->current_freq_table) += val;
	    (TMDY_PLAYMIDI->current_freq_table) -= floor((TMDY_PLAYMIDI->current_freq_table) / 12.0) * 12;
	    if(sync_restart(tmdy_struct, 1) != -1)
		jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (TMDY_PLAYMIDI->note_key_offset), 0);
	    continue;

	  case RC_SPEEDUP:
	    r = 1.0;
	    for(i = 0; i < val; i++)
		r *= SPEED_CHANGE_RATE;
	    sync_restart(tmdy_struct, 0);
	    (TMDY_PLAYMIDI->midi_time_ratio) /= r;
	    (TMDY_PLAYMIDI->current_sample) = (int32)((TMDY_PLAYMIDI->current_sample) / r + 0.5);
	    TMDY_MIDITRACE->trace_offset(tmdy_struct, (TMDY_PLAYMIDI->current_sample));
	    jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (TMDY_PLAYMIDI->midi_time_ratio) + 0.5, 0);
	    continue;

	  case RC_SPEEDDOWN:
	    r = 1.0;
	    for(i = 0; i < val; i++)
		r *= SPEED_CHANGE_RATE;
	    sync_restart(tmdy_struct, 0);
	    (TMDY_PLAYMIDI->midi_time_ratio) *= r;
	    (TMDY_PLAYMIDI->current_sample) = (int32)((TMDY_PLAYMIDI->current_sample) * r + 0.5);
	    TMDY_MIDITRACE->trace_offset(tmdy_struct, (TMDY_PLAYMIDI->current_sample));
	    jump_flag = 1;
	    ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (TMDY_PLAYMIDI->midi_time_ratio) + 0.5, 0);
	    continue;

	  case RC_VOICEINCR:
	    restore_voices(tmdy_struct, 0);
	    voice_increment(tmdy_struct, val);
	    if(sync_restart(tmdy_struct, 1) != -1)
		jump_flag = 1;
	    restore_voices(tmdy_struct, 1);
	    continue;

	  case RC_VOICEDECR:
	    restore_voices(tmdy_struct, 0);
	    if(sync_restart(tmdy_struct, 1) != -1)
	    {
		(TMDY_PLAYMIDI->voices) -= val;
		if((TMDY_PLAYMIDI->voices) < 0)
		    (TMDY_PLAYMIDI->voices) = 0;
		jump_flag = 1;
	    }
	    else
		voice_decrement(tmdy_struct, val);
	    restore_voices(tmdy_struct, 1);
	    continue;

	  case RC_TOGGLE_DRUMCHAN:
	    (TMDY_PLAYMIDI->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    if((TMDY_PLAYMIDI->midi_restart_time) == -1)
		(TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->current_sample);
	    SET_CHANNELMASK((TMDY_PLAYMIDI->drumchannel_mask), val);
	    SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannel_mask, val);
	    if(IS_SET_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), val))
	    {
		UNSET_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), val);
		UNSET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, val);
	    }
	    else
	    {
		SET_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), val);
		SET_CHANNELMASK(TMDY_READMIDI->current_file_info->drumchannels, val);
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    return RC_RELOAD;

	  case RC_TOGGLE_SNDSPEC:
#ifdef SUPPORT_SOUNDSPEC
	    if(view_soundspec_flag)
		close_soundspec();
	    else
		open_soundspec();
	    if(view_soundspec_flag || ctl_speana_flag)
	    {
		sync_restart(tmdy_struct, 0);
		soundspec_update_wave(NULL, -1);
	    }
#endif /* SUPPORT_SOUNDSPEC */
	    continue;

	  case RC_TOGGLE_CTL_SPEANA:
#ifdef SUPPORT_SOUNDSPEC
	    ctl_speana_flag = !ctl_speana_flag;
	    if(view_soundspec_flag || ctl_speana_flag)
	    {
		sync_restart(tmdy_struct, 0);
		soundspec_update_wave(NULL, -1);
	    }
#endif /* SUPPORT_SOUNDSPEC */
	    continue;

	  case RC_SYNC_RESTART:
	    sync_restart(tmdy_struct, val);
	    jump_flag = 1;
	    continue;

	  case RC_RELOAD:
	    (TMDY_PLAYMIDI->midi_restart_time) = TMDY_MIDITRACE->current_trace_samples(tmdy_struct);
	    if((TMDY_PLAYMIDI->midi_restart_time) == -1)
		(TMDY_PLAYMIDI->midi_restart_time) = (TMDY_PLAYMIDI->current_sample);
	    (TMDY_AQ->aq_flush)(tmdy_struct, 1);
	    return RC_RELOAD;

	  case RC_CHANGE_RATE:
	    if(playmidi_change_rate(tmdy_struct, val, 1))
		return RC_NONE;
	    return RC_RELOAD;

	  case RC_OUTPUT_CHANGED:
	    playmidi_output_changed(tmdy_struct, 0);
	    return RC_RELOAD;

	case RC_TOGGLE_MUTE:
		TOGGLE_CHANNELMASK(TMDY_PLAYMIDI->channel_mute, val);
		sync_restart(tmdy_struct, 0);
		jump_flag = 1;
		ctl_mode_event(tmdy_struct, CTLE_MUTE, 0,
				val, (IS_SET_CHANNELMASK(TMDY_PLAYMIDI->channel_mute, val)) ? 1 : 0);
		continue;

	case RC_SOLO_PLAY:
		COPY_CHANNELMASK(tmp_chbitmask, TMDY_PLAYMIDI->channel_mute);
		FILL_CHANNELMASK(TMDY_PLAYMIDI->channel_mute);
		UNSET_CHANNELMASK(TMDY_PLAYMIDI->channel_mute, val);
		if (! COMPARE_CHANNELMASK(tmp_chbitmask, TMDY_PLAYMIDI->channel_mute)) {
			sync_restart(tmdy_struct, 0);
			jump_flag = 1;
			for (i = 0; i < MAX_CHANNELS; i++)
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, 1);
			ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, val, 0);
		}
		continue;

	case RC_MUTE_CLEAR:
		COPY_CHANNELMASK(tmp_chbitmask, TMDY_PLAYMIDI->channel_mute);
		CLEAR_CHANNELMASK(TMDY_PLAYMIDI->channel_mute);
		if (! COMPARE_CHANNELMASK(tmp_chbitmask, TMDY_PLAYMIDI->channel_mute)) {
			sync_restart(tmdy_struct, 0);
			jump_flag = 1;
			for (i = 0; i < MAX_CHANNELS; i++)
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, 0);
		}
		continue;
	}
	if(TMDY_MAIN->intr)
	    return RC_QUIT;
	if((TMDY_PLAYMIDI->play_pause_flag))
	    usleep(300000);
    } while (rc != RC_NONE || (TMDY_PLAYMIDI->play_pause_flag));
    return jump_flag ? RC_JUMP : RC_NONE;
}

#ifdef USE_DSP_EFFECT
/* do_compute_data_midi() for new chorus */
static void do_compute_data_midi(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
	int i, j, uv, stereo, n, ch, note;
	int32 *vpblist[MAX_CHANNELS];
	int vc[MAX_CHANNELS];
	int channel_effect,channel_reverb,channel_chorus,channel_delay,channel_eq;
	int32 cnt = count * 2;
	
	stereo = ! ((TMDY_OUTPUT->play_mode->encoding) & PE_MONO);
	n = count * ((stereo) ? 8 : 4); /* in bytes */

	memset((TMDY_PLAYMIDI->buffer_pointer), 0, n);  
	memset((TMDY_PLAYMIDI->insertion_effect_buffer), 0, n);

	/* are effects valid? / don't supported in mono */
	channel_reverb = (((TMDY_PLAYMIDI->opt_reverb_control) == 1 || (TMDY_PLAYMIDI->opt_reverb_control) == 3)
			&& stereo);
	channel_chorus = ((TMDY_PLAYMIDI->opt_chorus_control) != 0 && stereo);
	channel_delay = ((TMDY_PLAYMIDI->opt_delay_control) > 0 && stereo);

	/* is EQ valid? */
	channel_eq = (TMDY_PLAYMIDI->opt_eq_control) && (TMDY_REVERB->eq_status.low_gain != 0x40 || TMDY_REVERB->eq_status.high_gain != 0x40);

	channel_effect = ((channel_reverb || channel_chorus
			|| channel_delay || channel_eq || (TMDY_PLAYMIDI->opt_insertion_effect)) && stereo);

	uv = (TMDY_PLAYMIDI->upper_voices);
	for(i=0;i<uv;i++) {
		if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
			(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[i].channel].lasttime = (TMDY_PLAYMIDI->current_sample) + count;
		}
	}

	if(channel_effect) {
		int buf_index = 0;
		
		if((TMDY_PLAYMIDI->reverb_buffer) == NULL) {	/* allocating buffer for channel effect */
			(TMDY_PLAYMIDI->reverb_buffer) = (char *)TMDY_COMMON->safe_malloc(tmdy_struct, MAX_CHANNELS * AUDIO_BUFFER_SIZE * 8);
		}

		for(i=0;i<MAX_CHANNELS;i++) {
			if(ISDRUMCHANNEL(i) && (TMDY_PLAYMIDI->opt_drum_effect) == 0) {
				vpblist[i] = (TMDY_PLAYMIDI->buffer_pointer);
			} else if((TMDY_PLAYMIDI->opt_insertion_effect) && (TMDY_PLAYMIDI->channel)[i].insertion_effect) {
				vpblist[i] = (TMDY_PLAYMIDI->insertion_effect_buffer);
			} else if((TMDY_PLAYMIDI->channel)[i].eq_on || ((TMDY_PLAYMIDI->channel)[i].reverb_level >= 0
					&& (TMDY_PLAYMIDI->current_sample) - (TMDY_PLAYMIDI->channel)[i].lasttime < REVERB_MAX_DELAY_OUT)
					|| (TMDY_PLAYMIDI->channel)[i].chorus_level > 0 || (TMDY_PLAYMIDI->channel)[i].delay_level > 0) {
				vpblist[i] = (int32*)((TMDY_PLAYMIDI->reverb_buffer) + buf_index);
				buf_index += n;
			} else {
				vpblist[i] = (TMDY_PLAYMIDI->buffer_pointer);
			}
		}

		if(buf_index) {memset((TMDY_PLAYMIDI->reverb_buffer), 0, buf_index);}
	}

	for(i=0;i<uv;i++) {
		if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
			int32 *vpb;
			
			if(channel_effect) {
				ch = (TMDY_PLAYMIDI->voice)[i].channel;
				vpb = vpblist[ch];
			} else {
				vpb = (TMDY_PLAYMIDI->buffer_pointer);
			}

			if(!IS_SET_CHANNELMASK(TMDY_PLAYMIDI->channel_mute, (TMDY_PLAYMIDI->voice)[i].channel)) {
				TMDY_MIX->mix_voice(tmdy_struct, vpb, i, count);
			} else {
				free_voice(tmdy_struct, i);
				ctl_note_event(tmdy_struct, i);
			}

			if((TMDY_PLAYMIDI->voice)[i].timeout > 0 && (TMDY_PLAYMIDI->voice)[i].timeout < (TMDY_PLAYMIDI->current_sample)) {
				/* timeout (See also "#extension timeout" line in *.cfg file */
				if((TMDY_PLAYMIDI->voice)[i].timeout > 1) {
					finish_note(tmdy_struct, i);
				} else {
					free_voice(tmdy_struct, i);
					ctl_note_event(tmdy_struct, i);
				}
			}
		}
	}

	while(uv > 0 && (TMDY_PLAYMIDI->voice)[uv - 1].status == VOICE_FREE)	{uv--;}
	(TMDY_PLAYMIDI->upper_voices) = uv;

	if(channel_effect) {
		if((TMDY_PLAYMIDI->opt_insertion_effect)) { 	/* insertion effect */
			/* applying insertion effect */
			TMDY_REVERB->do_insertion_effect(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt);
			/* sending insertion effect voice to channel effect */
			TMDY_REVERB->set_ch_chorus(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt, gs_ieffect.send_chorus);
			TMDY_REVERB->set_ch_delay(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt, gs_ieffect.send_delay);
			TMDY_REVERB->set_ch_reverb(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt,	gs_ieffect.send_reverb);
			if(gs_ieffect.send_eq_switch && channel_eq) {
				TMDY_REVERB->set_ch_eq(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt);
			} else {
				TMDY_REVERB->set_dry_signal(tmdy_struct, (TMDY_PLAYMIDI->insertion_effect_buffer), cnt);
			}
		}

		for(i=0;i<MAX_CHANNELS;i++) {	/* system effects */
			int32 *p;	
			p = vpblist[i];
			if(p != (TMDY_PLAYMIDI->buffer_pointer) && p != (TMDY_PLAYMIDI->insertion_effect_buffer)) {
				if(channel_chorus && (TMDY_PLAYMIDI->channel)[i].chorus_level > 0) {
					TMDY_REVERB->set_ch_chorus(tmdy_struct, p, cnt, (TMDY_PLAYMIDI->channel)[i].chorus_level);
				}
				if(channel_delay && (TMDY_PLAYMIDI->channel)[i].delay_level > 0) {
					TMDY_REVERB->set_ch_delay(tmdy_struct, p, cnt, (TMDY_PLAYMIDI->channel)[i].delay_level);
				}
				if(channel_reverb && (TMDY_PLAYMIDI->channel)[i].reverb_level > 0
					&& (TMDY_PLAYMIDI->current_sample) - (TMDY_PLAYMIDI->channel)[i].lasttime < REVERB_MAX_DELAY_OUT) {
					TMDY_REVERB->set_ch_reverb(tmdy_struct, p, cnt, (TMDY_PLAYMIDI->channel)[i].reverb_level);
				}
				if(channel_eq && (TMDY_PLAYMIDI->channel)[i].eq_on) {
					TMDY_REVERB->set_ch_eq(tmdy_struct, p, cnt);
				} else {
					TMDY_REVERB->set_dry_signal(tmdy_struct, p, cnt);
				}
			}
		}
		
		if(channel_reverb) {
			TMDY_REVERB->set_ch_reverb(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt, DEFAULT_REVERB_SEND_LEVEL);
		}
		TMDY_REVERB->set_dry_signal(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);

		/* mixing signal and applying system effects */ 
		TMDY_REVERB->mix_dry_signal(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);
		if(channel_eq) {TMDY_REVERB->do_ch_eq(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);}
		if(channel_chorus) {TMDY_REVERB->do_ch_chorus(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);}
		if(channel_delay) {TMDY_REVERB->do_ch_delay(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);}
		if(channel_reverb) {TMDY_REVERB->do_ch_reverb(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), cnt);}
	}

	(TMDY_PLAYMIDI->current_sample) += count;
}

#else
/* do_compute_data_midi() for traditionally chorus */
static void do_compute_data_midi(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
	int i, j, uv, stereo, n, ch, note;
	int32 *vpblist[MAX_CHANNELS];
	int vc[MAX_CHANNELS];
	int channel_reverb;
	int channel_effect;
	int32 cnt = count * 2;
	
	stereo = ! ((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
	n = count * ((stereo) ? 8 : 4); /* in bytes */
	channel_reverb = (((TMDY_PLAYMIDI->opt_reverb_control) == 1 || (TMDY_PLAYMIDI->opt_reverb_control) == 3)
			&& stereo);
		/* don't supported in mono */
	memset((TMDY_PLAYMIDI->buffer_pointer), 0, n);

	channel_effect = (((TMDY_PLAYMIDI->opt_reverb_control) || (TMDY_PLAYMIDI->opt_chorus_control)
			|| (TMDY_PLAYMIDI->opt_delay_control) || (TMDY_PLAYMIDI->opt_eq_control) || (TMDY_PLAYMIDI->opt_insertion_effect)) && stereo);
	uv = (TMDY_PLAYMIDI->upper_voices);
	for (i = 0; i < uv; i++)
		if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE)
			(TMDY_PLAYMIDI->channel)[(TMDY_PLAYMIDI->voice)[i].channel].lasttime = (TMDY_PLAYMIDI->current_sample) + count;

	if (channel_reverb) {
		int chbufidx;
		
		if (! (TMDY_PLAYMIDI->make_rvid_flag)) {
			make_rvid(tmdy_struct);
			(TMDY_PLAYMIDI->make_rvid_flag) = 1;
		}
		chbufidx = 0;
		for (i = 0; i < MAX_CHANNELS; i++) {
			vc[i] = 0;
			if ((TMDY_PLAYMIDI->channel)[i].reverb_id != -1
					&& (TMDY_PLAYMIDI->current_sample) - (TMDY_PLAYMIDI->channel)[i].lasttime
					< REVERB_MAX_DELAY_OUT) {
				if ((TMDY_PLAYMIDI->reverb_buffer) == NULL)
					(TMDY_PLAYMIDI->reverb_buffer) = (char *) TMDY_COMMON->safe_malloc(tmdy_struct, MAX_CHANNELS
							* AUDIO_BUFFER_SIZE * 8);
				if ((TMDY_PLAYMIDI->channel)[i].reverb_id != i)
					vpblist[i] = vpblist[(TMDY_PLAYMIDI->channel)[i].reverb_id];
				else {
					vpblist[i] = (int32 *) ((TMDY_PLAYMIDI->reverb_buffer) + chbufidx);
					chbufidx += n;
				}
			} else
				vpblist[i] = (TMDY_PLAYMIDI->buffer_pointer);
		}
		if (chbufidx)
			memset((TMDY_PLAYMIDI->reverb_buffer), 0, chbufidx);
	}
	for (i = 0; i < uv; i++)
		if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
			int32 *vpb;
			
			if (channel_reverb) {
				int ch = (TMDY_PLAYMIDI->voice)[i].channel;
				
				vpb = vpblist[ch];
				vc[ch] = 1;
			} else
				vpb = (TMDY_PLAYMIDI->buffer_pointer);
			if (! IS_SET_CHANNELMASK((TMDY_PLAYMIDI->channel_mute), (TMDY_PLAYMIDI->voice)[i].channel))
				TMDY_MIX->mix_voice(tmdy_struct, vpb, i, count);
			else {
				free_voice(tmdy_struct, i);
				ctl_note_event(tmdy_struct, i);
			}
			if ((TMDY_PLAYMIDI->voice)[i].timeout > 0 && (TMDY_PLAYMIDI->voice)[i].timeout < (TMDY_PLAYMIDI->current_sample)) {
				if ((TMDY_PLAYMIDI->voice)[i].timeout > 1)
					finish_note(tmdy_struct, i);
						/* timeout (See also "#extension timeout"
						line in *.cfg file */
				else {
					free_voice(tmdy_struct, i);
					ctl_note_event(tmdy_struct, i);
				}
			}
		}

	while (uv > 0 && (TMDY_PLAYMIDI->voice)[uv - 1].status == VOICE_FREE)
		uv--;
	(TMDY_PLAYMIDI->upper_voices) = uv;

	if (channel_reverb) {
		int k;
		
		k = count * 2; /* calclated buffer length in int32 */
		for (i = 0; i < MAX_CHANNELS; i++) {
			int32 *p;
			
			p = vpblist[i];
			if (p != (TMDY_PLAYMIDI->buffer_pointer) && (TMDY_PLAYMIDI->channel)[i].reverb_id == i)
				TMDY_REVERB->set_ch_reverb(tmdy_struct, p, k, (TMDY_PLAYMIDI->channel)[i].reverb_level);
		}
		TMDY_REVERB->set_ch_reverb(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), k, DEFAULT_REVERB_SEND_LEVEL);
		TMDY_REVERB->do_ch_reverb(tmdy_struct, (TMDY_PLAYMIDI->buffer_pointer), k);
	}
	(TMDY_PLAYMIDI->current_sample) += count;
}
#endif

static void do_compute_data_wav(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
    int i, stereo, n, file_byte, samples;

    stereo = !((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
    samples = (stereo ? (count * 2) : count );
    n = samples*4; /* in bytes */
    file_byte = samples*2; /*regard as 16bit*/

    memset((TMDY_PLAYMIDI->buffer_pointer), 0, n);

    TMDY_COMMON->tf_read(tmdy_struct, (TMDY_PLAYMIDI->wav_buffer), 1, file_byte, TMDY_READMIDI->current_file_info->pcm_tf);
    for( i=0; i<samples; i++ ){
    	(TMDY_PLAYMIDI->buffer_pointer)[i] = (LE_SHORT((TMDY_PLAYMIDI->wav_buffer)[i])) << 16;
    	(TMDY_PLAYMIDI->buffer_pointer)[i] /=4; /*level down*/
    }

    (TMDY_PLAYMIDI->current_sample) += count;
}

static void do_compute_data_aiff(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
    int i, stereo, n, file_byte, samples;

    stereo = !((TMDY_OUTPUT->play_mode)->encoding & PE_MONO);
    samples = (stereo ? (count * 2) : count );
    n = samples*4; /* in bytes */
    file_byte = samples*2; /*regard as 16bit*/

    memset((TMDY_PLAYMIDI->buffer_pointer), 0, n);

    TMDY_COMMON->tf_read(tmdy_struct, (TMDY_PLAYMIDI->wav_buffer), 1, file_byte, TMDY_READMIDI->current_file_info->pcm_tf);
    for( i=0; i<samples; i++ ){
    	(TMDY_PLAYMIDI->buffer_pointer)[i] = (BE_SHORT((TMDY_PLAYMIDI->wav_buffer)[i])) << 16;
    	(TMDY_PLAYMIDI->buffer_pointer)[i] /=4; /*level down*/
    }

    (TMDY_PLAYMIDI->current_sample) += count;
}

static void do_compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
   	switch(TMDY_READMIDI->current_file_info->pcm_mode)
    {
     	 case PCM_MODE_NON:
    		do_compute_data_midi(tmdy_struct, count);
      		break;
      	case PCM_MODE_WAV:
    		do_compute_data_wav(tmdy_struct, count);
        	break;
      	case PCM_MODE_AIFF:
    		do_compute_data_aiff(tmdy_struct, count);
        	break;
      	case PCM_MODE_AU:
        	break;
      	case PCM_MODE_MP3:
        	break;	
    } 
}

static int check_midi_play_end(tmdy_struct_ex_t *tmdy_struct, MidiEvent *e, int len)
{
    int i, type;

    for(i = 0; i < len; i++)
    {
	type = e[i].type;
	if(type == ME_NOTEON || type == ME_LAST || type == ME_WRD || type == ME_SHERRY)
	    return 0;
	if(type == ME_EOT)
	    return i + 1;
    }
    return 0;
}

int compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count);
static int midi_play_end(tmdy_struct_ex_t *tmdy_struct)
{
    int i, rc = RC_TUNE_END;

    (TMDY_PLAYMIDI->check_eot_flag) = 0;

    if((TMDY_PLAYMIDI->opt_realtime_playing) && (TMDY_PLAYMIDI->current_sample) == 0)
    {
	reset_voices(tmdy_struct);
	return RC_TUNE_END;
    }

    if((TMDY_PLAYMIDI->upper_voices) > 0)
    {
	int fadeout_cnt;

	rc = compute_data(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;

	for(i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
	    if((TMDY_PLAYMIDI->voice)[i].status & (VOICE_ON | VOICE_SUSTAINED))
		finish_note(tmdy_struct, i);
	if((TMDY_PLAYMIDI->opt_realtime_playing))
	    fadeout_cnt = 3;
	else
	    fadeout_cnt = 6;
	for(i = 0; i < fadeout_cnt && (TMDY_PLAYMIDI->upper_voices) > 0; i++)
	{
	    rc = compute_data(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate / 2);
	    if(RC_IS_SKIP_FILE(rc))
		goto midi_end;
	}

	/* kill voices */
	kill_all_voices(tmdy_struct);
	rc = compute_data(tmdy_struct, MAX_DIE_TIME);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
	(TMDY_PLAYMIDI->upper_voices) = 0;
    }

    /* clear reverb echo sound */
    TMDY_REVERB->init_reverb(tmdy_struct, (TMDY_OUTPUT->play_mode)->rate);
    for(i = 0; i < MAX_CHANNELS; i++)
    {
	(TMDY_PLAYMIDI->channel)[i].reverb_level = -1;
	(TMDY_PLAYMIDI->channel)[i].reverb_id = -1;
	(TMDY_PLAYMIDI->make_rvid_flag) = 1;
    }

    /* output null sound */
    if((TMDY_PLAYMIDI->opt_realtime_playing))
	rc = compute_data(tmdy_struct, (int32)((TMDY_OUTPUT->play_mode)->rate * PLAY_INTERLEAVE_SEC/2));
    else
	rc = compute_data(tmdy_struct, (int32)((TMDY_OUTPUT->play_mode)->rate * PLAY_INTERLEAVE_SEC));
    if(RC_IS_SKIP_FILE(rc))
	goto midi_end;

    compute_data(tmdy_struct, 0); /* flush buffer to device */

    if((TMDY_CONTROLS->ctl)->trace_playing)
    {
	rc = (TMDY_AQ->aq_flush)(tmdy_struct, 0); /* Wait until play out */
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
    }
    else
    {
	TMDY_MIDITRACE->trace_flush(tmdy_struct);
	rc = (TMDY_AQ->aq_soft_flush)(tmdy_struct);
	if(RC_IS_SKIP_FILE(rc))
	    goto midi_end;
    }

  midi_end:
    if(RC_IS_SKIP_FILE(rc))
	(TMDY_AQ->aq_flush)(tmdy_struct, 1);

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Playing time: ~%d seconds",
	      (TMDY_PLAYMIDI->current_sample)/(TMDY_OUTPUT->play_mode)->rate+2);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Notes cut: %d",
	      (TMDY_PLAYMIDI->cut_notes));
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "Notes lost totally: %d",
	      (TMDY_PLAYMIDI->lost_notes));
    if(RC_IS_SKIP_FILE(rc))
	return rc;
    return RC_TUNE_END;
}

/* count=0 means flush remaining buffered data to output device, then
   flush the device itself */
int compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count)
{
  int rc;

  if (!count)
    {
      if ((TMDY_PLAYMIDI->buffered_count))
      {
	  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		    "output data (%d)", (TMDY_PLAYMIDI->buffered_count));

#ifdef SUPPORT_SOUNDSPEC
	  soundspec_update_wave((TMDY_PLAYMIDI->common_buffer), (TMDY_PLAYMIDI->buffered_count));
#endif /* SUPPORT_SOUNDSPEC */

	  if((TMDY_AQ->aq_add)(tmdy_struct,(TMDY_PLAYMIDI->common_buffer), (TMDY_PLAYMIDI->buffered_count)) == -1)
	      return RC_ERROR;
      }
      (TMDY_PLAYMIDI->buffer_pointer)=(TMDY_PLAYMIDI->common_buffer);
      (TMDY_PLAYMIDI->buffered_count)=0;
      return RC_NONE;
    }

  while ((count+(TMDY_PLAYMIDI->buffered_count)) >= audio_buffer_size)
    {
      int i;

      if((rc = apply_controls(tmdy_struct)) != RC_NONE)
	  return rc;

      do_compute_data(tmdy_struct, audio_buffer_size-(TMDY_PLAYMIDI->buffered_count));
      count -= audio_buffer_size-(TMDY_PLAYMIDI->buffered_count);
      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		"output data (%d)", audio_buffer_size);

#ifdef SUPPORT_SOUNDSPEC
      soundspec_update_wave((TMDY_PLAYMIDI->common_buffer), audio_buffer_size);
#endif /* SUPPORT_SOUNDSPEC */

      /* fall back to linear interpolation when queue < 100% */
      if (! (TMDY_PLAYMIDI->opt_realtime_playing) && ((TMDY_OUTPUT->play_mode)->flag & PF_CAN_TRACE)) {
	  if (!(TMDY_AQ->aq_fill_buffer_flag) &&
	      100 * ((double)((TMDY_AQ->aq_filled)(tmdy_struct) + aq_soft_filled()) /
		     (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct)) < 99)
	      (TMDY_PLAYMIDI->reduce_quality_flag) = 1;
	  else
	      (TMDY_PLAYMIDI->reduce_quality_flag) = (TMDY_PLAYMIDI->no_4point_interpolation);
      }

#ifdef REDUCE_VOICE_TIME_TUNING
      /* Auto voice reduce implementation by Masanao Izumo */
      if((TMDY_PLAYMIDI->reduce_voice_threshold) &&
	 ((TMDY_OUTPUT->play_mode)->flag & PF_CAN_TRACE) &&
	 !(TMDY_AQ->aq_fill_buffer_flag) &&
	 (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct) > 0)
      {
	  /* Reduce voices if there is not enough audio device buffer */

          int nv, filled, filled_limit, rate, rate_limit;
//          static int last_filled;

	  filled = (TMDY_AQ->aq_filled)(tmdy_struct);

	  rate_limit = 75;
	  if((TMDY_PLAYMIDI->reduce_voice_threshold) >= 0)
	  {
	      filled_limit = (TMDY_OUTPUT->play_mode)->rate * (TMDY_PLAYMIDI->reduce_voice_threshold) / 1000
		  + 1; /* +1 disable zero */
	  }
	  else /* Use default threshold */
	  {
	      int32 maxfill;
	      maxfill = (TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct);
	      filled_limit = REDUCE_VOICE_TIME_TUNING;
	      if(filled_limit > maxfill / 5) /* too small audio buffer */
	      {
		  rate_limit -= 100 * audio_buffer_size / maxfill / 5;
		  filled_limit = 1;
	      }
	  }

	  /* Calculate rate as it is displayed in ncurs_c.c */
	  /* The old method of calculating rate resulted in very low values
	     when using the new high order interplation methods on "slow"
	     CPUs when the queue was being drained WAY too quickly.  This
	     caused premature voice reduction under Linux, even if the queue
	     was over 2000%, leading to major voice lossage. */
	  rate = (int)(((double)((TMDY_AQ->aq_filled)(tmdy_struct) + aq_soft_filled()) /
                  	(TMDY_AQ->aq_get_dev_queuesize)(tmdy_struct)) * 100 + 0.5);

          for(i = nv = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
	      if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE)
	          nv++;

	  if(! (TMDY_PLAYMIDI->opt_realtime_playing))
	  {
	      /* calculate ok_nv, the "optimum" max polyphony */
	      if ((TMDY_PLAYMIDI->auto_reduce_polyphony) && rate < 85) {
		/* average in current nv */
	        if ((rate == (TMDY_PLAYMIDI->old_rate) && nv > (TMDY_PLAYMIDI->min_bad_nv)) ||
	            (rate >= (TMDY_PLAYMIDI->old_rate) && rate < 20)) {
	        	(TMDY_PLAYMIDI->ok_nv_total) += nv;
	        	(TMDY_PLAYMIDI->ok_nv_counts)++;
	        }
	        /* increase polyphony when it is too low */
	        else if (nv == (TMDY_PLAYMIDI->voices) &&
	                 (rate > (TMDY_PLAYMIDI->old_rate) && filled > TMDY_PLAYMIDI->compute_data_last_filled)) {
	          		(TMDY_PLAYMIDI->ok_nv_total) += nv + 1;
	          		(TMDY_PLAYMIDI->ok_nv_counts)++;
	        }
	        /* reduce polyphony when loosing buffer */
	        else if (rate < 75 &&
	        	 (rate < (TMDY_PLAYMIDI->old_rate) && filled < TMDY_PLAYMIDI->compute_data_last_filled)) {
	        	(TMDY_PLAYMIDI->ok_nv_total) += (TMDY_PLAYMIDI->min_bad_nv);
	    		(TMDY_PLAYMIDI->ok_nv_counts)++;
	        }
	        else goto NO_RESCALE_NV;

		/* rescale ok_nv stuff every 1 seconds */
		if ((TMDY_PLAYMIDI->current_sample) >= (TMDY_PLAYMIDI->ok_nv_sample) && (TMDY_PLAYMIDI->ok_nv_counts) > 1) {
			(TMDY_PLAYMIDI->ok_nv_total) >>= 1;
			(TMDY_PLAYMIDI->ok_nv_counts) >>= 1;
			(TMDY_PLAYMIDI->ok_nv_sample) = (TMDY_PLAYMIDI->current_sample) + ((TMDY_OUTPUT->play_mode)->rate);
		}

		NO_RESCALE_NV:;
	      }
	  }

	  /* EAW -- if buffer is < 75%, start reducing some voices to
	     try to let it recover.  This really helps a lot, preserves
	     decent sound, and decreases the frequency of lost ON notes */
	  if ((! (TMDY_PLAYMIDI->opt_realtime_playing) && rate < rate_limit)
	      || filled < filled_limit)
	  {
	      if(filled <= TMDY_PLAYMIDI->compute_data_last_filled)
	      {
	          int v, kill_nv, temp_nv;

		  /* set bounds on "good" and "bad" nv */
		  if (! (TMDY_PLAYMIDI->opt_realtime_playing) && rate > 20 &&
		      nv < (TMDY_PLAYMIDI->min_bad_nv)) {
		  	(TMDY_PLAYMIDI->min_bad_nv) = nv;
	                if ((TMDY_PLAYMIDI->max_good_nv) < (TMDY_PLAYMIDI->min_bad_nv))
	                	(TMDY_PLAYMIDI->max_good_nv) = (TMDY_PLAYMIDI->min_bad_nv);
	          }

		  /* EAW -- count number of !ON voices */
		  /* treat chorus notes as !ON */
		  for(i = kill_nv = 0; i < (TMDY_PLAYMIDI->upper_voices); i++) {
		      if((TMDY_PLAYMIDI->voice)[i].status & VOICE_FREE ||
		         (TMDY_PLAYMIDI->voice)[i].cache != NULL)
		      		continue;
		      
		      if(((TMDY_PLAYMIDI->voice)[i].status & ~(VOICE_ON|VOICE_SUSTAINED) &&
			  !((TMDY_PLAYMIDI->voice)[i].status & ~(VOICE_DIE) &&
			    (TMDY_PLAYMIDI->voice)[i].sample->note_to_use)))
				kill_nv++;
		  }

		  /* EAW -- buffer is dangerously low, drasticly reduce
		     voices to a hopefully "safe" amount */
		  if (filled < filled_limit &&
		      ((TMDY_PLAYMIDI->opt_realtime_playing) || rate < 10)) {
		      FLOAT_T n;

		      /* calculate the drastic voice reduction */
		      if(nv > kill_nv) /* Avoid division by zero */
		      {
			  n = (FLOAT_T) nv / (nv - kill_nv);
			  temp_nv = (int)(nv - nv / (n + 1));

			  /* reduce by the larger of the estimates */
			  if (kill_nv < temp_nv && temp_nv < nv)
			      kill_nv = temp_nv;
		      }
		      else kill_nv = nv - 1; /* do not kill all the voices */
		  }
		  else {
		      /* the buffer is still high enough that we can throw
		         fewer voices away; keep the ON voices, use the
		         minimum "bad" nv as a floor on voice reductions */
		      temp_nv = nv - (TMDY_PLAYMIDI->min_bad_nv);
		      if (kill_nv > temp_nv)
		          kill_nv = temp_nv;
		  }

		  for(i = 0; i < kill_nv; i++)
		      v = reduce_voice(tmdy_struct);

		  /* lower max # of allowed voices to let the buffer recover */
		  if ((TMDY_PLAYMIDI->auto_reduce_polyphony)) {
		  	temp_nv = nv - kill_nv;
		  	(TMDY_PLAYMIDI->ok_nv) = (TMDY_PLAYMIDI->ok_nv_total) / (TMDY_PLAYMIDI->ok_nv_counts);

		  	/* decrease it to current nv left */
		  	if ((TMDY_PLAYMIDI->voices) > temp_nv && temp_nv > (TMDY_PLAYMIDI->ok_nv))
			    voice_decrement_conservative(tmdy_struct, (TMDY_PLAYMIDI->voices) - temp_nv);
			/* decrease it to ok_nv */
		  	else if ((TMDY_PLAYMIDI->voices) > (TMDY_PLAYMIDI->ok_nv) && temp_nv <= (TMDY_PLAYMIDI->ok_nv))
			    voice_decrement_conservative(tmdy_struct, (TMDY_PLAYMIDI->voices) - (TMDY_PLAYMIDI->ok_nv));
		  	/* increase the polyphony */
		  	else if ((TMDY_PLAYMIDI->voices) < (TMDY_PLAYMIDI->ok_nv))
			    voice_increment(tmdy_struct, (TMDY_PLAYMIDI->ok_nv) - (TMDY_PLAYMIDI->voices));
		  }

		  while((TMDY_PLAYMIDI->upper_voices) > 0 &&
			(TMDY_PLAYMIDI->voice)[(TMDY_PLAYMIDI->upper_voices) - 1].status == VOICE_FREE)
		      (TMDY_PLAYMIDI->upper_voices)--;
	      }
	      TMDY_PLAYMIDI->compute_data_last_filled = filled;
	  }
	  else {
	      if (! (TMDY_PLAYMIDI->opt_realtime_playing) && rate >= rate_limit &&
	          filled > TMDY_PLAYMIDI->compute_data_last_filled) {

		    /* set bounds on "good" and "bad" nv */
		    if (rate > 85 && nv > (TMDY_PLAYMIDI->max_good_nv)) {
		  	(TMDY_PLAYMIDI->max_good_nv) = nv;
		  	if ((TMDY_PLAYMIDI->min_bad_nv) > (TMDY_PLAYMIDI->max_good_nv))
		  	    (TMDY_PLAYMIDI->min_bad_nv) = (TMDY_PLAYMIDI->max_good_nv);
		    }

		    if ((TMDY_PLAYMIDI->auto_reduce_polyphony)) {
		    	/* reset ok_nv stuff when out of danger */
		    	(TMDY_PLAYMIDI->ok_nv_total) = (TMDY_PLAYMIDI->max_good_nv) * (TMDY_PLAYMIDI->ok_nv_counts);
			if ((TMDY_PLAYMIDI->ok_nv_counts) > 1) {
			    (TMDY_PLAYMIDI->ok_nv_total) >>= 1;
			    (TMDY_PLAYMIDI->ok_nv_counts) >>= 1;
			}

		    	/* restore max # of allowed voices to normal */
			restore_voices(tmdy_struct, 0);
		    }
	      }

	      TMDY_PLAYMIDI->compute_data_last_filled = filled_limit;
          }
          (TMDY_PLAYMIDI->old_rate) = rate;
      }
#endif

      if((TMDY_AQ->aq_add)(tmdy_struct, (TMDY_PLAYMIDI->common_buffer), audio_buffer_size) == -1)
	  return RC_ERROR;

      (TMDY_PLAYMIDI->buffer_pointer)=(TMDY_PLAYMIDI->common_buffer);
      (TMDY_PLAYMIDI->buffered_count)=0;
      if((TMDY_PLAYMIDI->current_event)->type != ME_EOT)
	  ctl_timestamp(tmdy_struct);

      /* check break signals */
      VOLATILE_TOUCH(TMDY_MAIN->intr);
      if(TMDY_MAIN->intr)
	  return RC_QUIT;

      if((TMDY_PLAYMIDI->upper_voices) == 0 && (TMDY_PLAYMIDI->check_eot_flag) &&
	 (i = check_midi_play_end(tmdy_struct, (TMDY_PLAYMIDI->current_event), EOT_PRESEARCH_LEN)) > 0)
      {
	  if(i > 1)
	      (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
			"Last %d MIDI events are ignored", i - 1);
	  return midi_play_end(tmdy_struct);
      }
    }
  if (count>0)
    {
      do_compute_data(tmdy_struct, count);
      (TMDY_PLAYMIDI->buffered_count) += count;
      (TMDY_PLAYMIDI->buffer_pointer) += ((TMDY_OUTPUT->play_mode)->encoding & PE_MONO) ? count : count*2;
    }
  return RC_NONE;
}

static void update_modulation_wheel(tmdy_struct_ex_t *tmdy_struct, int ch, int val)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices);
    for(i = 0; i < uv; i++)
	if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE && (TMDY_PLAYMIDI->voice)[i].channel == ch)
	{
	    /* Set/Reset mod-wheel */
	    (TMDY_PLAYMIDI->voice)[i].modulation_wheel = val;
	    (TMDY_PLAYMIDI->voice)[i].vibrato_delay = 0;
	    recompute_freq(tmdy_struct, i);
	}
}

static void drop_portamento(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices);

    (TMDY_PLAYMIDI->channel)[ch].porta_control_ratio = 0;
    for(i = 0; i < uv; i++)
	if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE &&
	   (TMDY_PLAYMIDI->voice)[i].channel == ch &&
	   (TMDY_PLAYMIDI->voice)[i].porta_control_ratio)
	{
	    (TMDY_PLAYMIDI->voice)[i].porta_control_ratio = 0;
	    recompute_freq(tmdy_struct, i);
	}
    (TMDY_PLAYMIDI->channel)[ch].last_note_fine = -1;
}

static void update_portamento_controls(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    if(!(TMDY_PLAYMIDI->channel)[ch].portamento ||
       ((TMDY_PLAYMIDI->channel)[ch].portamento_time_msb | (TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb)
       == 0)
	drop_portamento(tmdy_struct, ch);
    else
    {
	double mt, dc;
	int d;

	mt = midi_time_table[(TMDY_PLAYMIDI->channel)[ch].portamento_time_msb & 0x7F] *
	    midi_time_table2[(TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb & 0x7F] *
		PORTAMENTO_TIME_TUNING;
	dc = (TMDY_OUTPUT->play_mode)->rate * mt;
	d = (int)(1.0 / (mt * PORTAMENTO_CONTROL_RATIO));
	d++;
	(TMDY_PLAYMIDI->channel)[ch].porta_control_ratio = (int)(d * dc + 0.5);
	(TMDY_PLAYMIDI->channel)[ch].porta_dpb = d;
    }
}

static void update_portamento_time(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    int i, uv = (TMDY_PLAYMIDI->upper_voices);
    int dpb;
    int32 ratio;

    update_portamento_controls(tmdy_struct, ch);
    dpb = (TMDY_PLAYMIDI->channel)[ch].porta_dpb;
    ratio = (TMDY_PLAYMIDI->channel)[ch].porta_control_ratio;

    for(i = 0; i < uv; i++)
    {
	if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE &&
	   (TMDY_PLAYMIDI->voice)[i].channel == ch &&
	   (TMDY_PLAYMIDI->voice)[i].porta_control_ratio)
	{
	    (TMDY_PLAYMIDI->voice)[i].porta_control_ratio = ratio;
	    (TMDY_PLAYMIDI->voice)[i].porta_dpb = dpb;
	    recompute_freq(tmdy_struct, i);
	}
    }
}

static void update_legato_controls(tmdy_struct_ex_t *tmdy_struct, int ch)
{
	double mt, dc;
	int d;

	mt = 0.06250 * PORTAMENTO_TIME_TUNING * 0.3;
	dc = (TMDY_OUTPUT->play_mode)->rate * mt;
	d = (int)(1.0 / (mt * PORTAMENTO_CONTROL_RATIO));
	d++;
	(TMDY_PLAYMIDI->channel)[ch].porta_control_ratio = (int)(d * dc + 0.5);
	(TMDY_PLAYMIDI->channel)[ch].porta_dpb = d;
}

int play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
    int32 i, j, cet;
    int k, l, ch, orig_ch, port_ch, offset, layered;

    if((TMDY_OUTPUT->play_mode)->flag & PF_MIDI_EVENT)
	return (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_MIDI, ev);
    if(!((TMDY_OUTPUT->play_mode)->flag & PF_PCM_STREAM))
	return RC_NONE;

    (TMDY_PLAYMIDI->current_event) = ev;
    cet = MIDI_EVENT_TIME(ev);

    if((TMDY_CONTROLS->ctl)->verbosity >= VERB_DEBUG_SILLY)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		  "Midi Event %d: %s %d %d %d", cet,
		  event_name(tmdy_struct, ev->type), ev->channel, ev->a, ev->b);
    if(cet > (TMDY_PLAYMIDI->current_sample))
    {
	int rc;


    if(TMDY_PLAYMIDI->midi_streaming!=0){

    	if ( (cet - (TMDY_PLAYMIDI->current_sample)) * 1000 / (TMDY_OUTPUT->play_mode)->rate > (TMDY_PLAYMIDI->stream_max_compute) ) {
			kill_all_voices(tmdy_struct);
			/* reset_voices(tmdy_struct); *
/* 			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY, "play_event: discard %d samples", cet - (TMDY_PLAYMIDI->current_sample)); */
			(TMDY_PLAYMIDI->current_sample) = cet;
		}
    	
    }

	rc = compute_data(tmdy_struct, cet - (TMDY_PLAYMIDI->current_sample));
	ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);
    if(rc == RC_JUMP)
	{
		ctl_timestamp(tmdy_struct);
		return RC_NONE;
	}
	if(rc != RC_NONE)
	    return rc;
	}

#ifndef SUPPRESS_CHANNEL_LAYER
	orig_ch = ev->channel;
	layered = ! IS_SYSEX_EVENT_TYPE(ev->type);
	for (k = 0; k < MAX_CHANNELS; k += 16) {
		port_ch = (orig_ch + k) % MAX_CHANNELS;
		offset = (port_ch < 16) ? 0 : 16;
		for (l = offset; l < offset + 16; l++) {
			if (! layered && (k || l != offset))
				continue;
			if (layered) {
				if (! IS_SET_CHANNELMASK((TMDY_PLAYMIDI->channel)[l].channel_layer, port_ch)
						|| (TMDY_PLAYMIDI->channel)[l].port_select ^ (orig_ch >= 16))
					continue;
				ev->channel = l;
			}
#endif
	ch = ev->channel;

    switch(ev->type)
    {
	/* MIDI Events */
      case ME_NOTEOFF:
	note_off(tmdy_struct, ev);
	break;

      case ME_NOTEON:
	note_on(tmdy_struct, ev);
	break;

      case ME_KEYPRESSURE:
	adjust_pressure(tmdy_struct, ev);
	break;

      case ME_PROGRAM:
	midi_program_change(tmdy_struct, ch, ev->a);
	ctl_prog_event(tmdy_struct, ch);
	break;

      case ME_CHANNEL_PRESSURE:
	adjust_channel_pressure(tmdy_struct, ev);
	break;

      case ME_PITCHWHEEL:
	(TMDY_PLAYMIDI->channel)[ch].pitchbend = ev->a + ev->b * 128;
	(TMDY_PLAYMIDI->channel)[ch].pitchfactor = 0;
	/* Adjust pitch for notes already playing */
	adjust_pitch(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_PITCH_BEND, 1, ch, (TMDY_PLAYMIDI->channel)[ch].pitchbend);
	break;

	/* Controls */
      case ME_TONE_BANK_MSB:
	(TMDY_PLAYMIDI->channel)[ch].bank_msb = ev->a;
	break;

      case ME_TONE_BANK_LSB:
	(TMDY_PLAYMIDI->channel)[ch].bank_lsb = ev->a;
	break;

      case ME_MODULATION_WHEEL:
	(TMDY_PLAYMIDI->channel)[ch].modulation_wheel =
	    midi_cnv_vib_depth(tmdy_struct, ev->a);
	update_modulation_wheel(tmdy_struct, ch, (TMDY_PLAYMIDI->channel)[ch].modulation_wheel);
	ctl_mode_event(tmdy_struct, CTLE_MOD_WHEEL, 1, ch, (TMDY_PLAYMIDI->channel)[ch].modulation_wheel);
	break;

      case ME_MAINVOLUME:
	(TMDY_PLAYMIDI->channel)[ch].volume = ev->a;
	adjust_volume(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_VOLUME, 1, ch, ev->a);
	break;

      case ME_PAN:
	(TMDY_PLAYMIDI->channel)[ch].panning = ev->a;
	(TMDY_PLAYMIDI->channel)[ch].pan_random = 0;
	if((TMDY_PLAYMIDI->adjust_panning_immediately) && !(TMDY_PLAYMIDI->channel)[ch].pan_random)
	    adjust_panning(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_PANNING, 1, ch, ev->a);
	break;

      case ME_EXPRESSION:
	(TMDY_PLAYMIDI->channel)[ch].expression = ev->a;
	adjust_volume(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_EXPRESSION, 1, ch, ev->a);
	break;

      case ME_SUSTAIN:
	(TMDY_PLAYMIDI->channel)[ch].sustain = (ev->a >= 64);
	if(!ev->a)
	    drop_sustain(tmdy_struct, ch);
	ctl_mode_event(tmdy_struct, CTLE_SUSTAIN, 1, ch, ev->a >= 64);
	break;

      case ME_SOSTENUTO:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Sostenuto - this function is not supported.");
	break;

      case ME_LEGATO_FOOTSWITCH:
    (TMDY_PLAYMIDI->channel)[ch].legato = (ev->a >= 64);
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Legato Footswitch (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].legato);
	break;

      case ME_HOLD2:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Hold2 - this function is not supported.");
	break;

      case ME_BREATH:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Breath - this function is not supported.");
	break;

      case ME_FOOT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Foot - this function is not supported.");
	break;

      case ME_BALANCE:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Balance - this function is not supported.");
	break;

      case ME_PORTAMENTO_TIME_MSB:
	(TMDY_PLAYMIDI->channel)[ch].portamento_time_msb = ev->a;
	update_portamento_time(tmdy_struct, ch);
	break;

      case ME_PORTAMENTO_TIME_LSB:
	(TMDY_PLAYMIDI->channel)[ch].portamento_time_lsb = ev->a;
	update_portamento_time(tmdy_struct, ch);
	break;

      case ME_PORTAMENTO:
	(TMDY_PLAYMIDI->channel)[ch].portamento = (ev->a >= 64);
	if(!(TMDY_PLAYMIDI->channel)[ch].portamento)
	    drop_portamento(tmdy_struct, ch);
	break;

	  case ME_SOFT_PEDAL:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].soft_pedal = ev->a;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Soft Pedal (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].soft_pedal);
		  }
		  break;

	  case ME_HARMONIC_CONTENT:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].param_resonance = ev->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Harmonic Content (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].param_resonance);
		  }
		  break;

	  case ME_BRIGHTNESS:
		  if((TMDY_PLAYMIDI->opt_lpf_def)) {
			  (TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq = ev->a - 64;
			  (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Brightness (CH:%d VAL:%d)",ch,(TMDY_PLAYMIDI->channel)[ch].param_cutoff_freq);
		  }
		  break;

      case ME_DATA_ENTRY_MSB:
	if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    (TMDY_PLAYMIDI->channel)[ch].rpnmap[i] = ev->a;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_DATA_ENTRY_LSB:
	if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	/* Ignore */
	(TMDY_PLAYMIDI->channel)[ch].nrpn = -1;
	break;

      case ME_REVERB_EFFECT:
	if((TMDY_PLAYMIDI->opt_reverb_control))
	{
	    set_reverb_level(tmdy_struct, ch, ev->a);
	    ctl_mode_event(tmdy_struct, CTLE_REVERB_EFFECT, 1, ch, get_reverb_level(tmdy_struct, ch));
	}
	break;

      case ME_CHORUS_EFFECT:
	if((TMDY_PLAYMIDI->opt_chorus_control))
	{
	    if((TMDY_PLAYMIDI->opt_chorus_control) == 1)
		(TMDY_PLAYMIDI->channel)[ch].chorus_level = ev->a;
	    else
		(TMDY_PLAYMIDI->channel)[ch].chorus_level = -(TMDY_PLAYMIDI->opt_chorus_control);
	    ctl_mode_event(tmdy_struct, CTLE_CHORUS_EFFECT, 1, ch, get_chorus_level(tmdy_struct, ch));
		if(ev->a) {
			(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Chorus Send (CH:%d LEVEL:%d)",ch,ev->a);
		}
	}
	break;

      case ME_TREMOLO_EFFECT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Tremolo Send (CH:%d LEVEL:%d)",ch,ev->a);
	break;

      case ME_CELESTE_EFFECT:
	if((TMDY_PLAYMIDI->opt_delay_control)) {
		(TMDY_PLAYMIDI->channel)[ch].delay_level = ev->a;
		(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Delay Send (CH:%d LEVEL:%d)",ch,ev->a);
	}
	break;

	  case ME_ATTACK_TIME:
  	if(!(TMDY_PLAYMIDI->opt_tva_attack)) { break; }
	set_envelope_time(tmdy_struct, ch,ev->a,0);
	break;

	  case ME_RELEASE_TIME:
  	if(!(TMDY_PLAYMIDI->opt_tva_release)) { break; }
	set_envelope_time(tmdy_struct, ch,ev->a,3);
	break;

      case ME_PHASER_EFFECT:
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO,VERB_NOISY,"Phaser Send (CH:%d LEVEL:%d)",ch,ev->a);
	break;

      case ME_RPN_INC:
	if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    if((TMDY_PLAYMIDI->channel)[ch].rpnmap[i] < 127)
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[i]++;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_RPN_DEC:
	if((TMDY_PLAYMIDI->channel)[ch].rpn_7f7f_flag) /* disable */
	    break;
	if((i = last_rpn_addr(tmdy_struct, ch)) >= 0)
	{
	    if((TMDY_PLAYMIDI->channel)[ch].rpnmap[i] > 0)
		(TMDY_PLAYMIDI->channel)[ch].rpnmap[i]--;
	    update_rpn_map(tmdy_struct, ch, i, 1);
	}
	break;

      case ME_NRPN_LSB:
	(TMDY_PLAYMIDI->channel)[ch].lastlrpn = ev->a;
	(TMDY_PLAYMIDI->channel)[ch].nrpn = 1;
	break;

      case ME_NRPN_MSB:
	(TMDY_PLAYMIDI->channel)[ch].lastmrpn = ev->a;
	(TMDY_PLAYMIDI->channel)[ch].nrpn = 1;
	break;

      case ME_RPN_LSB:
	(TMDY_PLAYMIDI->channel)[ch].lastlrpn = ev->a;
	(TMDY_PLAYMIDI->channel)[ch].nrpn = 0;
	break;

      case ME_RPN_MSB:
	(TMDY_PLAYMIDI->channel)[ch].lastmrpn = ev->a;
	(TMDY_PLAYMIDI->channel)[ch].nrpn = 0;
	break;

      case ME_ALL_SOUNDS_OFF:
	all_sounds_off(tmdy_struct, ch);
	break;

      case ME_RESET_CONTROLLERS:
	reset_controllers(tmdy_struct, ch);
	redraw_controllers(tmdy_struct, ch);
	break;

      case ME_ALL_NOTES_OFF:
	all_notes_off(tmdy_struct, ch);
	break;

      case ME_MONO:
	(TMDY_PLAYMIDI->channel)[ch].mono = 1;
	all_notes_off(tmdy_struct, ch);
	break;

      case ME_POLY:
	(TMDY_PLAYMIDI->channel)[ch].mono = 0;
	all_notes_off(tmdy_struct, ch);
	break;

	/* TiMidity Extensionals */
      case ME_RANDOM_PAN:
	(TMDY_PLAYMIDI->channel)[ch].panning = TMDY_COMMON->int_rand(tmdy_struct, 128);
	(TMDY_PLAYMIDI->channel)[ch].pan_random = 1;
	if((TMDY_PLAYMIDI->adjust_panning_immediately) && !(TMDY_PLAYMIDI->channel)[ch].pan_random)
	    adjust_panning(tmdy_struct, ch);
	break;

      case ME_SET_PATCH:
	i = (TMDY_PLAYMIDI->channel)[ch].special_sample = (TMDY_PLAYMIDI->current_event)->a;
	if(TMDY_INSTRUM->special_patch[i] != NULL)
	    TMDY_INSTRUM->special_patch[i]->sample_offset = 0;
	ctl_prog_event(tmdy_struct, ch);
	break;

      case ME_TEMPO:
	(TMDY_PLAYMIDI->current_play_tempo) = ch + ev->b * 256 + ev->a * 65536;
	ctl_mode_event(tmdy_struct, CTLE_TEMPO, 1, (TMDY_PLAYMIDI->current_play_tempo), 0);
	break;

      case ME_CHORUS_TEXT:
      case ME_LYRIC:
      case ME_MARKER:
      case ME_INSERT_TEXT:
      case ME_TEXT:
      case ME_KARAOKE_LYRIC:
	i = ev->a | ((int)ev->b << 8);
	ctl_mode_event(tmdy_struct, CTLE_LYRIC, 1, i, 0);
	break;

      case ME_GSLCD:
	i = ev->a | ((int)ev->b << 8);
	ctl_mode_event(tmdy_struct, CTLE_GSLCD, 1, i, 0);
	break;

      case ME_MASTER_VOLUME:
	(TMDY_PLAYMIDI->master_volume_ratio) = (int32)ev->a + 256 * (int32)ev->b;
	adjust_master_volume(tmdy_struct);
	break;

      case ME_RESET:
	TMDY_READMIDI->change_system_mode(tmdy_struct, ev->a);
	reset_midi(tmdy_struct, 1);
	break;

      case ME_PATCH_OFFS:
	i = (TMDY_PLAYMIDI->channel)[ch].special_sample;
	if(TMDY_INSTRUM->special_patch[i] != NULL)
	    TMDY_INSTRUM->special_patch[i]->sample_offset =
		((TMDY_PLAYMIDI->current_event)->a | 256 * (TMDY_PLAYMIDI->current_event)->b);
	break;

      case ME_WRD:
	TMDY_MIDITRACE->push_midi_trace2(tmdy_struct, TMDY_WRD->wrd_midi_event,
			 ch, (TMDY_PLAYMIDI->current_event)->a | ((TMDY_PLAYMIDI->current_event)->b << 8));
	break;

      case ME_SHERRY:
	TMDY_MIDITRACE->push_midi_trace1(tmdy_struct, TMDY_WRD->wrd_sherry_event,
			 ch | ((TMDY_PLAYMIDI->current_event)->a<<8) | ((TMDY_PLAYMIDI->current_event)->b<<16));
	break;

      case ME_DRUMPART:
	if(midi_drumpart_change(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a))
	{
	    /* Update bank information */
	    midi_program_change(tmdy_struct, ch, (TMDY_PLAYMIDI->channel)[ch].program);
	    ctl_mode_event(tmdy_struct, CTLE_DRUMPART, 1, ch, ISDRUMCHANNEL(ch));
	    ctl_prog_event(tmdy_struct, ch);
	}
	break;

      case ME_KEYSHIFT:
	i = (int)(TMDY_PLAYMIDI->current_event)->a - 0x40;
	if(i != (TMDY_PLAYMIDI->channel)[ch].key_shift)
	{
	    all_sounds_off(tmdy_struct, ch);
	    (TMDY_PLAYMIDI->channel)[ch].key_shift = (int8)i;
	}
	break;

	case ME_KEYSIG:
		(TMDY_PLAYMIDI->current_keysig) = (TMDY_PLAYMIDI->current_event)->a + (TMDY_PLAYMIDI->current_event)->b * 16;
		ctl_mode_event(tmdy_struct, CTLE_KEYSIG, 1, (TMDY_PLAYMIDI->current_keysig), 0);
		(TMDY_PLAYMIDI->current_temper_keysig) = (TMDY_PLAYMIDI->current_keysig);
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_KEYSIG, 1, (TMDY_PLAYMIDI->current_temper_keysig), 0);
		if ((TMDY_PLAYMIDI->opt_force_keysig) != 8) {
			i = (TMDY_PLAYMIDI->current_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : -6);
			(TMDY_PLAYMIDI->note_key_offset) -= floor((TMDY_PLAYMIDI->note_key_offset) / 12.0) * 12;
			for (j = 0; j < (TMDY_PLAYMIDI->note_key_offset); j++)
				i += (i > 10) ? -5 : 7;
			j = (TMDY_PLAYMIDI->opt_force_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : 10);
			while (i != j && i != j + 12) {
				if (++(TMDY_PLAYMIDI->note_key_offset) > 6)
					(TMDY_PLAYMIDI->note_key_offset) -= 12;
				i += (i > 10) ? -5 : 7;
			}
			kill_all_voices(tmdy_struct);
			ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 1, (TMDY_PLAYMIDI->note_key_offset), 0);
		}
		i = (TMDY_PLAYMIDI->current_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : -9), j = 0;
		while (i != 7 && i != 19)
			i += (i < 7) ? 5 : -7, j++;
		j += (TMDY_PLAYMIDI->note_key_offset), j -= floor(j / 12.0) * 12;
		(TMDY_PLAYMIDI->current_freq_table) = j;
		break;

	case ME_SCALE_TUNING:
		TMDY_RECACHE->resamp_cache_refer_alloff(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->time);
		(TMDY_PLAYMIDI->channel)[ch].scale_tuning[(TMDY_PLAYMIDI->current_event)->a] = (TMDY_PLAYMIDI->current_event)->b;
		adjust_pitch(tmdy_struct, ch);
		break;

	case ME_BULK_TUNING_DUMP:
		set_single_note_tuning(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a, (TMDY_PLAYMIDI->current_event)->b, 0);
		break;

	case ME_SINGLE_NOTE_TUNING:
		set_single_note_tuning(tmdy_struct, ch, (TMDY_PLAYMIDI->current_event)->a, (TMDY_PLAYMIDI->current_event)->b, 1);
		break;

	case ME_TEMPER_KEYSIG:
		(TMDY_PLAYMIDI->current_temper_keysig) = (TMDY_PLAYMIDI->current_event)->a;
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_KEYSIG, 1, (TMDY_PLAYMIDI->current_temper_keysig), 0);
		i = (TMDY_PLAYMIDI->current_temper_keysig) + (((TMDY_PLAYMIDI->current_temper_keysig) < 8) ? 7 : -9);
		j = 0;
		while (i != 7 && i != 19)
			i += (i < 7) ? 5 : -7, j++;
		j += (TMDY_PLAYMIDI->note_key_offset), j -= floor(j / 12.0) * 12;
		(TMDY_PLAYMIDI->current_freq_table) = j;
		if ((TMDY_PLAYMIDI->current_event)->b)
			for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
				if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
					(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_TEMPER_TYPE:
		(TMDY_PLAYMIDI->channel)[ch].temper_type = (TMDY_PLAYMIDI->current_event)->a;
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, ch, (TMDY_PLAYMIDI->channel)[ch].temper_type);
		if ((TMDY_PLAYMIDI->temper_type_mute)) {
			if ((TMDY_PLAYMIDI->temper_type_mute) & 1 << (TMDY_PLAYMIDI->current_event)->a
					- (((TMDY_PLAYMIDI->current_event)->a >= 0x40) ? 0x3c : 0)) {
				SET_CHANNELMASK((TMDY_PLAYMIDI->channel_mute), ch);
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, ch, 1);
			} else {
				UNSET_CHANNELMASK((TMDY_PLAYMIDI->channel_mute), ch);
				ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, ch, 0);
			}
		}
		if ((TMDY_PLAYMIDI->current_event)->b)
			for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
				if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
					(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_MASTER_TEMPER_TYPE:
		for (i = 0; i < MAX_CHANNELS; i++) {
			(TMDY_PLAYMIDI->channel)[i].temper_type = (TMDY_PLAYMIDI->current_event)->a;
			ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 1, i, (TMDY_PLAYMIDI->channel)[i].temper_type);
		}
		if ((TMDY_PLAYMIDI->temper_type_mute)) {
			if ((TMDY_PLAYMIDI->temper_type_mute) & 1 << (TMDY_PLAYMIDI->current_event)->a
					- (((TMDY_PLAYMIDI->current_event)->a >= 0x40) ? 0x3c : 0)) {
				FILL_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));
				for (i = 0; i < MAX_CHANNELS; i++)
					ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, i, 1);
			} else {
				CLEAR_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));
				for (i = 0; i < MAX_CHANNELS; i++)
					ctl_mode_event(tmdy_struct, CTLE_MUTE, 1, i, 0);
			}
		}
		if ((TMDY_PLAYMIDI->current_event)->b)
			for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
				if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
					(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;

	case ME_SYSEX_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_SYSEX_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_SYSEX_GS_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_GS_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_SYSEX_GS_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_GS_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_SYSEX_XG_LSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_XG_LSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_SYSEX_XG_MSB:
		process_sysex_event(tmdy_struct, ME_SYSEX_XG_MSB,ch,(TMDY_PLAYMIDI->current_event)->a,(TMDY_PLAYMIDI->current_event)->b);
	    break;

	case ME_NOTE_STEP:
		i = ev->a + ((ev->b & 0x0f) << 8);
		j = ev->b >> 4;
		ctl_mode_event(tmdy_struct, CTLE_METRONOME, 1, i, j);
		if (TMDY_READMIDI->readmidi_wrd_mode)
			TMDY_WRD->wrdt->update_events(tmdy_struct);
		break;

      case ME_EOT:
	return midi_play_end(tmdy_struct);
    }
#ifndef SUPPRESS_CHANNEL_LAYER
		}
	}
	ev->channel = orig_ch;
#endif

    return RC_NONE;
}

static void set_single_note_tuning(tmdy_struct_ex_t *tmdy_struct, int part, int a, int b, int rt)
{
/*
	static int TMDY_PLAYMIDI->set_s_n_t_tp;	* tuning program number *
	static int TMDY_PLAYMIDI->set_s_n_t_kn;	* MIDI key number *
	static int TMDY_PLAYMIDI->set_s_n_t_st;	* the nearest equal-tempered semitone *
*/
	double f, fst;	/* fraction of semitone */
	int i;
	
	switch (part) {
	case 0:
		TMDY_PLAYMIDI->set_s_n_t_tp = a;
		break;
	case 1:
		TMDY_PLAYMIDI->set_s_n_t_kn = a;
		TMDY_PLAYMIDI->set_s_n_t_st = b;
		break;
	case 2:
		if (TMDY_PLAYMIDI->set_s_n_t_st == 0x7f && a == 0x7f && b == 0x7f)	/* no change */
			break;
		f = 440 * pow(2.0, (TMDY_PLAYMIDI->set_s_n_t_st - 69) / 12.0);
		fst = pow(2.0, (a << 7 | b) / 196608.0);
		freq_table_tuning[TMDY_PLAYMIDI->set_s_n_t_tp][TMDY_PLAYMIDI->set_s_n_t_kn] = f * fst * 1000 + 0.5;
		if (rt)
			for (i = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
				if ((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE) {
					(TMDY_PLAYMIDI->voice)[i].temper_instant = 1;
					recompute_freq(tmdy_struct, i);
				}
		break;
	}
}

static int play_midi(tmdy_struct_ex_t *tmdy_struct, MidiEvent *eventlist, int32 samples)
{
    int rc;
//    static int play_count = 0;

    if ((TMDY_OUTPUT->play_mode)->id_character == 'M') {
	int cnt;

	convert_mod_to_midi_file(tmdy_struct, eventlist);

	TMDY_PLAYMIDI->play_midi_play_count = 0;
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);	/* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
		      "%d memory blocks are free", cnt);
	return RC_NONE;
    }

    (TMDY_PLAYMIDI->sample_count) = samples;
    (TMDY_PLAYMIDI->event_list) = eventlist;
    (TMDY_PLAYMIDI->lost_notes) = (TMDY_PLAYMIDI->cut_notes) = 0;
    (TMDY_PLAYMIDI->check_eot_flag) = 1;

    TMDY_WRD->wrd_midi_event(tmdy_struct, -1, -1); /* For initialize */

    reset_midi(tmdy_struct, 0);
    if(!(TMDY_PLAYMIDI->opt_realtime_playing) &&
       TMDY_RECACHE->allocate_cache_size > 0 &&
       !IS_CURRENT_MOD_FILE &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
    {
	play_midi_prescan(tmdy_struct, eventlist);
	reset_midi(tmdy_struct, 0);
    }

    rc = (TMDY_AQ->aq_flush)(tmdy_struct, 0);
    if(RC_IS_SKIP_FILE(rc))
	return rc;

    skip_to(tmdy_struct, (TMDY_PLAYMIDI->midi_restart_time));

    if((TMDY_PLAYMIDI->midi_restart_time) > 0) { /* Need to update interface display */
      int i;
      for(i = 0; i < MAX_CHANNELS; i++)
	redraw_controllers(tmdy_struct, i);
    }
    rc = RC_NONE;
    for(;;)
    {
	(TMDY_PLAYMIDI->midi_restart_time) = 1;
	rc = play_event(tmdy_struct, (TMDY_PLAYMIDI->current_event));
	if(rc != RC_NONE)
	    break;
	if ((TMDY_PLAYMIDI->midi_restart_time))    /* don't skip the first event if == 0 */
	    (TMDY_PLAYMIDI->current_event)++;
    }

    if(TMDY_PLAYMIDI->play_midi_play_count++ > 3)
    {
	int cnt;
	TMDY_PLAYMIDI->play_midi_play_count = 0;
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct);	/* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE,
		      "%d memory blocks are free", cnt);
    }
    return rc;
}

static void read_header_wav(tmdy_struct_ex_t *tmdy_struct, struct timidity_file* tf)
{
    char buff[44];
    TMDY_COMMON->tf_read(tmdy_struct,  buff, 1, 44, tf);
}

static int read_header_aiff(tmdy_struct_ex_t *tmdy_struct, struct timidity_file* tf)
{
    char buff[5]="    ";
    int i;
    
    for( i=0; i<100; i++ ){
    	buff[0]=buff[1]; buff[1]=buff[2]; buff[2]=buff[3];
    	TMDY_COMMON->tf_read(tmdy_struct,  &buff[3], 1, 1, tf);
    	if( strcmp(buff,"SSND")==0 ){
            /*SSND chunk found */
    	    TMDY_COMMON->tf_read(tmdy_struct,  &buff[0], 1, 4, tf);
    	    TMDY_COMMON->tf_read(tmdy_struct,  &buff[0], 1, 4, tf);
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "aiff header read OK.");
	    return 0;
    	}
    }
    /*SSND chunk not found */
    return -1;
}

static int load_pcm_file_wav(tmdy_struct_ex_t *tmdy_struct)
{
    char *filename;

    if(strcmp((TMDY_PLAYMIDI->pcm_alternate_file), "auto") == 0)
    {
	filename = TMDY_COMMON->safe_malloc(tmdy_struct, strlen(TMDY_READMIDI->current_file_info->filename)+5);
	strcpy(filename, TMDY_READMIDI->current_file_info->filename);
	strcat(filename, ".wav");
    }
    else if(strlen((TMDY_PLAYMIDI->pcm_alternate_file)) >= 5 &&
	    strncasecmp((TMDY_PLAYMIDI->pcm_alternate_file) + strlen((TMDY_PLAYMIDI->pcm_alternate_file)) - 4,
			".wav", 4) == 0)
	filename = TMDY_COMMON->safe_strdup(tmdy_struct, (TMDY_PLAYMIDI->pcm_alternate_file));
    else
	return -1;

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "wav filename: %s", filename);
    TMDY_READMIDI->current_file_info->pcm_tf = TMDY_COMMON->open_file(tmdy_struct, filename, 0, OF_SILENT);
    if( TMDY_READMIDI->current_file_info->pcm_tf ){
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open successed.");
	read_header_wav(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
	TMDY_READMIDI->current_file_info->pcm_filename = filename;
	TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_WAV;
	return 0;
    }else{
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open failed.");
	free(filename);
	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
	return -1;
    }
}

static int load_pcm_file_aiff(tmdy_struct_ex_t *tmdy_struct)
{
    char *filename;

    if(strcmp((TMDY_PLAYMIDI->pcm_alternate_file), "auto") == 0)
    {
	filename = TMDY_COMMON->safe_malloc(tmdy_struct, strlen(TMDY_READMIDI->current_file_info->filename)+6);
	strcpy(filename, TMDY_READMIDI->current_file_info->filename);
	strcat( filename, ".aiff");
    }
    else if(strlen((TMDY_PLAYMIDI->pcm_alternate_file)) >= 6 &&
	    strncasecmp((TMDY_PLAYMIDI->pcm_alternate_file) + strlen((TMDY_PLAYMIDI->pcm_alternate_file)) - 5,
			".aiff", 5) == 0)
	filename = TMDY_COMMON->safe_strdup(tmdy_struct, (TMDY_PLAYMIDI->pcm_alternate_file));
    else
	return -1;

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "aiff filename: %s", filename);
    TMDY_READMIDI->current_file_info->pcm_tf = TMDY_COMMON->open_file(tmdy_struct, filename, 0, OF_SILENT);
    if( TMDY_READMIDI->current_file_info->pcm_tf ){
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open successed.");
	read_header_aiff(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
	TMDY_READMIDI->current_file_info->pcm_filename = filename;
	TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_AIFF;
	return 0;
    }else{
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		      "open failed.");
	free(filename);
	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
	return -1;
    }
}

static void load_pcm_file(tmdy_struct_ex_t *tmdy_struct)
{
    if( load_pcm_file_wav(tmdy_struct)==0 ) return; /*load OK*/
    if( load_pcm_file_aiff(tmdy_struct)==0 ) return; /*load OK*/
}

static int play_midi_load_file(tmdy_struct_ex_t *tmdy_struct, char *fn,
			       MidiEvent **event,
			       int32 *nsamples)
{
    int rc;
    struct timidity_file *tf;
    int32 nevents;

    *event = NULL;

    if(!strcmp(fn, "-"))
	(TMDY_PLAYMIDI->file_from_stdin) = 1;
    else
	(TMDY_PLAYMIDI->file_from_stdin) = 0;

    ctl_mode_event(tmdy_struct, CTLE_NOW_LOADING, 0, (long)fn, 0);
    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "MIDI file: %s", fn);
    if((tf = TMDY_READMIDI->open_midi_file(tmdy_struct, fn, 1, OF_VERBOSE)) == NULL)
    {
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, -1, 0);
	return RC_ERROR;
    }

    *event = NULL;
    rc = check_apply_control(tmdy_struct);
    if(RC_IS_SKIP_FILE(rc))
    {
	TMDY_COMMON->close_file(tmdy_struct, tf);
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 1, 0);
	return rc;
    }

    *event = TMDY_READMIDI->read_midi_file(tmdy_struct, tf, &nevents, nsamples, fn);
    TMDY_COMMON->close_file(tmdy_struct, tf);

    if(*event == NULL)
    {
	ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, -1, 0);
	return RC_ERROR;
    }

    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
	      "%d supported events, %d samples, time %d:%02d",
	      nevents, *nsamples,
	      *nsamples / (TMDY_OUTPUT->play_mode)->rate / 60,
	      (*nsamples / (TMDY_OUTPUT->play_mode)->rate) % 60);

    TMDY_READMIDI->current_file_info->pcm_mode = PCM_MODE_NON; /*initialize*/
    if((TMDY_PLAYMIDI->pcm_alternate_file) != NULL &&
       strcmp((TMDY_PLAYMIDI->pcm_alternate_file), "none") != 0 &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
	load_pcm_file(tmdy_struct);

    if(!IS_CURRENT_MOD_FILE &&
       ((TMDY_OUTPUT->play_mode)->flag&PF_PCM_STREAM))
    {
	/* FIXME: Instruments is not need for pcm_alternate_file. */

	/* Load instruments
	 * If (TMDY_PLAYMIDI->opt_realtime_playing), the instruments will be loaded later.
	 */
	if(!(TMDY_PLAYMIDI->opt_realtime_playing))
	{
	    rc = RC_NONE;
	    TMDY_INSTRUM->load_missing_instruments(tmdy_struct, &rc);
	    if(RC_IS_SKIP_FILE(rc))
	    {
		/* Instrument loading is terminated */
		ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 1, 0);
		TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);
		return rc;
	    }
	}
    }
    else
	TMDY_INSTRUM->clear_magic_instruments(tmdy_struct);	/* Clear load markers */

    ctl_mode_event(tmdy_struct, CTLE_LOADING_DONE, 0, 0, 0);

    return RC_NONE;
}

int play_midi_file(tmdy_struct_ex_t *tmdy_struct, char *fn)
{
    int i, j, rc;
//     static int last_rc = RC_NONE;
   MidiEvent *event;
    int32 nsamples;

    /* Set current file information */
    TMDY_READMIDI->current_file_info = TMDY_READMIDI->get_midi_file_info(tmdy_struct, fn, 1);

    rc = check_apply_control(tmdy_struct);
    if(RC_IS_SKIP_FILE(rc) && rc != RC_RELOAD)
	return rc;

    /* Reset key & speed each files */
    (TMDY_PLAYMIDI->current_keysig) = (TMDY_PLAYMIDI->current_temper_keysig) = (TMDY_PLAYMIDI->opt_init_keysig);
    (TMDY_PLAYMIDI->note_key_offset) = 0;
    (TMDY_PLAYMIDI->midi_time_ratio) = 1.0;
	for (i = 0; i < MAX_CHANNELS; i++) {
		for (j = 0; j < 12; j++)
			(TMDY_PLAYMIDI->channel)[i].scale_tuning[j] = 0;
		(TMDY_PLAYMIDI->channel)[i].prev_scale_tuning = 0;
		(TMDY_PLAYMIDI->channel)[i].temper_type = 0;
	}
    CLEAR_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));
	if ((TMDY_PLAYMIDI->temper_type_mute) & 1)
		FILL_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));

    /* Reset restart offset */
    (TMDY_PLAYMIDI->midi_restart_time) = 0;

#ifdef REDUCE_VOICE_TIME_TUNING
    /* Reset voice reduction stuff */
    (TMDY_PLAYMIDI->min_bad_nv) = 256;
    (TMDY_PLAYMIDI->max_good_nv) = 1;
    (TMDY_PLAYMIDI->ok_nv_total) = 32;
    (TMDY_PLAYMIDI->ok_nv_counts) = 1;
    (TMDY_PLAYMIDI->ok_nv) = 32;
    (TMDY_PLAYMIDI->ok_nv_sample) = 0;
    (TMDY_PLAYMIDI->old_rate) = -1;
    (TMDY_PLAYMIDI->reduce_quality_flag) = (TMDY_PLAYMIDI->no_4point_interpolation);
    restore_voices(tmdy_struct, 0);
#endif

	ctl_mode_event(tmdy_struct, CTLE_METRONOME, 0, 0, 0);
	ctl_mode_event(tmdy_struct, CTLE_KEYSIG, 0, (TMDY_PLAYMIDI->current_keysig), 0);
	ctl_mode_event(tmdy_struct, CTLE_TEMPER_KEYSIG, 0, (TMDY_PLAYMIDI->current_temper_keysig), 0);
	if ((TMDY_PLAYMIDI->opt_force_keysig) != 8) {
		i = (TMDY_PLAYMIDI->current_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : -6);
		j = (TMDY_PLAYMIDI->opt_force_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : 10);
		while (i != j && i != j + 12) {
			if (++(TMDY_PLAYMIDI->note_key_offset) > 6)
				(TMDY_PLAYMIDI->note_key_offset) -= 12;
			i += (i > 10) ? -5 : 7;
		}
	}
	ctl_mode_event(tmdy_struct, CTLE_KEY_OFFSET, 0, (TMDY_PLAYMIDI->note_key_offset), 0);
	i = (TMDY_PLAYMIDI->current_keysig) + (((TMDY_PLAYMIDI->current_keysig) < 8) ? 7 : -9), j = 0;
	while (i != 7 && i != 19)
		i += (i < 7) ? 5 : -7, j++;
	j += (TMDY_PLAYMIDI->note_key_offset), j -= floor(j / 12.0) * 12;
	(TMDY_PLAYMIDI->current_freq_table) = j;
	ctl_mode_event(tmdy_struct, CTLE_TEMPO, 0, (TMDY_PLAYMIDI->current_play_tempo), 0);
	ctl_mode_event(tmdy_struct, CTLE_TIME_RATIO, 0, 100 / (TMDY_PLAYMIDI->midi_time_ratio) + 0.5, 0);
	for (i = 0; i < MAX_CHANNELS; i++) {
		ctl_mode_event(tmdy_struct, CTLE_TEMPER_TYPE, 0, i, (TMDY_PLAYMIDI->channel)[i].temper_type);
		ctl_mode_event(tmdy_struct, CTLE_MUTE, 0, i, (TMDY_PLAYMIDI->temper_type_mute) & 1);
	}
  play_reload: /* Come here to reload MIDI file */
    rc = play_midi_load_file(tmdy_struct, fn, &event, &nsamples);
    if(RC_IS_SKIP_FILE(rc))
	goto play_end; /* skip playing */

//    TMDY_UTILS->mblock->init_mblock(tmdy_struct, &TMDY_PLAYMIDI->playmidi_pool);
    ctl_mode_event(tmdy_struct, CTLE_PLAY_START, 0, nsamples, 0);
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_START, NULL);
    rc = play_midi(tmdy_struct, event, nsamples);
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_END, NULL);
    ctl_mode_event(tmdy_struct, CTLE_PLAY_END, 0, 0, 0);
    TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_PLAYMIDI->playmidi_pool);

    for(i = 0; i < MAX_CHANNELS; i++)
	memset((TMDY_PLAYMIDI->channel)[i].drums, 0, sizeof((TMDY_PLAYMIDI->channel)[i].drums));

  play_end:
    if(TMDY_READMIDI->current_file_info->pcm_tf){
    	TMDY_COMMON->close_file(tmdy_struct, TMDY_READMIDI->current_file_info->pcm_tf);
    	TMDY_READMIDI->current_file_info->pcm_tf = NULL;
    	free( TMDY_READMIDI->current_file_info->pcm_filename );
    	TMDY_READMIDI->current_file_info->pcm_filename = NULL;
    }
    
    if(TMDY_WRD->wrdt->opened)
	TMDY_WRD->wrdt->end(tmdy_struct);

    if((TMDY_MAIN->free_instruments_afterwards))
    {
	int cnt;
	TMDY_INSTRUM->free_instruments(tmdy_struct, 0);
	cnt = TMDY_UTILS->mblock->free_global_mblock(tmdy_struct); /* free unused memory */
	if(cnt > 0)
	    (TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_VERBOSE, "%d memory blocks are free",
		      cnt);
    }

    TMDY_INSTRUM->free_special_patch(tmdy_struct, -1);

    if(event != NULL)
	free(event);
    if(rc == RC_RELOAD)
	goto play_reload;

    if(rc == RC_ERROR)
    {
	if(TMDY_READMIDI->current_file_info->file_type == IS_OTHER_FILE)
	    TMDY_READMIDI->current_file_info->file_type = IS_ERROR_FILE;
	if(TMDY_PLAYMIDI->last_rc == RC_REALLY_PREVIOUS)
	    return RC_REALLY_PREVIOUS;
    }
    TMDY_PLAYMIDI->last_rc = rc;
    return rc;
}

void dumb_pass_playing_list(tmdy_struct_ex_t *tmdy_struct, int number_of_files, char *list_of_files[])
{
    int i = 0;

    for(;;)
    {
	switch(play_midi_file(tmdy_struct, list_of_files[i]))
	{
	  case RC_REALLY_PREVIOUS:
	    if(i > 0)
		i--;
	    break;

	  default: /* An error or something */
	  case RC_NEXT:
	    if(i < number_of_files-1)
	    {
		i++;
		break;
	    }
	    (TMDY_AQ->aq_flush)(tmdy_struct, 0);

	    if(!((TMDY_CONTROLS->ctl)->flags & CTLF_LIST_LOOP))
		return;
	    i = 0;
	    break;

	    case RC_QUIT:
		return;
	}
    }
}

void default_ctl_lyric(tmdy_struct_ex_t *tmdy_struct, int lyricid)
{
    char *lyric;

    lyric = TMDY_READMIDI->event2string(tmdy_struct, lyricid);
    if(lyric != NULL)
	(TMDY_CONTROLS->ctl)->cmsg(tmdy_struct, CMSG_TEXT, VERB_VERBOSE, "%s", lyric + 1);
}

void ctl_mode_event(tmdy_struct_ex_t *tmdy_struct, int type, int trace, long arg1, long arg2)
{
    CtlEvent ce;
    ce.type = type;
    ce.v1 = arg1;
    ce.v2 = arg2;
    if(trace && (TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

void ctl_note_event(tmdy_struct_ex_t *tmdy_struct, int noteID)
{
    CtlEvent ce;
    ce.type = CTLE_NOTE;
    ce.v1 = (TMDY_PLAYMIDI->voice)[noteID].status;
    ce.v2 = (TMDY_PLAYMIDI->voice)[noteID].channel;
    ce.v3 = (TMDY_PLAYMIDI->voice)[noteID].note;
    ce.v4 = (TMDY_PLAYMIDI->voice)[noteID].velocity;
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_timestamp(tmdy_struct_ex_t *tmdy_struct)
{
    long i, secs, voices;
    CtlEvent ce;
//    static int TMDY_PLAYMIDI->last_secs = -1, TMDY_PLAYMIDI->last_voices = -1;

    secs = (long)((TMDY_PLAYMIDI->current_sample) / ((TMDY_PLAYMIDI->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    for(i = voices = 0; i < (TMDY_PLAYMIDI->upper_voices); i++)
	if((TMDY_PLAYMIDI->voice)[i].status != VOICE_FREE)
	    voices++;
    if(secs == TMDY_PLAYMIDI->last_secs && voices == TMDY_PLAYMIDI->last_voices)
	return;
    ce.type = CTLE_CURRENT_TIME;
    ce.v1 = TMDY_PLAYMIDI->last_secs = secs;
    ce.v2 = TMDY_PLAYMIDI->last_voices = voices;
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_updatetime(tmdy_struct_ex_t *tmdy_struct, int32 samples)
{
    long secs;
    secs = (long)(samples / ((TMDY_PLAYMIDI->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    ctl_mode_event(tmdy_struct, CTLE_CURRENT_TIME, 0, secs, 0);
    ctl_mode_event(tmdy_struct, CTLE_REFRESH, 0, 0, 0);
}

static void ctl_prog_event(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    CtlEvent ce;
    int bank, prog;

    if(IS_CURRENT_MOD_FILE)
    {
	bank = 0;
	prog = (TMDY_PLAYMIDI->channel)[ch].special_sample;
    }
    else
    {
	bank = (TMDY_PLAYMIDI->channel)[ch].bank;
	prog = (TMDY_PLAYMIDI->channel)[ch].program;
    }

    ce.type = CTLE_PROGRAM;
    ce.v1 = ch;
    ce.v2 = prog;
    ce.v3 = (long)channel_instrum_name(tmdy_struct, ch);
    ce.v4 = (bank |
	     ((TMDY_PLAYMIDI->channel)[ch].bank_lsb << 8) |
	     ((TMDY_PLAYMIDI->channel)[ch].bank_msb << 16));
    if((TMDY_CONTROLS->ctl)->trace_playing)
	TMDY_MIDITRACE->push_midi_trace_ce(tmdy_struct, (TMDY_CONTROLS->ctl)->event, &ce);
    else
	(TMDY_CONTROLS->ctl)->event(tmdy_struct, &ce);
}

static void ctl_pause_event(tmdy_struct_ex_t *tmdy_struct, int pause, int32 s)
{
    long secs;
    secs = (long)(s / ((TMDY_PLAYMIDI->midi_time_ratio) * (TMDY_OUTPUT->play_mode)->rate));
    ctl_mode_event(tmdy_struct, CTLE_PAUSE, 0, pause, secs);
}

char *channel_instrum_name(tmdy_struct_ex_t *tmdy_struct, int ch)
{
    char *comm;
    int bank, prog;

    if(ISDRUMCHANNEL(ch)) {
	bank = (TMDY_PLAYMIDI->channel)[ch].bank;
	if (TMDY_INSTRUM->drumset[bank] == NULL) return "";
	prog = 0;
	comm = TMDY_INSTRUM->drumset[bank]->tone[prog].comment;
	if (comm == NULL) return "";
	return comm;
    }

    if((TMDY_PLAYMIDI->channel)[ch].program == SPECIAL_PROGRAM)
	return "Special Program";

    if(IS_CURRENT_MOD_FILE)
    {
	int pr;
	pr = (TMDY_PLAYMIDI->channel)[ch].special_sample;
	if(pr > 0 &&
	   TMDY_INSTRUM->special_patch[pr] != NULL &&
	   TMDY_INSTRUM->special_patch[pr]->name != NULL)
	    return TMDY_INSTRUM->special_patch[pr]->name;
	return "MOD";
    }

    bank = (TMDY_PLAYMIDI->channel)[ch].bank;
    prog = (TMDY_PLAYMIDI->channel)[ch].program;
    TMDY_INSTRUM->instrument_map(tmdy_struct, (TMDY_PLAYMIDI->channel)[ch].mapID, &bank, &prog);
    if(TMDY_INSTRUM->tonebank[bank] == NULL)
	bank = 0;
    comm = TMDY_INSTRUM->tonebank[bank]->tone[prog].comment;
    if(comm == NULL)
	comm = TMDY_INSTRUM->tonebank[0]->tone[prog].comment;
    return comm;
}


/*
 * For MIDI stream player.
 */
void playmidi_stream_init(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
//    static int TMDY_PLAYMIDI->p_m_s_i_first = 1;

    (TMDY_PLAYMIDI->note_key_offset) = 0;
    (TMDY_PLAYMIDI->midi_time_ratio) = 1.0;
    CLEAR_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));
	if ((TMDY_PLAYMIDI->temper_type_mute) & 1)
		FILL_CHANNELMASK((TMDY_PLAYMIDI->channel_mute));
    (TMDY_PLAYMIDI->midi_restart_time) = 0;
    if(TMDY_PLAYMIDI->p_m_s_i_first)
    {
	TMDY_PLAYMIDI->p_m_s_i_first = 0;
//        TMDY_UTILS->mblock->init_mblock(tmdy_struct, &TMDY_PLAYMIDI->playmidi_pool);
	TMDY_READMIDI->current_file_info = TMDY_READMIDI->get_midi_file_info(tmdy_struct, "TiMidity", 1);
    TMDY_PLAYMIDI->midi_streaming=1;
    }
    else
        TMDY_UTILS->mblock->reuse_mblock(tmdy_struct, &TMDY_PLAYMIDI->playmidi_pool);

    /* Fill in TMDY_READMIDI->current_file_info */
    TMDY_READMIDI->current_file_info->readflag = 1;
    TMDY_READMIDI->current_file_info->seq_name = TMDY_COMMON->safe_strdup(tmdy_struct, "TiMidity server");
    TMDY_READMIDI->current_file_info->karaoke_title = TMDY_READMIDI->current_file_info->first_text = NULL;
    TMDY_READMIDI->current_file_info->mid = 0x7f;
    TMDY_READMIDI->current_file_info->hdrsiz = 0;
    TMDY_READMIDI->current_file_info->format = 0;
    TMDY_READMIDI->current_file_info->tracks = 0;
    TMDY_READMIDI->current_file_info->divisions = 192; /* ?? */
    TMDY_READMIDI->current_file_info->time_sig_n = 4; /* 4/ */
    TMDY_READMIDI->current_file_info->time_sig_d = 4; /* /4 */
    TMDY_READMIDI->current_file_info->time_sig_c = 24; /* clock */
    TMDY_READMIDI->current_file_info->time_sig_b = 8;  /* q.n. */
    TMDY_READMIDI->current_file_info->samples = 0;
    TMDY_READMIDI->current_file_info->max_channel = MAX_CHANNELS;
    TMDY_READMIDI->current_file_info->compressed = 0;
    TMDY_READMIDI->current_file_info->midi_data = NULL;
    TMDY_READMIDI->current_file_info->midi_data_size = 0;
    TMDY_READMIDI->current_file_info->file_type = IS_OTHER_FILE;

    (TMDY_PLAYMIDI->current_play_tempo) = 500000;
    (TMDY_PLAYMIDI->check_eot_flag) = 0;

    /* Setup default drums */
    COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannels), TMDY_READMIDI->current_file_info->drumchannels);
    COPY_CHANNELMASK((TMDY_PLAYMIDI->drumchannel_mask), TMDY_READMIDI->current_file_info->drumchannel_mask);
    for(i = 0; i < MAX_CHANNELS; i++)
	memset((TMDY_PLAYMIDI->channel)[i].drums, 0, sizeof((TMDY_PLAYMIDI->channel)[i].drums));
	TMDY_READMIDI->change_system_mode(tmdy_struct, DEFAULT_SYSTEM_MODE);
    reset_midi(tmdy_struct, 0);

    playmidi_tmr_reset(tmdy_struct);
}

void playmidi_tmr_reset(tmdy_struct_ex_t *tmdy_struct)
{
    int i;

    (TMDY_AQ->aq_flush)(tmdy_struct, 0);
    (TMDY_PLAYMIDI->current_sample) = 0;
    (TMDY_PLAYMIDI->buffered_count) = 0;
    (TMDY_PLAYMIDI->buffer_pointer) = (TMDY_PLAYMIDI->common_buffer);
    for(i = 0; i < MAX_CHANNELS; i++)
	(TMDY_PLAYMIDI->channel)[i].lasttime = 0;
    (TMDY_OUTPUT->play_mode)->acntl(tmdy_struct, PM_REQ_PLAY_START, NULL);
	    ;
}
	
	
	


playmidi_ex_t* new_playmidi(tmdy_struct_ex_t *tmdy_struct){
	playmidi_ex_t* playmidi_ex;

	playmidi_ex=(playmidi_ex_t *)TMDY_COMMON->safe_malloc(tmdy_struct, sizeof(playmidi_ex_t));


	playmidi_ex->control_ratio=0;
	playmidi_ex->amplification=DEFAULT_AMPLIFICATION;
	playmidi_ex->adjust_panning_immediately=1;
	playmidi_ex->voices=DEFAULT_VOICES;
	playmidi_ex->note_key_offset = 0;	/* For key up/down */
	playmidi_ex->midi_time_ratio = 1.0;	/* For speed up/down */
#ifdef MODULATION_WHEEL_ALLOW
	playmidi_ex->opt_modulation_wheel = 1;
#else
	playmidi_ex->opt_modulation_wheel = 0;
#endif /* MODULATION_WHEEL_ALLOW */

#ifdef PORTAMENTO_ALLOW
	playmidi_ex->opt_portamento = 1;
#else
	playmidi_ex->opt_portamento = 0;
#endif /* PORTAMENTO_ALLOW */

#ifdef NRPN_VIBRATO_ALLOW
	playmidi_ex->opt_nrpn_vibrato = 1;
#else
	playmidi_ex->opt_nrpn_vibrato = 0;
#endif /* NRPN_VIBRATO_ALLOW */
#ifdef REVERB_CONTROL_ALLOW
	playmidi_ex->opt_reverb_control = 1;
#else
#ifdef FREEVERB_CONTROL_ALLOW
	playmidi_ex->opt_reverb_control = 3;
#else
	playmidi_ex->opt_reverb_control = 0;
#endif /* FREEVERB_CONTROL_ALLOW */
#endif /* REVERB_CONTROL_ALLOW */
#ifdef CHORUS_CONTROL_ALLOW
	playmidi_ex->opt_chorus_control = 1;
#else
	playmidi_ex->opt_chorus_control = 0;
#endif /* CHORUS_CONTROL_ALLOW */
#ifdef SURROUND_CHORUS_ALLOW
	playmidi_ex->opt_surround_chorus = 1;
#else
	playmidi_ex->opt_surround_chorus = 0;
#endif /* SURROUND_CHORUS_ALLOW */
#ifdef GM_CHANNEL_PRESSURE_ALLOW
	playmidi_ex->opt_channel_pressure = 1;
#else
	playmidi_ex->opt_channel_pressure = 0;
#endif /* GM_CHANNEL_PRESSURE_ALLOW */
#ifdef VOICE_BY_VOICE_LPF_ALLOW
	playmidi_ex->opt_lpf_def = 1;
#else
	playmidi_ex->opt_lpf_def = 0;
#endif /* VOICE_BY_VOICE_LPF_ALLOW */
#ifdef OVERLAP_VOICE_ALLOW
	playmidi_ex->opt_overlap_voice_allow = 1;
#else
	playmidi_ex->opt_overlap_voice_allow = 0;
#endif /* OVERLAP_VOICE_ALLOW */
#ifdef TEMPER_CONTROL_ALLOW
	playmidi_ex->opt_temper_control = 1;
#else
	playmidi_ex->opt_temper_control = 0;
#endif /* TEMPER_CONTROL_ALLOW */
	playmidi_ex->opt_tva_attack = 0;	/* attack envelope control */
	playmidi_ex->opt_tva_decay = 0;	/* decay envelope control */
	playmidi_ex->opt_tva_release = 0;	/* release envelope control */
	playmidi_ex->opt_delay_control = 0;	/* CC#94 delay(celeste) effect control */
	playmidi_ex->opt_eq_control = 0;		/* channel equalizer control */
	playmidi_ex->opt_insertion_effect = 0;	/* insertion effect control */
	playmidi_ex->opt_drum_effect = 0;	/* drumpart effect control */
	playmidi_ex->opt_drum_power = 100;		/* coef. of drum amplitude */
	playmidi_ex->opt_amp_compensation = 0;
	playmidi_ex->opt_modulation_envelope = 0;
	playmidi_ex->current_play_tempo = 500000;
	playmidi_ex->opt_realtime_playing = 0;	
	playmidi_ex->reduce_voice_threshold = -1; /* msec */
	playmidi_ex->special_tonebank = -1;
	playmidi_ex->default_tonebank = 0;
	playmidi_ex->playmidi_seek_flag = 0;
	playmidi_ex->play_pause_flag = 0;
	playmidi_ex->auto_reduce_polyphony=1;
	playmidi_ex->reduce_quality_flag=0;
	playmidi_ex->no_4point_interpolation=0;
	playmidi_ex->current_keysig = 0;
	playmidi_ex->current_temper_keysig = 0;
	playmidi_ex->opt_init_keysig = 0;
	playmidi_ex->opt_force_keysig = 8;
	playmidi_ex->key_adjust = 0;
	playmidi_ex->opt_pure_intonation = 0;
	playmidi_ex->current_freq_table = 0;

	playmidi_ex->play_midi_file=play_midi_file;
	playmidi_ex->dumb_pass_playing_list=dumb_pass_playing_list;
	playmidi_ex->default_ctl_lyric=default_ctl_lyric;
	playmidi_ex->check_apply_control=check_apply_control;
	playmidi_ex->recompute_freq=recompute_freq;
	playmidi_ex->midi_drumpart_change=midi_drumpart_change;
	playmidi_ex->ctl_note_event=ctl_note_event;
	playmidi_ex->ctl_mode_event=ctl_mode_event;
	playmidi_ex->channel_instrum_name=channel_instrum_name;
	playmidi_ex->get_reverb_level=get_reverb_level;
	playmidi_ex->get_chorus_level=get_chorus_level;
	playmidi_ex->playmidi_output_changed=playmidi_output_changed;
	playmidi_ex->play_midi_load_instrument=play_midi_load_instrument;
	playmidi_ex->midi_program_change=midi_program_change;
	playmidi_ex->free_voice=free_voice;
	playmidi_ex->play_midi_setup_drums=play_midi_setup_drums;

/* For stream player */
	playmidi_ex->playmidi_stream_init=playmidi_stream_init;
	playmidi_ex->playmidi_tmr_reset=playmidi_tmr_reset;
	playmidi_ex->play_event=play_event;

	playmidi_ex->dup_tone_bank_element=dup_tone_bank_element;
	playmidi_ex->free_tone_bank_element=free_tone_bank_element;

	playmidi_ex->recompute_voice_filter=recompute_voice_filter;


/**** private valiables  ****/
#ifdef REDUCE_VOICE_TIME_TUNING
	
	playmidi_ex->max_good_nv = 1;
	playmidi_ex->min_bad_nv = 256;
	playmidi_ex->ok_nv_total = 32;
	playmidi_ex->ok_nv_counts = 1;
	playmidi_ex->ok_nv_sample = 0;
	playmidi_ex->ok_nv = 32;
	playmidi_ex->old_rate = -1;
#endif
	playmidi_ex->midi_streaming=0;
	playmidi_ex->stream_max_compute=500; /* compute time limit (in msec) when streaming */
	playmidi_ex->midi_restart_time = 0;
	TMDY_UTILS->mblock->init_mblock(tmdy_struct, &(playmidi_ex->playmidi_pool));
	playmidi_ex->make_rvid_flag = 0; /* For reverb optimization */
	/* Ring voice id for each notes.  This ID enables duplicated note. */
	playmidi_ex->voices=DEFAULT_VOICES;
	playmidi_ex->master_volume_ratio = 0xFFFF;
//	playmidi_ex->envelope_modify_rate = 1.0;

	playmidi_ex->pcm_alternate_file = NULL; /* NULL or "none": Nothing (default)
				  * "auto": Auto select
				  * filename: Use it
				  */
	playmidi_ex->reverb_buffer = NULL; /* MAX_CHANNELS*AUDIO_BUFFER_SIZE*8 */
	playmidi_ex->compensation_ratio = 1.0; /* compensation ratio */
	
//	playmidi_ex->common_buffer=malloc(sizeof(int32)*(AUDIO_BUFFER_SIZE*2));

	playmidi_ex->restore_voices_old_voices=-1;
	playmidi_ex->play_midi_play_count=0;
	playmidi_ex->last_rc = RC_NONE;
	playmidi_ex->last_secs = -1;
	playmidi_ex->last_voices = -1;
	playmidi_ex->p_m_s_i_first = 1;	
	
	return playmidi_ex;
}


void destroy_playmidi(playmidi_ex_t* playmidi){
	free(playmidi);
}

