/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2003 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include <stdio.h>
#if STDC_HEADERS
#include <stdlib.h>
#include <ctype.h>
#include <stddef.h>
#endif
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#ifdef __W32__
#include <windows.h>
#include <io.h>
#include <shlobj.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <fcntl.h> /* for open */

#ifdef HAVE_STDBOOL_H
#include <stdbool.h>
#endif

#ifndef __bool_true_false_are_defined
# ifdef bool
#  undef bool
# endif
# ifdef ture
#  undef ture
# endif
# ifdef false
#  undef false
# endif
# define bool int
# define false ((bool)0)
# define true (!false)
# define __bool_true_false_are_defined true
#endif /* C99 _Bool hack */

#ifdef BORLANDC_EXCEPTION
#include <excpt.h>
#endif /* BORLANDC_EXCEPTION */
#include <signal.h>

#if defined(__FreeBSD__) && !defined(__alpha__)
#include <floatingpoint.h> /* For FP exceptions */
#endif
#if defined(__NetBSD__) || defined(__OpenBSD__)
#include <ieeefp.h> /* For FP exceptions */
#endif

#include "tmdy_struct.h"
#include "interface.h"
#include "timidity.h"
#include "utils/tmdy_getopt.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "controls.h"
#include "tables.h"
#include "miditrace.h"
#include "reverb.h"
#ifdef SUPPORT_SOUNDSPEC
#include "soundspec.h"
#endif /* SUPPORT_SOUNDSPEC */
#include "resample.h"
#include "recache.h"
#include "arc.h"
#include "strtab.h"
#include "wrd.h"
#define DEFINE_GLOBALS
//#include "mid.defs" guha
#include "aq.h"
#include "mix.h"
#include "../libunimod/unimod.h"
#include "quantity.h"

#ifdef IA_W32GUI
#include "w32g.h"
#include "w32g_utl.h"
#endif

#ifndef __GNUC__
#define __attribute__(x) /* ignore */
#endif



#include "timidity_main.h"
#if defined(main) || defined(ANOTHER_MAIN)
#define MAIN_INTERFACE
#else
#define MAIN_INTERFACE static
#endif /* main */
#include "timidity_main_prv.h"



/* option enums */
enum {
	TIM_OPT_FIRST = 256,
	/* first entry */
	TIM_OPT_VOLUME = TIM_OPT_FIRST,
	TIM_OPT_DRUM_POWER,
	TIM_OPT_VOLUME_COMP,
	TIM_OPT_ANTI_ALIAS,
	TIM_OPT_BUFFER_FRAGS,
	TIM_OPT_CONTROL_RATIO,
	TIM_OPT_CONFIG_FILE,
	TIM_OPT_DRUM_CHANNEL,
	TIM_OPT_IFACE_PATH,
	TIM_OPT_EXT,
	TIM_OPT_MOD_WHEEL,
	TIM_OPT_PORTAMENTO,
	TIM_OPT_VIBRATO,
	TIM_OPT_CH_PRESS,
	TIM_OPT_VOICE_LPF,
	TIM_OPT_MOD_ENV,
	TIM_OPT_TRACE_TEXT,
	TIM_OPT_OVERLAP,
	TIM_OPT_TEMPER_CTRL,
	TIM_OPT_DEFAULT_MID,
	TIM_OPT_SYSTEM_MID,
	TIM_OPT_DEFAULT_BANK,
	TIM_OPT_FORCE_BANK,
	TIM_OPT_DEFAULT_PGM,
	TIM_OPT_FORCE_PGM,
	TIM_OPT_DELAY,
	TIM_OPT_CHORUS,
	TIM_OPT_REVERB,
	TIM_OPT_NS,
	TIM_OPT_RESAMPLE,
	TIM_OPT_EVIL,
	TIM_OPT_FAST_PAN,
	TIM_OPT_FAST_DECAY,
	TIM_OPT_SPECTROGRAM,
	TIM_OPT_KEYSIG,
	TIM_OPT_HELP,
	TIM_OPT_INTERFACE,
	TIM_OPT_VERBOSE,
	TIM_OPT_QUIET,
	TIM_OPT_TRACE,
	TIM_OPT_LOOP,
	TIM_OPT_RANDOM,
	TIM_OPT_SORT,
	TIM_OPT_BACKGROUND,
	TIM_OPT_RT_PRIO,
	TIM_OPT_SEQ_PORTS,
	TIM_OPT_REALTIME_LOAD,
	TIM_OPT_ADJUST_KEY,
	TIM_OPT_VOICE_QUEUE,
	TIM_OPT_PATCH_PATH,
	TIM_OPT_PCM_FILE,
	TIM_OPT_DECAY_TIME,
	TIM_OPT_INTERPOLATION,
	TIM_OPT_OUTPUT_MODE,
	TIM_OPT_OUTPUT_STEREO,
	TIM_OPT_OUTPUT_SIGNED,
	TIM_OPT_OUTPUT_16BIT,
	TIM_OPT_OUTPUT_FORMAT,
	TIM_OPT_OUTPUT_SWAB,
	TIM_OPT_OUTPUT_FILE,
	TIM_OPT_PATCH_FILE,
	TIM_OPT_POLYPHONY,
	TIM_OPT_POLY_REDUCE,
	TIM_OPT_MUTE,
	TIM_OPT_TEMPER_MUTE,
	TIM_OPT_AUDIO_BUFFER,
	TIM_OPT_CACHE_SIZE,
	TIM_OPT_SAMPLE_FREQ,
	TIM_OPT_ADJUST_TEMPO,
	TIM_OPT_CHARSET,
	TIM_OPT_UNLOAD_INST,
	TIM_OPT_VERSION,
	TIM_OPT_WRD,
	TIM_OPT_RCPCV_DLL,
	TIM_OPT_CONFIG_STR,
	TIM_OPT_FREQ_TABLE,
	TIM_OPT_PURE_INT,
	/* last entry */
	TIM_OPT_LAST = TIM_OPT_PURE_INT
};

static const char *optcommands =
		"4A:aB:b:C:c:D:d:E:eFfg:H:hI:i:jK:k:L:M:m:N:"
		"O:o:P:p:Q:q:R:S:s:T:t:UvW:"
#ifdef __W32__
		"w:"
#endif
		"x:Z:";		/* Only GJlnruVXYyz are remain... */
static const struct option longopts[] = {
	{ "volume",                 required_argument, NULL, TIM_OPT_VOLUME },
	{ "drum-power",             required_argument, NULL, TIM_OPT_DRUM_POWER },
	{ "no-volume-compensation", no_argument,       NULL, TIM_OPT_DRUM_POWER },
	{ "volume-compensation",    optional_argument, NULL, TIM_OPT_VOLUME_COMP },
	{ "no-anti-alias",          no_argument,       NULL, TIM_OPT_ANTI_ALIAS },
	{ "anti-alias",             optional_argument, NULL, TIM_OPT_ANTI_ALIAS },
	{ "buffer-fragments",       required_argument, NULL, TIM_OPT_BUFFER_FRAGS },
	{ "control-ratio",          required_argument, NULL, TIM_OPT_CONTROL_RATIO },
	{ "config-file",            required_argument, NULL, TIM_OPT_CONFIG_FILE },
	{ "drum-channel",           required_argument, NULL, TIM_OPT_DRUM_CHANNEL },
	{ "interface-path",         required_argument, NULL, TIM_OPT_IFACE_PATH },
	{ "ext",                    required_argument, NULL, TIM_OPT_EXT },
	{ "no-mod-wheel",           no_argument,       NULL, TIM_OPT_MOD_WHEEL },
	{ "mod-wheel",              optional_argument, NULL, TIM_OPT_MOD_WHEEL },
	{ "no-portamento",          no_argument,       NULL, TIM_OPT_PORTAMENTO },
	{ "portamento",             optional_argument, NULL, TIM_OPT_PORTAMENTO },
	{ "no-vibrato",             no_argument,       NULL, TIM_OPT_VIBRATO },
	{ "vibrato",                optional_argument, NULL, TIM_OPT_VIBRATO },
	{ "no-ch-pressure",         no_argument,       NULL, TIM_OPT_CH_PRESS },
	{ "ch-pressure",            optional_argument, NULL, TIM_OPT_CH_PRESS },
	{ "no-voice-lpf",           no_argument,       NULL, TIM_OPT_VOICE_LPF },
	{ "voice-lpf",              optional_argument, NULL, TIM_OPT_VOICE_LPF },
	{ "no-mod-envelope",        no_argument,       NULL, TIM_OPT_MOD_ENV },
	{ "mod-envelope",           optional_argument, NULL, TIM_OPT_MOD_ENV },
	{ "no-trace-text-meta",     no_argument,       NULL, TIM_OPT_TRACE_TEXT },
	{ "trace-text-meta",        optional_argument, NULL, TIM_OPT_TRACE_TEXT },
	{ "no-overlap-voice",       no_argument,       NULL, TIM_OPT_OVERLAP },
	{ "overlap-voice",          optional_argument, NULL, TIM_OPT_OVERLAP },
	{ "no-temper-control",      no_argument,       NULL, TIM_OPT_TEMPER_CTRL },
	{ "temper-control",         optional_argument, NULL, TIM_OPT_TEMPER_CTRL },
	{ "default-mid",            required_argument, NULL, TIM_OPT_DEFAULT_MID },
	{ "system-mid",             required_argument, NULL, TIM_OPT_SYSTEM_MID },
	{ "default-bank",           required_argument, NULL, TIM_OPT_DEFAULT_BANK },
	{ "force-bank",             required_argument, NULL, TIM_OPT_FORCE_BANK },
	{ "default-program",        required_argument, NULL, TIM_OPT_DEFAULT_PGM },
	{ "force-program",          required_argument, NULL, TIM_OPT_FORCE_PGM },
	{ "delay",                  required_argument, NULL, TIM_OPT_DELAY },
	{ "chorus",                 required_argument, NULL, TIM_OPT_CHORUS },
	{ "reverb",                 required_argument, NULL, TIM_OPT_REVERB },
	{ "noise-shaping",          required_argument, NULL, TIM_OPT_NS },
#ifndef FIXED_RESAMPLATION
	{ "resample",               required_argument, NULL, TIM_OPT_RESAMPLE },
#endif
	{ "evil",                   required_argument, NULL, TIM_OPT_EVIL },
	{ "no-fast-panning",        no_argument,       NULL, TIM_OPT_FAST_PAN },
	{ "fast-panning",           optional_argument, NULL, TIM_OPT_FAST_PAN },
	{ "no-fast-decay",          no_argument,       NULL, TIM_OPT_FAST_DECAY },
	{ "fast-decay",             optional_argument, NULL, TIM_OPT_FAST_DECAY },
	{ "spectrogram",            required_argument, NULL, TIM_OPT_SPECTROGRAM },
	{ "force-keysig",           required_argument, NULL, TIM_OPT_KEYSIG },
	{ "help",                   optional_argument, NULL, TIM_OPT_HELP },
	{ "interface",              required_argument, NULL, TIM_OPT_INTERFACE },
	{ "verbose",                optional_argument, NULL, TIM_OPT_VERBOSE },
	{ "quiet",                  optional_argument, NULL, TIM_OPT_QUIET },
	{ "no-trace",               no_argument,       NULL, TIM_OPT_TRACE },
	{ "trace",                  optional_argument, NULL, TIM_OPT_TRACE },
	{ "no-loop",                no_argument,       NULL, TIM_OPT_LOOP },
	{ "loop",                   optional_argument, NULL, TIM_OPT_LOOP },
	{ "no-random",              no_argument,       NULL, TIM_OPT_RANDOM },
	{ "random",                 optional_argument, NULL, TIM_OPT_RANDOM },
	{ "no-sort",                no_argument,       NULL, TIM_OPT_SORT },
	{ "sort",                   optional_argument, NULL, TIM_OPT_SORT },
#ifdef IA_ALSASEQ
	{ "no-background",          no_argument,       NULL, TIM_OPT_BACKGROUND },
	{ "background",             optional_argument, NULL, TIM_OPT_BACKGROUND },
	{ "realtime-priority",      required_argument, NULL, TIM_OPT_RT_PRIO },
	{ "sequencer-ports",        required_argument, NULL, TIM_OPT_SEQ_PORTS },
#endif
	{ "no-realtime-load",       no_argument,       NULL, TIM_OPT_REALTIME_LOAD },
	{ "realtime-load",          optional_argument, NULL, TIM_OPT_REALTIME_LOAD },
	{ "adjust-key",             required_argument, NULL, TIM_OPT_ADJUST_KEY },
	{ "voice-queue",            required_argument, NULL, TIM_OPT_VOICE_QUEUE },
	{ "patch-path",             required_argument, NULL, TIM_OPT_PATCH_PATH },
	{ "pcm-file",               required_argument, NULL, TIM_OPT_PCM_FILE },
	{ "decay-time",             required_argument, NULL, TIM_OPT_DECAY_TIME },
	{ "interpolation",          required_argument, NULL, TIM_OPT_INTERPOLATION },
	{ "output-mode",            required_argument, NULL, TIM_OPT_OUTPUT_MODE },
	{ "output-stereo",          no_argument,       NULL, TIM_OPT_OUTPUT_STEREO },
	{ "output-mono",            no_argument,       NULL, TIM_OPT_OUTPUT_STEREO },
	{ "output-signed",          no_argument,       NULL, TIM_OPT_OUTPUT_SIGNED },
	{ "output-unsigned",        no_argument,       NULL, TIM_OPT_OUTPUT_SIGNED },
	{ "output-16bit",           no_argument,       NULL, TIM_OPT_OUTPUT_16BIT },
	{ "output-8bit",            no_argument,       NULL, TIM_OPT_OUTPUT_16BIT },
	{ "output-linear",          no_argument,       NULL, TIM_OPT_OUTPUT_FORMAT },
	{ "output-ulaw",            no_argument,       NULL, TIM_OPT_OUTPUT_FORMAT },
	{ "output-alaw",            no_argument,       NULL, TIM_OPT_OUTPUT_FORMAT },
	{ "no-output-swab",         no_argument,       NULL, TIM_OPT_OUTPUT_SWAB },
	{ "output-swab",            optional_argument, NULL, TIM_OPT_OUTPUT_SWAB },
	{ "output-file",            required_argument, NULL, TIM_OPT_OUTPUT_FILE },
	{ "patch-file",             required_argument, NULL, TIM_OPT_PATCH_FILE },
	{ "polyphony",              required_argument, NULL, TIM_OPT_POLYPHONY },
	{ "no-polyphony-reduction", no_argument,       NULL, TIM_OPT_POLY_REDUCE },
	{ "polyphony-reduction",    optional_argument, NULL, TIM_OPT_POLY_REDUCE },
	{ "mute",                   required_argument, NULL, TIM_OPT_MUTE },
	{ "temper-mute",            required_argument, NULL, TIM_OPT_TEMPER_MUTE },
	{ "audio-buffer",           required_argument, NULL, TIM_OPT_AUDIO_BUFFER },
	{ "cache-size",             required_argument, NULL, TIM_OPT_CACHE_SIZE },
	{ "sampling-freq",          required_argument, NULL, TIM_OPT_SAMPLE_FREQ },
	{ "adjust-tempo",           required_argument, NULL, TIM_OPT_ADJUST_TEMPO },
	{ "output-charset",         required_argument, NULL, TIM_OPT_CHARSET },
	{ "no-unload-instruments",  no_argument,       NULL, TIM_OPT_UNLOAD_INST },
	{ "unload-instruments",     optional_argument, NULL, TIM_OPT_UNLOAD_INST },
	{ "version",                no_argument,       NULL, TIM_OPT_VERSION },
	{ "wrd",                    required_argument, NULL, TIM_OPT_WRD },
#ifdef __W32__
	{ "rcpcv-dll",              required_argument, NULL, TIM_OPT_RCPCV_DLL },
#endif
	{ "config-string",          required_argument, NULL, TIM_OPT_CONFIG_STR },
	{ "freq-table",             required_argument, NULL, TIM_OPT_FREQ_TABLE },
	{ "pure-intonation",        optional_argument, NULL, TIM_OPT_PURE_INT },
	{ NULL,                     no_argument,       NULL, '\0'     }
};
#define INTERACTIVE_INTERFACE_IDS "kmqagrwAWP"


//extern StringTable wrd_read_opts;

extern int SecondMode;

extern struct URL_module URL_module_file;
#ifndef __MACOS__
extern struct URL_module URL_module_dir;
#endif /* __MACOS__ */
#ifdef SUPPORT_SOCKET
extern struct URL_module URL_module_http;
extern struct URL_module URL_module_ftp;
extern struct URL_module URL_module_news;
extern struct URL_module URL_module_newsgroup;
#endif /* SUPPORT_SOCKET */
#ifdef HAVE_POPEN
extern struct URL_module URL_module_pipe;
#endif /* HAVE_POPEN */


static inline int parse_opt_A(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_drum_power(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_volume_comp(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_a(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_B(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_C(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_c(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_D(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_d(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_E(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_mod_wheel(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_portamento(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_vibrato(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_ch_pressure(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_voice_lpf(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_mod_env(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_trace_text(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_overlap_voice(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_temper_control(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_default_mid(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_system_mid(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_default_bank(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_force_bank(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_default_program(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_force_program(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int set_default_program(tmdy_struct_ex_t *timdy_struct, int);
static inline int parse_opt_delay(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_chorus(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_reverb(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_noise_shaping(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_resample(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_e(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_F(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_f(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_g(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_H(tmdy_struct_ex_t *timdy_struct, const char *);
__attribute__((noreturn))
static inline int parse_opt_h(tmdy_struct_ex_t *timdy_struct, const char *);
#ifdef IA_DYNAMIC
static inline void list_dyna_interface(tmdy_struct_ex_t *timdy_struct, FILE *, char *, char *);
static inline char *dynamic_interface_info(tmdy_struct_ex_t *timdy_struct, int);
#endif
static inline int parse_opt_i(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_verbose(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_quiet(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_trace(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_loop(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_random(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_sort(tmdy_struct_ex_t *timdy_struct, const char *);
#ifdef IA_ALSASEQ
static inline int parse_opt_background(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_rt_prio(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_seq_ports(tmdy_struct_ex_t *timdy_struct, const char *);
#endif
static inline int parse_opt_j(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_K(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_k(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_L(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_M(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_m(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_N(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_O(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_output_stereo(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_output_signed(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_output_16bit(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_output_format(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_output_swab(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_o(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_P(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_p(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_p1(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_Q(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_Q1(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_q(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_R(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_S(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_s(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_T(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_t(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_U(tmdy_struct_ex_t *timdy_struct, const char *);
__attribute__((noreturn))
static inline int parse_opt_v(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int parse_opt_W(tmdy_struct_ex_t *timdy_struct, char *);
#ifdef __W32__
static inline int parse_opt_w(tmdy_struct_ex_t *timdy_struct, const char *);
#endif
static inline int parse_opt_x(tmdy_struct_ex_t *timdy_struct, char *);
static inline void expand_escape_string(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_Z(tmdy_struct_ex_t *timdy_struct, char *);
static inline int parse_opt_Z1(tmdy_struct_ex_t *timdy_struct, const char *);
__attribute__((noreturn))
static inline int parse_opt_fail(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int set_value(tmdy_struct_ex_t *timdy_struct, int *, int, int, int, char *);
static inline int set_val_i32(tmdy_struct_ex_t *timdy_struct, int32 *, int32, int32, int32, char *);
static inline int set_channel_flag(tmdy_struct_ex_t *timdy_struct, ChannelBitMask *, int32, char *);
static inline int y_or_n_p(tmdy_struct_ex_t *timdy_struct, const char *);
static inline int set_flag(tmdy_struct_ex_t *timdy_struct, int32 *, int32, const char *);
static inline FILE *open_pager(tmdy_struct_ex_t *timdy_struct);
static inline void close_pager(tmdy_struct_ex_t *timdy_struct, FILE *);
static void interesting_message(tmdy_struct_ex_t *timdy_struct);

#ifdef IA_DYNAMIC
#include "dlutils.h"
#ifndef SHARED_LIB_PATH
#define SHARED_LIB_PATH PKGLIBDIR
#endif /* SHARED_LIB_PATH */
static char *dynamic_lib_root = SHARED_LIB_PATH;
#endif /* IA_DYNAMIC */

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif /* MAXPATHLEN */


#ifndef atof
extern double atof(const char *);
#endif

//#include "timidity_main_prv.h"

tmdy_struct_ex_t *tmdy_struct ; //guha

#include "mid.defs"
VOLATILE int intr = 0;

#ifdef __W32__
#pragma argsused
static BOOL WINAPI handler(DWORD dw)
{
#if defined(IA_WINSYN) || defined(IA_PORTMIDISYN)
//	if( (tmdy_struct->controls->ctl)->id_character == 'W' )
//		|| (tmdy_struct->controls->ctl)->id_character == 'P' )
//	{
    	rtsyn_midiports_close();
//	}
#endif
    printf ("***BREAK" NLS); fflush(stdout);
	intr++;
    return TRUE;
}
#endif


static void copybank(tmdy_struct_ex_t *tmdy_struct, ToneBank *to, ToneBank *from)
{
    int i;

    if(from == NULL)
	return;
    for(i = 0; i < 128; i++)
    {
	ToneBankElement *toelm, *fromelm;

	toelm   = &to->tone[i];
	fromelm = &from->tone[i];

	if(fromelm->name == NULL)
	    continue;

	if(toelm->name)
	    free(toelm->name);
	toelm->name = NULL;
	if(toelm->comment)
	    free(toelm->comment);
	toelm->comment = NULL;
	memcpy(toelm, fromelm, sizeof(ToneBankElement));
	if(toelm->name)
	    toelm->name = tmdy_struct->common->safe_strdup(tmdy_struct, toelm->name);
	if(toelm->comment)
	    toelm->comment = tmdy_struct->common->safe_strdup(tmdy_struct, toelm->comment);
	toelm->instrument = NULL;
	toelm->tune = NULL;
	toelm->tunenum = 0;
	toelm->envrate = toelm->envofs = NULL;
	toelm->envratenum = toelm->envofsnum = 0;
	toelm->trem = toelm->vib = NULL;
	toelm->tremnum = toelm->vibnum = 0;
	toelm->instype = 0;
    }
}

static float *config_parse_tune(tmdy_struct_ex_t *tmdy_struct, const char *cp, int *num)
{
	const char *p;
	float *tune_list;
	int i;
	
	/* count num */
	*num = 1, p = cp;
	while (p = strchr(p, ','))
		(*num)++, p++;
	/* alloc */
	tune_list = (float *) tmdy_struct->common->safe_malloc(tmdy_struct, (*num) * sizeof(float));
	/* regist */
	for (i = 0, p = cp; i < *num; i++, p++) {
		tune_list[i] = atof(p);
		if (! (p = strchr(p, ',')))
			break;
	}
	return tune_list;
}

static int **config_parse_envelope(tmdy_struct_ex_t *tmdy_struct, const char *cp, int *num)
{
	const char *p, *px;
	int **env_list;
	int i, j;
	
	/* count num */
	*num = 1, p = cp;
	while (p = strchr(p, ','))
		(*num)++, p++;
	/* alloc */
	env_list = (int **) tmdy_struct->common->safe_malloc(tmdy_struct, (*num) * sizeof(int *));
	for (i = 0; i < *num; i++)
		env_list[i] = (int *) tmdy_struct->common->safe_malloc(tmdy_struct, 6 * sizeof(int));
	/* init */
	for (i = 0; i < *num; i++)
		for (j = 0; j < 6; j++)
			env_list[i][j] = -1;
	/* regist */
	for (i = 0, p = cp; i < *num; i++, p++) {
		px = strchr(p, ',');
		for (j = 0; j < 6; j++, p++) {
			if (*p == ':')
				continue;
			env_list[i][j] = atoi(p);
			if (! (p = strchr(p, ':')))
				break;
			if (px && p > px)
				break;
		}
		if (! (p = px))
			break;
	}
	return env_list;
}

static Quantity **config_parse_modulation(tmdy_struct_ex_t *tmdy_struct, const char *name, int line, const char *cp, int *num, int mod_type)
{
	const char *p, *px, *err;
	char buf[128], *delim;
	Quantity **mod_list;
	int i, j;
	static const char * qtypestr[] = {"tremolo", "vibrato"};
	static const uint16 qtypes[] = {
		QUANTITY_UNIT_TYPE(TREMOLO_SWEEP), QUANTITY_UNIT_TYPE(TREMOLO_RATE), QUANTITY_UNIT_TYPE(DIRECT_INT),
		QUANTITY_UNIT_TYPE(VIBRATO_SWEEP), QUANTITY_UNIT_TYPE(VIBRATO_RATE), QUANTITY_UNIT_TYPE(DIRECT_INT)
	};
	
	/* count num */
	*num = 1, p = cp;
	while (p = strchr(p, ','))
		(*num)++, p++;
	/* alloc */
	mod_list = (Quantity **) tmdy_struct->common->safe_malloc(tmdy_struct, (*num) * sizeof(Quantity *));
	for (i = 0; i < *num; i++)
		mod_list[i] = (Quantity *) tmdy_struct->common->safe_malloc(tmdy_struct, 3 * sizeof(Quantity));
	/* init */
	for (i = 0; i < *num; i++)
		for (j = 0; j < 3; j++)
			INIT_QUANTITY(mod_list[i][j]);
	buf[sizeof buf - 1] = '\0';
	/* regist */
	for (i = 0, p = cp; i < *num; i++, p++) {
		px = strchr(p, ',');
		for (j = 0; j < 3; j++, p++) {
			if (*p == ':')
				continue;
			if ((delim = strpbrk(strncpy(buf, p, sizeof buf - 1), ":,")) != NULL)
				*delim = '\0';
			if (*buf != '\0' && (err = tmdy_struct->quantity->string_to_quantity(tmdy_struct, buf, &mod_list[i][j], qtypes[mod_type * 3 + j])) != NULL) {
				(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: line %d: %s: parameter %d of item %d: %s (%s)",
						name, line, qtypestr[mod_type], j+1, i+1, err, buf);
				tmdy_struct->common->free_ptr_list(tmdy_struct, mod_list, *num);
				mod_list = NULL;
				*num = 0;
				return NULL;
			}
			if (! (p = strchr(p, ':')))
				break;
			if (px && p > px)
				break;
		}
		if (! (p = px))
			break;
	}
	return mod_list;
}

static int set_gus_patchconf_opts(tmdy_struct_ex_t *tmdy_struct, char *name, int line, char *opts,
				  ToneBankElement *tone)
{
    char *cp;
    int k;

    if(!(cp = strchr(opts, '=')))
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: line %d: bad patch option %s",
		  name, line, opts);
	return 1;
    }

    *cp++ = 0;
    if(!strcmp(opts, "amp"))
    {
	k = atoi(cp);
	if((k < 0 || k > MAX_AMPLIFICATION) ||
	   (*cp < '0' || *cp > '9'))
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: amplification must be between "
		      "0 and %d", name, line, MAX_AMPLIFICATION);
	    return 1;
	}
	tone->amp = k;
    }
    else if(!strcmp(opts, "note"))
    {
	k = atoi(cp);
	if((k < 0 || k > 127) || (*cp < '0' || *cp > '9'))
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: note must be between 0 and 127",
		      name, line);
	    return 1;
	}
	tone->note = k;
    }
    else if(!strcmp(opts, "pan"))
    {
	if(!strcmp(cp, "center"))
	    k = 64;
	else if(!strcmp(cp, "left"))
	    k = 0;
	else if(!strcmp(cp, "right"))
	    k = 127;
	else
	    k = ((atoi(cp) + 100) * 100) / 157;
	if((k < 0 || k > 127) ||
	   (k == 0 && *cp != '-' && (*cp < '0' || *cp > '9')))
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: panning must be left, right, "
		      "center, or between -100 and 100",
		      name, line);
	    return 1;
	}
	tone->pan = k;
    }
    else if(!strcmp(opts, "keep"))
    {
	if(!strcmp(cp, "env"))
	    tone->strip_envelope = 0;
	else if(!strcmp(cp, "loop"))
	    tone->strip_loop = 0;
	else
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: keep must be env or loop",
		      name, line);
	    return 1;
	}
    }
    else if(!strcmp(opts, "strip"))
    {
	if(!strcmp(cp, "env"))
	    tone->strip_envelope = 1;
	else if(!strcmp(cp, "loop"))
	    tone->strip_loop = 1;
	else if(!strcmp(cp, "tail"))
	    tone->strip_tail = 1;
	else
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: strip must be "
		      "env, loop, or tail", name, line);
	    return 1;
	}
    }
    else if(!strcmp(opts, "comm"))
    {
	char *p;
	if(tone->comment)
	    free(tone->comment);
	p = tone->comment = tmdy_struct->common->safe_strdup(tmdy_struct, cp);
	while(*p)
	{
	    if(*p == ',') *p = ' ';
	    p++;
	}
    }
    else if(!strcmp(opts, "tune")) {
      tone->tune = config_parse_tune(tmdy_struct, cp, &tone->tunenum);
	} else if (! strcmp(opts, "rate"))
		tone->envrate = config_parse_envelope(tmdy_struct, cp, &tone->envratenum);
	else if (! strcmp(opts, "offset"))
		tone->envofs = config_parse_envelope(tmdy_struct, cp, &tone->envofsnum);
	else if (! strcmp(opts, "tremolo"))
	{
		if ((tone->trem = config_parse_modulation(tmdy_struct, name, line, cp, &tone->tremnum, 0)) == NULL)
			return 1;
	}
	else if (! strcmp(opts, "vibrato"))
	{
		if ((tone->vib = config_parse_modulation(tmdy_struct, name, line, cp, &tone->vibnum, 1)) == NULL)
			return 1;
	}
    else
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "%s: line %d: bad patch option %s",
		  name, line, opts);
	return 1;
    }
    return 0;
}


static void reinit_tone_bank_element(tmdy_struct_ex_t *tmdy_struct,ToneBankElement *tone)
{
    if (tone->name)
    {
	free(tone->name);
	tone->name = NULL;
    }
    if (tone->tunenum)
    {
	tone->tunenum = 0;
	free(tone->tune);
	tone->tune = NULL;
    }
    if (tone->envratenum)
    {
	tmdy_struct->common->free_ptr_list(tmdy_struct, tone->envrate, tone->envratenum);
	tone->envratenum = 0;
	tone->envrate = NULL;
    }
    if (tone->envofsnum)
    {
	tmdy_struct->common->free_ptr_list(tmdy_struct, tone->envofs, tone->envofsnum);
	tone->envofsnum = 0;
	tone->envofs = NULL;
    }
    if (tone->tremnum)
    {
	tmdy_struct->common->free_ptr_list(tmdy_struct, tone->trem, tone->tremnum);
	tone->tremnum = 0;
	tone->trem = NULL;
    }
    if (tone->vibnum)
    {
	tmdy_struct->common->free_ptr_list(tmdy_struct, tone->vib, tone->vibnum);
	tone->vibnum = 0;
	tone->vib = NULL;
    }
    tone->note = tone->pan =
	tone->strip_loop = tone->strip_envelope =
	tone->strip_tail = -1;
    tone->amp = -1;
    tone->legato = 0;
    tone->tva_level = -1;
}

#define SET_GUS_PATCHCONF_COMMENT
static int set_gus_patchconf(tmdy_struct_ex_t *tmdy_struct, char *name, int line,
			     ToneBankElement *tone, char *pat, char **opts)
{
    int j;
#ifdef SET_GUS_PATCHCONF_COMMENT
		char *old_name = NULL;

		if(tone != NULL && tone->name != NULL)
			old_name = tmdy_struct->common->safe_strdup(tmdy_struct, tone->name);
#endif
    reinit_tone_bank_element(tmdy_struct, tone);

    if(strcmp(pat, "%font") == 0) /* Font extention */
    {
	/* %font filename bank prog [note-to-use]
	 * %font filename 128 bank key
	 */

	if(opts[0] == NULL || opts[1] == NULL || opts[2] == NULL ||
	   (atoi(opts[1]) == 128 && opts[3] == NULL))
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: Syntax error", name, line);
	    return 1;
	}
	tone->name = tmdy_struct->common->safe_strdup(tmdy_struct, opts[0]);
	tone->instype = 1;
	if(atoi(opts[1]) == 128) /* drum */
	{
	    tone->font_bank = 128;
	    tone->font_preset = atoi(opts[2]);
	    tone->font_keynote = atoi(opts[3]);
	    opts += 4;
	}
	else
	{
	    tone->font_bank = atoi(opts[1]);
	    tone->font_preset = atoi(opts[2]);

	    if(opts[3] && isdigit(opts[3][0]))
	    {
		tone->font_keynote = atoi(opts[3]);
		opts += 4;
	    }
	    else
	    {
		tone->font_keynote = -1;
		opts += 3;
	    }
	}
    }
    else if(strcmp(pat, "%sample") == 0) /* Sample extention */
    {
	/* %sample filename */

	if(opts[0] == NULL)
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: Syntax error", name, line);
	    return 1;
	}
	tone->name = tmdy_struct->common->safe_strdup(tmdy_struct, opts[0]);
	tone->instype = 2;
	opts++;
    }
    else
    {
	tone->instype = 0;
	tone->name = tmdy_struct->common->safe_strdup(tmdy_struct, pat);
    }

    for(j = 0; opts[j] != NULL; j++)
    {
	int err;
	if((err = set_gus_patchconf_opts(tmdy_struct, name, line, opts[j], tone)) != 0)
	    return err;
    }
#ifdef SET_GUS_PATCHCONF_COMMENT
		if(tone->comment == NULL ||
			(old_name != NULL && strcmp(old_name,tone->comment) == 0))
		{
			if(tone->comment != NULL )
				free(tone->comment);
			tone->comment = tmdy_struct->common->safe_strdup(tmdy_struct, tone->name);
		}
		if(old_name != NULL)
			free(old_name);
#else
    if(tone->comment == NULL)
	tone->comment = tmdy_struct->common->safe_strdup(tmdy_struct, tone->name);
#endif
    return 0;
}

static int mapname2id(tmdy_struct_ex_t *tmdy_struct, char *name, int *isdrum)
{
    if(strcmp(name, "sc55") == 0)
    {
	*isdrum = 0;
	return SC_55_TONE_MAP;
    }

    if(strcmp(name, "sc55drum") == 0)
    {
	*isdrum = 1;
	return SC_55_DRUM_MAP;
    }

    if(strcmp(name, "sc88") == 0)
    {
	*isdrum = 0;
	return SC_88_TONE_MAP;
    }

    if(strcmp(name, "sc88drum") == 0)
    {
	*isdrum = 1;
	return SC_88_DRUM_MAP;
    }

    if(strcmp(name, "sc88pro") == 0)
    {
	*isdrum = 0;
	return SC_88PRO_TONE_MAP;
    }

    if(strcmp(name, "sc88prodrum") == 0)
    {
	*isdrum = 1;
	return SC_88PRO_DRUM_MAP;
    }

    if(strcmp(name, "xg") == 0)
    {
	*isdrum = 0;
	return XG_NORMAL_MAP;
    }

    if(strcmp(name, "xgsfx64") == 0)
    {
	*isdrum = 0;
	return XG_SFX64_MAP;
    }

    if(strcmp(name, "xgsfx126") == 0)
    {
	*isdrum = 1;
	return XG_SFX126_MAP;
    }

    if(strcmp(name, "xgdrum") == 0)
    {
	*isdrum = 1;
	return XG_DRUM_MAP;
    }
    return -1;
}

/* string[0] should not be '#' */
static int strip_trailing_comment(tmdy_struct_ex_t *tmdy_struct, char *string, int next_token_index)
{
    if (string[next_token_index - 1] == '#'	/* strip \1 in /^\S+(#*[ \t].*)/ */
	&& (string[next_token_index] == ' ' || string[next_token_index] == '\t'))
    {
	string[next_token_index] = '\0';	/* new c-string terminator */
	while(string[--next_token_index - 1] == '#')
	    ;
    }
    return next_token_index;
}

#define MAXWORDS 130
#define CHECKERRLIMIT \
  if(++errcnt >= 10) { \
    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, \
      "Too many errors... Give up read %s", name); \
    tmdy_struct->common->close_file(tmdy_struct, tf); return 1; }

MAIN_INTERFACE int read_config_file(tmdy_struct_ex_t *tmdy_struct, char *name, int self)
{
    struct timidity_file *tf;
    char tmp[1024], *w[MAXWORDS + 1], *cp;
    ToneBank *bank = NULL;
    int i, j, k, line = 0, words, errcnt = 0;
    static int rcf_count = 0;
    int dr = 0, bankno = 0;
    int extension_flag, param_parse_err;

    if(rcf_count > 50)
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Probable source loop in configuration files");
	return 2;
    }

    if(self)
    {
	tf = tmdy_struct->common->open_with_mem(tmdy_struct, name, (int32)strlen(name), OF_VERBOSE);
	name = "(configuration)";
    }
    else
	tf = tmdy_struct->common->open_file(tmdy_struct, name, 1, OF_VERBOSE);
    if(tf == NULL)
	return 1;

    errno = 0;
    while(tmdy_struct->common->tf_gets(tmdy_struct, tmp, sizeof(tmp), tf))
    {
	line++;
	if(strncmp(tmp, "#extension", 10) == 0) {
	    extension_flag = 1;
	    i = 10;
	}
	else
	{
	    extension_flag = 0;
	    i = 0;
	}

	while(isspace(tmp[i]))			/* skip /^\s*(?#)/ */
	    i++;
	if (tmp[i] == '#' || tmp[i] == '\0')	/* /^#|^$/ */
	    continue;
	j = strcspn(tmp + i, " \t\r\n\240");
	if (j == 0)
		j = strlen(tmp + i);
	j = strip_trailing_comment(tmdy_struct, tmp + i, j);
	tmp[i + j] = '\0';			/* terminate the first token */
	w[0] = tmp + i;
	i += j + 1;
	words = param_parse_err = 0;
	while(words < MAXWORDS - 1)		/* -1 : next arg */
	{
	    char *terminator;

	    while(isspace(tmp[i]))		/* skip /^\s*(?#)/ */
		i++;
	    if (tmp[i] == '\0'
		    || tmp[i] == '#')		/* /\s#/ */
		break;
	    if ((tmp[i] == '"' || tmp[i] == '\'')
		    && (terminator = strchr(tmp + i + 1, tmp[i])) != NULL)
	    {
		if (!isspace(terminator[1]) && terminator[1] != '\0')
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			"%s: line %d: there must be at least one whitespace between "
			"string terminator (%c) and the next parameter", name, line, tmp[i]);
		    CHECKERRLIMIT;
		    param_parse_err = 1;
		    break;
		}
		w[++words] = tmp + i + 1;
		i = terminator - tmp + 1;
		*terminator = '\0';
	    }
	    else	/* not terminated */
	    {
		j = strcspn(tmp + i, " \t\r\n\240");
		if (j > 0)
		    j = strip_trailing_comment(tmdy_struct, tmp + i, j);
		w[++words] = tmp + i;
		i += j;
		if (tmp[i] != '\0')		/* unless at the end-of-string (i.e. EOF) */
		    tmp[i++] = '\0';		/* terminate the token */
	    }
	}
	if (param_parse_err)
	    continue;
	w[++words] = NULL;

	/*
	 * #extension [something...]
	 */

	/* #extension comm program comment */
	if(strcmp(w[0], "comm") == 0)
	{
	    char *p;

	    if(words != 3)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum "
			  "set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension comm must be "
			  "between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(bank->tone[i].comment)
		free(bank->tone[i].comment);
	    p = bank->tone[i].comment = tmdy_struct->common->safe_strdup(tmdy_struct, w[2]);
	    while(*p)
	    {
		if(*p == ',') *p = ' ';
		p++;
	    }
	}
	/* #extension timeout program sec */
	else if(strcmp(w[0], "timeout") == 0)
	{
	    if(words != 3)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension timeout "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    bank->tone[i].loop_timeout = atoi(w[2]);
	}
	/* #extension copydrumset drumset */
	else if(strcmp(w[0], "copydrumset") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No copydrumset number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension copydrumset "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    copybank(tmdy_struct, bank, tmdy_struct->instrum->drumset[i]);
	}
	/* #extension copybank bank */
	else if(strcmp(w[0], "copybank") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No copybank number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension copybank "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    copybank(tmdy_struct, bank, tmdy_struct->instrum->tonebank[i]);
	}
	/* #extension HTTPproxy hostname:port */
	else if(strcmp(w[0], "HTTPproxy") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No proxy name given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    /* If network is not supported, this extension is ignored. */
#ifdef SUPPORT_SOCKET
	    url_http_proxy_host = tmdy_struct->common->safe_strdup(tmdy_struct, w[1]);
	    if((cp = strchr(url_http_proxy_host, ':')) == NULL)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    *cp++ = '\0';
	    if((url_http_proxy_port = atoi(cp)) <= 0)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Port number must be "
			  "positive number", name, line);
		CHECKERRLIMIT;
		continue;
	    }
#endif
	}
	/* #extension FTPproxy hostname:port */
	else if(strcmp(w[0], "FTPproxy") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No proxy name given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    /* If network is not supported, this extension is ignored. */
#ifdef SUPPORT_SOCKET
	    url_ftp_proxy_host = tmdy_struct->common->safe_strdup(tmdy_struct, w[1]);
	    if((cp = strchr(url_ftp_proxy_host, ':')) == NULL)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    *cp++ = '\0';
	    if((url_ftp_proxy_port = atoi(cp)) <= 0)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Port number "
			  "must be positive number", name, line);
		CHECKERRLIMIT;
		continue;
	    }
#endif
	}
	/* #extension mailaddr somebody@someware.domain.com */
	else if(strcmp(w[0], "mailaddr") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No mail address given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(strchr(w[1], '@') == NULL) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NOISY,
			  "%s: line %d: Warning: Mail address %s is not valid",
			  name, line);
	    }

	    /* If network is not supported, this extension is ignored. */
#ifdef SUPPORT_SOCKET
	    user_mailaddr = tmdy_struct->common->safe_strdup(tmdy_struct, w[1]);
#endif /* SUPPORT_SOCKET */
	}
	/* #extension opt [-]{option}[optarg] */
	else if (strcmp(w[0], "opt") == 0) {
		int c, longind, err;
		char *p, *cmd, *arg;
		
		if (words != 2 && words != 3) {
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
					"%s: line %d: Syntax error", name, line);
			CHECKERRLIMIT;
			continue;
		}
		if (*w[1] == '-') {
			int optind_save = optind;
			optind = 0;
			c = getopt_long(words, w, optcommands, longopts, &longind);
			err = set_tim_opt_long(tmdy_struct, c, optarg, longind);
			optind = optind_save;
		} else {
			/* backward compatibility */
			if ((p = strchr(optcommands, c = *(cmd = w[1]))) == NULL)
				err = 1;
			else {
				if (*(p + 1) == ':')
					arg = (words == 2) ? cmd + 1 : w[2];
				else
					arg = "";
				err = set_tim_opt_short(tmdy_struct, c, arg);
			}
		}
		if (err) {
			/* error */
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
					"%s: line %d: Invalid command line option",
					name, line);
			errcnt += err - 1;
			CHECKERRLIMIT;
			continue;
		}
	}
	/* #extension undef program */
	else if(strcmp(w[0], "undef") == 0)
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No undef number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension undef "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(bank->tone[i].name)
	    {
		free(bank->tone[i].name);
		    bank->tone[i].name = NULL;
	    }
	    if(bank->tone[i].comment)
	    {
		free(bank->tone[i].comment);
		bank->tone[i].comment = NULL;
	    }
			bank->tone[i].instype = 0;
	}
	/* #extension altassign numbers... */
	else if(strcmp(w[0], "altassign") == 0)
	{
	    ToneBank *bk;

	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before altassign", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No alternate assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    if(!dr) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
			  "%s: line %d: Warning: Not a drumset altassign"
			  " (ignored)",
			  name, line);
		continue;
	    }

	    bk = tmdy_struct->instrum->drumset[bankno];
	    bk->alt = tmdy_struct->instrum->add_altassign_string(tmdy_struct, bk->alt, w + 1, words - 1);
	}	/* #extension legato [program] [0 or 1] */
	else if(strcmp(w[0], "legato") == 0)
	{
	    if(words != 3)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension legato "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    bank->tone[i].legato = atoi(w[2]);
	}	/* #extension level program tva_level */
	else if(strcmp(w[0], "level") == 0)
	{
	    if(words != 3)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension level "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
		bank->tone[i].tva_level = atoi(w[2]);
	}
	else if(!strcmp(w[0], "soundfont"))
	{
	    int order, cutoff, isremove, reso, amp;
	    char *sf_file;

	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No soundfont file given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    sf_file = w[1];
	    order = cutoff = reso = amp = -1;
	    isremove = 0;
	    for(j = 2; j < words; j++)
	    {
		if(strcmp(w[j], "remove") == 0)
		{
		    isremove = 1;
		    break;
		}
		if(!(cp = strchr(w[j], '=')))
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: bad patch option %s",
			      name, line, w[j]);
		    CHECKERRLIMIT;
		    break;
		}
		*cp++=0;
		k = atoi(cp);
		if(!strcmp(w[j], "order"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: order must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    order = k;
		}
		else if(!strcmp(w[j], "cutoff"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: cutoff must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    cutoff = k;
		}
		else if(!strcmp(w[j], "reso"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: reso must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    reso = k;
		}
		else if(!strcmp(w[j], "amp"))
		{
		    amp = k;
		}
	    }
	    if(isremove)
		tmdy_struct->instrum->remove_soundfont(tmdy_struct, sf_file);
	    else
		add_soundfont(tmdy_struct, sf_file, order, cutoff, reso, amp);
	}
	else if(!strcmp(w[0], "font"))
	{
	    int bank, preset, keynote;
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: no font command", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!strcmp(w[1], "exclude"))
	    {
		if(words < 3)
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No bank/preset/key is given",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}
		bank = atoi(w[2]);
		if(words >= 4)
		    preset = atoi(w[3]) - tmdy_struct->instrum->progbase;
		else
		    preset = -1;
		if(words >= 5)
		    keynote = atoi(w[4]);
		else
		    keynote = -1;
		if(tmdy_struct->instrum->exclude_soundfont(tmdy_struct, bank, preset, keynote))
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No soundfont is given",
			      name, line);
		    CHECKERRLIMIT;
		}
	    }
	    else if(!strcmp(w[1], "order"))
	    {
		int order;
		if(words < 4)
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No order/bank is given",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}
		order = atoi(w[2]);
		bank = atoi(w[3]);
		if(words >= 5)
		    preset = atoi(w[4]) - tmdy_struct->instrum->progbase;
		else
		    preset = -1;
		if(words >= 6)
		    keynote = atoi(w[5]);
		else
		    keynote = -1;
		if(tmdy_struct->instrum->order_soundfont(tmdy_struct, bank, preset, keynote, order))
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No soundfont is given",
			      name, line);
		    CHECKERRLIMIT;
		}
	    }
	}
	else if(!strcmp(w[0], "progbase"))
	{
	    if(words < 2 || *w[1] < '0' || *w[1] > '9')
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    tmdy_struct->instrum->progbase = atoi(w[1]);
	}
	else if(!strcmp(w[0], "map")) /* map <name> set1 elem1 set2 elem2 */
	{
	    int arg[5], isdrum;

	    if(words != 6)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if((arg[0] = mapname2id(tmdy_struct, w[1], &isdrum)) == -1)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Invalid map name: %s", name, line, w[1]);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 2; i < 6; i++)
		arg[i - 1] = atoi(w[i]);
	    if(isdrum)
	    {
		arg[1] -= tmdy_struct->instrum->progbase;
		arg[3] -= tmdy_struct->instrum->progbase;
	    }
	    else
	    {
		arg[2] -= tmdy_struct->instrum->progbase;
		arg[4] -= tmdy_struct->instrum->progbase;
	    }

	    for(i = 1; i < 5; i++)
		if(arg[i] < 0 || arg[i] > 127)
		    break;
	    if(i != 5)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Invalid parameter", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    tmdy_struct->instrum->set_instrument_map(tmdy_struct, arg[0], arg[1], arg[2], arg[3], arg[4]);
	}

	/*
	 * Standard configurations
	 */
	else if(!strcmp(w[0], "dir"))
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No directory given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 1; i < words; i++)
		tmdy_struct->common->add_to_pathlist(tmdy_struct, w[i]);
	}
	else if(!strcmp(w[0], "source"))
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No file name given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 1; i < words; i++)
	    {
		int status;
		rcf_count++;
		status = read_config_file(tmdy_struct, w[i], 0);
		rcf_count--;
		if(status == 2)
		{
		    tmdy_struct->common->close_file(tmdy_struct, tf);
		    return 2;
		}
		else if(status != 0)
		{

		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else if(!strcmp(w[0], "default"))
	{
	    if(words != 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify exactly one patch name",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    strncpy(tmdy_struct->timidity_main->def_instr_name, w[1], 255);
	    tmdy_struct->timidity_main->def_instr_name[255] = '\0';
	    tmdy_struct->instrum->default_instrument_name = tmdy_struct->timidity_main->def_instr_name;
	}
	else if(!strcmp(w[0], "drumset"))
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No drum set number given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]) - tmdy_struct->instrum->progbase;
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Drum set must be between %d and %d",
			  name, line,
			  tmdy_struct->instrum->progbase, tmdy_struct->instrum->progbase + 127);
		CHECKERRLIMIT;
		continue;
	    }

	    tmdy_struct->instrum->alloc_instrument_bank(tmdy_struct, 1, i);

	    if(words == 2)
	    {
		bank = tmdy_struct->instrum->drumset[i];
		bankno = i;
		dr = 1;
	    }
	    else
	    {
		ToneBank *localbank;

		localbank = tmdy_struct->instrum->drumset[i];

		if(words < 4 || *w[2] < '0' || *w[2] > '9')
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: syntax error", name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		i = atoi(w[2]);
		if(i < 0 || i > 127)
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Drum number must be between "
			      "0 and 127",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		if(set_gus_patchconf(tmdy_struct, name, line, &localbank->tone[i],
				     w[3], w + 4))
		{
		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else if(!strcmp(w[0], "bank"))
	{
	    if(words < 2)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No bank number given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Tone bank must be between 0 and 127",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    tmdy_struct->instrum->alloc_instrument_bank(tmdy_struct, 0, i);

	    if(words == 2)
	    {
		bank = tmdy_struct->instrum->tonebank[i];
		bankno = i;
		dr = 0;
	    }
	    else
	    {
		ToneBank *localbank;

		localbank = tmdy_struct->instrum->tonebank[i];

		if(words < 4 || *w[2] < '0' || *w[2] > '9')
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: syntax error", name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		i = atoi(w[2]) - tmdy_struct->instrum->progbase;
		if(i < 0 || i > 127)
		{
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Program must be between "
			      "%d and %d",
			      name, line,
			      tmdy_struct->instrum->progbase, 127 + tmdy_struct->instrum->progbase);
		    CHECKERRLIMIT;
		    continue;
		}

		if(set_gus_patchconf(tmdy_struct, name, line, &localbank->tone[i],
				     w[3], w + 4))
		{
		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else
	{
	    if(words < 2 || *w[0] < '0' || *w[0] > '9')
	    {
		if(extension_flag)
		    continue;
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[0]);
	    if(!dr)
		i -= tmdy_struct->instrum->progbase;
	    if(i < 0 || i > 127)
	    {
		if(dr)
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Drum number must be between "
			      "0 and 127",
			      name, line);
		else
		    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Program must be between "
			      "%d and %d",
			      name, line,
			      tmdy_struct->instrum->progbase, 127 + tmdy_struct->instrum->progbase);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    if(set_gus_patchconf(tmdy_struct, name, line, &bank->tone[i], w[1], w + 2))
	    {
		CHECKERRLIMIT;
		continue;
	    }
	}
    }
    if(errno)
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		  "Can't read %s: %s", name, strerror(errno));
	errcnt++;
    }
    tmdy_struct->common->close_file(tmdy_struct, tf);
    return errcnt != 0;
}

#ifdef SUPPORT_SOCKET

#if defined(__W32__) && !defined(MAIL_NAME)
#define MAIL_NAME "anonymous"
#endif /* __W32__ */

#ifdef MAIL_NAME
#define get_username() MAIL_NAME
#else /* MAIL_NAME */
#include <pwd.h>
static char *get_username(tmdy_struct_ex_t *tmdy_struct)
{
    char *p;
    struct passwd *pass;

    /* USER
     * LOGIN
     * LOGNAME
     * getpwnam()
     */

    if((p = getenv("USER")) != NULL)
        return p;
    if((p = getenv("LOGIN")) != NULL)
        return p;
    if((p = getenv("LOGNAME")) != NULL)
        return p;

    pass = getpwuid(getuid());
    if(pass == NULL)
        return "nobody";
    return pass->pw_name;
}
#endif /* MAIL_NAME */

static void init_mail_addr(void)
{
    char addr[BUFSIZ];

    sprintf(addr, "%s%s", get_username(tmdy_struct), MAIL_DOMAIN);
    user_mailaddr = tmdy_struct->common->safe_strdup(tmdy_struct, addr);
}
#endif /* SUPPORT_SOCKET */

static int read_user_config_file(tmdy_struct_ex_t *tmdy_struct)
{
    char *home;
    char path[BUFSIZ];
    int opencheck;

#ifdef __W32__
/* HOME or home */
    home = getenv("HOME");
    if(home == NULL)
	home = getenv("home");
    if(home == NULL)
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		  "Warning: HOME environment is not defined.");
	return 0;
    }
/* .timidity.cfg or timidity.cfg */
    sprintf(path, "%s" PATH_STRING "timidity.cfg", home);
    if((opencheck = open(path, 0)) < 0)
    {
	sprintf(path, "%s" PATH_STRING "_timidity.cfg", home);
	if((opencheck = open(path, 0)) < 0)
	{
	    sprintf(path, "%s" PATH_STRING ".timidity.cfg", home);
	    if((opencheck = open(path, 0)) < 0)
	    {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "%s: %s",
			  path, strerror(errno));
		return 0;
	    }
	}
    }

    close(opencheck);
    return read_config_file(tmdy_struct, path, 0);
#else
    home = getenv("HOME");
    if(home == NULL)
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		  "Warning: HOME environment is not defined.");
	return 0;
    }
    sprintf(path, "%s" PATH_STRING ".timidity.cfg", home);

    if((opencheck = open(path, 0)) < 0)
    {
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY, "%s: %s",
		  path, strerror(errno));
	return 0;
    }

    close(opencheck);
    return read_config_file(tmdy_struct, path, 0);
#endif /* __W32__ */
}

MAIN_INTERFACE void tmdy_free_config(tmdy_struct_ex_t *tmdy_struct)
{
  ToneBank *bank;
  ToneBankElement *elm;

  int i, j;

  for (i = 0; i < 128; i++) {
    bank = tmdy_struct->instrum->tonebank[i];
    if (!bank)
      continue;
    for (j = 0; j < 128; j++) {
      elm = &bank->tone[j];
      if (elm->name)
	free(elm->name);
	elm->name = NULL;
      if (elm->comment)
	free(elm->comment);
	elm->comment = NULL;
	if (elm->tune)
		free(elm->tune);
	elm->tune = NULL;
	if (elm->envrate)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->envrate, elm->envratenum);
	elm->envrate = NULL;
	elm->envratenum = 0;
	if (elm->envofs)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->envofs, elm->envofsnum);
	elm->envofs = NULL;
	elm->envofsnum = 0;
	if (elm->trem)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->trem, elm->tremnum);
	elm->trem = NULL;
	elm->tremnum = 0;
	if (elm->vib)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->vib, elm->vibnum);
	elm->vib = NULL;
	elm->vibnum = 0;
	elm->instype = 0;
    }
    if (i > 0) {
      free(bank);
      tmdy_struct->instrum->tonebank[i] = NULL;
    }
  }

  for (i = 0; i < 128; i++) {
    bank = tmdy_struct->instrum->drumset[i];
    if (!bank)
      continue;
    for (j = 0; j < 128; j++) {
      elm = &bank->tone[j];
      if (elm->name)
	free(elm->name);
	elm->name = NULL;
      if (elm->comment)
	free(elm->comment);
	elm->comment = NULL;
	if (elm->tune)
		free(elm->tune);
	elm->tune = NULL;
	if (elm->envrate)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->envrate, elm->envratenum);
	elm->envrate = NULL;
	elm->envratenum = 0;
	if (elm->envofs)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->envofs, elm->envofsnum);
	elm->envofs = NULL;
	elm->envofsnum = 0;
	if (elm->trem)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->trem, elm->tremnum);
	elm->trem = NULL;
	elm->tremnum = 0;
	if (elm->vib)
		tmdy_struct->common->free_ptr_list(tmdy_struct, elm->vib, elm->vibnum);
	elm->vib = NULL;
	elm->vibnum = 0;
	elm->instype = 0;
    }
    if (i > 0) {
      free(bank);
      tmdy_struct->instrum->drumset[i] = NULL;
    }
  }

  tmdy_struct->instrum->free_instrument_map(tmdy_struct);
  tmdy_struct->common->clean_up_pathlist(tmdy_struct);
}

int set_extension_modes(tmdy_struct_ex_t *tmdy_struct, char *flag)
{
	return parse_opt_E(tmdy_struct, flag);
}

int set_ctl(tmdy_struct_ex_t *tmdy_struct,char *cp)
{
	return parse_opt_i(tmdy_struct, cp);
}

int set_play_mode(tmdy_struct_ex_t *tmdy_struct, char *cp)
{
	return parse_opt_O(tmdy_struct, cp);
}

int set_wrd(tmdy_struct_ex_t *tmdy_struct, char *w)
{
	return parse_opt_W(tmdy_struct, w);
}

#ifdef __W32__
int opt_evil_mode = 0;
#ifdef SMFCONV
int opt_rcpcv_dll = 0;
#endif	/* SMFCONV */
#endif	/* __W32__ */
static int   try_config_again = 0;
int32 opt_output_rate = 0;
static char *opt_output_name = NULL;
static StringTable opt_config_string;
#ifdef SUPPORT_SOUNDSPEC
static double spectrogram_update_sec = 0.0;
#endif /* SUPPORT_SOUNDSPEC */
int opt_buffer_fragments = -1;

MAIN_INTERFACE int set_tim_opt_short(tmdy_struct_ex_t *tmdy_struct, int c, char *optarg)
{
	int err = 0;
	
	switch (c) {
	case '4':
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"-4 option is obsoleted.  Please use -N");
		return 1;
	case 'A':
		if (*optarg != ',' && *optarg != 'a')
			err += parse_opt_A(tmdy_struct, optarg);
		if (strchr(optarg, ','))
			err += parse_opt_drum_power(tmdy_struct,strchr(optarg, ',') + 1);
		if (strchr(optarg, 'a'))
			tmdy_struct->playmidi->opt_amp_compensation = 1;
		return err;
	case 'a':
		tmdy_struct->instrum->antialiasing_allowed = 1;
		break;
	case 'B':
		return parse_opt_B(tmdy_struct, optarg);
	case 'C':
		return parse_opt_C(tmdy_struct, optarg);
	case 'c':
		return parse_opt_c(tmdy_struct, optarg);
	case 'D':
		return parse_opt_D(tmdy_struct, optarg);
	case 'd':
		return parse_opt_d(tmdy_struct, optarg);
	case 'E':
		return parse_opt_E(tmdy_struct, optarg);
	case 'e':
		return parse_opt_e(tmdy_struct, optarg);
	case 'F':
		tmdy_struct->playmidi->adjust_panning_immediately = (tmdy_struct->playmidi->adjust_panning_immediately) ? 0 : 1;
		break;
	case 'f':
		tmdy_struct->instrum->fast_decay = (tmdy_struct->instrum->fast_decay) ? 0 : 1;
		break;
	case 'g':
		return parse_opt_g(tmdy_struct, optarg);
	case 'H':
		return parse_opt_H(tmdy_struct, optarg);
	case 'h':
		return parse_opt_h(tmdy_struct, optarg);
	case 'I':
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"-I option is obsoleted.  Please use -Ei");
		return 1;
	case 'i':
		return parse_opt_i(tmdy_struct, optarg);
	case 'j':
		tmdy_struct->playmidi->opt_realtime_playing = (tmdy_struct->playmidi->opt_realtime_playing) ? 0 : 1;
		break;
	case 'K':
		return parse_opt_K(tmdy_struct, optarg);
	case 'k':
		return parse_opt_k(tmdy_struct, optarg);
	case 'L':
		return parse_opt_L(tmdy_struct, optarg);
	case 'M':
		return parse_opt_M(tmdy_struct, optarg);
	case 'm':
		return parse_opt_m(tmdy_struct, optarg);
	case 'N':
		return parse_opt_N(tmdy_struct, optarg);
	case 'O':
		return parse_opt_O(tmdy_struct, optarg);
	case 'o':
		return parse_opt_o(tmdy_struct, optarg);
	case 'P':
		return parse_opt_P(tmdy_struct, optarg);
	case 'p':
		if (*optarg != 'a')
			err += parse_opt_p(tmdy_struct, optarg);
		if (strchr(optarg, 'a'))
			tmdy_struct->playmidi->auto_reduce_polyphony = (tmdy_struct->playmidi->auto_reduce_polyphony) ? 0 : 1;
		return err;
	case 'Q':
		return parse_opt_Q(tmdy_struct, optarg);
	case 'q':
		return parse_opt_q(tmdy_struct, optarg);
	case 'R':
		return parse_opt_R(tmdy_struct, optarg);
	case 'S':
		return parse_opt_S(tmdy_struct, optarg);
	case 's':
		return parse_opt_s(tmdy_struct, optarg);
	case 'T':
		return parse_opt_T(tmdy_struct, optarg);
	case 't':
		return parse_opt_t(tmdy_struct, optarg);
	case 'U':
		tmdy_struct->timidity_main->free_instruments_afterwards = 1;
		break;
	case 'v':
		return parse_opt_v(tmdy_struct, optarg);
	case 'W':
		return parse_opt_W(tmdy_struct, optarg);
#ifdef __W32__
	case 'w':
		return parse_opt_w(tmdy_struct, optarg);
#endif
	case 'x':
		return parse_opt_x(tmdy_struct, optarg);
	case 'Z':
		if (strncmp(optarg, "pure", 4))
			return parse_opt_Z(tmdy_struct, optarg);
		else
			return parse_opt_Z1(tmdy_struct, optarg + 4);
	default:
		return 1;
	}
	return 0;
}

/* -------- getopt_long -------- */
MAIN_INTERFACE int set_tim_opt_long(tmdy_struct_ex_t *tmdy_struct, int c, char *optarg, int index)
{
	const struct option *the_option = &(longopts[index]);
	char *arg;
	
	if (c == '?')	/* getopt_long failed parsing */
		parse_opt_fail(tmdy_struct, optarg);
	else if (c < TIM_OPT_FIRST)
		return set_tim_opt_short(tmdy_struct, c, optarg);
	if (! strncmp(the_option->name, "no-", 3))
		arg = "no";		/* `reverse' switch */
	else
		arg = optarg;
	switch (c) {
	case TIM_OPT_VOLUME:
		return parse_opt_A(tmdy_struct, arg);
	case TIM_OPT_DRUM_POWER:
		return parse_opt_drum_power(tmdy_struct,arg);
	case TIM_OPT_VOLUME_COMP:
		return parse_opt_volume_comp(tmdy_struct, arg);
	case TIM_OPT_ANTI_ALIAS:
		return parse_opt_a(tmdy_struct,arg);
	case TIM_OPT_BUFFER_FRAGS:
		return parse_opt_B(tmdy_struct, arg);
	case TIM_OPT_CONTROL_RATIO:
		return parse_opt_C(tmdy_struct, arg);
	case TIM_OPT_CONFIG_FILE:
		return parse_opt_c(tmdy_struct, arg);
	case TIM_OPT_DRUM_CHANNEL:
		return parse_opt_D(tmdy_struct, arg);
	case TIM_OPT_IFACE_PATH:
		return parse_opt_d(tmdy_struct, arg);
	case TIM_OPT_EXT:
		return parse_opt_E(tmdy_struct, arg);
	case TIM_OPT_MOD_WHEEL:
		return parse_opt_mod_wheel(tmdy_struct, arg);
	case TIM_OPT_PORTAMENTO:
		return parse_opt_portamento(tmdy_struct, arg);
	case TIM_OPT_VIBRATO:
		return parse_opt_vibrato(tmdy_struct, arg);
	case TIM_OPT_CH_PRESS:
		return parse_opt_ch_pressure(tmdy_struct, arg);
	case TIM_OPT_VOICE_LPF:
		return parse_opt_voice_lpf(tmdy_struct, arg);
	case TIM_OPT_MOD_ENV:
		return parse_opt_mod_env(tmdy_struct, arg);
	case TIM_OPT_TRACE_TEXT:
		return parse_opt_trace_text(tmdy_struct, arg);
	case TIM_OPT_OVERLAP:
		return parse_opt_overlap_voice(tmdy_struct, arg);
	case TIM_OPT_TEMPER_CTRL:
		return parse_opt_temper_control(tmdy_struct, arg);
	case TIM_OPT_DEFAULT_MID:
		return parse_opt_default_mid(tmdy_struct, arg);
	case TIM_OPT_SYSTEM_MID:
		return parse_opt_system_mid(tmdy_struct, arg);
	case TIM_OPT_DEFAULT_BANK:
		return parse_opt_default_bank(tmdy_struct, arg);
	case TIM_OPT_FORCE_BANK:
		return parse_opt_force_bank(tmdy_struct, arg);
	case TIM_OPT_DEFAULT_PGM:
		return parse_opt_default_program(tmdy_struct, arg);
	case TIM_OPT_FORCE_PGM:
		return parse_opt_force_program(tmdy_struct, arg);
	case TIM_OPT_DELAY:
		return parse_opt_delay(tmdy_struct, arg);
	case TIM_OPT_CHORUS:
		return parse_opt_chorus(tmdy_struct, arg);
	case TIM_OPT_REVERB:
		return parse_opt_reverb(tmdy_struct, arg);
	case TIM_OPT_NS:
		return parse_opt_noise_shaping(tmdy_struct, arg);
#ifndef FIXED_RESAMPLATION
	case TIM_OPT_RESAMPLE:
		return parse_opt_resample(tmdy_struct, arg);
#endif
	case TIM_OPT_EVIL:
		return parse_opt_e(tmdy_struct, arg);
	case TIM_OPT_FAST_PAN:
		return parse_opt_F(tmdy_struct, arg);
	case TIM_OPT_FAST_DECAY:
		return parse_opt_f(tmdy_struct, arg);
	case TIM_OPT_SPECTROGRAM:
		return parse_opt_g(tmdy_struct, arg);
	case TIM_OPT_KEYSIG:
		return parse_opt_H(tmdy_struct, arg);
	case TIM_OPT_HELP:
		return parse_opt_h(tmdy_struct, arg);
	case TIM_OPT_INTERFACE:
		return parse_opt_i(tmdy_struct, arg);
	case TIM_OPT_VERBOSE:
		return parse_opt_verbose(tmdy_struct, arg);
	case TIM_OPT_QUIET:
		return parse_opt_quiet(tmdy_struct, arg);
	case TIM_OPT_TRACE:
		return parse_opt_trace(tmdy_struct, arg);
	case TIM_OPT_LOOP:
		return parse_opt_loop(tmdy_struct, arg);
	case TIM_OPT_RANDOM:
		return parse_opt_random(tmdy_struct, arg);
	case TIM_OPT_SORT:
		return parse_opt_sort(tmdy_struct, arg);
#ifdef IA_ALSASEQ
	case TIM_OPT_BACKGROUND:
		return parse_opt_background(tmdy_struct, arg);
	case TIM_OPT_RT_PRIO:
		return parse_opt_rt_prio(tmdy_struct, arg);
	case TIM_OPT_SEQ_PORTS:
		return parse_opt_seq_ports(tmdy_struct, arg);
#endif
	case TIM_OPT_REALTIME_LOAD:
		return parse_opt_j(tmdy_struct, arg);
	case TIM_OPT_ADJUST_KEY:
		return parse_opt_K(tmdy_struct, arg);
	case TIM_OPT_VOICE_QUEUE:
		return parse_opt_k(tmdy_struct, arg);
	case TIM_OPT_PATCH_PATH:
		return parse_opt_L(tmdy_struct, arg);
	case TIM_OPT_PCM_FILE:
		return parse_opt_M(tmdy_struct, arg);
	case TIM_OPT_DECAY_TIME:
		return parse_opt_m(tmdy_struct, arg);
	case TIM_OPT_INTERPOLATION:
		return parse_opt_N(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_MODE:
		return parse_opt_O(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_STEREO:
		if (! strcmp(the_option->name, "output-mono"))
			/* --output-mono == --output-stereo=no */
			arg = "no";
		return parse_opt_output_stereo(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_SIGNED:
		if (! strcmp(the_option->name, "output-unsigned"))
			/* --output-unsigned == --output-signed=no */
			arg = "no";
		return parse_opt_output_signed(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_16BIT:
		if (! strcmp(the_option->name, "output-8bit"))
			/* --output-8bit == --output-16bit=no */
			arg = "no";
		return parse_opt_output_16bit(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_FORMAT:
		if (! strcmp(the_option->name, "output-linear"))
			arg = "linear";
		else if (! strcmp(the_option->name, "output-ulaw"))
			arg = "ulaw";
		else if (! strcmp(the_option->name, "output-alaw"))
			arg = "alaw";
		return parse_opt_output_format(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_SWAB:
		return parse_opt_output_swab(tmdy_struct, arg);
	case TIM_OPT_OUTPUT_FILE:
		return parse_opt_o(tmdy_struct, arg);
	case TIM_OPT_PATCH_FILE:
		return parse_opt_P(tmdy_struct, arg);
	case TIM_OPT_POLYPHONY:
		return parse_opt_p(tmdy_struct, arg);
	case TIM_OPT_POLY_REDUCE:
		return parse_opt_p1(tmdy_struct, arg);
	case TIM_OPT_MUTE:
		return parse_opt_Q(tmdy_struct, arg);
	case TIM_OPT_TEMPER_MUTE:
		return parse_opt_Q1(tmdy_struct, arg);
	case TIM_OPT_AUDIO_BUFFER:
		return parse_opt_q(tmdy_struct, arg);
	case TIM_OPT_CACHE_SIZE:
		return parse_opt_S(tmdy_struct, arg);
	case TIM_OPT_SAMPLE_FREQ:
		return parse_opt_s(tmdy_struct, arg);
	case TIM_OPT_ADJUST_TEMPO:
		return parse_opt_T(tmdy_struct, arg);
	case TIM_OPT_CHARSET:
		return parse_opt_t(tmdy_struct, arg);
	case TIM_OPT_UNLOAD_INST:
		return parse_opt_U(tmdy_struct, arg);
	case TIM_OPT_VERSION:
		return parse_opt_v(tmdy_struct, arg);
	case TIM_OPT_WRD:
		return parse_opt_W(tmdy_struct, arg);
#ifdef __W32__
	case TIM_OPT_RCPCV_DLL:
		return parse_opt_w(tmdy_struct, arg);
#endif
	case TIM_OPT_CONFIG_STR:
		return parse_opt_x(tmdy_struct, arg);
	case TIM_OPT_FREQ_TABLE:
		return parse_opt_Z(tmdy_struct, arg);
	case TIM_OPT_PURE_INT:
		return parse_opt_Z1(tmdy_struct, arg);
	default:
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
				"[BUG] Inconceivable case branch %d", c);
		abort();
	}
}

static inline int parse_opt_A(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* amplify volume by n percent */
	return set_val_i32(tmdy_struct, &tmdy_struct->playmidi->amplification, atoi(arg), 0, MAX_AMPLIFICATION,
			"Amplification");
}

static inline int parse_opt_drum_power(tmdy_struct_ex_t *tmdy_struct,const char *arg)
{
	/* --drum-power */
	return set_val_i32(tmdy_struct, &tmdy_struct->playmidi->opt_drum_power, atoi(arg), 0, MAX_AMPLIFICATION,
			"Drum power");
}

static inline int parse_opt_volume_comp(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]volume-compensation */
	tmdy_struct->playmidi->opt_amp_compensation = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_a(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->instrum->antialiasing_allowed = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_B(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --buffer-fragments */
	const char *p;
	
	/* num */
	if (*arg != ',') {
		if (set_value(tmdy_struct, &opt_buffer_fragments, atoi(arg), 0, 1000,
				"Buffer Fragments (num)"))
			return 1;
	}
	/* bits */
	if (p = strchr(arg, ',')) {
		if (set_value(tmdy_struct, &audio_buffer_bits, atoi(++p), 1, AUDIO_BUFFER_BITS,
				"Buffer Fragments (bit)"))
			return 1;
	}
	return 0;
}

static inline int parse_opt_C(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	if (set_val_i32(tmdy_struct, &tmdy_struct->playmidi->control_ratio, atoi(arg), 0, MAX_CONTROL_RATIO,
			"Control ratio"))
		return 1;
	(tmdy_struct->timidity_main->opt_control_ratio) = tmdy_struct->playmidi->control_ratio;
	return 0;
}

static inline int parse_opt_c(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	if (read_config_file(tmdy_struct, arg, 0))
		return 1;
	(tmdy_struct->timidity_main->got_a_configuration) = 1;
	return 0;
}

static inline int parse_opt_D(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	return set_channel_flag(tmdy_struct, &tmdy_struct->playmidi->default_drumchannels, atoi(arg), "Drum channel");
}

static inline int parse_opt_d(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* dynamic lib root */
#ifdef IA_DYNAMIC
	if (dynamic_lib_root)
		free(dynamic_lib_root);
	dynamic_lib_root = tmdy_struct->common->safe_strdup(tmdy_struct, arg);
	return 0;
#else
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NOISY, "-d option is not supported");
	return 1;
#endif	/* IA_DYNAMIC */
}

static inline int parse_opt_E(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	/* undocumented option --ext */
	int err = 0;
	
	while (*arg) {
		switch (*arg) {
		case 'w':
			tmdy_struct->playmidi->opt_modulation_wheel = 1;
			break;
		case 'W':
			tmdy_struct->playmidi->opt_modulation_wheel = 0;
			break;
		case 'p':
			tmdy_struct->playmidi->opt_portamento = 1;
			break;
		case 'P':
			tmdy_struct->playmidi->opt_portamento = 0;
			break;
		case 'v':
			tmdy_struct->playmidi->opt_nrpn_vibrato = 1;
			break;
		case 'V':
			tmdy_struct->playmidi->opt_nrpn_vibrato = 0;
			break;
		case 's':
			tmdy_struct->playmidi->opt_channel_pressure = 1;
			break;
		case 'S':
			tmdy_struct->playmidi->opt_channel_pressure = 0;
			break;
		case 'l':
			tmdy_struct->playmidi->opt_lpf_def = 1;
			break;
		case 'L':
			tmdy_struct->playmidi->opt_lpf_def = 0;
			break;
		case 'e':
			tmdy_struct->playmidi->opt_modulation_envelope = 1;
			break;
		case 'E':
			tmdy_struct->playmidi->opt_modulation_envelope = 0;
			break;
		case 't':
			tmdy_struct->readmidi->opt_trace_text_meta_event = 1;
			break;
		case 'T':
			tmdy_struct->readmidi->opt_trace_text_meta_event = 0;
			break;
		case 'o':
			tmdy_struct->playmidi->opt_overlap_voice_allow = 1;
			break;
		case 'O':
			tmdy_struct->playmidi->opt_overlap_voice_allow = 0;
			break;
		case 'z':
			tmdy_struct->playmidi->opt_temper_control = 1;
			break;
		case 'Z':
			tmdy_struct->playmidi->opt_temper_control = 0;
			break;
		case 'm':
			if (parse_opt_default_mid(tmdy_struct, arg + 1))
				err++;
			arg += 2;
			break;
		case 'M':
			if (parse_opt_system_mid(tmdy_struct, arg + 1))
				err++;
			arg += 2;
			break;
		case 'b':
			if (parse_opt_default_bank(tmdy_struct, arg + 1))
				err++;
			while (isdigit(*(arg + 1)))
				arg++;
			break;
		case 'B':
			if (parse_opt_force_bank(tmdy_struct, arg + 1))
				err++;
			while (isdigit(*(arg + 1)))
				arg++;
			break;
		case 'i':
			if (parse_opt_default_program(tmdy_struct, arg + 1))
				err++;
			while (isdigit(*(arg + 1)) || *(arg + 1) == '/')
				arg++;
			break;
		case 'I':
			if (parse_opt_force_program(tmdy_struct, arg + 1))
				err++;
			while (isdigit(*(arg + 1)) || *(arg + 1) == '/')
				arg++;
			break;
		case 'F':
			if (strncmp(arg + 1, "delay=", 6) == 0) {
				if (parse_opt_delay(tmdy_struct, arg + 7))
					err++;
			} else if (strncmp(arg + 1, "chorus=", 7) == 0) {
				if (parse_opt_chorus(tmdy_struct, arg + 8))
					err++;
			} else if (strncmp(arg + 1, "reverb=", 7) == 0) {
				if (parse_opt_reverb(tmdy_struct, arg + 8))
					err++;
			} else if (strncmp(arg + 1, "ns=", 3) == 0) {
				if (parse_opt_noise_shaping(tmdy_struct, arg + 4))
					err++;
#ifndef FIXED_RESAMPLATION
			} else if (strncmp(arg + 1, "resamp=", 7) == 0) {
				if (parse_opt_resample(tmdy_struct, arg + 8))
					err++;
#endif
			}
			if (err) {
				(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR,
						VERB_NORMAL, "-E%s: unsupported effect", arg);
				return err;
			}
			return err;
		default:
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
					"-E: Illegal mode `%c'", *arg);
			err++;
			break;
		}
		arg++;
	}
	return err;
}

static inline int parse_opt_mod_wheel(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]mod-wheel */
	tmdy_struct->playmidi->opt_modulation_wheel = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_portamento(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]portamento */
	tmdy_struct->playmidi->opt_portamento = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_vibrato(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]vibrato */
	tmdy_struct->playmidi->opt_nrpn_vibrato = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_ch_pressure(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]ch-pressure */
	tmdy_struct->playmidi->opt_channel_pressure = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_voice_lpf(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]voice-lpf */
	tmdy_struct->playmidi->opt_lpf_def = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_mod_env(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]mod-envelope */
	tmdy_struct->playmidi->opt_modulation_envelope = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_trace_text(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]trace-text-meta */
	tmdy_struct->readmidi->opt_trace_text_meta_event = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_overlap_voice(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]overlap-voice */
	tmdy_struct->playmidi->opt_overlap_voice_allow = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_temper_control(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]temper-control */
	tmdy_struct->playmidi->opt_temper_control = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_default_mid(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	/* --default-mid */
	int val = tmdy_struct->common->str2mID(tmdy_struct, arg);
	
	if (! val) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Manufacture ID: Illegal value");
		return 1;
	}
	tmdy_struct->readmidi->opt_default_mid = val;
	return 0;
}

static inline int parse_opt_system_mid(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	/* --system-mid */
	int val = tmdy_struct->common->str2mID(tmdy_struct, arg);
	
	if (! val) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Manufacture ID: Illegal value");
		return 1;
	}
	tmdy_struct->readmidi->opt_system_mid = val;
	return 0;
}

static inline int parse_opt_default_bank(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --default-bank */
	if (set_value(tmdy_struct, &tmdy_struct->playmidi->default_tonebank, atoi(arg), 0, 0x7f, "Bank number"))
		return 1;
	tmdy_struct->playmidi->special_tonebank = -1;
	return 0;
}

static inline int parse_opt_force_bank(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --force-bank */
	if (set_value(tmdy_struct, &tmdy_struct->playmidi->special_tonebank, atoi(arg), 0, 0x7f, "Bank number"))
		return 1;
	return 0;
}

static inline int parse_opt_default_program(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --default-program */
	int prog, i;
	const char *p;
	
	if (set_value(tmdy_struct, &prog, atoi(arg), 0, 0x7f, "Program number"))
		return 1;
	if (p = strchr(arg, '/')) {
		if (set_value(tmdy_struct, &i, atoi(++p), 1, MAX_CHANNELS, "Program channel"))
			return 1;
		tmdy_struct->instrum->default_program[i - 1] = prog;
	} else
		for (i = 0; i < MAX_CHANNELS; i++)
			tmdy_struct->instrum->default_program[i] = prog;
	return 0;
}

static inline int parse_opt_force_program(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --force-program */
	const char *p;
	int i;
	
	if (set_value(tmdy_struct, &(tmdy_struct->timidity_main->def_prog), atoi(arg), 0, 0x7f, "Program number"))
		return 1;
	if ((tmdy_struct->controls->ctl)->opened)
		set_default_program(tmdy_struct, (tmdy_struct->timidity_main->def_prog));
	if (p = strchr(arg, '/')) {
		if (set_value(tmdy_struct, &i, atoi(++p), 1, MAX_CHANNELS, "Program channel"))
			return 1;
		tmdy_struct->instrum->default_program[i - 1] = SPECIAL_PROGRAM;
	} else
		for (i = 0; i < MAX_CHANNELS; i++)
			tmdy_struct->instrum->default_program[i] = SPECIAL_PROGRAM;
	return 0;
}

static inline int set_default_program(tmdy_struct_ex_t *tmdy_struct, int prog)
{
	int bank;
	Instrument *ip;
	
	bank = (tmdy_struct->playmidi->special_tonebank >= 0) ? tmdy_struct->playmidi->special_tonebank : tmdy_struct->playmidi->default_tonebank;
	if ((ip = tmdy_struct->playmidi->play_midi_load_instrument(tmdy_struct, 0, bank, prog)) == NULL)
		return 1;
	tmdy_struct->instrum->default_instrument = ip;
	return 0;
}

static inline int parse_opt_delay(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --delay */
	const char *p;
	
	switch (*arg) {
	case '0':
	case 'd':	/* disable */
		(tmdy_struct->timidity_main->effect_lr_mode) = -1;
		return 0;
	case 'l':	/* left */
		(tmdy_struct->timidity_main->effect_lr_mode) = 0;
		break;
	case 'r':	/* right */
		(tmdy_struct->timidity_main->effect_lr_mode) = 1;
		break;
	case 'b':	/* both */
		(tmdy_struct->timidity_main->effect_lr_mode) = 2;
		break;
	}
	if (p = strchr(arg, ','))
		if (((tmdy_struct->timidity_main->effect_lr_delay_msec) = atoi(++p)) < 0) {
			(tmdy_struct->timidity_main->effect_lr_delay_msec) = 0;
			(tmdy_struct->timidity_main->effect_lr_mode) = -1;
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Invalid delay parameter.");
			return 1;
		}
	return 0;
}

static inline int parse_opt_chorus(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --chorus */
	const char *p;
	
	switch (*arg) {
	case '0':
	case 'd':	/* disable */
		tmdy_struct->playmidi->opt_chorus_control = 0;
		tmdy_struct->playmidi->opt_surround_chorus = 0;
		break;
	case '1':
	case 'n':	/* normal */
	case '2':
	case 's':	/* surround */
		tmdy_struct->playmidi->opt_surround_chorus = (*arg == '2' || *arg == 's') ? 1 : 0;
		if (p = strchr(arg, ',')) {
			if (set_value(tmdy_struct, &tmdy_struct->playmidi->opt_chorus_control, atoi(++p), 0, 0x7f,
					"Chorus level"))
				return 1;
			tmdy_struct->playmidi->opt_chorus_control = -tmdy_struct->playmidi->opt_chorus_control;
		} else
			tmdy_struct->playmidi->opt_chorus_control = 1;
		break;
	default:
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Invalid chorus parameter.");
		return 1;
	}
	return 0;
}

static inline int parse_opt_reverb(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --reverb */
	const char *p;
	
	switch (*arg) {
	case '0':
	case 'd':	/* disable */
		tmdy_struct->playmidi->opt_reverb_control = 0;
		break;
	case '1':
	case 'n':	/* normal */
		if (p = strchr(arg, ',')) {
			if (set_value(tmdy_struct, &tmdy_struct->playmidi->opt_reverb_control, atoi(++p), 0, 0x7f,
					"Reverb level"))
				return 1;
			tmdy_struct->playmidi->opt_reverb_control = -tmdy_struct->playmidi->opt_reverb_control;
		} else
			tmdy_struct->playmidi->opt_reverb_control = 1;
		break;
	case '2':
	case 'g':	/* global */
		tmdy_struct->playmidi->opt_reverb_control = 2;
		break;
	case '3':
	case 'f':	/* freeverb */
		tmdy_struct->playmidi->opt_reverb_control = 3;
		break;
	default:
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Invalid reverb parameter.");
		return 1;
	}
	return 0;
}

/* Noise Shaping filter from
 * Kunihiko IMAI <imai@leo.ec.t.kanazawa-u.ac.jp>
 */
static inline int parse_opt_noise_shaping(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --noise-shaping */
	if (set_value(tmdy_struct, &tmdy_struct->aq->noise_sharp_type, atoi(arg), 0, 4, "Noise shaping type"))
		return 1;
	return 0;
}

static inline int parse_opt_resample(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --resample */
	switch (*arg) {
	case '0':
	case 'd':	/* disable */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 5);
		break;
	case '1':
	case 'l':	/* linear */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 4);
		break;
	case '2':
	case 'c':	/* cspline */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 0);
		break;
	case '3':
	case 'L':	/* lagrange */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 1);
		break;
	case '4':
	case 'n':	/* newton */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 3);
		break;
	case '5':
	case 'g':	/* guass */
		(tmdy_struct->resample->set_current_resampler)(tmdy_struct, 2);
		break;
	default:
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Invalid resample type %s", arg);
		return 1;
	}
	return 0;
}

static inline int parse_opt_e(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* evil */
#ifdef __W32__
	opt_evil_mode = 1;
	return 0;
#else
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "-e option is not supported");
	return 1;
#endif /* __W32__ */
}

static inline int parse_opt_F(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->playmidi->adjust_panning_immediately = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_f(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->instrum->fast_decay = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_g(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
#ifdef SUPPORT_SOUNDSPEC
	spectrogram_update_sec = atof(arg);
	if (spectrogram_update_sec <= 0) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"Invalid -g argument: `%s'", arg);
		return 1;
	}
	view_soundspec_flag = 1;
	return 0;
#else
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "-g option is not supported");
	return 1;
#endif	/* SUPPORT_SOUNDSPEC */
}

static inline int parse_opt_H(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* force keysig (number of sharp/flat) */
	int keysig;
	
	if (set_value(tmdy_struct, &keysig, atoi(arg), -7, 7,
			"Force keysig (number of sHarp(+)/flat(-))"))
		return 1;
	tmdy_struct->playmidi->opt_force_keysig = keysig;
	return 0;
}

__attribute__((noreturn))
static inline int parse_opt_h(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	static char *help_list[] = {
"TiMidity++ version %s (C) 1999-2003 Masanao Izumo <mo@goice.co.jp>",
"The original version (C) 1995 Tuukka Toivonen <tt@cgs.fi>",
"TiMidity is free software and comes with ABSOLUTELY NO WARRANTY.",
"",
#ifdef __W32__
"Win32 version by Davide Moretti <dave@rimini.com>",
"              and Daisuke Aoki <dai@y7.net>",
"",
#endif /* __W32__ */
"Usage:",
"  %s [options] filename [...]",
"",
#ifndef __W32__		/*does not work in Windows */
"  Use \"-\" as filename to read a MIDI file from stdin",
"",
#endif
"Options:",
"  -A n,m     --volume=n, --drum-power=m",
"               Amplify volume by n percent (may cause clipping),",
"                 and amplify drum power by m percent",
"     (a)     --[no-]volume-compensation",
"               Toggle amplify compensation (disabled by default)",
"  -a         --[no-]anti-alias",
"               Enable the anti-aliasing filter",
"  -B n,m     --buffer-fragments=n,m",
"               Set number of buffer fragments(n), and buffer size(2^m)",
"  -C n       --control-ratio=n",
"               Set ratio of sampling and control frequencies",
"  -c file    --config-file=file",
"               Read extra configuration file",
"  -D n       --drum-channel=n",
"               Play drums on channel n",
#ifdef IA_DYNAMIC
"  -d path    --interface-path=path",
"               Set dynamic interface module directory",
#endif /* IA_DYNAMIC */
"  -E mode    --ext=mode",
"               TiMidity sequencer extensional modes:",
"                 mode = w/W : Enable/Disable Modulation wheel",
"                        p/P : Enable/Disable Portamento",
"                        v/V : Enable/Disable NRPN Vibrato",
"                        s/S : Enable/Disable Channel pressure",
"                        l/L : Enable/Disable voice-by-voice LPF",
"                        e/E : Enable/Disable Modulation Envelope",
"                        t/T : Enable/Disable Trace Text Meta Event at playing",
"                        o/O : Enable/Disable Overlapped voice",
"                        z/Z : Enable/Disable Temperament control",
"                        m<HH>: Define default Manufacture ID <HH> in two hex",
"                        M<HH>: Define system Manufacture ID <HH> in two hex",
"                        b<n>: Use tone bank <n> as the default",
"                        B<n>: Always use tone bank <n>",
"                        i<n/m>: Use program <n> on channel <m> as the default",
"                        I<n/m>: Always use program <n> on channel <m>",
"                        F<args>: For effect.  See below for effect options",
"                   default: -E "
#ifdef MODULATION_WHEEL_ALLOW
"w"
#else
"W"
#endif /* MODULATION_WHEEL_ALLOW */
#ifdef PORTAMENTO_ALLOW
"p"
#else
"P"
#endif /* PORTAMENTO_ALLOW */
#ifdef NRPN_VIBRATO_ALLOW
"v"
#else
"V"
#endif /* NRPN_VIBRATO_ALLOW */
#ifdef GM_CHANNEL_PRESSURE_ALLOW
"s"
#else
"S"
#endif /* GM_CHANNEL_PRESSURE_ALLOW */
#ifdef VOICE_BY_VOICE_LPF_ALLOW
"l"
#else
"L"
#endif /* VOICE_BY_VOICE_LPF_ALLOW */
#ifdef MODULATION_ENVELOPE_ALLOW
"e"
#else
"E"
#endif /* MODULATION_ENVELOPE_ALLOW */
#ifdef ALWAYS_TRACE_TEXT_META_EVENT
"t"
#else
"T"
#endif /* ALWAYS_TRACE_TEXT_META_EVENT */
#ifdef OVERLAP_VOICE_ALLOW
"o"
#else
"O"
#endif /* OVERLAP_VOICE_ALLOW */
#ifdef TEMPER_CONTROL_ALLOW
"z"
#else
"Z"
#endif /* TEMPER_CONTROL_ALLOW */
,
#ifdef __W32__
"  -e         --evil",
"               Increase thread priority (evil) - be careful!",
#endif
"  -F         --[no-]fast-panning",
"               Disable/Enable fast panning (toggle on/off, default is on)",
"  -f         --[no-]fast-decay",
"               "
#ifdef FAST_DECAY
"Disable "
#else
"Enable "
#endif
"fast decay mode (toggle)",
#ifdef SUPPORT_SOUNDSPEC
"  -g sec     --spectrogram=sec",
"               Open Sound-Spectrogram Window",
#endif /* SUPPORT_SOUNDSPEC */
"  -H n       --force-keysig=n",
"               Force keysig number of sHarp(+)/flat(-) (-7..7)",
"  -h         --help",
"               Display this help message",
"  -i mode    --interface=mode",
"               Select user interface (see below for list)",
#ifdef IA_ALSASEQ
"             --realtime-priority=n (for alsaseq only)",
"               Set the realtime priority (0-100)",
"             --sequencer-ports=n (for alsaseq only)",
"               Set the number of opened sequencer ports (default is 4)",
#endif
"  -j         --[no-]realtime-load",
"               Realtime load instrument (toggle on/off)",
"  -K n       --adjust-key=n",
"               Adjust key by n half tone (-24..24)",
"  -k msec    --voice-queue=msec",
"               Specify audio queue time limit to reduce voice",
"  -L path    --patch-path=path",
"               Append dir to search path",
"  -M name    --pcm-file=name",
"               Specify PCM filename (*.wav or *.aiff) to be played or:",
"                 \"auto\" : Play *.mid.wav or *.mid.aiff",
"                 \"none\" : Disable this feature (default)",
"  -m msec    --decay-time=msec",
"               Minimum time for a full volume sustained note to decay,",
"                 0 disables",
"  -N n       --interpolation=n",
"               Set the interpolation parameter (depends on -EFresamp option)",
"                 Linear interpolation is used if audio queue < 99%%",
"                 cspline, lagrange:",
"                   Toggle 4-point interpolation (default on)",
"                 newton:",
"                   n'th order Newton polynomial interpolation, n=1-57 odd",
"                 gauss:",
"                   n+1 point Gauss-like interpolation, n=1-34 (default 25)",
"  -O mode    --output-mode=mode",
"               Select output mode and format (see below for list)",
"  -o file    --output-file=file",
"               Output to another file (or device/server) (Use \"-\" for stdout)",
"  -P file    --patch-file=file",
"               Use patch file for all programs",
"  -p n       --polyphony=n",
"               Allow n-voice polyphony.  Optional auto polyphony reduction",
"     (a)     --[no-]auto-poly-reduction",
"               Toggle automatic polyphony reduction.  Enabled by default",
"  -Q n[,...] --mute=n",
"               Ignore channel n (0: ignore all, -n: resume channel n)",
"     (t)     --temper-mute=n",
"               Quiet temperament type n (0..3: preset, 4..7: user-defined)",
"  -q sec/n   --audio-buffer=sec/n",
"               Specify audio buffer in seconds",
"                 sec: Maxmum buffer, n: Filled to start (default is 5.0/100%%)",
"                 (size of 100%% equals device buffer size)",
"  -R n         Pseudo reveb effect (set every instrument's release to n ms)",
"                 if n=0, n is set to 800",
"  -S n       --cache-size=n",
"               Cache size (0 means no cache)",
"  -s freq    --sampling-freq=freq",
"               Set sampling frequency to freq (Hz or kHz)",
"  -T n       --adjust-tempo=n",
"               Adjust tempo to n%%,",
"                 120=play MOD files with an NTSC Amiga's timing",
"  -t code    --output-charset=code",
"               Output text language code:",
"                 code=auto  : Auto conversion by `LANG' environment variable",
"                              (UNIX only)",
"                      ascii : Convert unreadable characters to '.' (0x2e)",
"                      nocnv : No conversion",
"                      1251  : Convert from windows-1251 to koi8-r",
#ifdef JAPANESE
"                      euc   : EUC-japan",
"                      jis   : JIS",
"                      sjis  : shift JIS",
#endif /* JAPANESE */
"  -U         --[no-]unload-instruments",
"               Unload instruments from memory between MIDI files",
"  -v         --version",
"               Display TiMidity version information",
"  -W mode    --wrd=mode",
"               Select WRD interface (see below for list)",
#ifdef __W32__
"  -w mode    --rcpcv-dll=mode",
"               Windows extensional modes:",
"                 mode=r/R : Enable/Disable rcpcv.dll",
#endif /* __W32__ */
"  -x str     --config-string=str",
"               Read configuration str from command line argument",
"  -Z file    --freq-table=file",
"               Load frequency table (Use \"pure\" for pure intonation)",
"  pure<n>(m) --pure-intonation=n(m)",
"               Initial keysig number <n> of sharp(+)/flat(-) (-7..7)",
"                 'm' stands for minor mode",
		NULL
	};
	static char *help_args[3];
	FILE *fp;
	int i, j;
	char *h;
	ControlMode *cmp, **cmpp;
	char mark[128];
	PlayMode *pmp, **pmpp;
	WRDTracer *wlp, **wlpp;
	
	fp = open_pager(tmdy_struct);
	help_args[0] = (tmdy_struct->common->timidity_version);
	help_args[1] = (tmdy_struct->common->program_name);
	help_args[2] = NULL;
	for (i = 0, j = 0; h = help_list[i]; i++) {
		if (strchr(h, '%')) {
			if (*(strchr(h, '%') + 1) != '%')
				fprintf(fp, h, help_args[j++]);
			else
				fprintf(fp, h);
		} else
			fputs(h, fp);
		fputs(NLS, fp);
	}
	fputs(NLS, fp);
	fputs("Effect options (-EF, --ext=F option):" NLS
"  -EFdelay=d   Disable delay effect (default)" NLS
"  -EFdelay=l   Enable Left delay" NLS
"    [,msec]      `msec' is optional to specify left-right delay time" NLS
"  -EFdelay=r   Enable Right delay" NLS
"    [,msec]      `msec' is optional to specify left-right delay time" NLS
"  -EFdelay=b   Enable rotate Both left and right" NLS
"    [,msec]      `msec' is optional to specify left-right delay time" NLS
"  -EFchorus=d  Disable MIDI chorus effect control" NLS
"  -EFchorus=n  Enable Normal MIDI chorus effect control" NLS
"    [,level]     `level' is optional to specify chorus level [0..127]" NLS
"                 (default)" NLS
"  -EFchorus=s  Surround sound, chorus detuned to a lesser degree" NLS
"    [,level]     `level' is optional to specify chorus level [0..127]" NLS
"  -EFreverb=d  Disable MIDI reverb effect control" NLS
"  -EFreverb=n  Enable Normal MIDI reverb effect control" NLS
"    [,level]     `level' is optional to specify reverb level [0..127]" NLS
"                 This effect is only available in stereo" NLS
"  -EFreverb=g  Global reverb effect" NLS
"  -EFreverb=f  Enable Freeverb MIDI reverb effect control" NLS
"                 This effect is only available in stereo (default)" NLS
"  -EFns=n      Enable the n th degree noise shaping filter" NLS
"                 n:[0..4] (for 8-bit linear encoding, default is 4)" NLS
"                 n:[0..2] (for 16-bit linear encoding, default is 2)" NLS, fp);
#ifndef FIXED_RESAMPLATION
	fputs(
"  -EFresamp=d  Disable resamplation" NLS
"  -EFresamp=l  Enable Linear resample algorithm" NLS
"  -EFresamp=c  Enable C-spline resample algorithm" NLS
"  -EFresamp=L  Enable Lagrange resample algorithm" NLS
"  -EFresamp=n  Enable Newton resample algorithm" NLS
"  -EFresamp=g  Enable Gauss-like resample algorithm (default)" NLS
"                 -EFresamp affects the behavior of -N option" NLS, fp);
#endif
	fputs(NLS, fp);
	fputs("Alternative TiMidity sequencer extensional mode long options:" NLS
"  --[no-]mod-wheel" NLS
"  --[no-]portamento" NLS
"  --[no-]vibrato" NLS
"  --[no-]ch-pressure" NLS
"  --[no-]voice-lpf" NLS
"  --[no-]mod-envelope" NLS
"  --[no-]trace-text-meta" NLS
"  --[no-]overlap-voice" NLS
"  --[no-]temper-control" NLS
"  --default-mid=<HH>" NLS
"  --system-mid=<HH>" NLS
"  --default-bank=n" NLS
"  --force-bank=n" NLS
"  --default-program=n/m" NLS
"  --force-program=n/m" NLS
"  --delay=(d|l|r|b)[,msec]" NLS
"  --chorus=(d|n|s)[,level]" NLS
"  --reverb=(d|n|g|f)[,level]" NLS
"  --noise-shaping=n" NLS, fp);
#ifndef FIXED_RESAMPLATION
	fputs("  --resample=(d|l|c|L|n|g)" NLS, fp);
#endif
	fputs(NLS, fp);
	fputs("Available interfaces (-i, --interface option):" NLS, fp);
	for (cmpp = (tmdy_struct->controls->ctl_list); cmp = *cmpp; cmpp++)
#ifdef IA_DYNAMIC
		if (cmp->id_character != tmdy_struct->timidity_main->dynamic_interface_id)
			fprintf(fp, "  -i%c          %s" NLS,
					cmp->id_character, cmp->id_name);
#else
		fprintf(fp, "  -i%c          %s" NLS,
				cmp->id_character, cmp->id_name);
#endif	/* IA_DYNAMIC */
#ifdef IA_DYNAMIC
	fprintf(fp, "Supported dynamic load interfaces (%s):" NLS,
			dynamic_lib_root);
	memset(mark, 0, sizeof(mark));
	for (cmpp = (tmdy_struct->controls->ctl_list); cmp = *cmpp; cmpp++)
		mark[(int) cmp->id_character] = 1;
	if (tmdy_struct->timidity_main->dynamic_interface_id != 0)
		mark[(int) tmdy_struct->timidity_main->dynamic_interface_id] = 0;
	list_dyna_interface(tmdy_struct, fp, dynamic_lib_root, mark);
#endif	/* IA_DYNAMIC */
	fputs(NLS, fp);
	fputs("Interface options (append to -i? option):" NLS
"  `v'          more verbose (cumulative)" NLS
"  `q'          quieter (cumulative)" NLS
"  `t'          trace playing" NLS
"  `l'          loop playing (some interface ignore this option)" NLS
"  `r'          randomize file list arguments before playing" NLS
"  `s'          sorting file list arguments before playing" NLS, fp);
#ifdef IA_ALSASEQ
	fputs("  `D'          daemonize TiMidity++ in background "
			"(for alsaseq only)" NLS, fp);
#endif
	fputs(NLS, fp);
	fputs("Alternative interface long options:" NLS
"  --verbose=n" NLS
"  --quiet=n" NLS
"  --[no-]trace" NLS
"  --[no-]loop" NLS
"  --[no-]random" NLS
"  --[no-]sort" NLS, fp);
#ifdef IA_ALSASEQ
	fputs("  --[no-]background" NLS, fp);
#endif
	fputs(NLS, fp);
	fputs("Available output modes (-O, --output-mode option):" NLS, fp);
	for (pmpp = (tmdy_struct->output->play_mode_list); pmp = *pmpp; pmpp++)
		fprintf(fp, "  -O%c          %s" NLS,
				pmp->id_character, pmp->id_name);
	fputs(NLS, fp);
	fputs("Output format options (append to -O? option):" NLS
"  `S'          stereo" NLS
"  `M'          monophonic" NLS
"  `s'          signed output" NLS
"  `u'          unsigned output" NLS
"  `1'          16-bit sample width" NLS
"  `8'          8-bit sample width" NLS
"  `l'          linear encoding" NLS
"  `U'          U-Law encoding" NLS
"  `A'          A-Law encoding" NLS
"  `x'          byte-swapped output" NLS, fp);
	fputs(NLS, fp);
	fputs("Alternative output format long options:" NLS
"  --output-stereo" NLS
"  --output-mono" NLS
"  --output-signed" NLS
"  --output-unsigned" NLS
"  --output-16bit" NLS
"  --output-8bit" NLS
"  --output-linear" NLS
"  --output-ulaw" NLS
"  --output-alaw" NLS
"  --[-no]output-swab" NLS, fp);
	fputs(NLS, fp);
	fputs("Available WRD interfaces (-W, --wrd option):" NLS, fp);
	for (wlpp = tmdy_struct->wrd->wrdt_list; wlp = *wlpp; wlpp++)
		fprintf(fp, "  -W%c          %s" NLS, wlp->id, wlp->name);
	fputs(NLS, fp);
	close_pager(tmdy_struct, fp);
	exit(EXIT_SUCCESS);
}

#ifdef IA_DYNAMIC
static inline void list_dyna_interface(tmdy_struct_ex_t *tmdy_struct, FILE *fp, char *path, char *mark)
{
	URL url;
	char fname[BUFSIZ], *info;
	int id;
	
	if ((url = url_dir_open(path)) == NULL)
		return;
	while (url_gets(url, fname, sizeof(fname)) != NULL)
		if (strncmp(fname, "interface_", 10) == 0) {
			id = fname[10];
			if (mark[id])
				continue;
			mark[id] = 1;
			if ((info = dynamic_interface_info(tmdy_struct, id)) == NULL)
				info = dynamic_interface_module(tmdy_struct, id);
			if (info != NULL)
				fprintf(fp, "  -i%c          %s" NLS, id, info);
		}
	url_close(url);
}

static inline char *dynamic_interface_info(tmdy_struct_ex_t *tmdy_struct, int id)
{
	static char libinfo[MAXPATHLEN];
	int fd, n;
	char *nl;
	
	sprintf(libinfo, "%s" PATH_STRING "interface_%c.txt",
			dynamic_lib_root, id);
	if ((fd = open(libinfo, 0)) < 0)
		return NULL;
	n = read(fd, libinfo, sizeof(libinfo) - 1);
	close(fd);
	if (n <= 0)
		return NULL;
	libinfo[n] = '\0';
	if ((nl = strchr(libinfo, '\n')) == libinfo)
		return NULL;
	if (nl != NULL) {
		*nl = '\0';
		if (*(nl - 1) == '\r')
			*(nl - 1) = '\0';
	}
	return libinfo;
}

char *dynamic_interface_module(tmdy_struct_ex_t *tmdy_struct, int id)
{
	static char shared_library[MAXPATHLEN];
	int fd;
	
	sprintf(shared_library, "%s" PATH_STRING "interface_%c%s",
			dynamic_lib_root, id, SHARED_LIB_EXT);
	if ((fd = open(shared_library, 0)) < 0)
		return NULL;
	close(fd);
	return shared_library;
}
#endif	/* IA_DYNAMIC */

static inline int parse_opt_i(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* interface mode */
	ControlMode *cmp, **cmpp;
	int found = 0;
	
	for (cmpp = (tmdy_struct->controls->ctl_list); cmp = *cmpp; cmpp++) {
		if (cmp->id_character == *arg) {
			found = 1;
			(tmdy_struct->controls->ctl) = cmp;
#if defined(IA_W32GUI) || defined(IA_W32G_SYN)
			cmp->verbosity = 1;
			cmp->trace_playing = 0;
			cmp->flags = 0;
#endif	/* IA_W32GUI */
			break;
		}
#ifdef IA_DYNAMIC
		if (cmp->id_character == tmdy_struct->timidity_main->dynamic_interface_id
				&& dynamic_interface_module(tmdy_struct, *arg)) {
			/* Dynamic interface loader */
			found = 1;
			(tmdy_struct->controls->ctl) = cmp;
			if (tmdy_struct->timidity_main->dynamic_interface_id != *arg) {
				cmp->id_character = tmdy_struct->timidity_main->dynamic_interface_id = *arg;
				cmp->verbosity = 1;
				cmp->trace_playing = 0;
				cmp->flags = 0;
			}
			break;
		}
#endif	/* IA_DYNAMIC */
	}
	if (! found) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"Interface `%c' is not compiled in.", *arg);
		return 1;
	}
	while (*(++arg))
		switch (*arg) {
		case 'v':
			cmp->verbosity++;
			break;
		case 'q':
			cmp->verbosity--;
			break;
		case 't':	/* toggle */
			cmp->trace_playing = (cmp->trace_playing) ? 0 : 1;
			break;
		case 'l':
			cmp->flags ^= CTLF_LIST_LOOP;
			break;
		case 'r':
			cmp->flags ^= CTLF_LIST_RANDOM;
			break;
		case 's':
			cmp->flags ^= CTLF_LIST_SORT;
			break;
		case 'a':
			cmp->flags ^= CTLF_AUTOSTART;
			break;
		case 'x':
			cmp->flags ^= CTLF_AUTOEXIT;
			break;
		case 'd':
			cmp->flags ^= CTLF_DRAG_START;
			break;
		case 'u':
			cmp->flags ^= CTLF_AUTOUNIQ;
			break;
		case 'R':
			cmp->flags ^= CTLF_AUTOREFINE;
			break;
		case 'C':
			cmp->flags ^= CTLF_NOT_CONTINUE;
			break;
#ifdef IA_ALSASEQ
		case 'D':
			cmp->flags ^= CTLF_DAEMONIZE;
			break;
#endif
		default:
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
					"Unknown interface option `%c'", *arg);
			return 1;
		}
	return 0;
}

static inline int parse_opt_verbose(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --verbose */
	(tmdy_struct->controls->ctl)->verbosity += (arg) ? atoi(arg) : 1;
	return 0;
}

static inline int parse_opt_quiet(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --quiet */
	(tmdy_struct->controls->ctl)->verbosity -= (arg) ? atoi(arg) : 1;
	return 0;
}

static inline int parse_opt_trace(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]trace */
	(tmdy_struct->controls->ctl)->trace_playing = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_loop(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]loop */
	return set_flag(tmdy_struct, &((tmdy_struct->controls->ctl)->flags), CTLF_LIST_LOOP, arg);
}

static inline int parse_opt_random(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]random */
	return set_flag(tmdy_struct, &((tmdy_struct->controls->ctl)->flags), CTLF_LIST_RANDOM, arg);
}

static inline int parse_opt_sort(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]sort */
	return set_flag(tmdy_struct, &((tmdy_struct->controls->ctl)->flags), CTLF_LIST_SORT, arg);
}

#ifdef IA_ALSASEQ
static inline int parse_opt_background(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]background */
	return set_flag(tmdy_struct, &((tmdy_struct->controls->ctl)->flags), CTLF_DAEMONIZE, arg);
}

static inline int parse_opt_rt_prio(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --realtime-priority */
	if (set_value(tmdy_struct, &opt_realtime_priority, atoi(arg), 0, 100,
			"Realtime priority"))
		return 1;
	return 0;
}

static inline int parse_opt_seq_ports(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --sequencer-ports */
	if (set_value(tmdy_struct, &opt_sequencer_ports, atoi(arg), 1, 16,
			"Number of sequencer ports"))
		return 1;
	return 0;
}
#endif

static inline int parse_opt_j(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->playmidi->opt_realtime_playing = y_or_n_p(tmdy_struct, arg);
}

static inline int parse_opt_K(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* key adjust */
	if (set_value(tmdy_struct, &tmdy_struct->playmidi->key_adjust, atoi(arg), -24, 24, "Key adjust"))
		return 1;
	return 0;
}

static inline int parse_opt_k(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->playmidi->reduce_voice_threshold = atoi(arg);
	return 0;
}

static inline int parse_opt_L(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	tmdy_struct->common->add_to_pathlist(tmdy_struct, arg);
	try_config_again = 1;
	return 0;
}

static inline int parse_opt_M(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	if ((tmdy_struct->timidity_main->pcm_alternate_file))
		free((tmdy_struct->timidity_main->pcm_alternate_file));
	(tmdy_struct->timidity_main->pcm_alternate_file) = tmdy_struct->common->safe_strdup(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_m(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->mix->min_sustain_time = atoi(arg);
	if (tmdy_struct->mix->min_sustain_time < 0)
		tmdy_struct->mix->min_sustain_time = 0;
	return 0;
}

static inline int parse_opt_N(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	int val;
	
	switch ( (tmdy_struct->resample->get_current_resampler)(tmdy_struct)) {
	case RESAMPLE_CSPLINE:
	case RESAMPLE_LAGRANGE:
		tmdy_struct->playmidi->no_4point_interpolation = y_or_n_p(tmdy_struct, arg);
		break;
	case RESAMPLE_NEWTON:
	case RESAMPLE_GAUSS:
		if (! (val = atoi(arg)))
			/* set to linear interpolation for compatibility */
			(tmdy_struct->resample->set_current_resampler)(tmdy_struct, RESAMPLE_LINEAR);
		else if ((tmdy_struct->resample->set_resampler_parm)(tmdy_struct, val)) {
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "Invalid -N value");
			return 1;
		}
		break;
	}
	return 0;
}

static inline int parse_opt_O(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* output mode */
	PlayMode *pmp, **pmpp;
	int found = 0;
	
	for (pmpp = (tmdy_struct->output->play_mode_list); pmp = *pmpp; pmpp++)
		if (pmp->id_character == *arg) {
			found = 1;
			(tmdy_struct->output->play_mode) = pmp;
			break;
		}
	if (! found) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"Playmode `%c' is not compiled in.", *arg);
		return 1;
	}
	while (*(++arg))
		switch (*arg) {
		case 'S':	/* stereo */
			pmp->encoding &= ~PE_MONO;
			break;
		case 'M':
			pmp->encoding |= PE_MONO;
			break;
		case 's':
			pmp->encoding |= PE_SIGNED;
			pmp->encoding &= ~(PE_ULAW | PE_ALAW);
			break;
		case 'u':
			pmp->encoding &= ~PE_SIGNED;
			pmp->encoding &= ~(PE_ULAW | PE_ALAW);
			break;
		case '1':	/* 1 for 16-bit */
			pmp->encoding |= PE_16BIT;
			pmp->encoding &= ~(PE_ULAW | PE_ALAW);
			break;
		case '8':
			pmp->encoding &= ~PE_16BIT;
			break;
		case 'l':	/* linear */
			pmp->encoding &= ~(PE_ULAW | PE_ALAW);
			break;
		case 'U':	/* uLaw */
			pmp->encoding |= PE_ULAW;
			pmp->encoding &=
					~(PE_SIGNED | PE_16BIT | PE_ALAW | PE_BYTESWAP);
			break;
		case 'A':	/* aLaw */
			pmp->encoding |= PE_ALAW;
			pmp->encoding &=
					~(PE_SIGNED | PE_16BIT | PE_ULAW | PE_BYTESWAP);
			break;
		case 'x':
			pmp->encoding ^= PE_BYTESWAP;	/* toggle */
			pmp->encoding &= ~(PE_ULAW | PE_ALAW);
			break;
		default:
			(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
					"Unknown format modifier `%c'", *arg);
			return 1;
		}
	return 0;
}

static inline int parse_opt_output_stereo(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --output-stereo, --output-mono */
	if (y_or_n_p(tmdy_struct, arg))
		/* I first thought --mono should be the syntax sugar to
		 * --stereo=no, but the source said stereo should be !PE_MONO,
		 * not mono should be !PE_STEREO.  Perhaps I took a wrong
		 * choice? -- mput
		 */
		(tmdy_struct->output->play_mode)->encoding &= ~PE_MONO;
	else
		(tmdy_struct->output->play_mode)->encoding |= PE_MONO;
	return 0;
}

static inline int parse_opt_output_signed(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --output-singed, --output-unsigned */
	if (set_flag(tmdy_struct, &((tmdy_struct->output->play_mode)->encoding), PE_SIGNED, arg))
		return 1;
	(tmdy_struct->output->play_mode)->encoding &= ~(PE_ULAW | PE_ALAW);
	return 0;
}

static inline int parse_opt_output_16bit(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --output-16bit, --output-8bit */
	if (set_flag(tmdy_struct, &((tmdy_struct->output->play_mode)->encoding), PE_16BIT, arg))
		return 1;
	if (y_or_n_p(tmdy_struct, arg))
		(tmdy_struct->output->play_mode)->encoding &= ~(PE_ULAW | PE_ALAW);
	return 0;
}

static inline int parse_opt_output_format(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --output-linear, --output-ulaw, --output-alaw */
	switch (*arg) {
	case 'l':	/* linear */
		(tmdy_struct->output->play_mode)->encoding &= ~(PE_ULAW | PE_ALAW);
		return 0;
	case 'u':	/* uLaw */
		(tmdy_struct->output->play_mode)->encoding |= PE_ULAW;
		(tmdy_struct->output->play_mode)->encoding &=
				~(PE_SIGNED | PE_16BIT | PE_ALAW | PE_BYTESWAP);
		return 0;
	case 'a':	/* aLaw */
		(tmdy_struct->output->play_mode)->encoding |= PE_ALAW;
		(tmdy_struct->output->play_mode)->encoding &=
				~(PE_SIGNED | PE_16BIT | PE_ULAW | PE_BYTESWAP);
		return 0;
	}
}

static inline int parse_opt_output_swab(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]output-swab */
	if (set_flag(tmdy_struct, &((tmdy_struct->output->play_mode)->encoding), PE_BYTESWAP, arg))
		return 1;
	(tmdy_struct->output->play_mode)->encoding &= ~(PE_ULAW | PE_ALAW);
	return 0;
}

static inline int parse_opt_o(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	if (opt_output_name)
		free(opt_output_name);
	opt_output_name = tmdy_struct->common->safe_strdup(tmdy_struct, url_expand_home_dir(arg));
	return 0;
}

static inline int parse_opt_P(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* set overriding instrument */
	strncpy(tmdy_struct->timidity_main->def_instr_name, arg, sizeof(tmdy_struct->timidity_main->def_instr_name) - 1);
	tmdy_struct->timidity_main->def_instr_name[sizeof(tmdy_struct->timidity_main->def_instr_name) - 1] = '\0';
	return 0;
}

static inline int parse_opt_p(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	if (set_value(tmdy_struct, &tmdy_struct->playmidi->voices, atoi(arg), 1, MAX_VOICES, "Polyphony"))
		return 1;
	return 0;
}

static inline int parse_opt_p1(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --[no-]polyphony-reduction */
	tmdy_struct->playmidi->auto_reduce_polyphony = y_or_n_p(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_Q(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	const char *p = arg;
	
	if (strchr(arg, 't'))
		/* backward compatibility */
		return parse_opt_Q1(tmdy_struct, arg);
	if (set_channel_flag(tmdy_struct, &tmdy_struct->readmidi->quietchannels, atoi(arg), "Quiet channel"))
		return 1;
	while (p = strchr(p, ','))
		if (set_channel_flag(tmdy_struct, &tmdy_struct->readmidi->quietchannels, atoi(++p), "Quiet channel"))
			return 1;
	return 0;
}

static inline int parse_opt_Q1(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --temper-mute */
	int prog;
	const char *p = arg;
	
	if (set_value(tmdy_struct, &prog, atoi(arg), 0, 7, "Temperament program number"))
		return 1;
	tmdy_struct->playmidi->temper_type_mute |= 1 << prog;
	while (p = strchr(p, ',')) {
		if (set_value(tmdy_struct, &prog, atoi(++p), 0, 7, "Temperament program number"))
			return 1;
		tmdy_struct->playmidi->temper_type_mute |= 1 << prog;
	}
	return 0;
}

static inline int parse_opt_q(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	char *max_buff = tmdy_struct->common->safe_strdup(tmdy_struct, arg);
	char *fill_buff = strchr(max_buff, '/');
	
	if (fill_buff != max_buff) {
		if ((tmdy_struct->timidity_main->opt_aq_max_buff))
			free((tmdy_struct->timidity_main->opt_aq_max_buff));
		(tmdy_struct->timidity_main->opt_aq_max_buff) = max_buff;
	}
	if (fill_buff) {
		*fill_buff = '\0';
		if ((tmdy_struct->timidity_main->opt_aq_fill_buff))
			free((tmdy_struct->timidity_main->opt_aq_fill_buff));
		(tmdy_struct->timidity_main->opt_aq_fill_buff) = ++fill_buff;
	}
	return 0;
}

static inline int parse_opt_R(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* I think pseudo reverb can now be retired... Computers are
	 * enough fast to do a full reverb, don't they?
	 */
	if (atoi(arg) == -1)	/* reset */
		tmdy_struct->instrum->modify_release = 0;
	else {
		if (set_val_i32(tmdy_struct, &(tmdy_struct->instrum->modify_release), atoi(arg), 0, MAX_MREL,
				"Modify Release"))
			return 1;
		if (tmdy_struct->instrum->modify_release == 0)
			tmdy_struct->instrum->modify_release = DEFAULT_MREL;
	}
	return 0;
}

static inline int parse_opt_S(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	int suffix = arg[strlen(arg) - 1];
	int32 figure;
	
	switch (suffix) {
	case 'M':
	case 'm':
		figure = 1 << 20;
		break;
	case 'K':
	case 'k':
		figure = 1 << 10;
		break;
	default:
		figure = 1;
		break;
	}
	tmdy_struct->recache->allocate_cache_size = atof(arg) * figure;
	return 0;
}

static inline int parse_opt_s(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* sampling rate */
	int32 freq;

	if ((freq = atoi(arg)) < 100)
		freq = atof(arg) * 1000 + 0.5;
	return set_val_i32(tmdy_struct, &opt_output_rate, freq,
			MIN_OUTPUT_RATE, MAX_OUTPUT_RATE, "Resampling frequency");
}

static inline int parse_opt_T(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* tempo adjust */
	int adjust;
	
	if (set_value(tmdy_struct, &adjust, atoi(arg), 10, 400, "Tempo adjust"))
		return 1;
	tmdy_struct->readmidi->tempo_adjust = 100.0 / adjust;
	return 0;
}

static inline int parse_opt_t(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	if ((tmdy_struct->common->output_text_code))
		free((tmdy_struct->common->output_text_code));
	(tmdy_struct->common->output_text_code) = tmdy_struct->common->safe_strdup(tmdy_struct, arg);
	return 0;
}

static inline int parse_opt_U(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	tmdy_struct->timidity_main->free_instruments_afterwards = y_or_n_p(tmdy_struct, arg);
	return 0;
}

__attribute__((noreturn))
static inline int parse_opt_v(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	const char *version_list[] = {
		"TiMidity++ version ", (tmdy_struct->common->timidity_version), NLS,
		NLS,
		"Copyright (C) 1999-2003 Masanao Izumo <mo@goice.co.jp>", NLS,
		"Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>", NLS,
		NLS,
#ifdef __W32__
		"Win32 version by Davide Moretti <dmoretti@iper.net>", NLS,
		"              and Daisuke Aoki <dai@y7.net>", NLS,
		NLS,
#endif	/* __W32__ */
		"This program is distributed in the hope that it will be useful,", NLS,
		"but WITHOUT ANY WARRANTY; without even the implied warranty of", NLS,
		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the", NLS,
		"GNU General Public License for more details.", NLS,
	};
	FILE *fp = open_pager(tmdy_struct);
	int i;
	
	for (i = 0; i < sizeof(version_list) / sizeof(char *); i++)
		fputs(version_list[i], fp);
	close_pager(tmdy_struct, fp);
	exit(EXIT_SUCCESS);
}

static inline int parse_opt_W(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	WRDTracer *wlp, **wlpp;
	
	if (*arg == 'R') {	/* for WRD reader options */
		put_string_table(&tmdy_struct->wrd->wrd_read_opts, arg + 1, strlen(arg + 1));
		return 0;
	}
	for (wlpp = tmdy_struct->wrd->wrdt_list; wlp = *wlpp; wlpp++)
		if (wlp->id == *arg) {
			tmdy_struct->wrd->wrdt = wlp;
			if ((tmdy_struct->timidity_main->wrdt_open_opts))
				free((tmdy_struct->timidity_main->wrdt_open_opts));
			(tmdy_struct->timidity_main->wrdt_open_opts) = tmdy_struct->common->safe_strdup(tmdy_struct, arg + 1);
			return 0;
		}
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			"WRD Tracer `%c' is not compiled in.", *arg);
	return 1;
}

#ifdef __W32__
static inline int parse_opt_w(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	switch (*arg) {
#ifdef SMFCONV
	case 'r':
		opt_rcpcv_dll = 1;
		return 0:
	case 'R':
		opt_rcpcv_dll = 0;
		return 0;
#else
	case 'r':
	case 'R':
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"-w%c option is not supported", *arg);
		return 1;
#endif	/* SMFCONV */
	default:
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL, "-w: Illegal mode `%c'", *arg);
		return 1;
	}
}
#endif	/* __W32__ */

static inline int parse_opt_x(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	StringTableNode *st;
	
	if ((st = put_string_table(&opt_config_string,
			arg, strlen(arg))) != NULL)
		expand_escape_string(tmdy_struct, st->string);
	return 0;
}

static inline void expand_escape_string(tmdy_struct_ex_t *tmdy_struct, char *s)
{
	char *t = s;
	
	if (s == NULL)
		return;
	for (t = s; *s; s++)
		if (*s == '\\') {
			switch (*++s) {
			case 'a':
				*t++ = '\a';
				break;
			case 'b':
				*t++ = '\b';
				break;
			case 't':
				*t++ = '\t';
				break;
			case 'n':
				*t++ = '\n';
				break;
			case 'f':
				*t++ = '\f';
				break;
			case 'v':
				*t++ = '\v';
				break;
			case 'r':
				*t++ = '\r';
				break;
			case '\\':
				*t++ = '\\';
				break;
			default:
				if (! (*t++ = *s))
					return;
				break;
			}
		} else
			*t++ = *s;
	*t = *s;
}

static inline int parse_opt_Z(tmdy_struct_ex_t *tmdy_struct, char *arg)
{
	/* load frequency table */
	return tmdy_struct->common->load_table(tmdy_struct, arg);
}

static inline int parse_opt_Z1(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* --pure-intonation */
	int keysig;
	
	tmdy_struct->playmidi->opt_pure_intonation = 1;
	if (*arg) {
		if (set_value(tmdy_struct, &keysig, atoi(arg), -7, -7,
				"Initial keysig (number of #(+)/b(-)[m(minor)])"))
			return 1;
		tmdy_struct->playmidi->opt_init_keysig = keysig;
		if (strchr(arg, 'm'))
			tmdy_struct->playmidi->opt_init_keysig += 16;
	}
}

__attribute__((noreturn))
static inline int parse_opt_fail(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	/* getopt_long failed to recognize any options */
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			"Could not understand option : try --help");
	exit(1);
}

static inline int set_value(tmdy_struct_ex_t *tmdy_struct, int *param, int i, int low, int high, char *name)
{
	int32 val;
	
	if (set_val_i32(tmdy_struct, &val, i, low, high, name))
		return 1;
	*param = val;
	return 0;
}

static inline int set_val_i32(tmdy_struct_ex_t *tmdy_struct, int32 *param,
		int32 i, int32 low, int32 high, char *name)
{
	if (i < low || i > high) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"%s must be between %ld and %ld", name, low, high);
		return 1;
	}
	*param = i;
	return 0;
}

static inline int set_channel_flag(tmdy_struct_ex_t *tmdy_struct, ChannelBitMask *flags, int32 i, char *name)
{
	if (i == 0) {
		FILL_CHANNELMASK(*flags);
		return 0;
	} else if (abs(i) > MAX_CHANNELS) {
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
				"%s must be between (-)1 and (-)%d, or 0",
						name, MAX_CHANNELS);
		return 1;
	}
	if (i > 0)
		SET_CHANNELMASK(*flags, i - 1);
	else
		UNSET_CHANNELMASK(*flags, -i - 1);
	return 0;
}

static inline int y_or_n_p(tmdy_struct_ex_t *tmdy_struct, const char *arg)
{
	if (arg) {
		switch (arg[0]) {
		case 'y':
		case 'Y':
		case 't':
		case 'T':
			return 1;
		case 'n':
		case 'N':
		case 'f':
		case 'F':
		default:
			return 0;
		}
	} else
		return 1;
}

static inline int set_flag(tmdy_struct_ex_t *tmdy_struct, int32 *fields, int32 bitmask, const char *arg)
{
	if (y_or_n_p(tmdy_struct, arg))
		*fields |= bitmask;
	else
		*fields &= ~bitmask;
	return 0;
}

static inline FILE *open_pager(tmdy_struct_ex_t *tmdy_struct)
{
#if ! defined(__MACOS__) && defined(HAVE_POPEN) && defined(HAVE_ISATTY) \
		&& ! defined(IA_W32GUI) && ! defined(IA_W32G_SYN)
	char *pager;
	
	if (isatty(1) && (pager = getenv("PAGER")) != NULL)
		return popen(pager, "w");
#endif
	return stdout;
}

static inline void close_pager(tmdy_struct_ex_t *tmdy_struct, FILE *fp)
{
#if ! defined(__MACOS__) && defined(HAVE_POPEN) && defined(HAVE_ISATTY) \
		&& ! defined(IA_W32GUI) && ! defined(IA_W32G_SYN)
	if (fp != stdout)
		pclose(fp);
#endif
}

static void interesting_message(tmdy_struct_ex_t *tmdy_struct)
{
	printf(
"TiMidity++ version %s -- MIDI to WAVE converter and player" NLS
"Copyright (C) 1999-2003 Masanao Izumo <mo@goice.co.jp>" NLS
"Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>" NLS
			NLS
#ifdef __W32__
"Win32 version by Davide Moretti <dmoretti@iper.net>" NLS
"              and Daisuke Aoki <dai@y7.net>" NLS
			NLS
#endif /* __W32__ */
"This program is free software; you can redistribute it and/or modify" NLS
"it under the terms of the GNU General Public License as published by" NLS
"the Free Software Foundation; either version 2 of the License, or" NLS
"(at your option) any later version." NLS
			NLS
"This program is distributed in the hope that it will be useful," NLS
"but WITHOUT ANY WARRANTY; without even the implied warranty of" NLS
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the" NLS
"GNU General Public License for more details." NLS
			NLS
"You should have received a copy of the GNU General Public License" NLS
"along with this program; if not, write to the Free Software" NLS
"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA" NLS
			NLS, (tmdy_struct->common->timidity_version));
}

/* -------- functions for getopt_long ends here --------- */

#ifdef HAVE_SIGNAL
static RETSIGTYPE sigterm_exit(int sig)
{
    char s[4];

    /* NOTE: Here, fprintf is dangerous because it is not re-enterance
     * function.  It is possible coredump if the signal is called in printf's.
     */

    write(2, "Terminated sig=0x", 17);
    s[0] = "0123456789abcdef"[(sig >> 4) & 0xf];
    s[1] = "0123456789abcdef"[sig & 0xf];
    s[2] = '\n';
    write(2, s, 3);

    tmdy_struct->common->safe_exit(tmdy_struct, 1);
}
#endif /* HAVE_SIGNAL */

static void timidity_arc_error_handler(char *error_message)
{
    if((tmdy_struct->common->open_file_noise_mode))
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL, "%s", error_message);
}

MAIN_INTERFACE void timidity_start_initialize(tmdy_struct_ex_t *tmdy_struct)
{
    int i;
    static int drums[] = DEFAULT_DRUMCHANNELS;
    static int is_first = 1;
#if defined(__FreeBSD__) && !defined(__alpha__)
    fp_except_t fpexp;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
    fp_except fpexp;
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
    fpexp = fpgetmask();
    fpsetmask(fpexp & ~(FP_X_INV|FP_X_DZ));
#endif

    if(!(tmdy_struct->common->output_text_code))
	(tmdy_struct->common->output_text_code) = tmdy_struct->common->safe_strdup(tmdy_struct, OUTPUT_TEXT_CODE);
    if(!(tmdy_struct->timidity_main->opt_aq_max_buff))
	(tmdy_struct->timidity_main->opt_aq_max_buff) = tmdy_struct->common->safe_strdup(tmdy_struct, "5.0");
    if(!(tmdy_struct->timidity_main->opt_aq_fill_buff))
	(tmdy_struct->timidity_main->opt_aq_fill_buff) = tmdy_struct->common->safe_strdup(tmdy_struct, "100%");

    /* Check the byte order */
    i = 1;
#ifdef LITTLE_ENDIAN
    if(*(char *)&i != 1)
#else
    if(*(char *)&i == 1)
#endif
    {
	fprintf(stderr, "Byte order is miss configured.\n");
	exit(1);
    }

    CLEAR_CHANNELMASK(tmdy_struct->readmidi->quietchannels);
    CLEAR_CHANNELMASK(tmdy_struct->playmidi->default_drumchannels);

    for(i = 0; drums[i] > 0; i++)
	SET_CHANNELMASK(tmdy_struct->playmidi->default_drumchannels, drums[i] - 1);
#if MAX_CHANNELS > 16
    for(i = 16; i < MAX_CHANNELS; i++)
	if(IS_SET_CHANNELMASK(tmdy_struct->playmidi->default_drumchannels, i & 0xF))
	    SET_CHANNELMASK(tmdy_struct->playmidi->default_drumchannels, i);
#endif

    if((tmdy_struct->common->program_name) == NULL)
	(tmdy_struct->common->program_name) = "TiMidity";
    uudecode_unquote_html = 1;
    for(i = 0; i < MAX_CHANNELS; i++)
    {
	tmdy_struct->instrum->default_program[i] = DEFAULT_PROGRAM;
	memset(tmdy_struct->playmidi->channel[i].drums, 0, sizeof(tmdy_struct->playmidi->channel[i].drums));
    }
    arc_error_handler = timidity_arc_error_handler;

    if((tmdy_struct->output->play_mode) == NULL)
    {
	char *output_id;
	int i;

	output_id = getenv("TIMIDITY_OUTPUT_ID");
#ifdef TIMIDITY_OUTPUT_ID
	if(output_id == NULL)
	    output_id = TIMIDITY_OUTPUT_ID;
#endif /* TIMIDITY_OUTPUT_ID */
	if(output_id != NULL)
	{
	    for(i = 0; (tmdy_struct->output->play_mode_list)[i]; i++)
		if((tmdy_struct->output->play_mode_list)[i]->id_character == *output_id)
		{
		    if (! (tmdy_struct->output->play_mode_list)[i]->detect ||
			(tmdy_struct->output->play_mode_list)[i]->detect(tmdy_struct)) {
			(tmdy_struct->output->play_mode) = (tmdy_struct->output->play_mode_list)[i];
			break;
		    }
		}
	}
    }

    if ((tmdy_struct->output->play_mode) == NULL) {
	/* try to detect the first available device */  
	for(i = 0; (tmdy_struct->output->play_mode_list)[i]; i++) {
	    /* check only the devices with detect callback */
	    if ((tmdy_struct->output->play_mode_list)[i]->detect) {
		if ((tmdy_struct->output->play_mode_list)[i]->detect(tmdy_struct)) {
		    (tmdy_struct->output->play_mode) = (tmdy_struct->output->play_mode_list)[i];
		    break;
		}
	    }
	}
    }
    
    if ((tmdy_struct->output->play_mode) == NULL) {
	fprintf(stderr, "Couldn't open output device" NLS);
	exit(1);
    }

    if(is_first) /* initialize once time */
    {
	(tmdy_struct->timidity_main->got_a_configuration) = 0;

#ifdef SUPPORT_SOCKET
	init_mail_addr();
	if(url_user_agent == NULL)
	{
	    url_user_agent =
		(char *)tmdy_struct->common->safe_malloc(tmdy_struct, 10 + strlen((tmdy_struct->common->timidity_version)));
	    strcpy(url_user_agent, "TiMidity-");
	    strcat(url_user_agent, (tmdy_struct->common->timidity_version));
	}
#endif /* SUPPORT_SOCKET */

	for(i = 0; tmdy_struct->timidity_main->url_module_list[i]; i++)
	    url_add_module(tmdy_struct->timidity_main->url_module_list[i]);
	init_string_table(&opt_config_string);
	init_freq_table();
	init_freq_table_tuning();
	init_freq_table_pytha();
	init_freq_table_meantone();
	init_freq_table_pureint();
	init_freq_table_user();
	init_bend_fine();
	init_bend_coarse();
	init_tables();
	init_attack_vol_table();
	init_sb_vol_table();
	init_convex_vol_table();
	init_def_vol_table();
	init_gs_vol_table();
	init_perceived_vol_table();
#ifdef SUPPORT_SOCKET
	url_news_connection_cache(URL_NEWS_CONN_CACHE);
#endif /* SUPPORT_SOCKET */
	for(i = 0; i < NSPECIAL_PATCH; i++)
	    tmdy_struct->instrum->special_patch[i] = NULL;
	init_midi_trace(tmdy_struct);
	tmdy_struct->common->int_rand(tmdy_struct, -1);	/* initialize random seed */
	tmdy_struct->common->int_rand(tmdy_struct, 42);	/* the 1st number generated is not very random */
	ML_RegisterAllLoaders ();
    }

    is_first = 0;
}

MAIN_INTERFACE int timidity_pre_load_configuration(tmdy_struct_ex_t *tmdy_struct)
{
#if defined(__W32__)
    /* Windows */
    char *strp;
    int check;
    char local[1024];

#if defined ( IA_W32GUI ) || defined ( IA_W32G_SYN )
    extern char *ConfigFile;
    if(!ConfigFile[0]) {
      GetWindowsDirectory(ConfigFile, 1023 - 13);
      strcat(ConfigFile, "\\TIMIDITY.CFG");
    }
    strncpy(local, ConfigFile, sizeof(local) - 1);
#else
    /* !IA_W32GUI */
    GetWindowsDirectory(local, 1023 - 13);
    strcat(local, "\\TIMIDITY.CFG");
#endif

    /* First, try read system configuration file.
     * Default is C:\WINDOWS\TIMIDITY.CFG
     */
    if((check = open(local, 0)) >= 0)
    {
	close(check);
	if(!read_config_file(tmdy_struct, local, 0))
	    (tmdy_struct->timidity_main->got_a_configuration) = 1;
    }

    /* Next, try read configuration file which is in the
     * TiMidity directory.
     */
    if(GetModuleFileName(NULL, local, 1023))
    {
        local[1023] = '\0';
	if(strp = strrchr(local, '\\'))
	{
	    *(++strp)='\0';
	    strcat(local,"TIMIDITY.CFG");
	    if((check = open(local, 0)) >= 0)
	    {
		close(check);
		if(!read_config_file(tmdy_struct, local, 0))
		    (tmdy_struct->timidity_main->got_a_configuration) = 1;
	    }
	}
    }

#else
    /* UNIX */
    if(!read_config_file(tmdy_struct, CONFIG_FILE, 0))
	(tmdy_struct->timidity_main->got_a_configuration) = 1;
#endif

    /* Try read configuration file which is in the
     * $HOME (or %HOME% for DOS) directory.
     * Please setup each user preference in $HOME/.timidity.cfg
     * (or %HOME%/timidity.cfg for DOS)
     */

    if(read_user_config_file(tmdy_struct))
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_NOISY,
		  "Warning: Can't read ~/.timidity.cfg correctly");
    return 0;
}

MAIN_INTERFACE int timidity_post_load_configuration(tmdy_struct_ex_t *tmdy_struct)
{
    int cmderr;

    cmderr = 0;
    if(!(tmdy_struct->timidity_main->got_a_configuration))
    {
	if(try_config_again && !read_config_file(tmdy_struct, CONFIG_FILE, 0))
	    (tmdy_struct->timidity_main->got_a_configuration) = 1;
    }

    if(opt_config_string.nstring > 0)
    {
	char **config_string_list;
	int i;

	config_string_list = make_string_array(&opt_config_string);
	if(config_string_list != NULL)
	{
	    for(i = 0; config_string_list[i]; i++)
	    {
		if(!read_config_file(tmdy_struct, config_string_list[i], 1))
		    (tmdy_struct->timidity_main->got_a_configuration) = 1;
		else
		    cmderr++;
	    }
	    free(config_string_list[0]);
	    free(config_string_list);
	}
    }

    if(!(tmdy_struct->timidity_main->got_a_configuration))
	cmderr++;
    return cmderr;
}

MAIN_INTERFACE void timidity_init_player(tmdy_struct_ex_t *tmdy_struct)
{
    /* Set play mode parameters */
    if(opt_output_rate != 0)
	(tmdy_struct->output->play_mode)->rate = opt_output_rate;
    else if((tmdy_struct->output->play_mode)->rate == 0)
	(tmdy_struct->output->play_mode)->rate = DEFAULT_RATE;

    /* save defaults */
    COPY_CHANNELMASK(tmdy_struct->playmidi->drumchannels, tmdy_struct->playmidi->default_drumchannels);
    COPY_CHANNELMASK(tmdy_struct->playmidi->drumchannel_mask, tmdy_struct->playmidi->default_drumchannel_mask);

    if(opt_buffer_fragments != -1)
    {
	if((tmdy_struct->output->play_mode)->flag & PF_BUFF_FRAGM_OPT)
	    (tmdy_struct->output->play_mode)->extra_param[0] = opt_buffer_fragments;
	else
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_WARNING, VERB_NORMAL,
		      "%s: -B option is ignored", (tmdy_struct->output->play_mode)->id_name);
    }

#ifdef SUPPORT_SOUNDSPEC
    if(view_soundspec_flag)
    {
	open_soundspec();
	soundspec_setinterval(spectrogram_update_sec);
    }
#endif /* SOUNDSPEC */
}

void timidity_init_aq_buff(tmdy_struct_ex_t *tmdy_struct)
{
    double time1, /* max buffer */
	   time2, /* init filled */
	   base;  /* buffer of device driver */

    if(!IS_STREAM_TRACE)
	return; /* Ignore */

    time1 = atof((tmdy_struct->timidity_main->opt_aq_max_buff));
    time2 = atof((tmdy_struct->timidity_main->opt_aq_fill_buff));
    base  = (double)(tmdy_struct->aq->aq_get_dev_queuesize)(tmdy_struct) / (tmdy_struct->output->play_mode)->rate;
    if(strchr((tmdy_struct->timidity_main->opt_aq_max_buff), '%'))
    {
	time1 = base * (time1 - 100) / 100.0;
	if(time1 < 0)
	    time1 = 0;
    }
    if(strchr((tmdy_struct->timidity_main->opt_aq_fill_buff), '%'))
	time2 = base * time2 / 100.0;
    (tmdy_struct->aq->aq_set_soft_queue)(tmdy_struct, time1, time2);
}

MAIN_INTERFACE int timidity_play_main(tmdy_struct_ex_t *tmdy_struct,int nfiles, char **files)
{
    int need_stdin = 0, need_stdout = 0;
    int i;
    int output_fail = 0;

    if(nfiles == 0 && !strchr(INTERACTIVE_INTERFACE_IDS, (tmdy_struct->controls->ctl)->id_character))
	return 0;

    if(opt_output_name)
    {
	(tmdy_struct->output->play_mode)->name = opt_output_name;
	if(!strcmp(opt_output_name, "-"))
	    need_stdout = 1;
    }

    for(i = 0; i < nfiles; i++)
	if (!strcmp(files[i], "-"))
	    need_stdin = 1;

    if((tmdy_struct->controls->ctl)->open(tmdy_struct, need_stdin, need_stdout))
    {
	fprintf(stderr, "Couldn't open %s (`%c')" NLS,
		(tmdy_struct->controls->ctl)->id_name, (tmdy_struct->controls->ctl)->id_character);
	(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
	return 3;
    }

    if(tmdy_struct->wrd->wrdt->open(tmdy_struct, (tmdy_struct->timidity_main->wrdt_open_opts)))
    {
	fprintf(stderr, "Couldn't open WRD Tracer: %s (`%c')" NLS,
		tmdy_struct->wrd->wrdt->name, tmdy_struct->wrd->wrdt->id);
	(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
	(tmdy_struct->controls->ctl)->close(tmdy_struct);
	return 1;
    }

#ifdef BORLANDC_EXCEPTION
    __try
    {
#endif /* BORLANDC_EXCEPTION */
#ifdef __W32__

#ifdef HAVE_SIGNAL
	signal(SIGTERM, sigterm_exit);
#endif
	SetConsoleCtrlHandler(handler, TRUE);

	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		  "Initialize for Critical Section");
	InitializeCriticalSection(&(tmdy_struct->timidity_main->critSect));
	if(opt_evil_mode)
	    if(!SetThreadPriority(GetCurrentThread(),
				  THREAD_PRIORITY_ABOVE_NORMAL))
		(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
			  "Error raising process priority");

#else
	/* UNIX */
#ifdef HAVE_SIGNAL
	signal(SIGINT, sigterm_exit);
	signal(SIGTERM, sigterm_exit);
#ifdef SIGPIPE
	signal(SIGPIPE, sigterm_exit);    /* Handle broken pipe */
#endif /* SIGPIPE */
#endif /* HAVE_SIGNAL */

#endif

	/* Open output device */
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		  "Open output: %c, %s",
		  (tmdy_struct->output->play_mode)->id_character,
		  (tmdy_struct->output->play_mode)->id_name);

	if ((tmdy_struct->output->play_mode)->flag & PF_PCM_STREAM) {
	    (tmdy_struct->output->play_mode)->extra_param[1] = (tmdy_struct->aq->aq_calc_fragsize)(tmdy_struct);
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		      "requesting fragment size: %d",
		      (tmdy_struct->output->play_mode)->extra_param[1]);
	}
#if !defined ( IA_W32GUI ) && !defined ( IA_W32G_SYN )
	if((tmdy_struct->output->play_mode)->open_output(tmdy_struct) < 0)
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		      "Couldn't open %s (`%c')",
		      (tmdy_struct->output->play_mode)->id_name, (tmdy_struct->output->play_mode)->id_character);
	    output_fail = 1;
	    (tmdy_struct->controls->ctl)->close(tmdy_struct);
	    return 2;
	}
#endif /* IA_W32GUI */
	if(!tmdy_struct->playmidi->control_ratio)
	{
	    tmdy_struct->playmidi->control_ratio = (tmdy_struct->output->play_mode)->rate / CONTROLS_PER_SECOND;
	    if(tmdy_struct->playmidi->control_ratio < 1)
		tmdy_struct->playmidi->control_ratio = 1;
	    else if (tmdy_struct->playmidi->control_ratio > MAX_CONTROL_RATIO)
		tmdy_struct->playmidi->control_ratio = MAX_CONTROL_RATIO;
	}

	tmdy_struct->instrum->init_load_soundfont(tmdy_struct);
	if(!output_fail)
	{
	    (tmdy_struct->aq->aq_setup)(tmdy_struct);
	    timidity_init_aq_buff(tmdy_struct);
	}
	if(tmdy_struct->recache->allocate_cache_size > 0)
	    tmdy_struct->recache->resamp_cache_reset(tmdy_struct);

	if ((tmdy_struct->timidity_main->def_prog) >= 0)
		set_default_program(tmdy_struct, (tmdy_struct->timidity_main->def_prog));
	if (*(tmdy_struct->timidity_main->def_instr_name))
		tmdy_struct->instrum->set_default_instrument(tmdy_struct, (tmdy_struct->timidity_main->def_instr_name));

	if((tmdy_struct->controls->ctl)->flags & CTLF_LIST_RANDOM)
	    tmdy_struct->common->randomize_string_list(tmdy_struct, files, nfiles);
	else if((tmdy_struct->controls->ctl)->flags & CTLF_LIST_SORT)
	    tmdy_struct->common->sort_pathname(tmdy_struct, files, nfiles);

	/* Return only when quitting */
	(tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_INFO, VERB_DEBUG_SILLY,
		  "pass_playing_list() nfiles=%d", nfiles);

	(tmdy_struct->controls->ctl)->pass_playing_list(tmdy_struct, nfiles, files);

	if(intr)
	    (tmdy_struct->aq->aq_flush)(tmdy_struct, 1);

#ifdef XP_UNIX
	return 0;
#endif /* XP_UNIX */

	(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
	(tmdy_struct->controls->ctl)->close(tmdy_struct);
	tmdy_struct->wrd->wrdt->close(tmdy_struct);
#ifdef __W32__
	DeleteCriticalSection (&(tmdy_struct->timidity_main->critSect));
#endif

#ifdef BORLANDC_EXCEPTION
    } __except(1) {
	fprintf(stderr, "\nError!!!\nUnexpected Exception Occured!\n");
	if((tmdy_struct->output->play_mode)->fd != -1)
	{
		(tmdy_struct->output->play_mode)->purge_output();
		(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
	}
	(tmdy_struct->controls->ctl)->close(tmdy_struct);
	tmdy_struct->wrd->wrdt->close(tmdy_struct);
	DeleteCriticalSection (&(tmdy_struct->timidity_main->critSect));
	exit(EXIT_FAILURE);
    }
#endif /* BORLANDC_EXCEPTION */

#ifdef SUPPORT_SOUNDSPEC
    if(view_soundspec_flag)
	close_soundspec();
#endif /* SUPPORT_SOUNDSPEC */

    free_archive_files();
#ifdef SUPPORT_SOCKET
    url_news_connection_cache(URL_NEWS_CLOSE_CACHE);
#endif /* SUPPORT_SOCKET */

    return 0;
}

#ifdef IA_W32GUI
int w32gSecondTiMidity(int opt, int argc, char **argv);
int w32gSecondTiMidityExit(void);
int w32gLoadDefaultPlaylist(void);
int w32gSaveDefaultPlaylist(void);
extern int volatile save_playlist_once_before_exit_flag;
#endif /* IA_W32GUI */

#if defined ( IA_W32GUI ) || defined ( IA_W32G_SYN )
static int CoInitializeOK = 0;
#endif

#ifndef __MACOS__
#ifdef __W32__ /* Windows */
#if ( (!defined(IA_W32GUI) || defined(__CYGWIN32__) || defined(__MINGW32__)) && !defined(IA_W32G_SYN) )
/* Cygwin or Console */
int __cdecl main(int argc, char **argv)
#else
/* _MSC_VER, _BORLANDC_ */
int win_main(int argc, char **argv)
#endif
#else /* UNIX */
int main(int argc, char **argv)
#endif
{
    int c, err;
    int nfiles;
    char **files;
    int main_ret;
    int longind;
//	tmdy_struct_ex_t * tmdy_struct;  //guha

#if defined(DANGEROUS_RENICE) && !defined(__W32__) && !defined(main)
    /*
     * THIS CODES MUST EXECUT BEGINNING OF MAIN FOR SECURITY.
     * DONT PUT ANY CODES ABOVE.
     */
#include <sys/resource.h>
    int uid;
#ifdef sun
    extern int setpriority(int which, id_t who, int prio);
    extern int setreuid(int ruid, int euid);
#endif /* sun */

    uid = getuid();
    if(setpriority(PRIO_PROCESS, 0, DANGEROUS_RENICE) < 0)
    {
	perror("setpriority");
	fprintf(stderr, "Couldn't set priority to %d.", DANGEROUS_RENICE);
    }
    setreuid(uid, uid);
#endif /* defined(DANGEROUS_RENICE) && !defined(__W32__) && !defined(main) */

	tmdy_struct=new_tmdy_struct();
	
#if defined(REDIRECT_STDOUT)
    memcpy(stdout, fopen(REDIRECT_STDOUT, "a+"), sizeof(FILE));
    printf("TiMidity++ start\n");fflush(stdout);
#endif

#ifdef main
    {
	static int maincnt = 0;
	if(maincnt++ > 0)
	{
	    argv++;
	    argc--;
	    while(argv[0][0] == '-') {
		argv++;
		argc--;
	    }
	    (tmdy_struct->controls->ctl)->pass_playing_list(tmdy_struct, argc, argv);
	    return 0;
	}
    }
#endif

#ifdef IA_DYNAMIC
    {
#ifdef XP_UNIX
	argv[0] = "netscape";
#endif /* XP_UNIX */
	tmdy_struct->timidity_main->dynamic_interface_id = 0;
	dl_init(argc, argv);
    }
#endif /* IA_DYNAMIC */

    if(((tmdy_struct->common->program_name)=tmdy_struct->common->pathsep_strrchr(tmdy_struct, argv[0]))) (tmdy_struct->common->program_name)++;
    else (tmdy_struct->common->program_name)=argv[0];

    if(strncmp((tmdy_struct->common->program_name),"timidity",8) == 0);
    else if(strncmp((tmdy_struct->common->program_name),"kmidi",5) == 0) set_ctl(tmdy_struct,"q");
    else if(strncmp((tmdy_struct->common->program_name),"tkmidi",6) == 0) set_ctl(tmdy_struct,"k");
    else if(strncmp((tmdy_struct->common->program_name),"gtkmidi",6) == 0) set_ctl(tmdy_struct,"g");
    else if(strncmp((tmdy_struct->common->program_name),"xmmidi",6) == 0) set_ctl(tmdy_struct,"m");
    else if(strncmp((tmdy_struct->common->program_name),"xawmidi",7) == 0) set_ctl(tmdy_struct,"a");
    else if(strncmp((tmdy_struct->common->program_name),"xskinmidi",9) == 0) set_ctl(tmdy_struct,"i");

    if(argc == 1 && !strchr(INTERACTIVE_INTERFACE_IDS, (tmdy_struct->controls->ctl)->id_character))
    {
	interesting_message(tmdy_struct);
	return 0;
    }

    timidity_start_initialize(tmdy_struct);
#if defined ( IA_W32GUI ) || defined ( IA_W32G_SYN )
    if(CoInitialize(NULL)==S_OK)
      CoInitializeOK = 1;
    w32g_initialize();

#ifdef IA_W32GUI
	/* Secondary TiMidity Execute */
	/*	FirstLoadIniFile(); */
	if(w32gSecondTiMidity(SecondMode,argc,argv)==FALSE){
		return 0;
	}
#endif
	
    for(c = 1; c < argc; c++)
    {
	if(is_directory(argv[c]))
	{
	    char *p;
	    p = (char *)tmdy_struct->common->safe_malloc(tmdy_struct, strlen(argv[c]) + 2);
	    strcpy(p, argv[c]);
	    directory_form(p);
	    argv[c] = p;
	}
    }
#endif

    if((err = timidity_pre_load_configuration(tmdy_struct)) != 0)
	return err;

	while ((c = getopt_long(argc, argv, optcommands, longopts, &longind)) > 0)
		if ((err = set_tim_opt_long(tmdy_struct, c, optarg, longind)) != 0)
			break;

    tmdy_struct->resample->initialize_resampler_coeffs(tmdy_struct);

    err += timidity_post_load_configuration(tmdy_struct);

    /* If there were problems, give up now */
    if(err || (optind >= argc &&
	       !strchr(INTERACTIVE_INTERFACE_IDS, (tmdy_struct->controls->ctl)->id_character)))
    {
	if(!(tmdy_struct->timidity_main->got_a_configuration))
	{
#ifdef __W32__
	    char config1[1024];
	    char config2[1024];

	    memset(config1, 0, sizeof(config1));
	    memset(config2, 0, sizeof(config2));
#if defined ( IA_W32GUI ) || defined ( IA_W32G_SYN )
	    {
		extern char *ConfigFile;
		strncpy(config1, ConfigFile, sizeof(config1) - 1);
	    }
#else
	    /* !IA_W32GUI */
	    GetWindowsDirectory(config1, 1023 - 13);
	    strcat(config1, "\\TIMIDITY.CFG");
#endif

	    if(GetModuleFileName(NULL, config2, 1023))
	    {
		char *strp;
		config2[1023] = '\0';
		if(strp = strrchr(config2, '\\'))
		{
		    *(++strp)='\0';
		    strcat(config2,"TIMIDITY.CFG");
		}
	    }

	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		      "%s: Can't read any configuration file.\nPlease check "
		      "%s or %s", (tmdy_struct->common->program_name), config1, config2);
#else
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_FATAL, VERB_NORMAL,
		      "%s: Can't read any configuration file.\nPlease check "
		      CONFIG_FILE, (tmdy_struct->common->program_name));
#endif /* __W32__ */
	}
	else
	{
	    (tmdy_struct->controls->ctl)->cmsg(tmdy_struct, CMSG_ERROR, VERB_NORMAL,
		      "Try %s -h for help", (tmdy_struct->common->program_name));
	}

#if !defined ( IA_W32GUI ) && !defined ( IA_W32G_SYN ) /* Try to continue if it is Windows version */
	return 1; /* problems with command line */
#endif
    }

    timidity_init_player(tmdy_struct);

    nfiles = argc - optind;
    files  = argv + optind;
    if(nfiles > 0 && (tmdy_struct->controls->ctl)->id_character != 'r' && (tmdy_struct->controls->ctl)->id_character != 'A' && (tmdy_struct->controls->ctl)->id_character != 'W' && (tmdy_struct->controls->ctl)->id_character != 'P')
	files = tmdy_struct->common->expand_file_archives(tmdy_struct, files, &nfiles);
	sleep(1); //to avoid dumb interface error

#ifndef IA_W32GUI
    main_ret = timidity_play_main(tmdy_struct,nfiles, files);
#ifdef IA_W32G_SYN
    if(CoInitializeOK)
      CoUninitialize();
#endif /* IA_W32G_SYN */
#else
	w32gLoadDefaultPlaylist();
    main_ret = timidity_play_main(tmdy_struct,nfiles, files);
	if(save_playlist_once_before_exit_flag) {
		save_playlist_once_before_exit_flag = 0;
		w32gSaveDefaultPlaylist();
	}
    w32gSecondTiMidityExit();
    if(CoInitializeOK)
      CoUninitialize();
#endif /* IA_W32GUI */

    tmdy_struct->instrum->free_instruments(tmdy_struct, 0);
    free_global_mblock();
    tmdy_struct->readmidi->free_all_midi_file_info(tmdy_struct);
	tmdy_struct->readmidi->free_userdrum(tmdy_struct);
	tmdy_struct->readmidi->free_userinst(tmdy_struct);
    tmdy_free_config(tmdy_struct);
	tmdy_struct->reverb->free_effect_buffers(tmdy_struct);
	
	destroy_tmdy_struct(tmdy_struct);
	
    return main_ret;
}
#endif /* __MACOS__ */



timidity_main_ex_t* new_timidity_main(tmdy_struct_ex_t *tmdy_struct){
	int i;	
	timidity_main_ex_t *timidity_main_ex;

	timidity_main_ex=(timidity_main_ex_t *)tmdy_struct->common->safe_malloc(tmdy_struct, sizeof(timidity_main_ex_t));

	timidity_main_ex->timidity_start_initialize=timidity_start_initialize;
	timidity_main_ex->timidity_pre_load_configuration=timidity_pre_load_configuration;
	timidity_main_ex->timidity_post_load_configuration=timidity_post_load_configuration;
	timidity_main_ex->timidity_init_player=timidity_init_player;
	timidity_main_ex->timidity_play_main=timidity_play_main;
	timidity_main_ex->timidity_init_aq_buff=timidity_init_aq_buff;
	timidity_main_ex->set_extension_modes=set_extension_modes;
	timidity_main_ex->set_ctl=set_ctl;
	timidity_main_ex->set_play_mode=set_play_mode;
	timidity_main_ex->set_wrd=set_wrd;
	timidity_main_ex->set_tim_opt_short=set_tim_opt_short;
	timidity_main_ex->set_tim_opt_long=set_tim_opt_long;
	

	timidity_main_ex->wrdt_open_opts = NULL;
	timidity_main_ex->opt_aq_max_buff = NULL;
	timidity_main_ex->opt_aq_fill_buff = NULL;
	timidity_main_ex->opt_control_ratio = 0; /* Save -C option */
#ifdef IA_DYNAMIC
	timidity_main_ex->dynamic_interface_module=dynamic_interface_module;
#endif	
	timidity_main_ex->read_config_file=read_config_file;

	i=0;
	timidity_main_ex->url_module_list[i++]=&URL_module_file;
#ifndef __MACOS__
    timidity_main_ex->url_module_list[i++]=&URL_module_dir;
#endif /* __MACOS__ */
#ifdef SUPPORT_SOCKET
    timidity_main_ex->url_module_list[i++]=&URL_module_http;
    timidity_main_ex->url_module_list[i++]=&URL_module_ftp;
    timidity_main_ex->url_module_list[i++]=&URL_module_news;
    timidity_main_ex->url_module_list[i++]=&URL_module_newsgroup;
#endif /* SUPPORT_SOCKET */
#if !defined(__MACOS__) && defined(HAVE_POPEN)
    timidity_main_ex->url_module_list[i++]=&URL_module_pipe;
#endif
#if defined(main) || defined(ANOTHER_MAIN)
    /* You can put some other modules */
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
    timidity_main_ex->url_module_list[i++]=NULL;
#endif /* main */
    timidity_main_ex->url_module_list[i++]=NULL;

timidity_main_ex->free_instruments_afterwards=0;
timidity_main_ex->def_prog = -1;
timidity_main_ex->def_instr_name[0]='\0';

timidity_main_ex->effect_lr_mode = -1;
/* 0: left delay
 * 1: right delay
 * 2: rotate
 * -1: not use
 */
timidity_main_ex->effect_lr_delay_msec = 25;

timidity_main_ex->pcm_alternate_file =NULL;

	
	return timidity_main_ex;	
}
void destroy_timidity_main(timidity_main_ex_t* timidity_main){
	free(timidity_main);
}

#include "common_prv.h" //guha  for sefe_malloc
#include "filter.h"
#include "mix.h"
#include "sffile.h"
#include "freq.h"
#include "mod.h"
#include "mod2midi.h"
#include "m2m.h"
#include "miditrace.h"
#include "wrd.h"
#include "quantity.h"
#include "reverb.h"
#include "rtsyn.h"

tmdy_struct_ex_t* new_tmdy_struct(void){
	tmdy_struct_ex_t *tmdy_struct_ex;

	
	tmdy_struct_ex=(tmdy_struct_ex_t *)safe_malloc(tmdy_struct, sizeof(tmdy_struct_ex_t));

	tmdy_struct_ex->common=new_common(tmdy_struct_ex);
	tmdy_struct_ex->timidity_main=new_timidity_main(tmdy_struct_ex);
	tmdy_struct_ex->output=new_output(tmdy_struct_ex);
	tmdy_struct_ex->aq=new_aq(tmdy_struct_ex);
	tmdy_struct_ex->controls=new_controls(tmdy_struct_ex);
	tmdy_struct_ex->instrum=new_instrum(tmdy_struct_ex);
	tmdy_struct_ex->resample=new_resample(tmdy_struct_ex);
	tmdy_struct_ex->readmidi=new_readmidi(tmdy_struct_ex);
	tmdy_struct_ex->playmidi=new_playmidi(tmdy_struct_ex);
	tmdy_struct_ex->filter=new_filter(tmdy_struct_ex);
	tmdy_struct_ex->mix=new_mix(tmdy_struct_ex);
	tmdy_struct_ex->sffile=new_sffile(tmdy_struct_ex);
	tmdy_struct_ex->freq=new_freq(tmdy_struct_ex);
	tmdy_struct_ex->mod=new_mod(tmdy_struct_ex);
	tmdy_struct_ex->mod2midi=new_mod2midi(tmdy_struct_ex);
	tmdy_struct_ex->m2m=new_m2m(tmdy_struct_ex);
	tmdy_struct_ex->miditrace=new_miditrace(tmdy_struct_ex);
	tmdy_struct_ex->wrd=new_wrd(tmdy_struct_ex);
	tmdy_struct_ex->recache=new_recache(tmdy_struct_ex);
	tmdy_struct_ex->quantity=new_quantity(tmdy_struct_ex);
	tmdy_struct_ex->reverb=new_reverb(tmdy_struct_ex);
	tmdy_struct_ex->rtsyn=new_rtsyn(tmdy_struct_ex);

	
	timidity_mutex_init(tmdy_struct_ex->busy);
		return tmdy_struct_ex;	
}
void destroy_tmdy_struct(tmdy_struct_ex_t* tmdy_struct){
	destroy_common(tmdy_struct->common);
	destroy_timidity_main(tmdy_struct->timidity_main);
	destroy_output(tmdy_struct->output);
	destroy_aq(tmdy_struct->aq);
	destroy_controls(tmdy_struct->controls);
	destroy_instrum(tmdy_struct->instrum);
	destroy_resample(tmdy_struct->resample);
	destroy_readmidi(tmdy_struct->readmidi);
	destroy_playmidi(tmdy_struct->playmidi);
	destroy_filter(tmdy_struct->filter);
	destroy_mix(tmdy_struct->mix);
	destroy_sffile(tmdy_struct->sffile);
	destroy_freq(tmdy_struct->freq);
	destroy_mod(tmdy_struct->mod);
	destroy_mod2midi(tmdy_struct->mod2midi);
	destroy_m2m(tmdy_struct->m2m);
	destroy_miditrace(tmdy_struct->miditrace);
	destroy_wrd(tmdy_struct->wrd);
	destroy_recache(tmdy_struct->recache);
	destroy_quantity(tmdy_struct->quantity);
	destroy_reverb(tmdy_struct->reverb);
	destroy_rtsyn(tmdy_struct->rtsyn);

	
	timidity_mutex_destroy(tmdy_struct->busy);
	free(tmdy_struct);
}



