/*
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2002 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    rtsyn_common.c
        Copyright (c) 2003  Keishi Suenaga <s_keishi@mutt.freemail.ne.jp>

    I referenced following sources.
        alsaseq_c.c - ALSA sequencer server interface
            Copyright (c) 2000  Takashi Iwai <tiwai@suse.de>
        readmidi.c
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#include "interface.h"

#include <stdio.h>

#include <stdarg.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#ifndef NO_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <math.h>
#include <signal.h>

#include "server_defs.h"

#ifdef __W32__
#include <windows.h>
#endif

#include "tmdy_struct.h"
#include "timidity.h"
#include "common.h"
#include "controls.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "recache.h"
#include "output.h"
#include "aq.h"
#include "timer.h"

#include "rtsyn.h"
#include "rtsyn_prv.h"




#define EX_RESET_NO 7
static char sysex_resets[EX_RESET_NO][11]={
		'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00',
		'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7',
		'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7',
		'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00' };
/*
#define EX_RESET_NO 9
static char sysex_resets[EX_RESET_NO][11]={
	'\xf0','\x7e','\x7f','\x09','\x00','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x01','\xf7','\x00','\x00','\x00','\x00','\x00', //gm1
	'\xf0','\x7e','\x7f','\x09','\x02','\xf7','\x00','\x00','\x00','\x00','\x00', //gm off
	'\xf0','\x7e','\x7f','\x09','\x03','\xf7','\x00','\x00','\x00','\x00','\x00', //gm2
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x00','\x41','\xf7', //GS
	'\xf0','\x41','\x10','\x42','\x12','\x40','\x00','\x7f','\x7f','\x41','\xf7', //GS off
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x00','\x01','\xf7', //88
	'\xf0','\x41','\x10','\x42','\x12','\x00','\x00','\x7f','\x01','\x00','\xf7', //88
	'\xf0','\x43','\x10','\x4c','\x00','\x00','\x7E','\x00','\xf7','\x00','\x00'  //XG on
	};
*/

static void seq_set_time(tmdy_struct_ex_t *tmdy_struct,MidiEvent *ev);



void rtsyn_gm_reset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	ev.type=ME_RESET;
	ev.a=GM_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);

}


void rtsyn_gs_reset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	ev.type=ME_RESET;
	ev.a=GS_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);
}


void rtsyn_xg_reset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	ev.type=ME_RESET;
	ev.a=XG_SYSTEM_MODE;
	ev.time=0;
	rtsyn_play_event(tmdy_struct,&ev);
}


void rtsyn_normal_reset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	ev.type=ME_RESET;
	ev.a=RTSYN->rtsyn_system_mode;
	rtsyn_play_event(tmdy_struct,&ev);
}
void rtsyn_gm_modeset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=GM_SYSTEM_MODE;
	ev.type=ME_RESET;
	ev.a=GM_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);
	tmdy_struct->readmidi->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}


void rtsyn_gs_modeset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=GS_SYSTEM_MODE;
	ev.type=ME_RESET;
	ev.a=GS_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);
	tmdy_struct->readmidi->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}


void rtsyn_xg_modeset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=XG_SYSTEM_MODE;
	ev.type=ME_RESET;
	ev.a=XG_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);
	tmdy_struct->readmidi->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}


void rtsyn_normal_modeset(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

	rtsyn_reset(tmdy_struct);
	RTSYN->rtsyn_system_mode=DEFAULT_SYSTEM_MODE;
	ev.type=ME_RESET;
	ev.a=GS_SYSTEM_MODE;
	rtsyn_play_event(tmdy_struct,&ev);
	tmdy_struct->readmidi->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
}

//extern int compute_data(tmdy_struct_ex_t *tmdy_struct, int32 count);

#ifdef USE_WINSYN_TIMER_I
#ifdef __W32__
VOID CALLBACK timercalc(UINT uTimerID, UINT uMsg, DWORD dwUser, DWORD dummy1, DWORD dummy2){
	MidiEvent ev;
	tmdy_struct_ex_t* tmdy_struct;
	double time_div, currenttime;
	
	tmdy_struct=(tmdy_struct_ex_t *)dwUser;	
	rtsyn_mutex_lock(RTSYN->timidityMUTEX);
	currenttime=get_current_calender_time();
	time_div=currenttime-RTSYN->starttime;
	if( time_div > (tmdy_struct->output->play_mode->rate)/TICKTIME_HZ){
		time_div=(tmdy_struct->output->play_mode->rate)/TICKTIME_HZ-3;
	}
	ev.time= ((double)tmdy_struct->playmidi->current_sample
			+ (tmdy_struct->output->play_mode->rate)*time_div+0.5);
	RTSYN->starttime=currenttime;
	ev.type = ME_NONE;
	tmdy_struct->playmidi->play_event(tmdy_struct, &ev);
//	compute_data(tmdy_struct,(tmdy_struct->output->play_mode->rate)*time_div);
	(tmdy_struct->aq->aq_fill_nonblocking)(tmdy_struct);
	RTSYN->rtsyn_reachtime=currenttime +  (double)(1.0f/TICKTIME_HZ);
	rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
	return;
}
#else
void *timercalc(void *arg){
 	MidiEvent ev;
	unsigned int slt;
	double reachtime,delay;
	tmdy_struct_ex_t* tmdy_struct;

	tmdy_struct=(tmdy_struct_ex_t *)arg;	
	delay=(double)(1.0/TICKTIME_HZ);
	while(RTSYN->thread_on_f==1){		
		rtsyn_mutex_lock(RTSYN->timidityMUTEX);
		reachtime=get_current_calender_time()+delay;
		ev.type = ME_NONE;
		seq_set_time(tmdy_struct. &ev);
		tmdy_struct->playmidi->play_event(tmdy_struct, &ev);
		(tmdy_struct->aq->aq_fill_nonblocking)(tmdy_struct);
		rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
		do{
			sleep(0);
		}while(get_current_calender_time()<reachtime);
	}
	return NULL;
}
#endif
#endif
void rtsyn_init(tmdy_struct_ex_t *tmdy_struct){
	int i,j;
	UINT port;
		/* set constants */
	tmdy_struct->playmidi->opt_realtime_playing = 1; /* Enable loading patch while playing */
	tmdy_struct->recache->allocate_cache_size = 0; /* Don't use pre-calclated samples */
	tmdy_struct->playmidi->auto_reduce_polyphony = 0;
	tmdy_struct->playmidi->current_keysig = tmdy_struct->playmidi->current_temper_keysig = tmdy_struct->playmidi->opt_init_keysig;
	tmdy_struct->playmidi->note_key_offset = 0;

	if (tmdy_struct->playmidi->opt_force_keysig != 8) {
		i = tmdy_struct->playmidi->current_keysig + ((tmdy_struct->playmidi->current_keysig < 8) ? 7 : -6);
		j = tmdy_struct->playmidi->opt_force_keysig + ((tmdy_struct->playmidi->current_keysig < 8) ? 7 : 10);
		while (i != j && i != j + 12) {
			if (++tmdy_struct->playmidi->note_key_offset > 6)
				tmdy_struct->playmidi->note_key_offset -= 12;
			i += (i > 10) ? -5 : 7;
		}
	}
	i = tmdy_struct->playmidi->current_keysig + ((tmdy_struct->playmidi->current_keysig < 8) ? 7 : -9), j = 0;
	while (i != 7 && i != 19)
		i += (i < 7) ? 5 : -7, j++;
	j += tmdy_struct->playmidi->note_key_offset, j -= floor(j / 12.0) * 12;
	tmdy_struct->playmidi->current_freq_table = j;

	RTSYN->active_sensing_flag=0;
	RTSYN->active_sensing_time=0;
#ifdef USE_WINSYN_TIMER_I
#ifdef __W32__
		RTSYN->timerID=0;
#else 
		timer_thread=NULL;
		RTSYN->thread_on_f=0;
#endif
#endif
	for(port=0;port<MAX_PORT;port++){
		RTSYN->portID[port]=-1;
	}
//	if(RTSYN->timidityMUTEX==NULL){
		rtsyn_mutex_init(RTSYN->timidityMUTEX);
//	}
	rtsyn_server_reset(tmdy_struct);
}

void rtsyn_close(tmdy_struct_ex_t *tmdy_struct){
#ifdef USE_WINSYN_TIMER_I
#ifdef __W32__
	timeKillEvent( RTSYN->timerID );
	RTSYN->timerID=0;
	timeEndPeriod(1);
#else
	RTSYN->thread_on_f=0;
	pthread_join(timer_thread, NULL);
	timer_thread=NULL;
#endif
#endif 

	rtsyn_mutex_destroy(RTSYN->timidityMUTEX);
}

void rtsyn_play_event(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
  int gch;
  int32 cet;
	rtsyn_mutex_lock(RTSYN->timidityMUTEX);
	gch = GLOBAL_CHANNEL_EVENT_TYPE(ev->type);
//	if(gch || !IS_SET_CHANNELMASK(tmdy_struct->readmidi->quietchannels, ev->channel) ){
//    if ( !seq_quit ) {
			ev->time=0;
			tmdy_struct->playmidi->play_event(tmdy_struct, ev);
//		}
//	}
	rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
}

void rtsyn_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_stop_playing(tmdy_struct);
	rtsyn_mutex_lock(RTSYN->timidityMUTEX);
	(tmdy_struct->instrum->free_instruments)(tmdy_struct, 0);        //also in rtsyn_server_reset
	free_global_mblock();
	rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
	rtsyn_server_reset(tmdy_struct);

//		printf("system reseted\n");
}

void rtsyn_server_reset(tmdy_struct_ex_t *tmdy_struct){
	rtsyn_mutex_lock(RTSYN->timidityMUTEX);
	(tmdy_struct->instrum->free_instruments)(tmdy_struct, 0);
	(tmdy_struct->output->play_mode)->close_output(tmdy_struct);	// PM_REQ_PLAY_START wlll called in tmdy_struct->playmidi->playmidi_stream_init()
	(tmdy_struct->output->play_mode)->open_output(tmdy_struct);	// but w32_a.c does not have it.
	tmdy_struct->readmidi->readmidi_read_init(tmdy_struct);
	tmdy_struct->playmidi->playmidi_stream_init(tmdy_struct);
	RTSYN->starttime=get_current_calender_time();
	tmdy_struct->playmidi->reduce_voice_threshold = 0; // * Disable auto reduction voice *
	tmdy_struct->playmidi->auto_reduce_polyphony = 0;
	rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
}

void rtsyn_stop_playing(tmdy_struct_ex_t *tmdy_struct)
{
	if(tmdy_struct->playmidi->upper_voices) {
		MidiEvent ev;
		ev.type = ME_EOT;
		ev.a = 0;
		ev.b = 0;
		rtsyn_play_event(tmdy_struct, &ev);
		rtsyn_mutex_lock(RTSYN->timidityMUTEX);
		(tmdy_struct->aq->aq_flush)(tmdy_struct, 1);
		rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
	}
}

static void seq_set_time(tmdy_struct_ex_t *tmdy_struct, MidiEvent *ev)
{
	double currenttime, time_div;
	
	currenttime=get_current_calender_time();
	time_div=currenttime-RTSYN->starttime;
	ev->time=((double) tmdy_struct->playmidi->current_sample
			+ (tmdy_struct->output->play_mode->rate)*time_div+0.5);
	RTSYN->starttime=currenttime;
	
	RTSYN->rtsyn_reachtime=currenttime +  (double)(1.0f/TICKTIME_HZ);
}

void rtsyn_play_calculate(tmdy_struct_ex_t *tmdy_struct){
	MidiEvent ev;

#ifdef USE_WINSYN_TIMER_I

#ifdef __W32__
	if(RTSYN->timerID==0){
		timeBeginPeriod(1);
		RTSYN->starttime=0;
		{
			DWORD data;
			UINT delay;
			delay=(1000/TICKTIME_HZ);
			data=(DWORD)tmdy_struct;
		 
		
			RTSYN->timerID = timeSetEvent( delay, 0, timercalc, data,
				TIME_PERIODIC | TIME_CALLBACK_FUNCTION );
        	if( !RTSYN->timerID ){
        		tmdy_struct->controls->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
        	}
		}
	}
#else
	if(timer_thread==NULL){
		RTSYN->thread_on_f=1;
		if(0!=pthread_create(&timer_thread,NULL,timercalc,(void *)tmdy_struct)){
        		tmdy_struct->controls->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL,"Fail to setup Timer Interrupt (winsyn) \n");
		}
	}
#endif /* __W32__ */

#else
	rtsyn_mutex_lock(RTSYN->timidityMUTEX);
	ev.type = ME_NONE;
	seq_set_time(tmdy_struct, &ev);
	tmdy_struct->playmidi->play_event(tmdy_struct, &ev);
	(tmdy_struct->aq->aq_fill_nonblocking)(tmdy_struct);
	rtsyn_mutex_unlock(RTSYN->timidityMUTEX);
#endif /* USE_WINSYN_TIMER_I */
	
	if(RTSYN->active_sensing_flag==~0 && (get_current_calender_time() > RTSYN->active_sensing_time+0.5)){
//normaly acitive sensing expiering time is 330ms(>300ms) but this loop is heavy
		(tmdy_struct->output->play_mode)->close_output(tmdy_struct);
		(tmdy_struct->output->play_mode)->open_output(tmdy_struct);
		tmdy_struct->controls->ctl->cmsg(tmdy_struct,   CMSG_ERROR, VERB_NORMAL,"Active Sensing Expired\n");
		RTSYN->active_sensing_flag=0;
	}
}
	
int rtsyn_play_one_data (tmdy_struct_ex_t *tmdy_struct, int port, int32 dwParam1){
	MidiEvent ev;

	ev.type = ME_NONE;
	ev.channel = dwParam1 & 0x0000000f;
	ev.channel = ev.channel+port*16;
	ev.a = (dwParam1 >> 8) & 0xff;
	ev.b = (dwParam1 >> 16) & 0xff;
	switch ((int) (dwParam1 & 0x000000f0)) {
	case 0x80:
		ev.type = ME_NOTEOFF;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0x90:
		ev.type = (ev.b) ? ME_NOTEON : ME_NOTEOFF;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xa0:
		ev.type = ME_KEYPRESSURE;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xb0:
		if (! tmdy_struct->readmidi->convert_midi_control_change(tmdy_struct, ev.channel, ev.a, ev.b, &ev))
		ev.type = ME_NONE;
		break;
	case 0xc0:
		ev.type = ME_PROGRAM;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xd0:
		ev.type = ME_CHANNEL_PRESSURE;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xe0:
		ev.type = ME_PITCHWHEEL;
//		rtsyn_play_event(tmdy_struct,&ev);
		break;
	case 0xf0:
#ifdef USE_PORTMIDI
		if ( (dwParam1 & 0x000000ff) == 0xf0) {
			//SysEx
			return 1;
		}
#endif
		if ((dwParam1 & 0x000000ff) == 0xf2) {
			ev.type = ME_PROGRAM;
//			rtsyn_play_event(tmdy_struct,&ev);
		}
#if 0
		if ((dwParam1 & 0x000000ff) == 0xf1)
			//MIDI Time Code Qtr. Frame (not need)
			printf("MIDI Time Code Qtr\n");
		if ((dwParam1 & 0x000000ff) == 0xf3)
			//Song Select(Song #) (not need)
		if ((dwParam1 & 0x000000ff) == 0xf6)
			//Tune request (not need)
			printf("Tune request\n");
		if ((dwParam1 & 0x000000ff) == 0xf8)
			//Timing Clock (not need)
			printf("Timing Clock\n");
		if ((dwParam1&0x000000ff)==0xfa)
			//Start
		if ((dwParam1 & 0x000000ff) == 0xfb)
			//Continue
		if ((dwParam1 & 0x000000ff) == 0xfc) {
			//Stop
			printf("Stop\n");
		}
#endif
		if ((dwParam1 & 0x000000ff) == 0xfe) {
			//Active Sensing
//			printf("Active Sensing\n");
			RTSYN->active_sensing_flag = ~0;
			RTSYN->active_sensing_time = get_current_calender_time();
		}
		if ((dwParam1 & 0x000000ff) == 0xff) {
			//System Reset
			printf("System Reset\n");
		}
		break;
	default:
//		printf("Unsup/ed event %d\n", aevp->type);
		break;
	}
	if (ev.type != ME_NONE) {
		rtsyn_play_event(tmdy_struct,&ev);
	}
	return 0;
}


void rtsyn_play_one_sysex (tmdy_struct_ex_t *tmdy_struct, char *sysexbuffer, int exlen ){
	int i,j,chk,ne;
	MidiEvent ev;
	MidiEvent evm[260];

	if(sysexbuffer[exlen-1] == '\xf7'){            // I don't konw why this need
		for(i=0;i<EX_RESET_NO;i++){
			chk=0;
			for(j=0;(j<exlen)&&(j<11);j++){
				if(chk==0 && sysex_resets[i][j]!=sysexbuffer[j]){
					chk=~0;
				}
			}
			if(chk==0){
				 rtsyn_server_reset(tmdy_struct);
			}
		}
/*
		printf("SyeEx length=%x bytes \n", exlen);
		for(i=0;i<exlen;i++){
			printf("%x ",sysexbuffer[i]);
		}
		printf("\n");
*/
		if(tmdy_struct->readmidi->parse_sysex_event(tmdy_struct, sysexbuffer+1,exlen-1,&ev)){
			if(ev.type==ME_RESET && RTSYN->rtsyn_system_mode!=DEFAULT_SYSTEM_MODE){
				ev.a=RTSYN->rtsyn_system_mode;
				tmdy_struct->readmidi->change_system_mode(tmdy_struct, RTSYN->rtsyn_system_mode);
				rtsyn_play_event(tmdy_struct,&ev);
			}else{
				rtsyn_play_event(tmdy_struct,&ev);
			}
		}
		if(ne=tmdy_struct->readmidi->parse_sysex_event_multi(tmdy_struct, sysexbuffer+1,exlen-1, evm)){
			for (i = 0; i < ne; i++){
					rtsyn_play_event(tmdy_struct,&evm[i]);
			}
		}
	}
}




rtsyn_ex_t* new_rtsyn(tmdy_struct_ex_t *tmdy_struct){
	int i;
	rtsyn_ex_t* rtsyn_ex;

	rtsyn_ex=(rtsyn_ex_t *)tmdy_struct->common->safe_malloc(tmdy_struct, sizeof(rtsyn_ex_t));
	
/* rtsyn_common.c */
rtsyn_ex->rtsyn_gm_reset=rtsyn_gm_reset;
rtsyn_ex->rtsyn_gs_reset=rtsyn_gs_reset;
rtsyn_ex->rtsyn_xg_reset=rtsyn_xg_reset;
rtsyn_ex->rtsyn_normal_reset=rtsyn_normal_reset;
rtsyn_ex->rtsyn_gm_modeset=rtsyn_gm_modeset;
rtsyn_ex->rtsyn_gs_modeset=rtsyn_gs_modeset;
rtsyn_ex->rtsyn_xg_modeset=rtsyn_xg_modeset;
rtsyn_ex->rtsyn_normal_modeset=rtsyn_normal_modeset;
rtsyn_ex->rtsyn_init=rtsyn_init;
rtsyn_ex->rtsyn_close=rtsyn_close;
rtsyn_ex->rtsyn_play_event=rtsyn_play_event;
rtsyn_ex->rtsyn_server_reset=rtsyn_server_reset;
rtsyn_ex->rtsyn_reset=rtsyn_reset;
rtsyn_ex->rtsyn_stop_playing=rtsyn_stop_playing;
rtsyn_ex->rtsyn_play_one_data=rtsyn_play_one_data;
rtsyn_ex->rtsyn_play_one_sysex=rtsyn_play_one_sysex;
rtsyn_ex->rtsyn_play_calculate=rtsyn_play_calculate;
rtsyn_ex->rtsyn_add_midi_port=rtsyn_add_midi_port;
rtsyn_ex->rtsyn_delete_midi_port=rtsyn_delete_midi_port;
/*rtsyn_(interface).c*/
rtsyn_ex->rtsyn_get_port_list=rtsyn_get_port_list;
rtsyn_ex->rtsyn_synth_start=rtsyn_synth_start;
rtsyn_ex->rtsyn_synth_stop=rtsyn_synth_stop;
rtsyn_ex->rtsyn_play_some_data=rtsyn_play_some_data;
#if defined(IA_WINSYN) || defined(IA_W32G_SYN)
rtsyn_ex->rtsyn_buf_check=rtsyn_buf_check;
#endif
rtsyn_ex->rtsyn_midiports_close=rtsyn_midiports_close;

	return rtsyn_ex;
}
void destroy_rtsyn(rtsyn_ex_t* rtsyn){	
	free(rtsyn);
}
