/*
 *  message.hpp
 *  serialize
 *
 *  Created by Jason Toffaletti on Sat Apr 19 2003.
 *  Copyright (c) 2003 __MyCompanyName__. All rights reserved.
 *
 */

#include "flattypes.hpp"
#include <boost/shared_ptr.hpp>
#include <boost/any.hpp>
#include <sstream>

namespace jason {

class container : public boost::shared_ptr<flat::flattenable>
{
public:
	container(flat::flattenable *p, bool flat = false) : 
		boost::shared_ptr<flat::flattenable>(p), m_flat(flat) {}
	bool is_flat() const { return m_flat; }
protected:
	bool m_flat;
};

#define FLAT_MESSAGE_TYPE	(flat::flat_type_code)'jmsg'

template <typename key_var_type, typename key_type>
class message : public flat::flattenable
{
public:
	typedef container value_type;
	typedef std::map<key_type, value_type> message_map;
	
	flat::flattenable_uint32 what;
	
	message() : what(0) {}
	message(const message &m) : what(m.what), m_map(m.m_map) {}
	virtual ~message() {}
	
	const flat::flat_type_code type_code() const { return FLAT_MESSAGE_TYPE; }
	const bool is_fixed_size() const { return false; }
	
	void add_bool(const key_var_type &in_key, const bool in_value)
	{
		value_type p(new flat::flattenable_bool(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_sint8(const key_var_type &in_key, const sint8 in_value)
	{
		value_type p(new flat::flattenable_sint8(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_uint8(const key_var_type &in_key, const uint8 in_value)
	{
		value_type p(new flat::flattenable_uint8(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_sint16(const key_var_type &in_key, const sint16 in_value)
	{
		value_type p(new flat::flattenable_sint16(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_uint16(const key_var_type &in_key, const uint16 in_value)
	{
		value_type p(new flat::flattenable_uint16(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_sint32(const key_var_type &in_key, const sint32 in_value)
	{
		value_type p(new flat::flattenable_sint32(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_uint32(const key_var_type &in_key, const uint32 in_value)
	{
		value_type p(new flat::flattenable_uint32(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_sint64(const key_var_type &in_key, const sint64 in_value)
	{
		value_type p(new flat::flattenable_sint64(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_uint64(const key_var_type &in_key, const uint64 in_value)
	{
		value_type p(new flat::flattenable_uint64(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	void add_string(const key_var_type &in_key, const std::string &in_value)
	{
		value_type p(new flat::flattenable_string(in_value), true);
		m_map.insert(std::make_pair(in_key, p));
	}
	
	template <typename T> void add_object(const key_var_type &in_key, const T &in_value)
	{
		value_type p(new T(in_value));
		m_map.insert(std::make_pair(in_key, p));
	}
	
	bool find_bool(const key_var_type &in_key, bool &out_value)
	{
		flat::flattenable_bool tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_sint8(const key_var_type &in_key, sint8 &out_value)
	{
		flat::flattenable_sint8 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_uint8(const key_var_type &in_key, uint8 &out_value)
	{
		flat::flattenable_uint8 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_sint16(const key_var_type &in_key, sint16 &out_value)
	{
		flat::flattenable_sint16 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_uint16(const key_var_type &in_key, uint16 &out_value)
	{
		flat::flattenable_uint16 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_sint32(const key_var_type &in_key, sint32 &out_value)
	{
		flat::flattenable_sint32 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_uint32(const key_var_type &in_key, uint32 &out_value)
	{
		flat::flattenable_uint32 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_sint64(const key_var_type &in_key, sint64 &out_value)
	{
		flat::flattenable_sint64 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_uint64(const key_var_type &in_key, uint64 &out_value)
	{
		flat::flattenable_uint64 tmp;
		if (find_object(in_key, tmp))
		{
			out_value = tmp;
			return true;
		}
		return false;
	}
	
	bool find_string(const key_var_type &in_key, std::string &out_value)
	{
		typename message_map::iterator iter = m_map.find(in_key);
		if (iter == m_map.end())
			return false;
		out_value = *dynamic_cast<flat::flattenable_string *>(iter->second.get());
		return true;
	}
	
	template <typename T> bool find_object(const key_var_type &in_key, T &in_value)
	{
		typename message_map::iterator iter = m_map.find(in_key);
		if (iter == m_map.end())
			return false;
		if (iter->second.is_flat())
		{
			// this object was unflattened and it still flat
			std::istringstream strm(*dynamic_cast<flat::flattenable_string *>(iter->second.get()));
			in_value.unflatten(strm);
		}
		else
			in_value = *dynamic_cast<T *>(iter->second.get());
		return true;
	}
	
	const unsigned int flattened_size() const
	{
		// 2 uint32 the what and the num items
		unsigned int flat_size = (sizeof(uint32) * 2);
		typename message_map::const_iterator iter = m_map.begin();
		while (iter != m_map.end())
		{
			flat_size += iter->first.flattened_size();
			if (!iter->second.is_flat())
				// a uint32 for the size of the flattenable
				flat_size += sizeof(uint32);
			flat_size += iter->second->flattened_size();
			iter++;
		}
		return flat_size;
	}
	
	void flatten(std::ostream &strm) const
	{
		// first the header which contains:
		// the what field
		what.flatten(strm);
		
		// the number of key, value pairs (objects) stored in the message
		flat::flattenable_uint32 nitems = m_map.size();
		nitems.flatten(strm);
		
		typename message_map::const_iterator iter = m_map.begin();
		while (iter != m_map.end())
		{
			iter->first.flatten(strm);
			if (!iter->second.is_flat())
			{
				// if it isn't already a flattened object then we
				// need to save the size
				flat::flattenable_uint32 flat_size = iter->second->flattened_size();
				flat_size.flatten(strm);
			}
			iter->second->flatten(strm);
			iter++;
		}
	}
	
	void unflatten(std::istream &strm)
	{
		m_map.clear();
		
		what.unflatten(strm);
		
		flat::flattenable_uint32 nitems;
		nitems.unflatten(strm);
		
		for (unsigned int i = 0; i < nitems; i++)
		{
			key_type key;
			key.unflatten(strm);
			
			value_type buf(new flat::flattenable_string(), true);
			buf->unflatten(strm);
			
			m_map.insert(make_pair(key, buf));
		}
	}

protected:
	message_map m_map;
};

} // end namespace jason