// Basic wrapper around a blocking TCP socket for client and server operations

#ifndef TCP_SOCKET_H
#define TCP_SOCKET_H

#include <stdexcept>
#include <string>

#ifndef WIN32
#include <netinet/in.h>
#include <sys/socket.h>
#endif

#define SOCKET_INVALID_PORT			0
#define SOCKET_INVALID_DESCRIPTOR	-1

using namespace std;

class socket_error : public runtime_error
{
    public:
        explicit socket_error(int inErrorCode)
            : runtime_error(""), mErrorCode(inErrorCode)
        {
            snprintf(mErrorBuf, 128, "%s", strerror(mErrorCode));
        }
        
        virtual const char *what() const throw ()
        {
            return mErrorBuf;
        }
        
        const int error_code() const
        {
            return mErrorCode;
        }
        
    private:
        
        int mErrorCode;
        char mErrorBuf[128];
};

class TCPSocket
{
	public:
		TCPSocket();
		TCPSocket(int inDescriptor, struct sockaddr_in* inRemoteAddress);
		virtual ~TCPSocket();
		
		void Open();
		virtual void Close();
        virtual void Shutdown();
        bool IsConnected();
        int GetAvailBytes();
        virtual bool SetNonBlock(bool nonBlock);
        bool SetAsync(bool isAsync);
        void SetLinger(bool inOnOff);
        void SetKeepAlive(bool inOn);
		unsigned int GetSndBufSize();
		unsigned int GetRcvBufSize();
        int GetError();
		
        void RecvBytes(void *outBytes, u_int32_t inCount) throw (socket_error);
		void SendBytes(const void *inBytes, u_int32_t inCount) throw (socket_error);
		
        int Descriptor()
			{ return mDescriptor; }
        const string& RemoteHostName()
            { return mRemoteHostName; }
		u_int32_t RemoteHost()
			{ return mRemoteHost; }
		u_int16_t RemotePort()
            { return mRemotePort; }
		
        virtual void Connect(const u_int32_t inAddress, const u_int16_t inPort);
		virtual void Bind(u_int16_t inPort = 0, u_int32_t inAddress = INADDR_ANY);
        virtual void Listen(u_int32_t inMaxPendingConnections = 5);
        
        void Select(bool &ioRead, bool &ioWrite);
        
		int Accept( struct sockaddr_in& acceptAddr );
        
	protected:
        int Recv(void *outBytes, u_int32_t inCount, u_int32_t inFlags) throw ();
        int Send(const void *inBytes, u_int32_t inCount) throw ();
        
		int mDescriptor;
		string mRemoteHostName;
		u_int32_t mRemoteHost;
		unsigned short mRemotePort;
        bool mConnected;
		bool mListening;
};

#endif // TCP_SOCKET_H


