/**
 *  @file DirScan.cpp
 *
 *  @brief Implementation of DirScan (q.v.) and helper functions
 */ 
// RCS ID line follows -- this is updated by CVS
// $Id: DirScan.cpp 3944 2006-12-11 22:06:57Z kimmov $

#include "stdafx.h"
#include <shlwapi.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "DirScan.h"
#include "CompareStats.h"
#include "common/unicoder.h"
#include "DiffContext.h"
#include "DiffWrapper.h"
#include "FileFilterHelper.h"
#include "logfile.h"
#include "paths.h"
#include "FileTransform.h"
#include "codepage.h"
#include "DiffItemList.h"
#include "PathContext.h"
#include "IAbortable.h"
#include "DiffFileData.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

// Static types (ie, types only used locally)
/**
 * @brief directory or file info for one row in diff result
 * @note times are seconds since January 1, 1970.
 */
struct fentry
{
	CString name; /**< Item name */
	__int64 mtime; /**< Last modify time */
	__int64 ctime; /**< Creation modify time */
	__int64 size; /**< File size */
	int attrs; /**< Item attributes */
};
typedef CArray<fentry, fentry&> fentryArray;

// Static functions (ie, functions only used locally)
void CompareDiffItem(DIFFITEM di, CDiffContext * pCtxt);
static void LoadFiles(const CString & sDir, fentryArray * dirs, fentryArray * files);
void LoadAndSortFiles(const CString & sDir, fentryArray * dirs, fentryArray * files, bool casesensitive);
static void Sort(fentryArray * dirs, bool casesensitive);;
static int collstr(const CString & s1, const CString & s2, bool casesensitive);
static void StoreDiffData(DIFFITEM &di, CDiffContext * pCtxt,
		const DiffFileData * pDiffFileData);
static void AddToList(const CString & sLeftDir, const CString & sRightDir, const fentry * lent, const fentry * rent,
	int code, DiffItemList * pList, CDiffContext *pCtxt);
static void AddToList(const CString & sLeftDir, const CString & sMiddleDir, const CString & sRightDir, const fentry * lent, const fentry * ment, const fentry * rent,
	int code, DiffItemList * pList, CDiffContext *pCtxt);
static void UpdateDiffItem(DIFFITEM & di, BOOL & bExists, CDiffContext *pCtxt);

static __int64 FiletimeToTimeT(FILETIME time);

/** @brief cmpmth is a typedef for a pointer to a method */
typedef int (CString::*cmpmth)(LPCTSTR sz) const;
/** @brief CALL_MEMBER_FN calls a method through a pointer to a method */
#define CALL_MEMBER_FN(object,ptrToMember)  ((object).*(ptrToMember))

/**
 * @brief Help minimize memory footprint by sharing CStringData if possible.
 * 
 * Use OPTIMIZE_SHARE_CSTRINGDATA to conditionally include code that is merely
 * intended to minimize memory footprint by having two CStrings share one
 * CStringData if possible. The rule is that semantics must be identical
 * regardless of whether OPTIMIZE_SHARE_CSTRINGDATA(X) expands to X or to
 * nothing. If you suspect some bug to be related to this kind of optimization,
 * then you can simply change OPTIMIZE_SHARE_CSTRINGDATA to expand to nothing,
 * recompile, and see if bug disappears.
 */
#define OPTIMIZE_SHARE_CSTRINGDATA(X) X

/**
 * @brief Collect file- and directory-names to list.
 * 
 * @param [in] paths Root paths of compare
 * @param [in] subdir Current subdirectory under root paths
 * @param [in,out] list List where found items are added
 * @param [in] casesensitive Is filename compare casesensitive?
 * @param [in] depth Levels of subdirectories to scan, -1 scans all
 * @param [in] pCtxt Compare context
 * @param [in] piAbortable Interface allowing compare to be aborted
 * @return 1 normally, -1 if compare was aborted
 */
int DirScan_GetItems(const PathContext &paths, const CString subdir[], DiffItemList *pList,
		bool casesensitive, int depth, CDiffContext * pCtxt)
{
	static const TCHAR backslash[] = _T("\\");
	int nDirs = paths.GetSize();
	bool scanunpaireddir = (bool)pCtxt->m_bScanUnpairedDir;

	CString sDir[3];
	CString subprefix[3];

	int nIndex;
	for (nIndex = 0; nIndex < nDirs; nIndex++)
		sDir[nIndex] = paths.GetPath(nIndex);

	if (!subdir[0].IsEmpty())
	{
		for (nIndex = 0; nIndex < paths.GetSize(); nIndex++)
			sDir[nIndex] += backslash + subdir[nIndex];
		subprefix[0] = subdir[0] + backslash;
		for (nIndex = 1; nIndex < paths.GetSize(); nIndex++)
		{
			// minimize memory footprint by having left/rightsubprefix share CStringData if possible
			subprefix[nIndex] = OPTIMIZE_SHARE_CSTRINGDATA
			(
				(LPCTSTR)subdir[0] == (LPCTSTR)subdir[nIndex] ? subprefix[0] : 
			) subdir[nIndex] + backslash;
		}
	}

	fentryArray dirs[3], files[3];
	for (nIndex = 0; nIndex < paths.GetSize(); nIndex++)
		LoadAndSortFiles(sDir[nIndex], &dirs[nIndex], &files[nIndex], casesensitive);

	// Allow user to abort scanning
	if (pCtxt->ShouldAbort())
		return -1;

	// Handle directories
	// i points to current directory in left list (leftDirs)
	// j points to current directory in right list (rightDirs)

	// If there is only one directory on each side, and no files
	// then pretend the directories have the same name
	bool bTreatDirAsEqual;
	if (nDirs < 3)
	{
		bTreatDirAsEqual = 
		  (dirs[0].GetSize() == 1)
		&& (dirs[1].GetSize() == 1)
		&& (files[0].GetSize() == 0)
		&& (files[1].GetSize() == 0)
		;
	}
	else
	{
		bTreatDirAsEqual = 
		  (dirs[0].GetSize() == 1)
		&& (dirs[1].GetSize() == 1)
		&& (dirs[2].GetSize() == 1)
		&& (files[0].GetSize() == 0)
		&& (files[1].GetSize() == 0)
		&& (files[2].GetSize() == 0)
		;
	}

	int i=0, j=0, k=0;
	int nDiffCode;
	while (1)
	{
		if (pCtxt->ShouldAbort())
			return -1;

		if (i >= dirs[0].GetSize() && j >= dirs[1].GetSize() && (nDirs < 3 || k >= dirs[2].GetSize()))
			break;

		// Comparing directories leftDirs[i].name to rightDirs[j].name
TCHAR buf[1024];
if (nDirs == 2)
	wsprintf(buf, _T("%s %s\n"), (i < dirs[0].GetSize()) ? dirs[0][i].name : "", (j < dirs[1].GetSize()) ? dirs[1][j].name : "");
else
	wsprintf(buf, _T("%s %s %s\n"), (i < dirs[0].GetSize()) ? dirs[0][i].name : "", (j < dirs[1].GetSize()) ?  dirs[1][j].name : "", (k < dirs[2].GetSize()) ? dirs[2][k].name : "");
OutputDebugString(buf);

		if (!bTreatDirAsEqual)
		{
			if (i<dirs[0].GetSize() && (j==dirs[1].GetSize() || collstr(dirs[0][i].name, dirs[1][j].name, casesensitive)<0)
				&& (nDirs < 3 ||      (k==dirs[2].GetSize() || collstr(dirs[0][i].name, dirs[2][k].name, casesensitive)<0) ))
			{
				nDiffCode = DIFFCODE::FIRST | DIFFCODE::DIR;
				if (!scanunpaireddir)
				{
					if (nDirs < 3)
						AddToList(subdir[0], subdir[1], &dirs[0][i], 0, nDiffCode, pList, pCtxt);
					else
						AddToList(subdir[0], subdir[1], subdir[2], &dirs[0][i], 0, 0, nDiffCode, pList, pCtxt);
					// Advance left pointer over left-only entry, and then retest with new pointers
					++i;
					continue;
				}
			}
			else if (j<dirs[1].GetSize() && (i==dirs[0].GetSize() || collstr(dirs[1][j].name, dirs[0][i].name, casesensitive)<0)
				&& (nDirs < 3 ||      (k==dirs[2].GetSize() || collstr(dirs[1][j].name, dirs[2][k].name, casesensitive)<0) ))
			{
				nDiffCode = DIFFCODE::SECOND | DIFFCODE::DIR;
				if (!scanunpaireddir)
				{
					if (nDirs < 3)
						AddToList(subdir[0], subdir[1], 0, &dirs[1][j], nDiffCode, pList, pCtxt);
					else
						AddToList(subdir[0], subdir[1], subdir[2], 0, &dirs[1][j], 0, nDiffCode, pList, pCtxt);
					// Advance right pointer over right-only entry, and then retest with new pointers
					++j;
					continue;
				}
			}
			else if (nDirs < 3)
			{
				nDiffCode = DIFFCODE::BOTH | DIFFCODE::DIR;
			}
			else
			{
				if (k<dirs[2].GetSize() && (i==dirs[0].GetSize() || collstr(dirs[2][k].name, dirs[0][i].name, casesensitive)<0)
					&&                     (j==dirs[1].GetSize() || collstr(dirs[2][k].name, dirs[1][j].name, casesensitive)<0) )
				{
					nDiffCode = DIFFCODE::THIRD | DIFFCODE::DIR;
					if (!scanunpaireddir)
					{
						AddToList(subdir[0], subdir[1], subdir[2], 0, 0, &dirs[2][k], nDiffCode, pList, pCtxt);
						++k;
						// Advance right pointer over right-only entry, and then retest with new pointers
						continue;
	
					}	
	
				}
				else if ((i<dirs[0].GetSize() && j<dirs[1].GetSize() && collstr(dirs[0][i].name, dirs[1][j].name, casesensitive) == 0)
				    && (k==dirs[2].GetSize() || collstr(dirs[2][k].name, dirs[0][i].name, casesensitive) != 0))
				{
					nDiffCode = DIFFCODE::FIRST | DIFFCODE::SECOND | DIFFCODE::DIR;
					if (!scanunpaireddir)
					{
						AddToList(subdir[0], subdir[1], subdir[2], &dirs[0][i], &dirs[1][j], 0, nDiffCode, pList, pCtxt);
						++i;
						++j;
						continue;	
					}
				}
				else if ((i<dirs[0].GetSize() && k<dirs[2].GetSize() && collstr(dirs[0][i].name, dirs[2][k].name, casesensitive) == 0)
				    && (j==dirs[1].GetSize() || collstr(dirs[1][j].name, dirs[2][k].name, casesensitive) != 0))
				{
					nDiffCode = DIFFCODE::FIRST | DIFFCODE::THIRD | DIFFCODE::DIR;
					if (!scanunpaireddir)
					{
						AddToList(subdir[0], subdir[1], subdir[2], &dirs[0][i], 0, &dirs[2][k], nDiffCode, pList, pCtxt);
						++i;
						++k;
						continue;
					}
				}
				else if ((j<dirs[1].GetSize() && k<dirs[2].GetSize() && collstr(dirs[1][j].name, dirs[2][k].name, casesensitive) == 0)
				    && (i==dirs[0].GetSize() || collstr(dirs[0][i].name, dirs[1][j].name, casesensitive) != 0))
				{
					nDiffCode = DIFFCODE::SECOND | DIFFCODE::THIRD | DIFFCODE::DIR;
					if (!scanunpaireddir)
					{
						AddToList(subdir[0], subdir[1], subdir[2], 0, &dirs[1][j], &dirs[2][k], nDiffCode, pList, pCtxt);
						++j;
						++k;
						continue;
					}
				}
				else
				{
					nDiffCode = DIFFCODE::ALL | DIFFCODE::DIR;
				}
			}
		}
		else
		{
			if (nDirs < 3)
				nDiffCode = DIFFCODE::BOTH | DIFFCODE::DIR;
			else
				nDiffCode = DIFFCODE::ALL | DIFFCODE::DIR;
		}

		// add to list
		if (!depth)
		{
			if (nDirs < 3)
				AddToList(subdir[0], subdir[1], 
					nDiffCode & DIFFCODE::FIRST  ? &dirs[0][i] : NULL, 
					nDiffCode & DIFFCODE::SECOND ? &dirs[1][j] : NULL,
					nDiffCode, pList, pCtxt);
			else
				AddToList(subdir[0], subdir[1], subdir[2], 
					nDiffCode & DIFFCODE::FIRST  ? &dirs[0][i] : NULL,
					nDiffCode & DIFFCODE::SECOND ? &dirs[1][j] : NULL,
					nDiffCode & DIFFCODE::THIRD  ? &dirs[2][k] : NULL,
					nDiffCode, pList, pCtxt);
		}
		else
		{
			// Recursive compare
			CString leftnewsub;
			CString rightnewsub;
			CString middlenewsub;
			if (nDirs < 3)
			{
				leftnewsub  = (nDiffCode & DIFFCODE::FIRST)  ? subprefix[0] + dirs[0][i].name : subprefix[0] + dirs[1][j].name;
				rightnewsub = (nDiffCode & DIFFCODE::SECOND) ? subprefix[1] + dirs[1][j].name : subprefix[1] + dirs[0][i].name;
			}
			else
			{
				leftnewsub   = subprefix[0];
				if (nDiffCode & DIFFCODE::FIRST)       leftnewsub += dirs[0][i].name;
				else if (nDiffCode & DIFFCODE::SECOND) leftnewsub += dirs[1][j].name;
				else if (nDiffCode & DIFFCODE::THIRD)  leftnewsub += dirs[2][k].name;
				middlenewsub = subprefix[1];
				if (nDiffCode & DIFFCODE::SECOND)      middlenewsub += dirs[1][j].name;
				else if (nDiffCode & DIFFCODE::FIRST)  middlenewsub += dirs[0][i].name;
				else if (nDiffCode & DIFFCODE::THIRD)  middlenewsub += dirs[2][k].name;
				rightnewsub  = subprefix[2];
				if (nDiffCode & DIFFCODE::THIRD)       rightnewsub += dirs[2][k].name;
				else if (nDiffCode & DIFFCODE::FIRST)  rightnewsub += dirs[0][i].name;
				else if (nDiffCode & DIFFCODE::SECOND) rightnewsub += dirs[1][j].name;
			}
			if (nDirs < 3)
			{
				// Test against filter so we don't include contents of filtered out directories
				// Also this is only place we can test for both-sides directories in recursive compare
				if (!pCtxt->m_piFilterGlobal->includeDir(leftnewsub, rightnewsub))
				{
					nDiffCode |= DIFFCODE::SKIPPED;
					AddToList(subdir[0], subdir[1], 
						nDiffCode & DIFFCODE::FIRST  ? &dirs[0][i] : NULL, 
						nDiffCode & DIFFCODE::SECOND ? &dirs[1][j] : NULL,
						nDiffCode, pList, pCtxt);
				}
				else
				{
					// Scan recursively all subdirectories too, we are not adding folders
					CString subdir[3];
					subdir[0] = leftnewsub;
					subdir[1] = rightnewsub;
					if (DirScan_GetItems(paths, subdir, pList, casesensitive,
							depth - 1, pCtxt) == -1)
					{
						return -1;
					}
				}
			}
			else
			{
				// Test against filter so we don't include contents of filtered out directories
				// Also this is only place we can test for both-sides directories in recursive compare
				if (!pCtxt->m_piFilterGlobal->includeDir(leftnewsub, middlenewsub, rightnewsub))
				{
					nDiffCode |= DIFFCODE::SKIPPED;
					AddToList(subdir[0], subdir[1], subdir[2], 
						nDiffCode & DIFFCODE::FIRST  ? &dirs[0][i] : NULL,
						nDiffCode & DIFFCODE::SECOND ? &dirs[1][j] : NULL,
						nDiffCode & DIFFCODE::THIRD  ? &dirs[2][k] : NULL,
						nDiffCode, pList, pCtxt);
				}
				else
				{
					// Scan recursively all subdirectories too, we are not adding folders
					CString subdir[3];
					subdir[0] = leftnewsub;
					subdir[1] = middlenewsub;
					subdir[2] = rightnewsub;
					if (DirScan_GetItems(paths, subdir, pList, casesensitive,
							depth - 1, pCtxt) == -1)
					{
						return -1;
					}
				}
			}
		}
		if (nDiffCode & DIFFCODE::FIRST)
			i++;
		if (nDiffCode & DIFFCODE::SECOND)
			j++;
		if (nDiffCode & DIFFCODE::THIRD)
			k++;
	}
	// Handle files
	// i points to current file in left list (files[0])
	// j points to current file in right list (files[1])
	i=0, j=0, k=0;
	while (1)
	{
		if (pCtxt->ShouldAbort())
			return -1;


		// Comparing file files[0][i].name to files[1][j].name
TCHAR buf[1024];
if (nDirs == 2)
	wsprintf(buf, _T("%s %s\n"), (i < files[0].GetSize()) ? files[0][i].name : "", (j < files[1].GetSize()) ? files[1][j].name : "");
else
	wsprintf(buf, _T("%s %s %s\n"), (i < files[0].GetSize()) ? files[0][i].name : "", (j < files[1].GetSize()) ?  files[1][j].name : "", (k < files[2].GetSize()) ? files[2][k].name : "");
OutputDebugString(buf);
		
		if (i<files[0].GetSize() && (j==files[1].GetSize() ||
				collstr(files[0][i].name, files[1][j].name, casesensitive) < 0)
			&& (nDirs < 3 || 
				(k==files[2].GetSize() || collstr(files[0][i].name, files[2][k].name, casesensitive)<0) ))
		{
			if (nDirs < 3)
			{
				const int nDiffCode = DIFFCODE::FIRST | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], &files[0][i], 0, nDiffCode, pList, pCtxt);
			}
			else
			{
				const int nDiffCode = DIFFCODE::FIRST | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], &files[0][i], 0, 0, nDiffCode, pList, pCtxt);
			}
			// Advance left pointer over left-only entry, and then retest with new pointers
			++i;
			continue;
		}
		if (j<files[1].GetSize() && (i==files[0].GetSize() ||
				collstr(files[0][i].name, files[1][j].name, casesensitive) > 0)
			&& (nDirs < 3 ||
				(k==files[2].GetSize() || collstr(files[1][j].name, files[2][k].name, casesensitive)<0) ))
		{
			const int nDiffCode = DIFFCODE::SECOND | DIFFCODE::FILE;
			if (nDirs < 3)
				AddToList(subdir[0], subdir[1], 0, &files[1][j], nDiffCode, pList, pCtxt);
			else
				AddToList(subdir[0], subdir[1], subdir[2], 0, &files[1][j], 0, nDiffCode, pList, pCtxt);
			// Advance right pointer over right-only entry, and then retest with new pointers
			++j;
			continue;
		}
		if (nDirs == 3)
		{
			if (k<files[2].GetSize() && (i==files[0].GetSize() ||
					collstr(files[2][k].name, files[0][i].name, casesensitive)<0)
				&& (j==files[1].GetSize() || collstr(files[2][k].name, files[1][j].name, casesensitive)<0) )
			{
				int nDiffCode = DIFFCODE::THIRD | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], 0, 0, &files[2][k], nDiffCode, pList, pCtxt);
				++k;
				// Advance right pointer over right-only entry, and then retest with new pointers
				continue;
			}

			if ((i<files[0].GetSize() && j<files[1].GetSize() && collstr(files[0][i].name, files[1][j].name, casesensitive) == 0)
			    && (k==files[2].GetSize() || collstr(files[0][i].name, files[2][k].name, casesensitive) != 0))
			{
				int nDiffCode = DIFFCODE::FIRST | DIFFCODE::SECOND | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], &files[0][i], &files[1][j], 0, nDiffCode, pList, pCtxt);
				++i;
				++j;
				continue;
			}
			else if ((i<files[0].GetSize() && k<files[2].GetSize() && collstr(files[0][i].name, files[2][k].name, casesensitive) == 0)
			    && (j==files[1].GetSize() || collstr(files[1][j].name, files[2][k].name, casesensitive) != 0))
			{
				int nDiffCode = DIFFCODE::FIRST | DIFFCODE::THIRD | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], &files[0][i], 0, &files[2][k], nDiffCode, pList, pCtxt);
				++i;
				++k;
				continue;
			}
			else if ((j<files[1].GetSize() && k<files[2].GetSize() && collstr(files[1][j].name, files[2][k].name, casesensitive) == 0)
			    && (i==files[0].GetSize() || collstr(files[0][i].name, files[1][j].name, casesensitive) != 0))
			{
				int nDiffCode = DIFFCODE::SECOND | DIFFCODE::THIRD | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], 0, &files[1][j], &files[2][k], nDiffCode, pList, pCtxt);
				++j;
				++k;
				continue;
			}
		}
		if (i<files[0].GetSize())
		{
			if (nDirs < 3)
			{
				ASSERT(j<files[1].GetSize());
				const int nDiffCode = DIFFCODE::BOTH | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], &files[0][i], &files[1][j], nDiffCode, pList, pCtxt);
				++i;
				++j;
				continue;
			}
			else
			{
				ASSERT(j<files[1].GetSize());
				ASSERT(k<files[2].GetSize());
				const int nDiffCode = DIFFCODE::ALL | DIFFCODE::FILE;
				AddToList(subdir[0], subdir[1], subdir[2], &files[0][i], &files[1][j], &files[2][k], nDiffCode, pList, pCtxt);
				++i;
				++j;
				++k;
				continue;
			}
		}
		break;
	}
	return 1;
}

/**
 * @brief Compare DiffItems in list and add results to compare context.
 *
 * @param list [in] List of items to compare
 * @param pCtxt [in,out] Compare context: contains list where results are added.
 * @param piAbortable [in] Interface allowing to abort compare
 * @return 1 if compare finished, -1 if compare was aborted
 */
int DirScan_CompareItems(DiffItemList & list, CDiffContext * pCtxt)
{
	int res = 1;
	POSITION pos = list.GetFirstDiffPosition();
	
	while (pos != NULL)
	{
		if (pCtxt->ShouldAbort())
		{
			res = -1;
			break;
		}

		DIFFITEM di = list.GetNextDiffPosition(pos);
		CompareDiffItem(di, pCtxt);
	}
	return res;
}

/**
 * @brief Compare DiffItems in context marked for rescan.
 *
 * @param pCtxt [in,out] Compare context: contains list of items.
 * @param piAbortable [in] Interface allowing to abort compare
 * @return 1 if compare finished, -1 if compare was aborted
 */
int DirScan_CompareItems(CDiffContext * pCtxt)
{
	int res = 1;
	POSITION pos = pCtxt->GetFirstDiffPosition();
	
	while (pos != NULL)
	{
		if (pCtxt->ShouldAbort())
		{
			res = -1;
			break;
		}

		POSITION oldPos = pos;
		DIFFITEM di = pCtxt->GetNextDiffPosition(pos);
		if (di.isScanNeeded())
		{
			BOOL bItemsExist = TRUE;
			pCtxt->RemoveDiff(oldPos);
			UpdateDiffItem(di, bItemsExist, pCtxt);
			if (bItemsExist)
				CompareDiffItem(di, pCtxt);
		}
	}
	return res;
}

/**
 * @brief Update diffitem file/dir infos.
 *
 * Re-tests dirs/files if sides still exists, and updates infos for
 * existing sides. This assumes filenames, or paths are not changed.
 * Since in normal situations (I can think of) they cannot change
 * after first compare.
 *
 * @param [in,out] di DiffItem to update.
 * @param [out] bExists Set to
 *  - TRUE if one of items exists so diffitem is valid
 *  - FALSE if items were deleted, so diffitem is not valid
 * @param [in] pCtxt Compare context
 */
void UpdateDiffItem(DIFFITEM & di, BOOL & bExists, CDiffContext *pCtxt)
{
	// Clear side-info and file-infos
	di.diffFileInfo[0].Clear();
	di.diffFileInfo[1].Clear();
	di.setSideBoth(); // FIXME: DIRTY HACK for UpdateInfoFromDiskHalf
	BOOL bLeftExists = pCtxt->UpdateInfoFromDiskHalf(di, 0);
	BOOL bRightExists = pCtxt->UpdateInfoFromDiskHalf(di, 1);
	bExists = bLeftExists || bRightExists;
	if (bLeftExists)
	{
		if (bRightExists)
			di.setSideBoth();
		else
			di.setSideFirst();
	}
	else
	{
		if (bRightExists)
			di.setSideSecond();
		else
			di.setSideNone();
	}
}

/**
 * @brief Compare two diffitems and add results to difflist in context.
 *
 * This function does the actual compare for previously gathered list of
 * items. Basically we:
 * - ignore items matching filefilters
 * - add non-ignored directories (no compare for directory items)
 * - add  unique files
 * - compare files
 *
 * @param [in] di DiffItem to compare
 * @param [in,out] pCtxt Compare context: contains difflist, encoding info etc.
 * @todo For date compare, maybe we should use creation date if modification
 * date is missing?
 */
void CompareDiffItem(DIFFITEM di, CDiffContext * pCtxt)
{
	int nDirs = pCtxt->GetCompareDirs();
	// Clear rescan-request flag (not set by all codepaths)
	di.diffcode &= ~DIFFCODE::NEEDSCAN;
	// Is it a directory?
	if (di.isDirectory())
	{
		// 1. Test against filters
		if (
			(nDirs == 2 && pCtxt->m_piFilterGlobal->includeDir(di.sFilename[0], di.sFilename[1])) ||
			(nDirs == 3 && pCtxt->m_piFilterGlobal->includeDir(di.sFilename[0], di.sFilename[1], di.sFilename[2]))
			)
			di.diffcode |= DIFFCODE::INCLUDED;
		else
			di.diffcode |= DIFFCODE::SKIPPED;
		// We don't actually 'compare' directories, just add non-ignored
		// directories to list.
		StoreDiffData(di, pCtxt, NULL);
	}
	else
	{
		// 1. Test against filters
		if (
			(nDirs == 2 && pCtxt->m_piFilterGlobal->includeFile(di.sFilename[0], di.sFilename[1])) ||
			(nDirs == 3 && pCtxt->m_piFilterGlobal->includeFile(di.sFilename[0], di.sFilename[1], di.sFilename[2]))
			)
		{
			di.diffcode |= DIFFCODE::INCLUDED;
			// 2. Add unique files
			// We must compare unique files to itself to detect encoding
			if (di.isSideFirst() || di.isSideSecond() || (nDirs > 2 && di.isSideThird()))
			{
				if (pCtxt->m_nCompMethod != CMP_DATE &&
					pCtxt->m_nCompMethod != CMP_DATE_SIZE &&
					pCtxt->m_nCompMethod != CMP_SIZE)
				{
					DiffFileData diffdata;
					int diffCode = diffdata.prepAndCompareFiles(pCtxt, di);

					// Add possible binary flag for unique items
					if (diffCode & DIFFCODE::BIN)
						di.diffcode |= DIFFCODE::BIN;
					StoreDiffData(di, pCtxt, &diffdata);
				}
				else
				{
					StoreDiffData(di, pCtxt, NULL);
				}
			}
			// 3. Compare two files
			else if (pCtxt->m_nCompMethod == CMP_DATE ||
				pCtxt->m_nCompMethod == CMP_DATE_SIZE)
			{
				// Compare by modified date
				// Check that we have both filetimes
				if (di.diffFileInfo[0].mtime != 0 && di.diffFileInfo[1].mtime != 0 && (nDirs < 3 || di.diffFileInfo[2].mtime != 0))
				{
					int i;
					__int64 nTimeDiff[2];
					for (i = 0; i < nDirs - 1; i++)
					{
						// Compare by modified date
						nTimeDiff[i] = di.diffFileInfo[i].mtime - di.diffFileInfo[i + 1].mtime;
						// Remove sign
						nTimeDiff[i] = (nTimeDiff[i] > 0 ? nTimeDiff[i] : -nTimeDiff[i]);
						if (pCtxt->m_bIgnoreSmallTimeDiff)
						{
							// If option to ignore small timediffs (couple of seconds)
							// is set, decrease absolute difference by allowed diff
							nTimeDiff[i] -= SmallTimeDiff;
						}
					}
					if (nTimeDiff[0] <= 0 && (nDirs < 3 || nTimeDiff[1] <= 0))
						di.diffcode |= DIFFCODE::TEXT | DIFFCODE::SAME;
					else
						di.diffcode |= DIFFCODE::TEXT | DIFFCODE::DIFF;
				}
				else
				{
					// Filetimes for item(s) could not be read. So we have to
					// set error status, unless we have DATE_SIZE -compare
					// when we have still hope for size compare..
					if (pCtxt->m_nCompMethod == CMP_DATE_SIZE)
						di.diffcode |= DIFFCODE::TEXT | DIFFCODE::SAME;
					else
						di.diffcode |= DIFFCODE::TEXT | DIFFCODE::CMPERR;
				}
				
				// This is actual CMP_DATE_SIZE method..
				// If file sizes differ mark them different
				if ( pCtxt->m_nCompMethod == CMP_DATE_SIZE && di.isResultSame())
				{
					if (di.diffFileInfo[0].size != di.diffFileInfo[1].size && (nDirs < 3 || di.diffFileInfo[1].size != di.diffFileInfo[2].size))
					{
						di.diffcode &= ~DIFFCODE::SAME;
						di.diffcode |= DIFFCODE::DIFF;
					}
				}

				// report result back to caller
				StoreDiffData(di, pCtxt, NULL);
			}
			else if (pCtxt->m_nCompMethod == CMP_SIZE)
			{
				// Compare by size
				if (di.diffFileInfo[0].size == di.diffFileInfo[1].size && (nDirs < 3 || di.diffFileInfo[1].size == di.diffFileInfo[2].size))
					di.diffcode |= DIFFCODE::SAME;
				else
					di.diffcode |= DIFFCODE::DIFF;

				// report result back to caller
				StoreDiffData(di, pCtxt, NULL);
			}
			else
			{
				// Really compare
				DiffFileData diffdata;
				di.diffcode |= diffdata.prepAndCompareFiles(pCtxt, di);
				// report result back to caller
				StoreDiffData(di, pCtxt, &diffdata);
			}
		}
		else
		{
			di.diffcode |= DIFFCODE::SKIPPED;
			StoreDiffData(di, pCtxt, NULL);
		}
	}
}

/**
 * @brief Load arrays with all directories & files in specified dir
 */
void LoadAndSortFiles(const CString & sDir, fentryArray * dirs, fentryArray * files, bool casesensitive)
{
	LoadFiles(sDir, dirs, files);
	Sort(dirs, casesensitive);
	Sort(files, casesensitive);
}

/**
 * @brief Convert time in type FILETIME to type int (time_t compatible).
 * @param [in] time Time in FILETIME type.
 * @return Time in time_t compiliant integer.
 */
static __int64 FiletimeToTimeT(FILETIME time)
{
	const __int64 SecsTo100ns = 10000000;
	const __int64 SecsBetweenEpochs = 11644473600;
	__int64 converted_time;
	converted_time = ((__int64)time.dwHighDateTime << 32) + time.dwLowDateTime;
	converted_time -= (SecsBetweenEpochs * SecsTo100ns);
	converted_time /= SecsTo100ns;
	return converted_time;
}

/**
 * @brief Find files and subfolders from given folder.
 * This function saves all files and subfolders in given folder to arrays.
 * We use 64-bit version of stat() to get times since find doesn't return
 * valid times for very old files (around year 1970). Even stat() seems to
 * give negative time values but we can live with that. Those around 1970
 * times can happen when file is created so that it  doesn't get valid
 * creation or modificatio dates.
 * @param [in] sDir Base folder for files and subfolders.
 * @param [in, out] dirs Array where subfolders are stored.
 * @param [in, out] files Array where files are stored.
 */
void LoadFiles(const CString & sDir, fentryArray * dirs, fentryArray * files)
{
	CString sPattern = sDir;
	sPattern.TrimRight(_T("\\"));
	sPattern += _T("\\*.*");

	WIN32_FIND_DATA ff;
	HANDLE h = FindFirstFile(sPattern, &ff);
	if (h != INVALID_HANDLE_VALUE)
	{
		do
		{
			DWORD dwIsDirectory = ff.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY;
			if (dwIsDirectory && StrStr(_T(".."), ff.cFileName))
				continue;

			fentry ent;

			// Save filetimes as seconds since January 1, 1970
			// Note that times can be < 0 if they are around that 1970..
			// Anyway that is not sensible case for normal files so we can
			// just use zero for their time.
			ent.ctime = FiletimeToTimeT(ff.ftCreationTime);
			if (ent.ctime < 0)
				ent.ctime = 0;
			ent.mtime = FiletimeToTimeT(ff.ftLastWriteTime);
			if (ent.mtime < 0)
				ent.mtime = 0;

			if (ff.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
				ent.size = -1;  // No size for directories
			else
			{
				ent.size = ((__int64)ff.nFileSizeHigh << 32) + ff.nFileSizeLow;
			}

			ent.name = ff.cFileName;
			ent.attrs = ff.dwFileAttributes;
			(dwIsDirectory ? dirs : files) -> Add(ent);
		} while (FindNextFile(h, &ff));
		FindClose(h);
	}
}

/**
 * @brief case-sensitive collate function for qsorting an array
 */
static int __cdecl cmpstring(const void *elem1, const void *elem2)
{
	const fentry * s1 = static_cast<const fentry *>(elem1);
	const fentry * s2 = static_cast<const fentry *>(elem2);
	return s1->name.Collate(s2->name);
}

/**
 * @brief case-insensitive collate function for qsorting an array
 */
static int __cdecl cmpistring(const void *elem1, const void *elem2)
{
	const fentry * s1 = static_cast<const fentry *>(elem1);
	const fentry * s2 = static_cast<const fentry *>(elem2);
	return s1->name.CollateNoCase(s2->name);
}

/**
 * @brief sort specified array
 */
void Sort(fentryArray * dirs, bool casesensitive)
{
	fentry * data = dirs->GetData();
	if (!data) return;
	int (__cdecl *comparefnc)(const void *elem1, const void *elem2) = (casesensitive ? cmpstring : cmpistring);
	qsort(data, dirs->GetSize(), sizeof(dirs->GetAt(0)), comparefnc);
}

/**
 * @brief  Compare (NLS aware) two strings, either case-sensitive or case-insensitive as caller specifies
 */
static int collstr(const CString & s1, const CString & s2, bool casesensitive)
{
	if (casesensitive)
		return s1.Collate(s2);
	else
		return s1.CollateNoCase(s2);
}

/**
 * @brief Send one file or directory result back through the diff context
 */
static void StoreDiffData(DIFFITEM &di, CDiffContext * pCtxt,
		const DiffFileData * pDiffFileData)
{
	if (pDiffFileData)
	{
		// Set text statistics
		if (di.isSideLeftOrBoth())
			di.diffFileInfo[0].m_textStats = pDiffFileData->m_textStats[0];
		if (di.isSideRightOrBoth())
			di.diffFileInfo[1].m_textStats = pDiffFileData->m_textStats[1];

		di.nsdiffs = pDiffFileData->m_ndiffs;
		di.nidiffs = pDiffFileData->m_ntrivialdiffs;

		if (!di.isSideFirst())
		{
			di.diffFileInfo[1].encoding = pDiffFileData->m_FileLocation[1].encoding;
		}
		
		if (!di.isSideSecond())
		{
			di.diffFileInfo[0].encoding = pDiffFileData->m_FileLocation[0].encoding;
		}
	}

	gLog.Write
	(
		CLogFile::LCOMPAREDATA, _T("name=<%s>, leftdir=<%s>, rightdir=<%s>, code=%d"),
		(LPCTSTR)di.sFilename[0], (LPCTSTR)_T("di.diffFileInfo[0].spath"), (LPCTSTR)_T("di.diffFileInfo[1].spath"), di.diffcode
	);
	pCtxt->AddDiff(di);
}

/**
 * @brief Add one compare item to list.
 */
static void AddToList(const CString & sLeftDir, const CString & sRightDir,
	const fentry * lent, const fentry * rent,
	int code, DiffItemList * pList, CDiffContext *pCtxt)
{
	AddToList(sLeftDir, sRightDir, sLeftDir, lent, rent, 0, code, pList, pCtxt);
}

/**
 * @brief Add one compare item to list.
 */
static void AddToList(const CString & sLeftDir, const CString & sMiddleDir, const CString & sRightDir,
	const fentry * lent, const fentry * ment, const fentry * rent,
	int code, DiffItemList * pList, CDiffContext *pCtxt)
{
	// We must store both paths - we cannot get paths later
	// and we need unique item paths for example when items
	// change to identical

	DIFFITEM di;

	di.sSubdir[0] = sLeftDir;
	di.sSubdir[1] = sMiddleDir;
	di.sSubdir[2] = sRightDir;

	if (lent)
	{
		di.sFilename[0] = lent->name;
		di.diffFileInfo[0].mtime = lent->mtime;
		di.diffFileInfo[0].ctime = lent->ctime;
		di.diffFileInfo[0].size = lent->size;
		di.diffFileInfo[0].flags.attributes = lent->attrs;
	}
	else
	{
		// Don't break CDirView::DoCopyRightToLeft()
		if (rent)
			di.sFilename[0] = rent->name;
		else
			di.sFilename[0] = ment->name;
	}

	if (ment)
	{
		di.sFilename[1] = OPTIMIZE_SHARE_CSTRINGDATA
		(
			ment && di.sFilename[0] == ment->name ? di.sFilename[0] :
		) ment->name;
		di.diffFileInfo[1].mtime = ment->mtime;
		di.diffFileInfo[1].ctime = ment->ctime;
		di.diffFileInfo[1].size = ment->size;
		di.diffFileInfo[1].flags.attributes = ment->attrs;
	}
	else
	{
		// Don't break CDirView::DoCopyLeftToRight()
		if (lent)
			di.sFilename[1] = lent->name;
		else
			di.sFilename[1] = rent->name;
	}

	if (rent)
	{
		di.sFilename[2] = OPTIMIZE_SHARE_CSTRINGDATA
		(
			lent && di.sFilename[0] == lent->name ? di.sFilename[0] :
		) rent->name;
		di.diffFileInfo[2].mtime = rent->mtime;
		di.diffFileInfo[2].ctime = rent->ctime;
		di.diffFileInfo[2].size = rent->size;
		di.diffFileInfo[2].flags.attributes = rent->attrs;
	}
	else
	{
		// Don't break CDirView::DoCopyLeftToRight()
		if (lent)
			di.sFilename[2] = lent->name;
		else
			di.sFilename[2] = ment->name;
	}

	di.diffcode = code;

	gLog.Write
	(
		CLogFile::LCOMPAREDATA, _T("name=<%s>, leftdir=<%s>, rightdir=<%s>, code=%d"),
		(LPCTSTR)di.sFilename[0], (LPCTSTR)_T("di.diffFileInfo[0].spath"), (LPCTSTR)_T("di.diffFileInfo[1].spath"), code
	);
	pCtxt->m_pCompareStats->IncreaseTotalItems();
	pList->AddDiff(di);
}

void // static
DirScan_InitializeDefaultCodepage()
{
	// Set thread default codepage
	// This is application-wide initialization
	// but neither MainFrame nor MergeApp included the needed headers
	DiffFileData::SetDefaultCodepage(getDefaultCodepage());
}

