/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tika.pipes;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.tika.metadata.TikaCoreProperties;
import org.apache.tika.pipes.FetchEmitTuple;
import org.apache.tika.pipes.PipesConfigBase;
import org.apache.tika.pipes.PipesResult;
import org.apache.tika.pipes.PipesServer;
import org.apache.tika.pipes.emitter.EmitData;
import org.apache.tika.utils.ProcessUtils;
import org.apache.tika.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PipesClient
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(PipesClient.class);
    private static final int MAX_BYTES_BEFORE_READY = 20000;
    private static AtomicInteger CLIENT_COUNTER = new AtomicInteger(0);
    private Process process;
    private final PipesConfigBase pipesConfig;
    private DataOutputStream output;
    private DataInputStream input;
    private final int pipesClientId;
    private final ExecutorService executorService = Executors.newFixedThreadPool(1);
    private int filesProcessed = 0;

    public PipesClient(PipesConfigBase pipesConfig) {
        this.pipesConfig = pipesConfig;
        this.pipesClientId = CLIENT_COUNTER.getAndIncrement();
    }

    public int getFilesProcessed() {
        return this.filesProcessed;
    }

    private boolean ping() {
        if (this.process == null || !this.process.isAlive()) {
            return false;
        }
        try {
            this.output.write(PipesServer.STATUS.PING.getByte());
            this.output.flush();
            int ping = this.input.read();
            if (ping == PipesServer.STATUS.PING.getByte()) {
                return true;
            }
        }
        catch (IOException e) {
            return false;
        }
        return false;
    }

    @Override
    public void close() throws IOException {
        if (this.process != null) {
            this.process.destroyForcibly();
        }
        this.executorService.shutdownNow();
    }

    public PipesResult process(FetchEmitTuple t) throws IOException {
        if (!this.ping()) {
            this.restart();
        }
        if (this.pipesConfig.getMaxFilesProcessedPerProcess() > 0 && this.filesProcessed >= this.pipesConfig.getMaxFilesProcessedPerProcess()) {
            LOG.info("restarting server after hitting max files: " + this.filesProcessed);
            this.restart();
        }
        return this.actuallyProcess(t);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PipesResult actuallyProcess(FetchEmitTuple t) {
        long start = System.currentTimeMillis();
        FutureTask<PipesResult> futureTask = new FutureTask<PipesResult>(() -> {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(bos);){
                objectOutputStream.writeObject(t);
            }
            byte[] bytes = bos.toByteArray();
            this.output.write(PipesServer.STATUS.CALL.getByte());
            this.output.writeInt(bytes.length);
            this.output.write(bytes);
            this.output.flush();
            if (LOG.isTraceEnabled()) {
                LOG.trace("timer -- write tuple: {} ms", (Object)(System.currentTimeMillis() - start));
            }
            long readStart = System.currentTimeMillis();
            PipesResult result = this.readResults(t, start);
            if (LOG.isTraceEnabled()) {
                LOG.trace("timer -- read result: {} ms", (Object)(System.currentTimeMillis() - readStart));
            }
            return result;
        });
        try {
            this.executorService.execute(futureTask);
            PipesResult pipesResult = futureTask.get(this.pipesConfig.getTimeoutMillis(), TimeUnit.MILLISECONDS);
            return pipesResult;
        }
        catch (InterruptedException e) {
            this.process.destroyForcibly();
            PipesResult pipesResult = PipesResult.INTERRUPTED_EXCEPTION;
            return pipesResult;
        }
        catch (ExecutionException e) {
            LOG.error("pipesClientId=" + this.pipesClientId + " execution exception", e);
            long elapsed = System.currentTimeMillis() - start;
            this.destroyWithPause();
            if (!this.process.isAlive() && 17 == this.process.exitValue()) {
                LOG.warn("pipesClientId={} server timeout: {} in {} ms", this.pipesClientId, t.getId(), elapsed);
                PipesResult pipesResult = PipesResult.TIMEOUT;
                return pipesResult;
            }
            try {
                this.process.waitFor(500L, TimeUnit.MILLISECONDS);
                if (this.process.isAlive()) {
                    LOG.warn("pipesClientId={} crash: {} in {} ms with no exit code available", this.pipesClientId, t.getId(), elapsed);
                } else {
                    LOG.warn("pipesClientId={} crash: {} in {} ms with exit code {}", this.pipesClientId, t.getId(), elapsed, this.process.exitValue());
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            PipesResult pipesResult = PipesResult.UNSPECIFIED_CRASH;
            return pipesResult;
        }
        catch (TimeoutException e) {
            long elapsed = System.currentTimeMillis() - start;
            this.process.destroyForcibly();
            LOG.warn("pipesClientId={} client timeout: {} in {} ms", this.pipesClientId, t.getId(), elapsed);
            PipesResult pipesResult = PipesResult.TIMEOUT;
            return pipesResult;
        }
        finally {
            futureTask.cancel(true);
        }
    }

    private void destroyWithPause() {
        try {
            this.process.waitFor(200L, TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException interruptedException) {
        }
        finally {
            this.process.destroyForcibly();
        }
    }

    private PipesResult readResults(FetchEmitTuple t, long start) throws IOException {
        int statusByte = this.input.read();
        long millis = System.currentTimeMillis() - start;
        PipesServer.STATUS status = null;
        try {
            status = PipesServer.STATUS.lookup(statusByte);
        }
        catch (IllegalArgumentException e) {
            throw new IOException("problem reading response from server " + (Object)((Object)status));
        }
        switch (status) {
            case OOM: {
                LOG.warn("pipesClientId={} oom: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return PipesResult.OOM;
            }
            case TIMEOUT: {
                LOG.warn("pipesClientId={} server response timeout: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return PipesResult.TIMEOUT;
            }
            case EMIT_EXCEPTION: {
                LOG.warn("pipesClientId={} emit exception: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.readMessage(PipesResult.STATUS.EMIT_EXCEPTION);
            }
            case EMITTER_NOT_FOUND: {
                LOG.warn("pipesClientId={} emitter not found: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.readMessage(PipesResult.STATUS.NO_EMITTER_FOUND);
            }
            case FETCHER_NOT_FOUND: {
                LOG.warn("pipesClientId={} fetcher not found: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.readMessage(PipesResult.STATUS.NO_FETCHER_FOUND);
            }
            case FETCHER_INITIALIZATION_EXCEPTION: {
                LOG.warn("pipesClientId={} fetcher initialization exception: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.readMessage(PipesResult.STATUS.FETCHER_INITIALIZATION_EXCEPTION);
            }
            case FETCH_EXCEPTION: {
                LOG.warn("pipesClientId={} fetch exception: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.readMessage(PipesResult.STATUS.FETCH_EXCEPTION);
            }
            case PARSE_SUCCESS: {
                LOG.info("pipesClientId={} parse success: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return this.deserializeEmitData();
            }
            case PARSE_EXCEPTION_NO_EMIT: {
                return this.readMessage(PipesResult.STATUS.PARSE_EXCEPTION_NO_EMIT);
            }
            case EMIT_SUCCESS: {
                LOG.info("pipesClientId={} emit success: {} in {} ms", this.pipesClientId, t.getId(), millis);
                return PipesResult.EMIT_SUCCESS;
            }
            case EMIT_SUCCESS_PARSE_EXCEPTION: {
                return this.readMessage(PipesResult.STATUS.EMIT_SUCCESS_PARSE_EXCEPTION);
            }
            case EMPTY_OUTPUT: {
                return PipesResult.EMPTY_OUTPUT;
            }
            case READY: 
            case CALL: 
            case PING: 
            case FAILED_TO_START: {
                throw new IOException("Not expecting this status: " + (Object)((Object)status));
            }
        }
        throw new IOException("Need to handle procesing for: " + (Object)((Object)status));
    }

    private PipesResult readMessage(PipesResult.STATUS status) throws IOException {
        int length = this.input.readInt();
        byte[] bytes = new byte[length];
        this.input.readFully(bytes);
        String msg = new String(bytes, StandardCharsets.UTF_8);
        return new PipesResult(status, msg);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PipesResult deserializeEmitData() throws IOException {
        int length = this.input.readInt();
        byte[] bytes = new byte[length];
        this.input.readFully(bytes);
        try (ObjectInputStream objectInputStream = new ObjectInputStream(new ByteArrayInputStream(bytes));){
            EmitData emitData = (EmitData)objectInputStream.readObject();
            String stack = this.getStack(emitData);
            if (StringUtils.isBlank(stack)) {
                PipesResult pipesResult = new PipesResult(emitData);
                return pipesResult;
            }
            PipesResult pipesResult = new PipesResult(emitData, stack);
            return pipesResult;
        }
        catch (ClassNotFoundException e) {
            LOG.error("class not found exception deserializing data", e);
            throw new RuntimeException(e);
        }
    }

    private String getStack(EmitData emitData) {
        if (emitData.getMetadataList() == null || emitData.getMetadataList().size() < 1) {
            return "";
        }
        return emitData.getMetadataList().get(0).get(TikaCoreProperties.CONTAINER_EXCEPTION);
    }

    private void restart() throws IOException {
        if (this.process != null) {
            this.process.destroyForcibly();
            LOG.info("restarting process");
        } else {
            LOG.info("starting process");
        }
        ProcessBuilder pb = new ProcessBuilder(this.getCommandline());
        pb.redirectError(ProcessBuilder.Redirect.INHERIT);
        this.process = pb.start();
        this.input = new DataInputStream(this.process.getInputStream());
        this.output = new DataOutputStream(this.process.getOutputStream());
        FutureTask<Integer> futureTask = new FutureTask<Integer>(() -> {
            int read;
            int b = this.input.read();
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            for (read = 1; read < 20000 && b != PipesServer.STATUS.READY.getByte(); ++read) {
                if (b == -1) {
                    throw new RuntimeException("Couldn't start server: read EOF before 'ready' byte.\n Make absolutely certain that your logger is not writing to stdout.");
                }
                bos.write(b);
                b = this.input.read();
            }
            if (read >= 20000) {
                throw new RuntimeException("Couldn't start server: read too many bytes before 'ready' byte.\n Make absolutely certain that your logger is not writing to stdout.\n Message read: " + new String(bos.toByteArray(), StandardCharsets.ISO_8859_1));
            }
            if (bos.size() > 0) {
                LOG.warn("From forked process before start byte: {}", (Object)new String(bos.toByteArray(), StandardCharsets.ISO_8859_1));
            }
            return 1;
        });
        this.executorService.submit(futureTask);
        try {
            futureTask.get(this.pipesConfig.getStartupTimeoutMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            this.process.destroyForcibly();
            return;
        }
        catch (ExecutionException e) {
            LOG.error("couldn't start server", e);
            this.process.destroyForcibly();
            throw new RuntimeException(e);
        }
        catch (TimeoutException e) {
            LOG.error("couldn't start server in time", e);
            this.process.destroyForcibly();
            throw new RuntimeException(e);
        }
        finally {
            futureTask.cancel(true);
        }
    }

    private String[] getCommandline() {
        List<String> configArgs = this.pipesConfig.getForkedJvmArgs();
        boolean hasClassPath = false;
        boolean hasHeadless = false;
        boolean hasExitOnOOM = false;
        boolean hasLog4j = false;
        String origGCString = null;
        String newGCLogString = null;
        for (String arg : configArgs) {
            if (arg.startsWith("-Djava.awt.headless")) {
                hasHeadless = true;
            }
            if (arg.equals("-cp") || arg.equals("--classpath")) {
                hasClassPath = true;
            }
            if (arg.equals("-XX:+ExitOnOutOfMemoryError") || arg.equals("-XX:+CrashOnOutOfMemoryError")) {
                hasExitOnOOM = true;
            }
            if (arg.startsWith("-Dlog4j.configuration")) {
                hasLog4j = true;
            }
            if (!arg.startsWith("-Xloggc:")) continue;
            origGCString = arg;
            newGCLogString = arg.replace("${pipesClientId}", "id-" + this.pipesClientId);
        }
        if (origGCString != null && newGCLogString != null) {
            configArgs.remove(origGCString);
            configArgs.add(newGCLogString);
        }
        ArrayList<String> commandLine = new ArrayList<String>();
        String javaPath = this.pipesConfig.getJavaPath();
        commandLine.add(ProcessUtils.escapeCommandLine(javaPath));
        if (!hasClassPath) {
            commandLine.add("-cp");
            commandLine.add(System.getProperty("java.class.path"));
        }
        if (!hasHeadless) {
            commandLine.add("-Djava.awt.headless=true");
        }
        if (hasExitOnOOM) {
            LOG.warn("I notice that you have an exit/crash on OOM. If you run heavy external processes like tesseract, this setting may result in orphaned processes which could be disastrous for performance.");
        }
        if (!hasLog4j) {
            commandLine.add("-Dlog4j.configurationFile=classpath:pipes-fork-server-default-log4j2.xml");
        }
        commandLine.add("-DpipesClientId=" + this.pipesClientId);
        commandLine.addAll(configArgs);
        commandLine.add("org.apache.tika.pipes.PipesServer");
        commandLine.add(ProcessUtils.escapeCommandLine(this.pipesConfig.getTikaConfig().toAbsolutePath().toString()));
        commandLine.add(Long.toString(this.pipesConfig.getMaxForEmitBatchBytes()));
        commandLine.add(Long.toString(this.pipesConfig.getTimeoutMillis()));
        commandLine.add(Long.toString(this.pipesConfig.getShutdownClientAfterMillis()));
        LOG.debug("commandline: {}", (Object)commandLine);
        return commandLine.toArray(new String[0]);
    }
}

