<?php

define('MSG_COPYRIGHT_SHORT', '(C) 2009 '.BLOG_ADMIN);
define('MSG_COPYRIGHT_LONG', 'Copyright '.MSG_COPYRIGHT_SHORT.' All Rights Reserved.');

function twtr($message){

//OAuth Start


/*
 * Abraham Williams (abraham@abrah.am) http://abrah.am
 *
 * The first PHP Library to support OAuth for Twitter's REST API.
 */

/* Load OAuth lib. You can find it at http://oauth.net */
//require_once('OAuth.php');

// vim: foldmethod=marker

/* Generic exception class
 */
class OAuthException extends Exception {
  // pass
}

class OAuthConsumer {
  public $key;
  public $secret;

  function __construct($key, $secret, $callback_url=NULL) {
    $this->key = $key;
    $this->secret = $secret;
    $this->callback_url = $callback_url;
  }

  function __toString() {
    return "OAuthConsumer[key=$this->key,secret=$this->secret]";
  }
}

class OAuthToken {
  // access tokens and request tokens
  public $key;
  public $secret;

  /**
   * key = the token
   * secret = the token secret
   */
  function __construct($key, $secret) {
    $this->key = $key;
    $this->secret = $secret;
  }

  /**
   * generates the basic string serialization of a token that a server
   * would respond to request_token and access_token calls with
   */
  function to_string() {
    return "oauth_token=" .
           OAuthUtil::urlencode_rfc3986($this->key) .
           "&oauth_token_secret=" .
           OAuthUtil::urlencode_rfc3986($this->secret);
  }

  function __toString() {
    return $this->to_string();
  }
}

/**
 * A class for implementing a Signature Method
 * See section 9 ("Signing Requests") in the spec
 */
abstract class OAuthSignatureMethod {
  /**
   * Needs to return the name of the Signature Method (ie HMAC-SHA1)
   * @return string
   */
  abstract public function get_name();

  /**
   * Build up the signature
   * NOTE: The output of this function MUST NOT be urlencoded.
   * the encoding is handled in OAuthRequest when the final
   * request is serialized
   * @param OAuthRequest $request
   * @param OAuthConsumer $consumer
   * @param OAuthToken $token
   * @return string
   */
  abstract public function build_signature($request, $consumer, $token);

  /**
   * Verifies that a given signature is correct
   * @param OAuthRequest $request
   * @param OAuthConsumer $consumer
   * @param OAuthToken $token
   * @param string $signature
   * @return bool
   */
  public function check_signature($request, $consumer, $token, $signature) {
    $built = $this->build_signature($request, $consumer, $token);
    return $built == $signature;
  }
}

/**
 * The HMAC-SHA1 signature method uses the HMAC-SHA1 signature algorithm as defined in [RFC2104] 
 * where the Signature Base String is the text and the key is the concatenated values (each first 
 * encoded per Parameter Encoding) of the Consumer Secret and Token Secret, separated by an '&' 
 * character (ASCII code 38) even if empty.
 *   - Chapter 9.2 ("HMAC-SHA1")
 */
class OAuthSignatureMethod_HMAC_SHA1 extends OAuthSignatureMethod {
  function get_name() {
    return "HMAC-SHA1";
  }

  public function build_signature($request, $consumer, $token) {
    $base_string = $request->get_signature_base_string();
    $request->base_string = $base_string;

    $key_parts = array(
      $consumer->secret,
      ($token) ? $token->secret : ""
    );

    $key_parts = OAuthUtil::urlencode_rfc3986($key_parts);
    $key = implode('&', $key_parts);

    return base64_encode(hash_hmac('sha1', $base_string, $key, true));
  }
}

/**
 * The PLAINTEXT method does not provide any security protection and SHOULD only be used 
 * over a secure channel such as HTTPS. It does not use the Signature Base String.
 *   - Chapter 9.4 ("PLAINTEXT")
 */
class OAuthSignatureMethod_PLAINTEXT extends OAuthSignatureMethod {
  public function get_name() {
    return "PLAINTEXT";
  }

  /**
   * oauth_signature is set to the concatenated encoded values of the Consumer Secret and 
   * Token Secret, separated by a '&' character (ASCII code 38), even if either secret is 
   * empty. The result MUST be encoded again.
   *   - Chapter 9.4.1 ("Generating Signatures")
   *
   * Please note that the second encoding MUST NOT happen in the SignatureMethod, as
   * OAuthRequest handles this!
   */
  public function build_signature($request, $consumer, $token) {
    $key_parts = array(
      $consumer->secret,
      ($token) ? $token->secret : ""
    );

    $key_parts = OAuthUtil::urlencode_rfc3986($key_parts);
    $key = implode('&', $key_parts);
    $request->base_string = $key;

    return $key;
  }
}

/**
 * The RSA-SHA1 signature method uses the RSASSA-PKCS1-v1_5 signature algorithm as defined in 
 * [RFC3447] section 8.2 (more simply known as PKCS#1), using SHA-1 as the hash function for 
 * EMSA-PKCS1-v1_5. It is assumed that the Consumer has provided its RSA public key in a 
 * verified way to the Service Provider, in a manner which is beyond the scope of this 
 * specification.
 *   - Chapter 9.3 ("RSA-SHA1")
 */
abstract class OAuthSignatureMethod_RSA_SHA1 extends OAuthSignatureMethod {
  public function get_name() {
    return "RSA-SHA1";
  }

  // Up to the SP to implement this lookup of keys. Possible ideas are:
  // (1) do a lookup in a table of trusted certs keyed off of consumer
  // (2) fetch via http using a url provided by the requester
  // (3) some sort of specific discovery code based on request
  //
  // Either way should return a string representation of the certificate
  protected abstract function fetch_public_cert(&$request);

  // Up to the SP to implement this lookup of keys. Possible ideas are:
  // (1) do a lookup in a table of trusted certs keyed off of consumer
  //
  // Either way should return a string representation of the certificate
  protected abstract function fetch_private_cert(&$request);

  public function build_signature($request, $consumer, $token) {
    $base_string = $request->get_signature_base_string();
    $request->base_string = $base_string;

    // Fetch the private key cert based on the request
    $cert = $this->fetch_private_cert($request);

    // Pull the private key ID from the certificate
    $privatekeyid = openssl_get_privatekey($cert);

    // Sign using the key
    $ok = openssl_sign($base_string, $signature, $privatekeyid);

    // Release the key resource
    openssl_free_key($privatekeyid);

    return base64_encode($signature);
  }

  public function check_signature($request, $consumer, $token, $signature) {
    $decoded_sig = base64_decode($signature);

    $base_string = $request->get_signature_base_string();

    // Fetch the public key cert based on the request
    $cert = $this->fetch_public_cert($request);

    // Pull the public key ID from the certificate
    $publickeyid = openssl_get_publickey($cert);

    // Check the computed signature against the one passed in the query
    $ok = openssl_verify($base_string, $decoded_sig, $publickeyid);

    // Release the key resource
    openssl_free_key($publickeyid);

    return $ok == 1;
  }
}

class OAuthRequest {
  private $parameters;
  private $http_method;
  private $http_url;
  // for debug purposes
  public $base_string;
  public static $version = '1.0';
  public static $POST_INPUT = 'php://input';

  function __construct($http_method, $http_url, $parameters=NULL) {
    @$parameters or $parameters = array();
    $parameters = array_merge( OAuthUtil::parse_parameters(parse_url($http_url, PHP_URL_QUERY)), $parameters);
    $this->parameters = $parameters;
    $this->http_method = $http_method;
    $this->http_url = $http_url;
  }


  /**
   * attempt to build up a request from what was passed to the server
   */
  public static function from_request($http_method=NULL, $http_url=NULL, $parameters=NULL) {
    $scheme = (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] != "on")
              ? 'http'
              : 'https';
    @$http_url or $http_url = $scheme .
                              '://' . $_SERVER['HTTP_HOST'] .
                              ':' .
                              $_SERVER['SERVER_PORT'] .
                              $_SERVER['REQUEST_URI'];
    @$http_method or $http_method = $_SERVER['REQUEST_METHOD'];

    // We weren't handed any parameters, so let's find the ones relevant to
    // this request.
    // If you run XML-RPC or similar you should use this to provide your own
    // parsed parameter-list
    if (!$parameters) {
      // Find request headers
      $request_headers = OAuthUtil::get_headers();

      // Parse the query-string to find GET parameters
      $parameters = OAuthUtil::parse_parameters($_SERVER['QUERY_STRING']);

      // It's a POST request of the proper content-type, so parse POST
      // parameters and add those overriding any duplicates from GET
      if ($http_method == "POST"
          && @strstr($request_headers["Content-Type"],
                     "application/x-www-form-urlencoded")
          ) {
        $post_data = OAuthUtil::parse_parameters(
          file_get_contents(self::$POST_INPUT)
        );
        $parameters = array_merge($parameters, $post_data);
      }

      // We have a Authorization-header with OAuth data. Parse the header
      // and add those overriding any duplicates from GET or POST
      if (@substr($request_headers['Authorization'], 0, 6) == "OAuth ") {
        $header_parameters = OAuthUtil::split_header(
          $request_headers['Authorization']
        );
        $parameters = array_merge($parameters, $header_parameters);
      }

    }

    return new OAuthRequest($http_method, $http_url, $parameters);
  }

  /**
   * pretty much a helper function to set up the request
   */
  public static function from_consumer_and_token($consumer, $token, $http_method, $http_url, $parameters=NULL) {
    @$parameters or $parameters = array();
    $defaults = array("oauth_version" => OAuthRequest::$version,
                      "oauth_nonce" => OAuthRequest::generate_nonce(),
                      "oauth_timestamp" => OAuthRequest::generate_timestamp(),
                      "oauth_consumer_key" => $consumer->key);
    if ($token)
      $defaults['oauth_token'] = $token->key;

    $parameters = array_merge($defaults, $parameters);

    return new OAuthRequest($http_method, $http_url, $parameters);
  }

  public function set_parameter($name, $value, $allow_duplicates = true) {
    if ($allow_duplicates && isset($this->parameters[$name])) {
      // We have already added parameter(s) with this name, so add to the list
      if (is_scalar($this->parameters[$name])) {
        // This is the first duplicate, so transform scalar (string)
        // into an array so we can add the duplicates
        $this->parameters[$name] = array($this->parameters[$name]);
      }

      $this->parameters[$name][] = $value;
    } else {
      $this->parameters[$name] = $value;
    }
  }

  public function get_parameter($name) {
    return isset($this->parameters[$name]) ? $this->parameters[$name] : null;
  }

  public function get_parameters() {
    return $this->parameters;
  }

  public function unset_parameter($name) {
    unset($this->parameters[$name]);
  }

  /**
   * The request parameters, sorted and concatenated into a normalized string.
   * @return string
   */
  public function get_signable_parameters() {
    // Grab all parameters
    $params = $this->parameters;

    // Remove oauth_signature if present
    // Ref: Spec: 9.1.1 ("The oauth_signature parameter MUST be excluded.")
    if (isset($params['oauth_signature'])) {
      unset($params['oauth_signature']);
    }

    return OAuthUtil::build_http_query($params);
  }

  /**
   * Returns the base string of this request
   *
   * The base string defined as the method, the url
   * and the parameters (normalized), each urlencoded
   * and the concated with &.
   */
  public function get_signature_base_string() {
    $parts = array(
      $this->get_normalized_http_method(),
      $this->get_normalized_http_url(),
      $this->get_signable_parameters()
    );

    $parts = OAuthUtil::urlencode_rfc3986($parts);

    return implode('&', $parts);
  }

  /**
   * just uppercases the http method
   */
  public function get_normalized_http_method() {
    return strtoupper($this->http_method);
  }

  /**
   * parses the url and rebuilds it to be
   * scheme://host/path
   */
  public function get_normalized_http_url() {
    $parts = parse_url($this->http_url);

    $port = @$parts['port'];
    $scheme = $parts['scheme'];
    $host = $parts['host'];
    $path = @$parts['path'];

    $port or $port = ($scheme == 'https') ? '443' : '80';

    if (($scheme == 'https' && $port != '443')
        || ($scheme == 'http' && $port != '80')) {
      $host = "$host:$port";
    }
    return "$scheme://$host$path";
  }

  /**
   * builds a url usable for a GET request
   */
  public function to_url() {
    $post_data = $this->to_postdata();
    $out = $this->get_normalized_http_url();
    if ($post_data) {
      $out .= '?'.$post_data;
    }
    return $out;
  }

  /**
   * builds the data one would send in a POST request
   */
  public function to_postdata() {
    return OAuthUtil::build_http_query($this->parameters);
  }

  /**
   * builds the Authorization: header
   */
  public function to_header($realm=null) {
    $first = true;
	if($realm) {
      $out = 'Authorization: OAuth realm="' . OAuthUtil::urlencode_rfc3986($realm) . '"';
      $first = false;
    } else
      $out = 'Authorization: OAuth';

    $total = array();
    foreach ($this->parameters as $k => $v) {
      if (substr($k, 0, 5) != "oauth") continue;
      if (is_array($v)) {
        throw new OAuthException('Arrays not supported in headers');
      }
      $out .= ($first) ? ' ' : ',';
      $out .= OAuthUtil::urlencode_rfc3986($k) .
              '="' .
              OAuthUtil::urlencode_rfc3986($v) .
              '"';
      $first = false;
    }
    return $out;
  }

  public function __toString() {
    return $this->to_url();
  }


  public function sign_request($signature_method, $consumer, $token) {
    $this->set_parameter(
      "oauth_signature_method",
      $signature_method->get_name(),
      false
    );
    $signature = $this->build_signature($signature_method, $consumer, $token);
    $this->set_parameter("oauth_signature", $signature, false);
  }

  public function build_signature($signature_method, $consumer, $token) {
    $signature = $signature_method->build_signature($this, $consumer, $token);
    return $signature;
  }

  /**
   * util function: current timestamp
   */
  private static function generate_timestamp() {
    return time();
  }

  /**
   * util function: current nonce
   */
  private static function generate_nonce() {
    $mt = microtime();
    $rand = mt_rand();

    return md5($mt . $rand); // md5s look nicer than numbers
  }
}

class OAuthServer {
  protected $timestamp_threshold = 300; // in seconds, five minutes
  protected $version = '1.0';             // hi blaine
  protected $signature_methods = array();

  protected $data_store;

  function __construct($data_store) {
    $this->data_store = $data_store;
  }

  public function add_signature_method($signature_method) {
    $this->signature_methods[$signature_method->get_name()] =
      $signature_method;
  }

  // high level functions

  /**
   * process a request_token request
   * returns the request token on success
   */
  public function fetch_request_token(&$request) {
    $this->get_version($request);

    $consumer = $this->get_consumer($request);

    // no token required for the initial token request
    $token = NULL;

    $this->check_signature($request, $consumer, $token);

    // Rev A change
    $callback = $request->get_parameter('oauth_callback');
    $new_token = $this->data_store->new_request_token($consumer, $callback);

    return $new_token;
  }

  /**
   * process an access_token request
   * returns the access token on success
   */
  public function fetch_access_token(&$request) {
    $this->get_version($request);

    $consumer = $this->get_consumer($request);

    // requires authorized request token
    $token = $this->get_token($request, $consumer, "request");

    $this->check_signature($request, $consumer, $token);

    // Rev A change
    $verifier = $request->get_parameter('oauth_verifier');
    $new_token = $this->data_store->new_access_token($token, $consumer, $verifier);

    return $new_token;
  }

  /**
   * verify an api call, checks all the parameters
   */
  public function verify_request(&$request) {
    $this->get_version($request);
    $consumer = $this->get_consumer($request);
    $token = $this->get_token($request, $consumer, "access");
    $this->check_signature($request, $consumer, $token);
    return array($consumer, $token);
  }

  // Internals from here
  /**
   * version 1
   */
  private function get_version(&$request) {
    $version = $request->get_parameter("oauth_version");
    if (!$version) {
      // Service Providers MUST assume the protocol version to be 1.0 if this parameter is not present. 
      // Chapter 7.0 ("Accessing Protected Ressources")
      $version = '1.0';
    }
    if ($version !== $this->version) {
      throw new OAuthException("OAuth version '$version' not supported");
    }
    return $version;
  }

  /**
   * figure out the signature with some defaults
   */
  private function get_signature_method(&$request) {
    $signature_method =
        @$request->get_parameter("oauth_signature_method");

    if (!$signature_method) {
      // According to chapter 7 ("Accessing Protected Ressources") the signature-method
      // parameter is required, and we can't just fallback to PLAINTEXT
      throw new OAuthException('No signature method parameter. This parameter is required');
    }

    if (!in_array($signature_method,
                  array_keys($this->signature_methods))) {
      throw new OAuthException(
        "Signature method '$signature_method' not supported " .
        "try one of the following: " .
        implode(", ", array_keys($this->signature_methods))
      );
    }
    return $this->signature_methods[$signature_method];
  }

  /**
   * try to find the consumer for the provided request's consumer key
   */
  private function get_consumer(&$request) {
    $consumer_key = @$request->get_parameter("oauth_consumer_key");
    if (!$consumer_key) {
      throw new OAuthException("Invalid consumer key");
    }

    $consumer = $this->data_store->lookup_consumer($consumer_key);
    if (!$consumer) {
      throw new OAuthException("Invalid consumer");
    }

    return $consumer;
  }

  /**
   * try to find the token for the provided request's token key
   */
  private function get_token(&$request, $consumer, $token_type="access") {
    $token_field = @$request->get_parameter('oauth_token');
    $token = $this->data_store->lookup_token(
      $consumer, $token_type, $token_field
    );
    if (!$token) {
      throw new OAuthException("Invalid $token_type token: $token_field");
    }
    return $token;
  }

  /**
   * all-in-one function to check the signature on a request
   * should guess the signature method appropriately
   */
  private function check_signature(&$request, $consumer, $token) {
    // this should probably be in a different method
    $timestamp = @$request->get_parameter('oauth_timestamp');
    $nonce = @$request->get_parameter('oauth_nonce');

    $this->check_timestamp($timestamp);
    $this->check_nonce($consumer, $token, $nonce, $timestamp);

    $signature_method = $this->get_signature_method($request);

    $signature = $request->get_parameter('oauth_signature');
    $valid_sig = $signature_method->check_signature(
      $request,
      $consumer,
      $token,
      $signature
    );

    if (!$valid_sig) {
      throw new OAuthException("Invalid signature");
    }
  }

  /**
   * check that the timestamp is new enough
   */
  private function check_timestamp($timestamp) {
    if( ! $timestamp )
      throw new OAuthException(
        'Missing timestamp parameter. The parameter is required'
      );
    
    // verify that timestamp is recentish
    $now = time();
    if (abs($now - $timestamp) > $this->timestamp_threshold) {
      throw new OAuthException(
        "Expired timestamp, yours $timestamp, ours $now"
      );
    }
  }

  /**
   * check that the nonce is not repeated
   */
  private function check_nonce($consumer, $token, $nonce, $timestamp) {
    if( ! $nonce )
      throw new OAuthException(
        'Missing nonce parameter. The parameter is required'
      );

    // verify that the nonce is uniqueish
    $found = $this->data_store->lookup_nonce(
      $consumer,
      $token,
      $nonce,
      $timestamp
    );
    if ($found) {
      throw new OAuthException("Nonce already used: $nonce");
    }
  }

}

class OAuthDataStore {
  function lookup_consumer($consumer_key) {
    // implement me
  }

  function lookup_token($consumer, $token_type, $token) {
    // implement me
  }

  function lookup_nonce($consumer, $token, $nonce, $timestamp) {
    // implement me
  }

  function new_request_token($consumer, $callback = null) {
    // return a new token attached to this consumer
  }

  function new_access_token($token, $consumer, $verifier = null) {
    // return a new access token attached to this consumer
    // for the user associated with this token if the request token
    // is authorized
    // should also invalidate the request token
  }

}

class OAuthUtil {
  public static function urlencode_rfc3986($input) {
  if (is_array($input)) {
    return array_map(array('OAuthUtil', 'urlencode_rfc3986'), $input);
  } else if (is_scalar($input)) {
    return str_replace(
      '+',
      ' ',
      str_replace('%7E', '~', rawurlencode($input))
    );
  } else {
    return '';
  }
}


  // This decode function isn't taking into consideration the above
  // modifications to the encoding process. However, this method doesn't
  // seem to be used anywhere so leaving it as is.
  public static function urldecode_rfc3986($string) {
    return urldecode($string);
  }

  // Utility function for turning the Authorization: header into
  // parameters, has to do some unescaping
  // Can filter out any non-oauth parameters if needed (default behaviour)
  public static function split_header($header, $only_allow_oauth_parameters = true) {
    $pattern = '/(([-_a-z]*)=("([^"]*)"|([^,]*)),?)/';
    $offset = 0;
    $params = array();
    while (preg_match($pattern, $header, $matches, PREG_OFFSET_CAPTURE, $offset) > 0) {
      $match = $matches[0];
      $header_name = $matches[2][0];
      $header_content = (isset($matches[5])) ? $matches[5][0] : $matches[4][0];
      if (preg_match('/^oauth_/', $header_name) || !$only_allow_oauth_parameters) {
        $params[$header_name] = OAuthUtil::urldecode_rfc3986($header_content);
      }
      $offset = $match[1] + strlen($match[0]);
    }

    if (isset($params['realm'])) {
      unset($params['realm']);
    }

    return $params;
  }

  // helper to try to sort out headers for people who aren't running apache
  public static function get_headers() {
    if (function_exists('apache_request_headers')) {
      // we need this to get the actual Authorization: header
      // because apache tends to tell us it doesn't exist
      $headers = apache_request_headers();

      // sanitize the output of apache_request_headers because
      // we always want the keys to be Cased-Like-This and arh()
      // returns the headers in the same case as they are in the
      // request
      $out = array();
      foreach( $headers AS $key => $value ) {
        $key = str_replace(
            " ",
            "-",
            ucwords(strtolower(str_replace("-", " ", $key)))
          );
        $out[$key] = $value;
      }
    } else {
      // otherwise we don't have apache and are just going to have to hope
      // that $_SERVER actually contains what we need
      $out = array();
      if( isset($_SERVER['CONTENT_TYPE']) )
        $out['Content-Type'] = $_SERVER['CONTENT_TYPE'];
      if( isset($_ENV['CONTENT_TYPE']) )
        $out['Content-Type'] = $_ENV['CONTENT_TYPE'];

      foreach ($_SERVER as $key => $value) {
        if (substr($key, 0, 5) == "HTTP_") {
          // this is chaos, basically it is just there to capitalize the first
          // letter of every word that is not an initial HTTP and strip HTTP
          // code from przemek
          $key = str_replace(
            " ",
            "-",
            ucwords(strtolower(str_replace("_", " ", substr($key, 5))))
          );
          $out[$key] = $value;
        }
      }
    }
    return $out;
  }

  // This function takes a input like a=b&a=c&d=e and returns the parsed
  // parameters like this
  // array('a' => array('b','c'), 'd' => 'e')
  public static function parse_parameters( $input ) {
    if (!isset($input) || !$input) return array();

    $pairs = explode('&', $input);

    $parsed_parameters = array();
    foreach ($pairs as $pair) {
      $split = explode('=', $pair, 2);
      $parameter = OAuthUtil::urldecode_rfc3986($split[0]);
      $value = isset($split[1]) ? OAuthUtil::urldecode_rfc3986($split[1]) : '';

      if (isset($parsed_parameters[$parameter])) {
        // We have already recieved parameter(s) with this name, so add to the list
        // of parameters with this name

        if (is_scalar($parsed_parameters[$parameter])) {
          // This is the first duplicate, so transform scalar (string) into an array
          // so we can add the duplicates
          $parsed_parameters[$parameter] = array($parsed_parameters[$parameter]);
        }

        $parsed_parameters[$parameter][] = $value;
      } else {
        $parsed_parameters[$parameter] = $value;
      }
    }
    return $parsed_parameters;
  }

  public static function build_http_query($params) {
    if (!$params) return '';

    // Urlencode both keys and values
    $keys = OAuthUtil::urlencode_rfc3986(array_keys($params));
    $values = OAuthUtil::urlencode_rfc3986(array_values($params));
    $params = array_combine($keys, $values);

    // Parameters are sorted by name, using lexicographical byte value ordering.
    // Ref: Spec: 9.1.1 (1)
    uksort($params, 'strcmp');

    $pairs = array();
    foreach ($params as $parameter => $value) {
      if (is_array($value)) {
        // If two or more parameters share the same name, they are sorted by their value
        // Ref: Spec: 9.1.1 (1)
        natsort($value);
        foreach ($value as $duplicate_value) {
          $pairs[] = $parameter . '=' . $duplicate_value;
        }
      } else {
        $pairs[] = $parameter . '=' . $value;
      }
    }
    // For each parameter, the name is separated from the corresponding value by an '=' character (ASCII code 61)
    // Each name-value pair is separated by an '&' character (ASCII code 38)
    return implode('&', $pairs);
  }
}



/**
 * Twitter OAuth class
 */
class TwitterOAuth {
  /* Contains the last HTTP status code returned. */
  public $http_code;
  /* Contains the last API call. */
  public $url;
  /* Set up the API root URL. */
  public $host = "https://api.twitter.com/1/";
  /* Set timeout default. */
  public $timeout = 30;
  /* Set connect timeout. */
  public $connecttimeout = 30; 
  /* Verify SSL Cert. */
  public $ssl_verifypeer = FALSE;
  /* Respons format. */
  public $format = 'json';
  /* Decode returned json data. */
  public $decode_json = TRUE;
  /* Contains the last HTTP headers returned. */
  public $http_info;
  /* Set the useragnet. */
  public $useragent = 'TwitterOAuth v0.2.0-beta2';
  /* Immediately retry the API call if the response was not successful. */
  //public $retry = TRUE;




  /**
   * Set API URLS
   */
  function accessTokenURL()  { return 'https://api.twitter.com/oauth/access_token'; }
  function authenticateURL() { return 'https://twitter.com/oauth/authenticate'; }
  function authorizeURL()    { return 'https://twitter.com/oauth/authorize'; }
  function requestTokenURL() { return 'https://api.twitter.com/oauth/request_token'; }

  /**
   * Debug helpers
   */
  function lastStatusCode() { return $this->http_status; }
  function lastAPICall() { return $this->last_api_call; }

  /**
   * construct TwitterOAuth object
   */
  function __construct($consumer_key, $consumer_secret, $oauth_token = NULL, $oauth_token_secret = NULL) {
    $this->sha1_method = new OAuthSignatureMethod_HMAC_SHA1();
    $this->consumer = new OAuthConsumer($consumer_key, $consumer_secret);
    if (!empty($oauth_token) && !empty($oauth_token_secret)) {
      $this->token = new OAuthConsumer($oauth_token, $oauth_token_secret);
    } else {
      $this->token = NULL;
    }
  }


  /**
   * Get a request_token from Twitter
   *
   * @returns a key/value array containing oauth_token and oauth_token_secret
   */
  function getRequestToken($oauth_callback = NULL) {
    $parameters = array();
    if (!empty($oauth_callback)) {
      $parameters['oauth_callback'] = $oauth_callback;
    } 
    $request = $this->oAuthRequest($this->requestTokenURL(), 'GET', $parameters);
    $token = OAuthUtil::parse_parameters($request);
    $this->token = new OAuthConsumer($token['oauth_token'], $token['oauth_token_secret']);
    return $token;
  }

  /**
   * Get the authorize URL
   *
   * @returns a string
   */
  function getAuthorizeURL($token, $sign_in_with_twitter = TRUE) {
    if (is_array($token)) {
      $token = $token['oauth_token'];
    }
    if (empty($sign_in_with_twitter)) {
      return $this->authorizeURL() . "?oauth_token={$token}";
    } else {
       return $this->authenticateURL() . "?oauth_token={$token}";
    }
  }

  /**
   * Exchange request token and secret for an access token and
   * secret, to sign API calls.
   *
   * @returns array("oauth_token" => "the-access-token",
   *                "oauth_token_secret" => "the-access-secret",
   *                "user_id" => "9436992",
   *                "screen_name" => "abraham")
   */
  function getAccessToken($oauth_verifier = FALSE) {
    $parameters = array();
    if (!empty($oauth_verifier)) {
      $parameters['oauth_verifier'] = $oauth_verifier;
    }
    $request = $this->oAuthRequest($this->accessTokenURL(), 'GET', $parameters);
    $token = OAuthUtil::parse_parameters($request);
    $this->token = new OAuthConsumer($token['oauth_token'], $token['oauth_token_secret']);
    return $token;
  }

  /**
   * One time exchange of username and password for access token and secret.
   *
   * @returns array("oauth_token" => "the-access-token",
   *                "oauth_token_secret" => "the-access-secret",
   *                "user_id" => "9436992",
   *                "screen_name" => "abraham",
   *                "x_auth_expires" => "0")
   */  
  function getXAuthToken($username, $password) {
    $parameters = array();
    $parameters['x_auth_username'] = $username;
    $parameters['x_auth_password'] = $password;
    $parameters['x_auth_mode'] = 'client_auth';
    $request = $this->oAuthRequest($this->accessTokenURL(), 'POST', $parameters);
    $token = OAuthUtil::parse_parameters($request);
    $this->token = new OAuthConsumer($token['oauth_token'], $token['oauth_token_secret']);
    return $token;
  }

  /**
   * GET wrapper for oAuthRequest.
   */
  function get($url, $parameters = array()) {
    $response = $this->oAuthRequest($url, 'GET', $parameters);
    if ($this->format === 'json' && $this->decode_json) {
      return json_decode($response);
    }
    return $response;
  }
  
  /**
   * POST wrapper for oAuthRequest.
   */
  function post($url, $parameters = array()) {
    $response = $this->oAuthRequest($url, 'POST', $parameters);
    if ($this->format === 'json' && $this->decode_json) {
      return json_decode($response);
    }
    return $response;
  }

  /**
   * DELETE wrapper for oAuthReqeust.
   */
  function delete($url, $parameters = array()) {
    $response = $this->oAuthRequest($url, 'DELETE', $parameters);
    if ($this->format === 'json' && $this->decode_json) {
      return json_decode($response);
    }
    return $response;
  }

  /**
   * Format and sign an OAuth / API request
   */
  function oAuthRequest($url, $method, $parameters) {
    if (strrpos($url, 'https://') !== 0 && strrpos($url, 'http://') !== 0) {
      $url = "{$this->host}{$url}.{$this->format}";
    }
    $request = OAuthRequest::from_consumer_and_token($this->consumer, $this->token, $method, $url, $parameters);
    $request->sign_request($this->sha1_method, $this->consumer, $this->token);
    switch ($method) {
    case 'GET':
      return $this->http($request->to_url(), 'GET');
    default:
      return $this->http($request->get_normalized_http_url(), $method, $request->to_postdata());
    }
  }

  /**
   * Make an HTTP request
   *
   * @return API results
   */
  function http($url, $method, $postfields = NULL) {
    $this->http_info = array();
    $ci = curl_init();
    /* Curl settings */
    curl_setopt($ci, CURLOPT_USERAGENT, $this->useragent);
    curl_setopt($ci, CURLOPT_CONNECTTIMEOUT, $this->connecttimeout);
    curl_setopt($ci, CURLOPT_TIMEOUT, $this->timeout);
    curl_setopt($ci, CURLOPT_RETURNTRANSFER, TRUE);
    curl_setopt($ci, CURLOPT_HTTPHEADER, array('Expect:'));
    curl_setopt($ci, CURLOPT_SSL_VERIFYPEER, $this->ssl_verifypeer);
    curl_setopt($ci, CURLOPT_HEADERFUNCTION, array($this, 'getHeader'));
    curl_setopt($ci, CURLOPT_HEADER, FALSE);

    switch ($method) {
      case 'POST':
        curl_setopt($ci, CURLOPT_POST, TRUE);
        if (!empty($postfields)) {
          curl_setopt($ci, CURLOPT_POSTFIELDS, $postfields);
        }
        break;
      case 'DELETE':
        curl_setopt($ci, CURLOPT_CUSTOMREQUEST, 'DELETE');
        if (!empty($postfields)) {
          $url = "{$url}?{$postfields}";
        }
    }

    curl_setopt($ci, CURLOPT_URL, $url);
    $response = curl_exec($ci);
    $this->http_code = curl_getinfo($ci, CURLINFO_HTTP_CODE);
    $this->http_info = array_merge($this->http_info, curl_getinfo($ci));
    $this->url = $url;
    curl_close ($ci);
    return $response;
  }

  /**
   * Get the header info to store.
   */
  function getHeader($ch, $header) {
    $i = strpos($header, ':');
    if (!empty($i)) {
      $key = str_replace('-', '_', strtolower(substr($header, 0, $i)));
      $value = trim(substr($header, $i + 2));
      $this->http_header[$key] = $value;
    }
    return strlen($header);
  }
}


//OAuth End

 $message = strip_tags(to_utf8($message));
 if(strlen($message)>140){
  $message = to_utf8(mb_substr($message,0,135).' ...');
 }

 $to = new TwitterOAuth(CONSUMER_KEY, CONSUMER_SECRET, ACCESS_TOKEN, ACCESS_TOKEN_SECRET);
 $req = $to->OAuthRequest("https://twitter.com/statuses/update.xml","POST",array("status"=>$message));

 $loggingdata = date('YmdHis')."\t".h(__FILE__)."\t".h(gethostbyaddr(h($_SERVER['REMOTE_ADDR'])))."(".h($_SERVER['REMOTE_ADDR']).")\t".h($_SERVER['HTTP_USER_AGENT'])."\t".h(@$_SERVER['HTTP_REFERER'])."\t".CONSUMER_KEY."\t".CONSUMER_SECRET."\t".ACCESS_TOKEN."\t".ACCESS_TOKEN_SECRET."\n";
 $loggingdata = h($loggingdata);
 $fplog = fopen(BLOG_LOGGING_DIR.'/loggingtw.cgi','a+b');
 flock($fplog,LOCK_EX);
 fwrite($fplog,$loggingdata);
 fclose($fplog);
}

function mixi($title, $message){
 $message = strip_tags(to_utf8($message));
 if(strlen($message)>10000){
  $message = to_utf8(mb_substr($message,0,9995).' ...');
 }
 //$nonce       = pack('H*', sha1(md5(time().rand().posix_getpid())));
 $nonce       = pack('H*', sha1(md5(time().rand().uniqid())));

 $created     = date('Y-m-d\TH:i:s\Z');
 $digest      = base64_encode(pack('H*', sha1($nonce . $created . MIXI_PW)));
 $wsse_text   = 'UsernameToken Username="%s", PasswordDigest="%s", Nonce="%s", Created="%s"';
 $wsse_header = sprintf($wsse_text, MIXI_LGIN_ID, $digest, base64_encode($nonce), $created);
 $url         = 'http://mixi.jp/atom/diary/member_id=' . MIXI_USER_ID;
 $post_data
        = "<?xml version='1.0' encoding='utf-8'?>"
        . "<entry xmlns='http://www.w3.org/2007/app'>"
        . "<title>".to_utf8($title)."</title>"
        . "<summary>".to_utf8($message)."</summary>"
        . "</entry>";

$request = new HTTP_Request($location);
$request->setMethod(HTTP_REQUEST_METHOD_POST);
$request->addHeader('X-WSSE', $wsse_header);
$request->setBody($post_data);
$request->sendRequest();

$res_code = $request->getResponseCode();
echo "$res_code\n";
 $loggingdata = date('YmdHis')."\t".h(__FILE__)."\t".h(gethostbyaddr(h($_SERVER['REMOTE_ADDR'])))."(".h($_SERVER['REMOTE_ADDR']).")\t".h($_SERVER['HTTP_USER_AGENT'])."\t".h(@$_SERVER['HTTP_REFERER'])."\t".MIXI_LGIN_ID."\t".MIXI_USER_ID."\t".MIXI_PW."\n";
 $loggingdata = h($loggingdata);
 $fplog = fopen(BLOG_LOGGING_DIR.'/loggingmx.cgi','a+b');
 flock($fplog,LOCK_EX);
 fwrite($fplog,$loggingdata);
 fclose($fplog);
}

function load_comment_data($filename) {
 if (!file_exists(COMMENT_DATA_DIR.'/'.$filename)) {return array();}
 $fp = fopen(COMMENT_DATA_DIR.'/'.$filename, 'r');
 if (!$fp) {die(LIB_COMMENTLIB_CANNOTOPEN);};
 if (!flock($fp, LOCK_SH)) {die(LIB_COMMENTLIB_CANNOTLOCK);}

 $comments = array();
 while (!feof($fp)) {
  $line = fgets($fp);
  $line = rtrim($line);
  if ($line == '') {continue;}
  list($date, $deleted, $name, $url, $comment) = explode("\t", $line, 5);

  $comments[] = array(
   'date' => $date,
   'deleted' => $deleted,
   'name' => $name,
   'url' => $url,
   'comment' => $comment,
  );
 }

  fclose($fp);
  return $comments;
}

function import_comment_data() {
 $fields = array('name', 'url', 'comment', 'filename');
 $comment_data = array();
 foreach ($fields as $field) {
  if (isset($_POST[$field])) {
   $comment_data[$field] = rtrim($_POST[$field]);
  } else {
   $comment_data[$field] = NULL;
  }
 }
 $comment_data['comment'] = str_replace("\n", '<br>', htmlspecialchars($comment_data['comment']));
 foreach (array('name', 'url', 'comment', 'filename') as $field) {
  $comment_data[$field] = preg_replace('/[\n\r\t\0]/', ' ', $comment_data[$field]);
 }

 $entry_data = load_entry_data($comment_data['filename']);

 return $comment_data;
}

function is_valid_comment_data($comment_data) {
 $result = array();

 $fields = array(
  'name' => LIB_COMMENTLIB_ISVALID_NAME,
  'url' => LIB_COMMENTLIB_ISVALID_URI,
  'comment' => LIB_COMMENTLIB_ISVALID_COMMENT,
  'filename' => LIB_COMMENTLIB_ISVALID_FILENAME,
 );

 $requires = array('name', 'comment', 'filename');
 foreach ($requires as $field) {
  if ($comment_data[$field] == '') {
   $result[] = LIB_COMMENTLIB_BLANK1.$fields[$field].LIB_COMMENTLIB_BLANK2;
  }
 }
 if (count($result) > 0) {return $result;}

 $lengths = array(
  'name' => 40,
  'url' => 100,
  'comment' => 4000,
 );
 foreach ($lengths as $field => $size) {
  if (strlen($comment_data[$field]) > $size) {
   $result[] = LIB_COMMENTLIB_TOOLONG1.$fields[$field].LIB_COMMENTLIB_TOOLONG2;
  }
 }

 if (!preg_match('/^[0-9A-Za-z\-_]+\.txt$/', $comment_data['filename'])) {
  $result[] = LIB_COMMENTLIB_BADCHAR1.$fields['filename'].LIB_COMMENTLIB_BADCHAR2;
 }

 if (!preg_match('/.+@.+/', $comment_data['url'])) {
  $result[] = LIB_COMMENTLIB_BADURICHAR1.$fields['url'].LIB_COMMENTLIB_BADURICHAR2;
 }

 if (!file_exists(ENTRY_DATA_DIR.'/'.$comment_data['filename'])) {
  $result[] = LIB_COMMENTLIB_NOFILE;
 }

 if (count($result) == 0) {return NULL;}

 return $result;
}

function add_comment_data($comment_data) {
 $filename = $comment_data['filename'];
 $fp = fopen(COMMENT_DATA_DIR.'/'.$filename, 'a');
 if (!$fp) {die(LIB_COMMENTLIB_CANNOTOPEN);}
 if (!flock($fp, LOCK_EX)) {die(LIB_COMMENTLIB_CANNOTLOCK);}

 $now = time();
 $deleted = 0;
 $line = mb_convert_encoding($now."\t".$deleted."\t".$comment_data['name']."\t".$comment_data['url']."\t".$comment_data['comment']."\n", "UTF-8", "auto");

 fwrite($fp, $line);
 fclose($fp);
}

function e($str) {
 echo $str;
}

function h($str) {
 return htmlspecialchars($str);
}

function u($str) {
 return urlencode($str);
}

function eh($str) {
 e(h($str));
}

function eu($str) {
 e(u($str));
}

function text_to_html($text) {
 $result = htmlspecialchars($text);
 $result = str_replace("\n", '<br>'."\n", $result);
 $result = preg_replace('/http:\/\/[0-9a-z_,.:;&=+*%$#!?@()~\'\/-]+/i', '<a href="$0" class="autolinked">$0</a>', $result);
 $result = preg_replace('/\s+([a-z0-9_\-.]+@([a-z0-9_\-]+\.)+[a-z]+)\s+/i', '<a href="mailto:$0" class="autolinked">$0</a>', $result);

 return $result;
}

function show_template($_filename, $_data = array()) {
 extract($_data);
 include TEMPLATE_DIR.'/'.$_filename;
}

function capture_template($_filename, $_data = array()) {
 ob_start();
 show_template($_filename, $_data);
 $result = ob_get_contents();
 ob_end_clean();
 return $result;
}

function unmagicquote() {
 if (get_magic_quotes_gpc()) {
  foreach (array('_GET', '_POST', '_COOKIE') as $varname) {
   foreach ($GLOBALS[$varname] as $key => $value) {
    if ( preg_match('/\0/',$value) ) { die( MSG_ILLEGAL_ACCESS ); }
    $GLOBALS[$varname][$key] = stripslashes($value);
   }
  }
 } else {
  foreach (array('_GET', '_COOKIE') as $varname) {
   foreach ($GLOBALS[$varname] as $key => $value) {
    if ( preg_match('/\0/',$value) ) { die( MSG_ILLEGAL_ACCESS ); }
   }
  }
 }
}

function to_euc($str, $from = 'auto') {
 if (is_array($str)) {
  $result = array();
  foreach ($str as $key => $value) {
   $result[$key] = to_euc($value, $from);
  }
  return $result;
 } else {
  return mb_convert_encoding($str, 'euc-jp', $from);
 }
}

function to_utf8($str, $from = 'auto') {
 if (is_array($str)) {
  $result = array();
  foreach ($str as $key => $value) {
   $result[$key] = to_utf8($value, $from);
  }
  return $result;
 } else {
  return mb_convert_encoding($str, 'utf8', $from);
 }
}

function fileindex($dir = BLOG_ROOT_DIR) {
 $filelist = scandir($dir);
 if ($filelist[0] === '.') array_shift($filelist);
 if ($filelist[0] === '..') array_shift($filelist);
 return $filelist;
}

function status2_fileindex($dir) {
 $dirs = fileindex($dir);
 foreach ($dirs as $key => $value) {
  $fileexsit = file_exists($dir.'/'.$value) ? '1' : '0';
  $filesize  = number_format(filesize($dir.'/'.$value));
  $filemtime = date('y/n/j H:i:s', filemtime($dir.'/'.$value));
  $filer  = is_readable($dir.'/'.$value) ? '1' : '0';
  $filew  = is_writable($dir.'/'.$value) ? '1' : '0';
  if (mb_strpos($value, '.') === FALSE) {
   echo '<tr><td><b>'.$value.'</b></td><td><b>'.$filesize.'</b></td>';
   echo '<td><b>'.$filemtime.'</b></td><td><b>'.$filer.'</b></td>';
   echo '<td><b>'.$filew.'</b></td></tr>';
  } else {
   echo '<tr><td>'.$value.'</td><td>'.$filesize.'</td>';
   echo '<td>'.$filemtime.'</td><td>'.$filer.'</td><td>'.$filew.'</td></tr>';
  }
 }
}

function debuglog($type = 'ERROR', $msg = '', $di = '0') {
 $logdata = date('YmdHis')."\t".
  h($type)."\t".
  h(@__FILE__)."\t".
  h(@__FUNCTION__)."\t".
  h(@__CLASS__)."\t".
  h(@__METHOD__)."\t".
  h(@gethostbyaddr($_SERVER['REMOTE_ADDR']))."(".
  h(@$_SERVER['REMOTE_ADDR']).")\t".
  h(@$_SERVER['HTTP_USER_AGENT'])."\t".
  h(@$_SERVER['HTTP_REFERER']);
 if (!$msg == '') { $logdata = $logdata."\t".h($msg)."\t"; }
 $logdata = $logdata."\n";
 $logdata = h($logdata);
 $fplog = fopen(BLOG_LOGGING_DIR.'/logging_all_'.date('Ymd').'.cgi','a+b');
 flock($fplog,LOCK_EX);
 fwrite($fplog,$logdata);
 fclose($fplog);
 if ($di == '1') { die($msg); }
}

function t($str) {
    return mb_ereg_replace('^(\s|　)+|(\s|　)+$','', $str);
}

class EntryIndex
{
 var $_index;

 var $_sortkey;
 var $_sortorder;

 function EntryIndex() {
  $this->_index = array();
 }

 function &getIndex() {
  return $this->_index;
 }

 function sync($entry_dir = ENTRY_DATA_DIR) {
  $this->_index = array();

  $files = glob($entry_dir.'/*.txt');
  if (!$files) {
   return;
  }

  foreach ($files as $filename) {
   $fileinfo = pathinfo($filename);
   $filename = $fileinfo['basename'];

   $this->update($filename);
  }

  return $this->_index;
 }

 function save($indexfile = ENTRY_INDEX_FILE) {
  $fp = fopen($indexfile, 'w');
  if (!$fp) {die(LIB_ENTRYINDEX_CANNOTOPEN);}
  if (!flock($fp, LOCK_EX)) {die(LIB_ENTRYINDEX_CANNOTLOCK);}

  foreach ($this->_index as $filename => $entry_data) {
   $line = $filename."\t".$entry_data['date']."\t".$entry_data['title']."\n";
   fwrite($fp, $line);
  }

  fclose($fp);
 }

 function load($indexfile = ENTRY_INDEX_FILE) {
  $this->_index = array();

  if (file_exists($indexfile)) {
   if (filesize($indexfile) > 0) {
    $fp = fopen($indexfile, 'r');
   } else {
    $fp = fopen($indexfile, 'w+');
   }
  } else {
   $fp = fopen($indexfile, 'w+');
   echo '<a href="admin.php?cmd=makeindex">'.TMP_ADMINMENU_REINDEX.'</a>';
   exit();
  }

  if (!flock($fp, LOCK_SH)) {die(LIB_ENTRYINDEX_CANNOTLOCK);}
  while (!feof($fp)) {
   $line = fgets($fp);
   if (trim($line) == '') {continue;}

   list($filename, $date, $title) = explode("\t", $line, 3);

   $this->_index[$filename] = array(
    'date' => intval($date),
    'title' => rtrim($title),
   );
  }
  fclose($fp);

  return $this->_index;
 }

 function update($entryfile) {
  $entry_data = load_entry_data($entryfile);
  $this->_index[$entry_data['filename']] = array(
   'date' => $entry_data['date'],
   'title' => $entry_data['title'],
  );
  return $this->_index;
 }

 function remove($entryfile) {
  if (isset($this->_index[$entryfile])) {
   unset($this->_index[$entryfile]);
  } else {
   die(LIB_ENTRYINDEX_NOFILE);
  }
 }

 function select($key, $value) {
  $result = array();
  foreach ($this->_index as $filename => $entry_data) {
   switch ($key) {
   case 'date':
    $date = date('Ymd', $entry_data['date']);
    if (preg_match('/^'.preg_quote($value).'/', $date)) {
     $result[$filename] = $entry_data;
    }
    break;
   case 'title':
    $title = $entry_data['title'];
    if (preg_match('/'.preg_quote($value).'/', $title)) {
     $result[$filename] = $entry_data;
    }
    break;
   }
  }
  $this->_index = $result;
  return $this->_index;
 }

 function sort($key, $order = TRUE) {
  $this->_sortkey = $key;
  $this->_sortorder = $order;
  uksort($this->_index, array(&$this, '_sort_helper'));
  return $this->_index;
 }

 function _sort_helper($filenameA, $filenameB) {
  $valueA = $this->_index[$filenameA][$this->_sortkey];
  $valueB = $this->_index[$filenameB][$this->_sortkey];
  switch ($this->_sortkey) {
  case 'date':
   if ($valueA == $valueB) {return 0;}
   return ($valueA > $valueB ? 1 : -1) * ($this->_sortorder ? 1 : -1);
   break;
  case 'title':
   return strcmp($valueA, $valueB) * ($this->_sortorder ? 1 : -1);
   break;
  }
 }
}

function render_entry_body($entry_data) {
 switch ($entry_data['format']) {
  case 'HTML':
   return $entry_data['body'];
   break;
  case 'TEXT':
   return text_to_html($entry_data['body']);
   break;
  default:
   die(MSG_ILLEGAL_ACCESS);
   break;
 }
}

function load_entry_data($filename) {
 $fp = fopen(ENTRY_DATA_DIR . '/' . $filename, 'r');
 if (!$fp) {die(LIB_ENTRYLIB_CANNOTOPEN);}

 if (!flock($fp, LOCK_SH)) {die(LIB_ENTRYLIB_CANNOTLOCK);}

 $entry_data = array();

 while ($line = fgets($fp)) {
  $line = rtrim($line, "\r\n");
  if ($line == '') {break;}

  list($name, $value) = explode(':', $line, 2);
  $name = strtolower(trim($name));
  $value = trim($value);

  $entry_data[$name] = $value;
 }

 $entry_data['body'] = '';
 while ($line = fgets($fp)) {
  $entry_data['body'] .= $line;
 }

 fclose($fp);

 $entry_data['filename'] = $filename;

 return $entry_data;
}

function save_entry_data($entry_data) {
 $filename = ENTRY_DATA_DIR . '/' . $entry_data['filename'];

 $fp = fopen($filename, 'w');
 if (!$fp) {die(LIB_ENTRYLIB_CANNOTOPEN);}
 if (!flock($fp, LOCK_EX)) {die(LIB_ENTRYLIB_CANNOTLOCK);}

 $headers = array('title', 'date', 'format');
 foreach ($headers as $header) {
  fwrite($fp, @mb_convert_encoding($header, "UTF-8", "auto").': ' . @mb_convert_encoding($entry_data[$header], "UTF-8", "auto") . "\n");
 }

 fwrite($fp, "\n");

 fwrite($fp, @mb_convert_encoding($entry_data['body'], "UTF-8", "auto"));
 fclose($fp);

 $index =& new EntryIndex();
 $index->load();
 $index->update($entry_data['filename']);
 $index->save();
}

function delete_entry_data($entry_data) {
 unlink(ENTRY_DATA_DIR . '/' . $entry_data['filename']);
 $index =& new EntryIndex();
 $index->load();
 $index->remove($entry_data['filename']);
 $index->save();
}

function get_entry_list() {
 $index =& new EntryIndex();
 return $index->load();
}

function get_recent_entry_list($max = 10) {
 $index =& new EntryIndex();
 $index->load();
 $entry_list = get_entry_list();
 $entry_list = array();
 foreach (array_slice($index->sort('date', FALSE), 0, $max) as $filename => $entry_data) {
  $entry_list[] = load_entry_data($filename);
 }
 return $entry_list;
}

function _sort_recent_entry_list($entry_data1, $entry_data2) {
 if ($entry_data1['date'] == $entry_data2['date']) {return 0;}
 return ($entry_data1['date'] > $entry_data2['date']) ? -1 : 1;
}

function is_valid_entry_data($entry_data) {
 $result = array();

 $fields = array(
  'cmd' => LIB_ENTRYLIB_ISVALID_CMD,
  'filename' => LIB_ENTRYLIB_ISVALID_FILENAME,
  'title' => LIB_ENTRYLIB_ISVALID_TITLE,
  'date' => LIB_ENTRYLIB_ISVALID_DATE,
  'format' => LIB_ENTRYLIB_ISVALID_FORMAT,
  'body' => LIB_ENTRYLIB_ISVALID_BODY,
 );

 $requires = array('cmd', 'filename', 'title', 'date', 'format', 'body');
 foreach ($requires as $name) {
  if (!isset($entry_data[$name]) || $entry_data[$name] == '') {
   $result[] = LIB_ENTRYLIB_BLANK1.$fields[$name].LIB_ENTRYLIB_BLANK2;
  }
 }
 if (count($result) > 0) {return $result;}

 $lengths = array(
  'filename' => 50,
  'title' => 100,
  'date' => 32,
  'format' => 10,
  'body' => 100000,
 );
 foreach ($lengths as $name => $length) {
  if (strlen($entry_data[$name]) > $length) {
   $result[] = LIB_ENTRYLIB_TOOLONG1.$fields[$name].LIB_ENTRYLIB_TOOLONG2;
  }
 }
 if (count($result) > 0) {return $result;}

 if (!preg_match('/^[0-9A-Za-z\-_]+\.txt$/', $entry_data['filename'])) {
  $result[] = LIB_ENTRYLIB_BADCHAR1.$fields['filename'].LIB_ENTRYLIB_BADCHAR2;
 }

 if ($entry_data['date'] <= 0) {
  $result[] = LIB_ENTRYLIB_BADDATE1.$fields['date'].LIB_ENTRYLIB_BADDATE2;
 }

 $formats = array('TEXT', 'HTML');
 if (!in_array($entry_data['format'], $formats)) {
  $result[] = LIB_ENTRYLIB_BADFORMAT1.$fields['format'].LIB_ENTRYLIB_BADFORMAT2;
 }

 if ($entry_data['cmd'] == 'new' && file_exists(ENTRY_DATA_DIR . '/' . $entry_data['filename'])) {
  $result[] = LIB_ENTRYLIB_EXISTEDYET;
 }

 if ($entry_data['cmd'] == 'edit' && !file_exists(ENTRY_DATA_DIR . '/' . $entry_data['filename'])) {
  $result[] = LIB_ENTRYLIB_NOFILE;
 }

 if (count($result) > 0) {return $result;}

 return NULL;
}

function init_entry_data() {
 $data = array(
  'cmd' => 'new',
  'filename' => date('YmdHis').'.txt',
  'title' => '',
  'date' => time(),
  'format' => 'TEXT',
  'body' => ''
 );
 return $data;
}

function import_entry_data() {
 $data = array(
  'cmd' => isset($_POST['cmd']) ? $_POST['cmd'] : 'new',
  'filename' => isset($_POST['filename']) ? $_POST['filename'] : '',
  'title' => isset($_POST['title']) ? $_POST['title'] : '',
  'date' => isset($_POST['date']) ? strtotime($_POST['date']) : time(),
  'format' => isset($_POST['format']) ? $_POST['format'] : 'TEXT',
  'body' => isset($_POST['body']) ? $_POST['body'] : ''
 );
 return $data;
}

function send_update_ping($ping_url) {
 $request_xml = get_update_ping_request();

 $request_length = strlen($request_xml);

 $ch = curl_init($ping_url);
 $urlinfo = parse_url($ping_url);
 $header = 'POST '.$urlinfo['path'].' HTTP/1.1'."\r\n";
 $header .= 'HOST: '.$urlinfo['host']."\r\n";
 $header .= 'Content-Type: text/xml; charset=UTF-8'."\r\n";
 $header .= 'Content-Length: '.$request_length."\r\n";
 curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $header."\r\n".$request_xml);

 ob_start();
 curl_exec($ch);
 curl_close($ch);
 $result_xml = ob_get_contents();
 ob_end_clean();

 return get_error_from_result_xml($result_xml);
}

function get_update_ping_request() {
 $blog_name = mb_convert_encoding(BLOG_NAME, 'utf8', 'UTF-8');
 $blog_url = BLOG_ROOT_URL;

 $request_xml  = '<'.'?xml version="1.0" encoding="utf-8" ?'.'>'."\n";
 $request_xml .= '<methodCall>'."\n";
 $request_xml .= '<methodName>weblogUpdates.ping</methodName>'."\n";
 $request_xml .= '<params>'."\n";
 $request_xml .= '<param><value>'
     .htmlspecialchars($blog_name)
     .'</value></param>'."\n";
 $request_xml .= '<param><value>'
     .htmlspecialchars($blog_url)
     .'</value></param>'."\n";
 $request_xml .= '</params>'."\n";
 $request_xml .= '</methodCall>'."\n";

 return $request_xml;
}

function get_error_from_result_xml($result_xml) {
 $pattern = '|<name>faultCode</name>\s*<value>\s*<int>(.*?)</int>\s*</value>|';
 if (preg_match($pattern, $result_xml, $matches)) {
  $faultCode = intval($matches[1]);
  if ($faultCode == 0) {
   return NULL;
  } else {
   if (preg_match('|<name>faultString</name>\s*<value>\s*<string>(.*)</string>\s*</value>|m', $result_xml, $matches)) {
    return 'ERROR: '.$matches[1];
   } else {
    return 'ERROR: UNKNOWN ERROR';
   }
  }
 } else {
  return 'ERROR: UNKNOWN ERROR';
 }
}

function pvcount($str) {
 $loggingdata = date('YmdHis')."\t".h(__FILE__)."\t".h(gethostbyaddr(h($_SERVER['REMOTE_ADDR'])))."(".h($_SERVER['REMOTE_ADDR']).")\t".h($_SERVER['HTTP_USER_AGENT'])."\t".h(@$_SERVER['HTTP_REFERER'])."\n";
 $loggingdata = h($loggingdata);
 $fplog = fopen(BLOG_LOGGING_DIR.'/logging_'.date('Ymd').'.cgi','a+b');
 flock($fplog,LOCK_EX);
 fwrite($fplog,$loggingdata);
 fclose($fplog);

 $fp = fopen(BLOG_COUNT_DIR.'/pvcount_'.$str.'.cgi','a+b');
 flock($fp,LOCK_EX);
 $count = fgets($fp);
 $count = $count + 1;
 ftruncate($fp,0);
 fwrite($fp,$count);
 fclose($fp);
 echo "$count";
}

function qr($str) {
 $qrdata = rawurlencode($str);
 $parts = array(
  'cht' => 'qr',
  'chs' => QRCODE_SIZE,
  'choe' => QRCODE_ENCODE,
  'chl' => '$qrdata',
 );
 $qruri = QRCODE_GOOGLEURI;
 $query = '';
 foreach ( $parts as $key => $val ) {
  if ($query != '') {
   $query .= '&amp;';
  }
  $query .= "$key=$val";
 }
 $qruri .= $query;
 echo '<img src="' . $qruri . '" class="qrcode">';
}

function add_referrer_data($entry_data) {
 $referreruris = array();
 if (isset($_SERVER['HTTP_REFERER'])) { $referreruri = h($_SERVER['HTTP_REFERER']); } else { $referreruri = "From bookmark"; }
 if (EXCEPTED_STRING1 == '') { $excepted_uri1 = 'http://windblog.sourceforge.jp/windblog/'; } else { $excepted_uri1 = EXCEPTED_STRING1; }
 if (EXCEPTED_STRING2 == '') { $excepted_uri2 = 'http://windblog.sourceforge.jp/windblog/'; } else { $excepted_uri2 = EXCEPTED_STRING2; }
 if (EXCEPTED_STRING3 == '') { $excepted_uri3 = 'http://windblog.sourceforge.jp/windblog/'; } else { $excepted_uri3 = EXCEPTED_STRING3; }
 if (mb_strpos($referreruri,$excepted_uri1) === FALSE) {
 if (mb_strpos($referreruri,$excepted_uri2) === FALSE) {
 if (mb_strpos($referreruri,$excepted_uri3) === FALSE) {
 if (mb_strpos($referreruri,'admin.php') === FALSE) {
   $filename = BLOG_REFERRER_DIR.'/'.$entry_data['filename'];
   $fp = fopen($filename, 'a+b');
   if (!$fp) {die('---');};
   if (!flock($fp, LOCK_SH)) {die('---');}
   $referreruris2 = file_get_contents($filename);
   if (mb_strpos($referreruris2,$referreruri) === FALSE) {
    fwrite($fp,$referreruri."\n");
    fclose($fp);
   }
 }}}}
}

function load_referrer_data($entry_data) {
 if (!file_exists(BLOG_REFERRER_DIR.'/'.$entry_data['filename'])) {return array();}
 $fp = @fopen(BLOG_REFERRER_DIR.'/'.$entry_data['filename'], 'rb');
 if (!$fp) {die('---');};
 if (!flock($fp, LOCK_SH)) {die('---');}
 $referreruris = array();
 while (!feof($fp)) {
  $line = rtrim(fgets($fp));
  if ($line == '') {continue;}
  $referreruris[] = $line;
 }
 fclose($fp);
 return $referreruris;
}

function send_trackback_ping($entry_data, $ping_urls) {
 $excerpt = strip_tags(render_entry_body($entry_data));
 $excerpt = mb_substr($excerpt, 0, 252);
 $excerpt = $excerpt . '...';

 $data = array(
  'blog_name' => BLOG_NAME,
  'title' => $entry_data['title'],
  'url' => BLOG_ROOT_URL.'entry.php/'.$entry_data['filename'],
  'excerpt' => $excerpt
 );

 $postdata = '';
 foreach ($data as $key => $value) {
  $value = mb_convert_encoding($value, 'utf-8', 'UTF-8');
  $postdata .= $key.'='.urlencode($value).'&';
 }

 $result = array();
 foreach ($ping_urls as $ping_url) {
  $ch = curl_init($ping_url);
  curl_setopt($ch, CURLOPT_POST, 1);
  curl_setopt($ch, CURLOPT_POSTFIELDS, $postdata);

  ob_start();
  curl_exec($ch);
  $xml = ob_get_contents();
  ob_end_clean();

  curl_close($ch);

  if (preg_match('|<error>(.+?)</error>|', $xml, $matches)) {
   $is_error = $matches[1];
   if ($is_error > 0) {
    if (preg_match('|<message>(.+)?</message>|', $xml, $matches)) {
     $error_msg = mb_convert_encoding($matches[1], 'UTF-8', 'auto');
    } else {
     $error_msg = 'UNKNOWN ERROR';
    }
   }
  } else {
   $is_error = 1;
   $error_msg = 'UNKNOWN ERROR';
  }

  if ($is_error > 0) {$result[] = $error_msg;}
 }
 return $result;

}

function add_trackback_data($trackback_data) {
 $filename = $trackback_data['filename'];
 $fp = fopen(TRACKBACK_DATA_DIR.'/'.$filename, 'a');
 if (!$fp) {die(LIB_TBLIB_CANNOTOPEN);}
 if (!flock($fp, LOCK_EX)) {die(LIB_TBLIB_CANNOTLOCK);}

 $now = time();
 $deleted = 0;
 $line = $now."\t".
  $deleted."\t".
  $trackback_data['blog_name']."\t".
  $trackback_data['title']."\t".
  $trackback_data['url']."\t".
  $trackback_data['excerpt']."\n";

 fwrite($fp, $line);
 fclose($fp);
}

function is_valid_trackback_data($trackback_data) {
 if (!preg_match('/^[0-9A-Za-z\-_]+\.txt$/', $trackback_data['filename'])) {
  return LIB_TBLIB_ISVALID_ILLEGAL_FILENAME;
 }
 if (!file_exists(ENTRY_DATA_DIR.'/'.$trackback_data['filename'])) {
  return LIB_TBLIB_ISVALID_ILLEGAL_LOADFILENAME;
 }

 if (!preg_match('|^https?://|i', $trackback_data['url'])) {
  return LIB_TBLIB_ISVALID_ILLEGAL_URI;
 }
 return NULL;
}

function import_trackback_data() {
 $trackback_data = array();

 if (!isset($_SERVER['PATH_INFO'])) {return $trackback_data;}
 $trackback_data['filename'] = substr($_SERVER['PATH_INFO'], 1);

 foreach (array('blog_name', 'title', 'url', 'excerpt') as $key) {
  $value = isset($_POST[$key]) ? $_POST[$key] : NULL;
  $value = mb_convert_encoding($value, 'UTF-8', 'auto');
  $value = preg_replace('|[\r\n\0\t]|', ' ', $value);
  $trackback_data[$key] = $value;
 }

 return $trackback_data;
}

function load_trackback_data($filename) {
 if (!file_exists(TRACKBACK_DATA_DIR.'/'.$filename)) {return array();}
 $fp = fopen(TRACKBACK_DATA_DIR.'/'.$filename, 'r');
 if (!$fp) {die(LIB_TBLIB_CANNOTOPEN);};
 if (!flock($fp, LOCK_SH)) {die(LIB_TBLIB_CANNOTLOCK);}

 $trackbacks = array();
 while (!feof($fp)) {
  $line = fgets($fp);
  $line = rtrim($line);
  if ($line == '') {continue;}
  list($date, $deleted, $blogname, $title, $url, $excerpt) = explode("\t", $line, 6);

  $trackbacks[] = array(
   'date' => $date,
   'deleted' => $deleted,
   'blogname' => $blogname,
   'title' => $title,
   'url' => $url,
   'excerpt' => $excerpt,
  );
 }

  fclose($fp);
  return $trackbacks;
}

function trackback_auto_discovery($url) {
 $body = file_get_contents($url);

 if (preg_match_all('|<rdf:RDF.+?</rdf:RDF>|s', $body, $matches)) {
  if (count($matches[0]) == 1) {
   $rdf_data = parse_rdf($matches[0][0]);
   if (isset($rdf_data['trackback:ping'])) {
    return $rdf_data['trackback:ping'];
   }
  } else {
   foreach ($matches[0] as $rdf_text) {
    $rdf_data = parse_rdf($rdf_text);
    if (isset($rdf_data['dc:identifier']) &&
     $rdf_data['dc:identifier'] == $url) {
     if (isset($rdf_data['trackback:ping'])) {
      return $rdf_data['trackback:ping'];
     }
    }
   }
  }
 }
 return NULL;

}

function parse_rdf($rdf) {
 $result = array();

 $pattern = '|dc:identifier="(.+?)"|';
 if (preg_match($pattern, $rdf, $matches)) {
  $result['dc:identifier'] = $matches[1];
 }

 $pattern = '|trackback:ping="(.+?)"|';
 if (preg_match($pattern, $rdf, $matches)) {
  $result['trackback:ping'] = $matches[1];
 }

 return $result;
}
