#include "stdafx.h"
#include "WinCS.h"
#include "APIGraphicsWindow.h"
#include "APIReportDlg.h"

IMPLEMENT_DYNAMIC(CAPIGraphicsWindow, CWnd)

BEGIN_MESSAGE_MAP(CAPIGraphicsWindow, CWnd)
	ON_WM_CREATE()
	ON_WM_CLOSE()
	ON_WM_DESTROY()
	ON_MESSAGE(WM_SHOW_LOG, &CAPIGraphicsWindow::OnShowLog)
	ON_MESSAGE(WM_OUTPUT_REPORT, &CAPIGraphicsWindow::OnOutputReport)
END_MESSAGE_MAP()

//////////////////////////////////////////////////////////////////////////

CAPIGraphicsWindow::CAPIGraphicsWindow() :
	m_nWidth(0),
	m_nHeight(0),
	m_dwNodes(0),
	m_dwProcessors(0),
	m_fDrawing(FALSE),
	m_fView(FALSE),
	m_fSystem(FALSE),
	m_fSaveLog(FALSE)
{
	ZeroMemory(&m_CalcSize, sizeof(m_CalcSize));
	m_Timer.hTimer = NULL;
	m_hIcon	= AfxGetApp()->LoadIcon(IDI_PLUGIN);
}

CAPIGraphicsWindow::~CAPIGraphicsWindow()
{
}

BOOL CAPIGraphicsWindow::ForceClose()
{
	CSingleLock sl(&m_DrawLock, TRUE);
	
	if (IsWindow(m_hWnd) && m_fDrawing)
	{
		m_fDrawing = FALSE;
		m_fView	   = FALSE;
		StopTimer();
		SendMessage(WM_CLOSE);
		return TRUE;
	}

	return FALSE;
}

void CAPIGraphicsWindow::FinishDrawing()
{
	CString cs, csTime;

	m_fDrawing = FALSE;

	if (IsWindow(m_hWnd))
	{
		m_Stopwatch.GetHMS(csTime);
		cs.Format(_T("%s (%s) - (Completed)"), m_csWndName, csTime);
		SetWindowText(cs);
	}
}

// static
void WINAPI CAPIGraphicsWindow::doRuntimeTimer(LPVOID pvContext, BOOLEAN fTimeout)
{
	((CAPIGraphicsWindow *)pvContext)->RuntimeTimer();
}

void CAPIGraphicsWindow::RuntimeTimer()
{
	CString cs, csTime;

	if (IsWindow(m_hWnd) && m_fDrawing)
	{
		m_Stopwatch.GetHMS(csTime);

		if (m_fSystem)
		{
			cs.Format(_T("%s (%s) - (%d / %d)"), m_csWndName, csTime, m_nRecvCount, m_nTotalCount);
		}
		else
		{
			cs.Format(_T("%s (%s) - (Processing)"), m_csWndName, csTime);
		}
		
		SetWindowText(cs);
	}
}

void CAPIGraphicsWindow::GetTimer(CString &cs)
{
	m_Stopwatch.GetHMS(cs);
}

BOOL CAPIGraphicsWindow::StartTimer()
{
	StopTimer();

	CSingleLock sl(&m_Timer.cs, TRUE);
	m_Stopwatch.Start();

	if (!CreateTimerQueueTimer(&m_Timer.hTimer, NULL, doRuntimeTimer, this, 1000, 1000, 0))
	{
		m_Timer.hTimer = NULL;
		return FALSE;
	}

	return TRUE;
}

void CAPIGraphicsWindow::StopTimer()
{
	CSingleLock sl(&m_Timer.cs, TRUE);

	if (m_Timer.hTimer != NULL)
	{
		DeleteTimerQueueTimer(NULL, m_Timer.hTimer, NULL);
		m_Timer.hTimer = NULL;
	}
}

BOOL CAPIGraphicsWindow::Create(const CString &csName, int cx, int cy, BOOL bFront)
{
	CString strClass;
	DWORD   dwStyle, dwExStyle;
	CRect   rect;

	try
	{
		strClass = AfxRegisterWndClass(0, 0, 0, 0);
	}
	catch (CResourceException *e) 
	{
		TRACE0("***** ERROR: AfxRegisterWndClass *****\n");
		e->Delete();
		return FALSE;
	}

	if (!strClass.IsEmpty())
	{
		dwExStyle = bFront ? WS_EX_PALETTEWINDOW : WS_EX_OVERLAPPEDWINDOW;
		dwStyle   = (WS_POPUP | WS_CAPTION | WS_VISIBLE | WS_SYSMENU |WS_BORDER);
		
		m_csWndName = csName;
		m_nWidth    = cx;
		m_nHeight   = cy;
		rect.SetRect(0, 0, m_nWidth, m_nHeight);

		if (CreateEx(dwExStyle, strClass, m_csWndName, dwStyle, rect, NULL, NULL))
		{
			ResetWindow();
			return TRUE;
		}
		else
		{
			TRACE1("***** ERROR: CreateEx(%d) *****\n", GetLastError());
			return FALSE;
		}
	}

	return FALSE;
}

int CAPIGraphicsWindow::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
	{
		return -1;
	}

	SetIcon(m_hIcon, TRUE);
	SetIcon(m_hIcon, FALSE);

	m_fDrawing = TRUE;
	m_fView	   = TRUE;

	return 0;
}

void CAPIGraphicsWindow::OnClose()
{
	if (m_fDrawing)
	{
		return;
	}

	CWnd::OnClose();
}

void CAPIGraphicsWindow::OnDestroy()
{
	CWnd::OnDestroy();

	m_nWidth   = 0;
	m_nHeight  = 0;
	m_fDrawing = FALSE;
	m_fView	   = FALSE;
}

BOOL CAPIGraphicsWindow::SetPixel(int x, int y, COLORREF color)
{
	if (IsWindow(m_hWnd) && m_fView)
	{
		CSingleLock sl(&m_DrawLock, TRUE);
		CClientDC dc(this);

		if (m_fDrawing)
		{
			ASSERT(IsWindow(m_hWnd));
			dc.SetPixel(x, y, color);
			return TRUE;
		}
	}
	
	return FALSE;
}

BOOL CAPIGraphicsWindow::SetXLine(int y, int cx, LPCOLORREF lpColor)
{
	if (IsWindow(m_hWnd) && m_fView)
	{
		CSingleLock sl(&m_DrawLock, TRUE);
		CClientDC dc(this);
		CBitmap Bmp, *pBmp;
		CDC MemDC, *pDC;

		if (m_fDrawing)
		{
			pDC = GetDC();
			MemDC.CreateCompatibleDC(pDC);
			Bmp.CreateCompatibleBitmap(pDC, cx, 1);
			pBmp = MemDC.SelectObject(&Bmp);

			for (int i = 0; i < cx; i++)
			{
				MemDC.SetPixel(i, 0, lpColor[i]);
			}

			dc.BitBlt(0, y, cx, 1, &MemDC, 0, 0, SRCCOPY);
			MemDC.SelectObject(pBmp);
			MemDC.DeleteDC();
			Bmp.DeleteObject();
			ReleaseDC(pDC);

			return TRUE;
		}
	}
	
	return FALSE;
}

BOOL CAPIGraphicsWindow::SetYLine(int x, int cy, LPCOLORREF lpColor)
{
	if (IsWindow(m_hWnd) && m_fView)
	{
		CSingleLock sl(&m_DrawLock, TRUE);
		CClientDC dc(this);
		CBitmap Bmp, *pBmp;
		CDC MemDC, *pDC;

		if (m_fDrawing)
		{
			ASSERT(IsWindow(m_hWnd));

			pDC = GetDC();
			MemDC.CreateCompatibleDC(pDC);
			Bmp.CreateCompatibleBitmap(pDC, 1, cy);
			pBmp = MemDC.SelectObject(&Bmp);

			for (int i = 0; i < cy; i++)
			{
				MemDC.SetPixel(0, i, lpColor[i]);
			}

			dc.BitBlt(x, 0, 1, cy, &MemDC, 0, 0, SRCCOPY);
			MemDC.SelectObject(pBmp);
			MemDC.DeleteDC();
			Bmp.DeleteObject();
			ReleaseDC(pDC);

			return TRUE;
		}
	}

	return FALSE;
}

void CAPIGraphicsWindow::SetWindowSize(int cx, int cy, int x, int y)
{
	CSingleLock sl(&m_DrawLock, TRUE);

	if (IsWindow(m_hWnd))
	{
		MoveWindow(x, y, cx, cy);
	}
}

void CAPIGraphicsWindow::SetCenterWindow()
{
	CSingleLock sl(&m_DrawLock, TRUE);

	if (IsWindow(m_hWnd))
	{
		CenterWindow(GetDesktopWindow());
	}
}

void CAPIGraphicsWindow::ResetWindow()
{
	if (IsWindow(m_hWnd) && m_fView)
	{
		CSingleLock sl(&m_DrawLock, TRUE);
		CClientDC dc(this);
		CRect rect;

		if (m_fDrawing)
		{
			GetClientRect(&rect);
			dc.FillSolidRect(&rect, RESET_WINDOW_COLOR);
		}
	}
}

void CAPIGraphicsWindow::OutputReportList()
{
	CAPIReportDlg dlg(this);
	UINT nSize = m_ReportList.GetSize();

	if (IsWindow(m_hWnd) && nSize > 0)
	{
		if (m_fSaveLog)
		{
			SaveReport();
		}
		
		dlg.SetReportList(m_ReportList);
		dlg.SetLog(m_fSaveLog);
		dlg.DoModal();
	}
}

void CAPIGraphicsWindow::InitRenderingCount(UINT nTotal)
{
	CSingleLock sl(&m_RgCountLock, TRUE);
	m_nRecvCount  = 0;
	m_nTotalCount = nTotal;
}

void CAPIGraphicsWindow::AddRenderingCount()
{
	m_RgCountLock.Lock();
	m_nRecvCount++;
	m_RgCountLock.Unlock();
	RuntimeTimer();	// Redraw
}

BOOL CAPIGraphicsWindow::SaveReport()
{
	if (IsWindow(m_hWnd) && m_fView)
	{
		CSingleLock sl(&m_DrawLock, TRUE);
		SYSTEMTIME  st;
		CClientDC   dc(this);
		CBitmap	    bmp;
		CString     cs, csReport, csText, csTime, csImgPath, csTxtPath, csDirPath;
		HANDLE		hFile;
		double		dbAvg, dbLate, dbFast;
		CImage	    img;
		DWORD		dwFileSize, dwWriteSize;
		TCHAR		szLogPath[_MAX_PATH * 2];
		CRect	    cr;
		UINT		nSize;
		CDC		    cdc;

		GetLocalTime(&st);
		csDirPath.Format(_T("./Log/Plugin/%s_%04d%02d%02d%02d%02d%02d"), 
						 m_csWndName,
						 st.wYear, st.wMonth, st.wDay,
						 st.wHour, st.wMinute, st.wSecond);
		csTxtPath.Format(_T("%s/index.html"), csDirPath);
		csImgPath.Format(_T("%s/img.png"), csDirPath);
		_tfullpath(szLogPath, csTxtPath, MC_ARYSIZE(szLogPath) - sizeof(TCHAR));
		m_LogPath.Format(_T("%s"), szLogPath);

		if (!CreateDirectory(csDirPath, NULL))
		{
			return FALSE;
		}

		// Image
		GetClientRect(&cr);
		bmp.CreateCompatibleBitmap(&dc, cr.Width(), cr.Height());
		cdc.CreateCompatibleDC(&dc);
		cdc.SelectObject(bmp);
		cdc.BitBlt(0, 0, cr.Width(), cr.Height(), &dc, 0, 0, SRCCOPY);
		img.Attach(static_cast<HBITMAP>(bmp));
		img.Save(csImgPath);

		// HTML
		nSize  = m_ReportList.GetSize();
		dbAvg  = 0.0;
		dbLate = 0.0;
		dbFast = static_cast<double>(INT_MAX);

		for (UINT i = 0; i < nSize; i++)
		{
			cs.Format(LOG_TEMPLATE_HTML_PLUGIN_DATA,
					  m_ReportList[i].wNodeID,
					  m_ReportList[i].wSubID,
					  DwToIPAddress(m_ReportList[i].dwAddress),
					  m_ReportList[i].dbRuntime);
			csReport += cs;

			dbAvg += m_ReportList[i].dbRuntime;
			dbLate = max(dbLate, m_ReportList[i].dbRuntime);
			dbFast = min(dbFast, m_ReportList[i].dbRuntime);
		}

		dbAvg = dbAvg / nSize;
		csTime.Format(_T("%04d/%02d/%02d %02d:%02d:%02d"),
					  st.wYear, st.wMonth, st.wDay,
					  st.wHour, st.wMinute, st.wSecond);
		csText.Format(LOG_TEMPLATE_HTML_PLUGIN_CONTAINER,
					  m_csWndName,								// Title
					  m_csWndName,								// Plugin name
					  m_fSystem ? _T("System") : _T("Single"),	// Mode
					  csTime,									// DateTime
					  m_dwNodes,								// Nodes
					  m_dwProcessors,							// Processors
					  m_CalcSize.nWidth,						// Width
					  m_CalcSize.nHeight,						// Height
					  m_CalcSize.nRange,						// Range
					  dbAvg,									// Average
					  dbFast,									// Fastest
					  dbLate,									// Latest
					  csReport);								// Report data

		hFile = CreateFile(csTxtPath, GENERIC_WRITE, 0, NULL, CREATE_NEW,
						   FILE_ATTRIBUTE_NORMAL, NULL);

		if (hFile != INVALID_HANDLE_VALUE)
		{
			dwFileSize = csText.GetLength() * sizeof(TCHAR);
			WriteFile(hFile, csText, dwFileSize, &dwWriteSize, NULL);
			CloseHandle(hFile);
			return TRUE;
		}
	}
	
	return FALSE;
}

LRESULT CAPIGraphicsWindow::OnShowLog(WPARAM wParam, LPARAM lParam)
{
	SHELLEXECUTEINFO sei;
	CString			 cs;

	ZeroMemory(&sei, sizeof(sei));
	sei.cbSize = sizeof(SHELLEXECUTEINFO);						
	sei.hwnd   = m_hWnd;											
	sei.nShow  = SW_SHOWNORMAL;									
	sei.fMask  = SEE_MASK_NOCLOSEPROCESS;						
	sei.lpFile = MC_CStoSTR(m_LogPath);								

	if(!ShellExecuteEx(&sei) || reinterpret_cast<int>(sei.hInstApp) <= 32)
	{	
		TRACE1("***** ERROR: OnShowLog(%d) *****\n", GetLastError());
		cs.LoadString(IDS_WCS_ERROR_SHOW_LOG);
		AfxMessageBox(cs);
		return FALSE;
	}

	WaitForSingleObject(sei.hProcess, INFINITE);
	CloseHandle(sei.hProcess);

	return TRUE;
}

LRESULT CAPIGraphicsWindow::OnOutputReport(WPARAM wParam, LPARAM lParam)
{
	return OutputReportList(), TRUE;
}

//////////////////////////////////////////////////////////////////////////

