package jp.ac.dendai.cdl.mori.wikie.util;

import java.io.*;
import java.util.*;
import java.util.regex.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;

import org.apache.commons.codec.net.*;
import org.apache.commons.lang.*;
import org.apache.hadoop.conf.*;
import org.apache.hadoop.fs.*;
import org.apache.hadoop.fs.FileSystem;

/**
 * Wikipediaエントリのタイトルを正規化するクラス
 * @author Mori
 *
 */
public class WikipediaNormalizer {
    /**
     * 名前空間文字列と番号の対応を示すHashMap
     */
    private HashMap<String, Integer> namespaceMap;

    /**
     * XMLファイルのPathからNormalizerを生成する
     * @param sourceFilePath　処理対象のXMLファイルを示すPath
     */
    public WikipediaNormalizer (Path sourceFilePath) {
        try {
            Configuration conf = new Configuration();
            FileSystem fs = FileSystem.get(conf);
            FSDataInputStream fsdis = fs.open(sourceFilePath);
            this.namespaceMap = createNamespaceMap(fsdis);
        }
        catch (Exception e) {
        }
    }

    /**
     * XMLファイルを示すFileからNormalizerを生成する
     * @param sourceFilePath　処理対象のXMLファイルを示すFile
     */
    public WikipediaNormalizer (File sourceFilePath) {
        try {
            FileInputStream fis = new FileInputStream(sourceFilePath);
            this.namespaceMap = createNamespaceMap(fis);
        }
        catch (Exception e) {
        }
    }

    /**
     * XMLファイルのヘッダ部分から名前空間Mapを生成する
     * @param isr 対象ファイルの読み込むInputStream
     * @return 名前空間文字列と番号の対応を示すHashMap
     */
    public static HashMap<String, Integer> createNamespaceMap(InputStream isr) {
        try {
            HashMap<String, Integer> namespacMap = new HashMap<String, Integer>();

            BufferedReader reader = new BufferedReader(new InputStreamReader(isr, WikIE.UTF8));
            String line = new String();
            namespacMap.put("", WikIE.ARTICLE_NS_NUM);
            while (!(line = reader.readLine().trim()).equals("</" + WikIE.NAMESPACES_ELEMENT + ">")) {
                Pattern pattern = Pattern.compile("<" + WikIE.NAMESPACE_ELEMENT + " +key=\"(-*[0-9]+?)\">(.*?)" + "</" + WikIE.NAMESPACE_ELEMENT + ">");
                Matcher matcher = pattern.matcher(line);
                if (matcher.find()) {
                    int number = Integer.parseInt(matcher.group(1));
                    String namespace = matcher.group(2).toLowerCase();
                    namespacMap.put(namespace, number);
                }
                namespacMap.put("image", WikIE.IMAGE_NS_NUM);
            }
            reader.close();
            return namespacMap;
        }
        catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }


    /**
     * タイトルにスペース・アンダースコアの除去規則を適用し、
     * デコード・アンエスケープをする。
     * @param title Wikipediaエントリのタイトル
     * @return 処理後のタイトル
     */
    public static String decode(String title) {
            title = title.trim();
            title = title.replaceAll("_", " ");
            title = title.replaceAll(" {2,}", " ");
            title = decodePart(title);
        while (!title.equals(StringEscapeUtils.unescapeHtml(title))) {
            title = StringEscapeUtils.unescapeHtml(title);
        }
        return title;
    }

    public static String decodePart(String title) {
        URLCodec codec = new URLCodec(WikIE.UTF8);
        Pattern pattern = Pattern.compile("([a-zA-Z0-9%]+)");
        Matcher matcher = pattern.matcher(title);
        while (matcher.find()) {
            try {
                String encoded = matcher.group(1);
                String decoded = codec.decode(encoded);
                if (decoded.length() < encoded.length()) {
                    title = title.replaceAll(encoded, codec.decode(encoded));
                }
            }
            catch (Exception e) {
                // TODO: handle exception
            }
        }
        return title;
    }

    public static String decodeAnchor(String link) {
        String tmp = link.replaceAll("\\.", "%");
        URLCodec codec = new URLCodec(WikIE.UTF8);
        Pattern pattern = Pattern.compile("([a-zA-Z0-9%]+)");
        Matcher matcher = pattern.matcher(tmp);
        while (matcher.find()) {
            try {
                String encoded = matcher.group(1);
                String decoded = codec.decode(encoded);
                if (decoded.length() < encoded.length()) {
                    tmp = tmp.replaceAll(encoded, codec.decode(encoded));
                }
            }
            catch (Exception e) {
                // TODO: handle exception
            }
        }
        if (tmp.length() < link.length())
            return tmp;
        else
            return link;
    }

    /**
     * 制御文字を除去した文字列を返す
     * @param text 対象文字列
     * @return 制御文字を除去した文字列
     */
    public static String removeNonPrintingCharacter(String text) {
        //制御文字を除去する。(MediaWikiにもある処理）
        char[] target = {'\n', '\r', '\u0000', '\u200e', '\u200f', '\u2028', '\u2029'};
        for (char c : target) {
            text = text.replaceAll(Character.toString(c), "");
        }
        return text;
    }

    /**
     * タイトルのうち名前空間部分を返す
     * @param title　Wikipediaエントリのタイトル
     * @return　名前空間部分
     */
    public String getNamespaceString(String title) {
        Integer ns = getNamespaceNumber(title);
        if (ns == 0) {
            return "";
        }
        else {
            return getPrefix(title);
        }
    }

    /**
     * タイトルのうち:で区切られている先頭部分を返す
     * @param title　Wikipediaエントリのタイトル
     * @return　:で区切られている先頭部分
     * @throws ArrayIndexOutOfBoundsException
     */
    public String getPrefix(String title) throws ArrayIndexOutOfBoundsException {
        String[] splited = title.split(":");
        if (0 < splited.length && 0 < title.indexOf(":")) {
            return splited[0].trim();
        }
        else {
            return "";
        }
    }

    /**
     * タイトルのうち名前空間でない部分（ページ名）を返す
     * @param title　Wikipediaエントリのタイトル
     * @return　名前空間でない部分（ページ名）を返す
     * @throws StringIndexOutOfBoundsException
     */
    public String getPageName(String title) throws StringIndexOutOfBoundsException {
        if (getNamespaceNumber(title) == WikIE.ARTICLE_NS_NUM) {
            return title;
        }
        else {
            String[] splited = title.split(":");
            StringBuffer pageName = new StringBuffer();
            for (int i = 1; i < splited.length; i++) {
                pageName.append(splited[i] + ":");
            }
            pageName.deleteCharAt(pageName.length()-1);
            return pageName.toString().trim();
        }
    }

    /**
     * タイトルが属する名前空間の番号を返す
     * @param title　Wikipediaエントリのタイトル
     * @return 名前空間の番号
     */
    public int getNamespaceNumber(String title) {
        String prefix = getPrefix(title).toLowerCase();
        if (prefix.length() == 0) {
            return 0;
        }
        else {
            Integer ns = namespaceMap.get(prefix);
            if (ns == null) {
                return 0;
            }
            else {
                return ns;
            }
        }
    }

    /**
     * 名前空間を正規化する
     * @param namespace　名前空間部分
     * @return 正規化された名前空間部分
     */
    public String normalizeNamespace(String namespace) {
        return namespace.toLowerCase();
    }

    /**
     * ページ名を正規化する
     * @param pageName　ページ名
     * @return　正規化されたページ名
     */
    public String normalizePageName(String pageName) {
        String initial = ((Character)pageName.charAt(0)).toString();
        if (initial.toUpperCase().length() > initial.length()) {
            return pageName;
        }
        else {
            StringBuffer tmp = new StringBuffer(pageName);
            return  tmp.replace(0, 1, initial.toUpperCase()).toString();
        }
    }

    /**
     * タイトルを正規化する
     * @param title　Wikipediaエントリのタイトル
     * @return　正規化されたタイトル
     */
    public String normalize(String title) {
        if (title.length() == 0) return title;
        title = decode(title);
        try {
            String namespace = getNamespaceString(title);
            String pageName = getPageName(title);
            if (pageName.equals(":"))
                return "";
            else if (namespace.length() == 0)
                return normalizePageName(pageName);
            else
                return normalizeNamespace(namespace) + ":" + normalizePageName(pageName);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            return "";
        }
        catch (StringIndexOutOfBoundsException e) {
            return "";
        }
    }




}
