package jp.ac.dendai.cdl.mori.wikie.mapper;

import java.io.*;
import java.util.*;

import jp.ac.dendai.cdl.mori.wikie.main.*;
import jp.ac.dendai.cdl.mori.wikie.util.*;

import org.apache.hadoop.io.*;
import org.apache.hadoop.mapred.*;

/**
 * edge処理用Mapper
 * @author Mori
 *
 */
public class EdgeMapper extends WikipediaMapper
implements Mapper<LongWritable, Text, Text, Text>{

    /**
     * このMapperが出力するキーは常に「エントリのタイトル」である。
     * 記事・カテゴリ・リダイレクトをエントリとする。
     * 処理中のpage要素のタイトルのほかに、
     * 上位カテゴリのタイトルと、
     * リダイレクト先のエントリのタイトルも「エントリのタイトル」として出力をつくる。
     * Reducerでは例えば次のようなキーと値のペアを受け取ることになる。
     * category:言語    edge\t123
     * category:言語    entry\tnode\t456   <--これが他のedgeが向かっているエントリ
     * category:言語    edge\t789
     * category:言語    edge\t098
     * リダイレクト関係なら次のようになる
     * アメリカ合衆国      target\t444
     * アメリカ合衆国      entry\tleaf\t444   <--これが他のredirectが向かっているエントリ
     * アメリカ合衆国      target\t555
     * アメリカ合衆国      target\t666
     */
    @Override
    public void map(LongWritable key, Text value, OutputCollector<Text, Text> output, Reporter reporter) throws IOException {
        try {
            String page = WikipediaNormalizer.removeNonPrintingCharacter(value.toString());
            String title = normalizer.normalize(WikipediaMapper.getValueFromElementName(WikIE.TITLE_ELEMENT, page, 0, 1));
            String id  = WikipediaMapper.getValueFromElementName(WikIE.ID_ELEMENT, page, 0, 1);
            String text = WikipediaMapper.getValueFromElementName(WikIE.TEXT_ELEMENT, page, 0, 1);
            String kind = linkChecker.getKind(title, text);

            if (kind != null) {
                if (kind.equals(WikIE.LEAF) || kind.equals(WikIE.NODE)) {
                    ArrayList<String> categoryArrayList = linkChecker.getCategory(text);
                    output.collect(new Text(title), new Text(WikIE.ENTRY + "\t" + kind + "\t" + id));
                    Iterator<String> itr = categoryArrayList.iterator();
                    while (itr.hasNext()) {
                        String category = itr.next();
                        output.collect(new Text(category), new Text(WikIE.EDGE + "\t" + id));
                    }
                }
                else if (kind.equals(WikIE.REDIRECT)) {
                    String redirect = linkChecker.isRedirect(text);
                    output.collect(new Text(title), new Text(WikIE.ENTRY + "\t" + WikIE.REDIRECT + "\t" + id));
                    output.collect(new Text(redirect), new Text(WikIE.TARGET + "\t" + id));
                }
            }
        }
        catch (NullPointerException e) {
            // TODO: handle exception
        }
    }



}
