﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using WhereTrainBuild.MapUtil.Data;

namespace WhereTrainBuild.Script
{
    /// <summary>
    /// OUDIAツール
    /// </summary>
    public class OuDiaTool
    {
        /// <summary>
        /// ファクトリ
        /// </summary>
        protected IFactory m_factory = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public OuDiaTool()
        {
        }

        /// <summary>
        /// ロード
        /// </summary>
        /// <param name="filename"></param>
        public void Load( string filename, IFactory factory )
        {
            m_factory = factory;

            using ( var fs = new FileStream(filename,FileMode.Open,FileAccess.Read) )
            {
                using (var sr = new StreamReader(fs, Encoding.GetEncoding("Shift_JIS")))
                {
                    while( sr.EndOfStream == false )
                    {
                        var line = sr.ReadLine();

                        Analyze(line);
                    }
                }
            }
        }

        /// <summary>
        /// ファイルタイプ
        /// </summary>
        protected string m_filetype = string.Empty;

        /// <summary>
        /// パック
        /// </summary>
        protected List<string> m_pack = new List<string>();

        /// <summary>
        /// 解析
        /// </summary>
        /// <param name="line"></param>
        protected void Analyze( string line )
        {
            if (m_filetype == string.Empty)
            {
                if (line.IndexOf("FileType") == 0)
                {
                    OnFileType(line);
                }
            }
            else
            {
                var planelist = line.Trim();

                if (planelist.Length > 0)
                {
                    //末尾.検出
                    if (planelist[planelist.Length - 1] == '.')
                    {
                        if(m_pack.Count > 0 )
                        {
                            //パック完了
                            OnPack(m_pack);
                        }

                        m_pack.Clear();
                    }

                    //コメント判定
                    if(planelist.Length > 1)
                        m_pack.Add(planelist);
                }
            }
        }

        /// <summary>
        /// ファイルタイプ
        /// </summary>
        /// <param name="line"></param>
        protected void OnFileType(string line)
        {
            var filetypesplit = line.Split('=');
            m_filetype = filetypesplit[1].Trim();
        }

        /// <summary>
        /// パック
        /// </summary>
        /// <param name="pack"></param>
        protected void OnPack( List<string> pack )
        {
            switch (pack[0])
            {
                case "Rosen.":
                    break;
                case "Eki.":
                    OnEki(pack);
                    break;
                case "Ressyasyubetsu.":
                    OnRessyasyubetsu(pack);
                    break;
                case "Dia.":
                    OnDia(pack);
                    break;
                case "Nobori.":
                    m_linename = "上り";
                    OnNobori(pack);
                    break;
                case "Kudari.":
                    m_linename = "下り";
                    OnKudari(pack);
                    break;
                case "Ressya.":
                    OnRessya(pack);
                    break;
                case "DispProp.":
                    break;
                default:

                    var lll = pack[0];

                    break;
            }
        }

        /// <summary>
        /// パックデコード
        /// </summary>
        /// <param name="pack"></param>
        /// <returns></returns>
        protected Dictionary<string, string> DecodePack(List<string> pack)
        {
            Dictionary<string, string> tbl = new Dictionary<string, string>();
            foreach (var line in pack)
            {
                var split = line.Split('=');
                if(split.Length == 2 )
                    tbl[split[0]] = split[1];
            }

            return tbl;
        }

        /// <summary>
        /// 駅
        /// </summary>
        /// <param name="pack"></param>
        protected void OnEki(List<string> pack)
        {
            var tbl = DecodePack(pack);
            var name = tbl["Ekimei"];

            var target = m_factory.GetStationManager().Get(name);
            if (target == null)
            {
                var list = m_factory.GetStationManager().StationList().Where((station) => { return station.Name.IndexOf(name) >= 0 || name.IndexOf(station.Name) >= 0; });
                if(list.Count()>0)
                    target = list.First();
            }
            if (target == null)
                throw new Exception(string.Format("駅が見つからない {0}",name));
            m_line.Add(target);
        }

        /// <summary>
        /// 駅リスト
        /// </summary>
        protected List<StationInfoData> m_line = new List<StationInfoData>();

        /// <summary>
        /// 電車種別
        /// </summary>
        protected class Syubetsu
        {
            /// <summary>
            /// 種類
            /// </summary>
            public string Kind = string.Empty;

            /// <summary>
            /// 略称
            /// </summary>
            public string Simple = string.Empty;
        }

        /// <summary>
        /// 電車種別リスト
        /// </summary>
        protected List<Syubetsu> m_syubetsu = new List<Syubetsu>();

        /// <summary>
        /// 列車種別
        /// </summary>
        /// <param name="pack"></param>
        protected void OnRessyasyubetsu(List<string> pack)
        {
            var tbl = DecodePack(pack);

            var trainkind = new Syubetsu();

            trainkind.Kind = tbl["Syubetsumei"];
            if( tbl.ContainsKey("Ryakusyou") == true )
                trainkind.Simple = tbl["Ryakusyou"];

            m_syubetsu.Add(trainkind);
        }

        /// <summary>
        /// ダイヤ種別
        /// </summary>
        protected string m_setname = string.Empty;

        /// <summary>
        /// 上り
        /// </summary>
        /// <param name="pack"></param>
        protected void OnNobori(List<string> pack)
        {
            var firstline = m_factory.GetNetwork().GetLine(m_setname, (ll) => ll.Name == "下り");
            var secondline = firstline.Clone() as TrainLine;
            secondline.Display = "上り";
            secondline.Name = "上り";
            secondline.Reverse();

            m_factory.GetNetwork().AddLine(m_setname,secondline);
        }

        /// <summary>
        /// 下り
        /// </summary>
        /// <param name="pack"></param>
        protected void OnKudari(List<string> pack)
        {
            var line = new TrainLine();
            line.Display = "下り";
            line.Name = "下り";

            StationInfoData laststation = null;
            foreach (var station in m_line)
            {
                if (laststation != null)
                {
                    var targetpath = m_factory.GetNetwork().GetPathList().SingleOrDefault(ppp => ppp.StationA.UniqID == laststation.UniqID && ppp.StationB.UniqID == station.UniqID);
                    if (targetpath == null)
                    {
                        var rev = m_factory.GetNetwork().GetPathList().SingleOrDefault(ppp => ppp.StationB.UniqID == laststation.UniqID && ppp.StationA.UniqID == station.UniqID);
                        if (rev == null)
                            throw new Exception(string.Format("{0}->{1}が、見つからない", laststation.Name, station.Name));

                        rev = (rev.Clone() as TrainPath);
                        rev.Reverse();
                        targetpath = rev;
                    }

                    line.AddPath(targetpath);
                }

                laststation = station;
            }

            m_factory.GetNetwork().AddLine(m_setname,line);
        }

        /// <summary>
        /// ダイヤ種別
        /// </summary>
        /// <param name="pack"></param>
        protected void OnDia(List<string> pack)
        {
            var tbl = DecodePack(pack);
            m_setname = tbl["DiaName"];

            if (m_setname != "平日" || m_setname != "土休日")
            {
                //なんとかする
                if (m_setname.IndexOf("休") >= 0 || m_setname.IndexOf("祝") >= 0 || m_setname.IndexOf("土") >= 0)
                    m_setname = "土日祝日";
                else
                    m_setname = "平日";
            }
        }

        /// <summary>
        /// 方面
        /// </summary>
        protected string m_linename = string.Empty;

        /// <summary>
        /// 列車
        /// </summary>
        /// <param name="pack"></param>
        protected void OnRessya(List<string> pack)
        {
            var tbl = DecodePack(pack);

            var houkou = "下り";
            if( tbl["Houkou"] == "Nobori" )
                houkou = "上り";

            var name = tbl["Ressyabangou"];
            var display = tbl["Ressyamei"];
            var syubetsu = tbl["Syubetsu"];

            var jikoku = tbl["EkiJikoku"];

            //方面
            var line = m_factory.GetNetwork().GetLineList(m_setname).SingleOrDefault(lll => lll.Name == m_linename);
            
            //電車
            var train = new TrainInfo();
            train.Name = name;
            train.Display = display;
            train.Kind = m_syubetsu[int.Parse(syubetsu)].Kind;

            line.AddTrain(train);

            bool bStart = false;

            //時刻表
            var jtbl = jikoku.Split(',');
            for (int iIdx = 0; iIdx < jtbl.Length; iIdx++)
            {
                if (jtbl[iIdx].Trim() == string.Empty)
                    continue;

                //駅取得
                StationInfoData station = null;
                if( iIdx == jtbl.Length -1 )
                    station = line.GetPathList()[iIdx-1].StationB;
                else
                    station = line.GetPathList()[iIdx].StationA;

                if (bStart == false)
                {
                    train.StartStation = station.Name;
                    bStart = true;
                }
                train.EndStation = station.Name;

                //計画
                var plan = new SceduleManager.Plan();
                plan.Order = iIdx;
                plan.Station = station;

                var timetbl = jtbl[iIdx].Split(';');
                var kind = int.Parse(timetbl[0]);

                if (kind == 1)
                {
                    var time = timetbl[1];

                    if (time.IndexOf("/") < 0)
                    {
                        //発車時刻のみ
                        plan.StartTime = DecodeTime(time);
                        plan.AliveTime = plan.StartTime.Subtract(new TimeSpan(0, 1, 0));
                    }
                    else
                    {
                        var inout = time.Split('/');
                        if (inout[1].Trim() == string.Empty)
                        {
                            //到着時刻のみ
                            plan.AliveTime = DecodeTime(inout[0]);
                            plan.StartTime = DecodeTime(inout[0]);
                        }
                        else
                        {
                            //到着、発車両方
                            plan.AliveTime = DecodeTime(inout[0]);
                            plan.StartTime = DecodeTime(inout[1]);
                        }
                    }

                    //登録
                    train.Scedule.Add(plan);
                }
            }

            if (train.Display == string.Empty)
            {
                train.Display = string.Format("{0} 行き", train.EndStation);
            }
        }

        /// <summary>
        /// 時刻文字列デコード
        /// </summary>
        /// <param name="timestr"></param>
        /// <returns></returns>
        protected TimeSpan DecodeTime(string timestr)
        {
            var timeval = int.Parse(timestr);

            int iHour = timeval / 100;
            int iMinute = timeval % 100;

            var time = new TimeSpan(iHour,iMinute,0);

            if( time < m_factory.OverTime )
                time = time.Add( new TimeSpan(24,0,0));

            return time;
        }
    }
}
