/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *    RandomCommittee.java
 *    Copyright (C) 2003 Eibe Frank
 *
 */

package weka.classifiers.meta;

import weka.classifiers.*;
import weka.classifiers.trees.RandomTree;
import java.util.Enumeration;
import java.util.Vector;
import java.util.Random;

import weka.core.Instance;
import weka.core.Instances;
import weka.core.OptionHandler;
import weka.core.WeightedInstancesHandler;
import weka.core.Option;
import weka.core.Utils;
import weka.core.Randomizable;
import weka.core.UnsupportedAttributeTypeException;

/**
 * Class for creating a committee of random classifiers. The base
 * classifier (that forms the committee members) needs to implement
 * the Randomizable interface.
 *
 * Valid options are:<p>
 *
 * -W classname <br>
 * Specify the full class name of a base classifier as the basis for 
 * the random committee (required).<p>
 *
 * -I num <br>
 * Set the number of committee members (default 10). <p>
 *
 * -S seed <br>
 * Random number seed for the randomization process (default 1). <p>
 *
 * Options after -- are passed to the designated classifier.<p>
 *
 * @author Eibe Frank (eibe@cs.waikato.ac.nz)
 * @version $Revision: 1.5.2.2 $
 */
public class RandomCommittee extends RandomizableIteratedSingleClassifierEnhancer
  implements WeightedInstancesHandler {
    
  /**
   * Constructor.
   */
  public RandomCommittee() {
    
    m_Classifier = new weka.classifiers.trees.RandomTree();
  }

  /**
   * String describing default classifier.
   */
  protected String defaultClassifierString() {
    
    return "weka.classifiers.trees.RandomTree";
  }

  /**
   * Returns a string describing classifier
   * @return a description suitable for
   * displaying in the explorer/experimenter gui
   */
  public String globalInfo() {
 
    return weka.LocalString.get("Class for building an ensemble of randomizable base classifiers. Each ")
      + weka.LocalString.get("base classifiers is built using a different random number seed (but based ")
      + weka.LocalString.get("one the same data). The final prediction is a straight average of the ")
      + weka.LocalString.get("predictions generated by the individual base classifiers.");
  }

  /**
   * Builds the committee of randomizable classifiers.
   *
   * @param data the training data to be used for generating the
   * bagged classifier.
   * @exception Exception if the classifier could not be built successfully
   */
  public void buildClassifier(Instances data) throws Exception {

    if (data.numInstances() == 0) {
      throw new IllegalArgumentException(weka.LocalString.get("RandomTree: zero training instances or all ") +
					 weka.LocalString.get("instances have missing class!"));
    }

    if (!(m_Classifier instanceof Randomizable)) {
      throw new IllegalArgumentException(weka.LocalString.get("Base learner must implement Randomizable!"));
    }

    m_Classifiers = Classifier.makeCopies(m_Classifier, m_NumIterations);

    Random random = data.getRandomNumberGenerator(m_Seed);
    for (int j = 0; j < m_Classifiers.length; j++) {

      // Set the random number seed for the current classifier.
      ((Randomizable) m_Classifiers[j]).setSeed(random.nextInt());
      
      // Build the classifier.
      m_Classifiers[j].buildClassifier(data);
    }
  }

  /**
   * Calculates the class membership probabilities for the given test
   * instance.
   *
   * @param instance the instance to be classified
   * @return preedicted class probability distribution
   * @exception Exception if distribution can't be computed successfully 
   */
  public double[] distributionForInstance(Instance instance) throws Exception {

    double [] sums = new double [instance.numClasses()], newProbs; 
    
    for (int i = 0; i < m_NumIterations; i++) {
      if (instance.classAttribute().isNumeric() == true) {
	sums[0] += m_Classifiers[i].classifyInstance(instance);
      } else {
	newProbs = m_Classifiers[i].distributionForInstance(instance);
	for (int j = 0; j < newProbs.length; j++)
	  sums[j] += newProbs[j];
      }
    }
    if (instance.classAttribute().isNumeric() == true) {
      sums[0] /= (double)m_NumIterations;
      return sums;
    } else if (Utils.eq(Utils.sum(sums), 0)) {
      return sums;
    } else {
      Utils.normalize(sums);
      return sums;
    }
  }

  /**
   * Returns description of the committee.
   *
   * @return description of the committee as a string
   */
  public String toString() {
    
    if (m_Classifiers == null) {
      return weka.LocalString.get("RandomCommittee: No model built yet.");
    }
    StringBuffer text = new StringBuffer();
    text.append(weka.LocalString.get("All the base classifiers: \n\n"));
    for (int i = 0; i < m_Classifiers.length; i++)
      text.append(m_Classifiers[i].toString() + "\n\n");

    return text.toString();
  }

  /**
   * Main method for testing this class.
   *
   * @param argv the options
   */
  public static void main(String [] argv) {
   
    try {
      System.out.println(Evaluation.
			 evaluateModel(new RandomCommittee(), argv));
    } catch (Exception e) {
      e.printStackTrace();
      System.err.println(e.getMessage());
    }
  }
}
