#
# $Id$
# Copyright Narushima Hironori. All rights reserved.
#

require 'fileutils'
require 'webpub'

module Webpub

class PublisherEnvironmentException < StandardError
end

class PublishManager

	attr_reader :plugin_registory, :workspace, :plugins_entry_filepath, :listeners

	def initialize
		@plugin_registory = PublisherRegistory.instance
		@workspace = Workspace.instance
		
		@default_ignores = ['.publish', 'CVS', 'Thumbs.db']
		@listeners = []
	end

	def seek_publish_from(pub_to)
		wp = @workspace.web_projects.select { |wp| /^#{wp[:publish_dir]}/i === pub_to }
		return nil if wp.size != 1
		wp = wp[0]
		
		rel_path = pub_to[wp[:publish_dir].size .. -1]
		from = wp[:htsources_dir] + rel_path
		return from if File.file?(from)
		
		Dir.glob(wp.path + '/**/.publish').sort { |a, b|
			b.count('/') <=> a.count('/')
		}.each { |mapfile|
			wp.publish_description_factory.publish_mapping(mapfile).each { |mapping|
				next unless mapping[:attributes]['publish_to']
				to = File.join(wp[:publish_dir], mapping[:attributes]['publish_to'].gsub(/^\//, '') )
				if to == pub_to
					return File.join(File.dirname(mapfile), mapping[:pattern])
				end
				
				glob = File.join(File.dirname(mapfile), mapping[:pattern])
				Dir.glob(glob) { |from|
					if /\*/ === to
						return from if File.basename(from, '.*') == File.basename(pub_to, '.*')
					else
						return from if File.basename(from) == File.basename(pub_to)
					end
				}
			}
		}
		nil
	end
	
	def plugins_entry_filepath
		raise PublisherEnvironmentException, "can't define publish plugin cache file location: workspace path is not specified" unless @workspace.path
		@workspace.path + '/.metadata/.plugins/com.narucy.webpub.core/.cache'
	end
	
	def check_env
		raise PublisherEnvironmentException, "publishers not found" if @plugin_registory.ids.empty?
		raise PublisherEnvironmentException, "workspace is not specified" unless @workspace.path
	end
	
	def store_plugins_entry
		file = plugins_entry_filepath()
		FileUtils.mkdir_p(File.dirname(file))
		@plugin_registory.store_publisher_entry_file(file)
	end
	
	def load_plugins_entry
		@plugin_registory.regist_plugins_from_entry_file(plugins_entry_filepath())
	end
	
	def sort_by_webproject(files)
		matches = Hash.new { |hash, key|
			hash[key] = []
		}
		files.each { |f|
			wp = @workspace.web_projects.select { |wp| /^#{wp.path}/i === f }
			if wp.size != 1
				raise ArgumentError, "specify file isn't included in web project: #{f}"
			end
			matches[wp[0]] << f
		}
		matches
	end
	
	def publish(source_files)
		check_env()
		
		# revalidate sources entry
		source_files = [source_files] unless source_files.is_a?(Array)
		source_files.reject! { |f|
			@default_ignores.any? { |pattern| f.split('/').any? { |p| File::fnmatch?(pattern, p) } }
		}
		
		# publish file path to source file path
		source_files.map! { |f|
			f = File.expand_path(f)
			if from = seek_publish_from(f)
				f = from
			end
			f
		}
		
		ok_descs = []
		failed_descs = []
		
		sort_by_webproject(source_files).each { |wp, files|
			$: << wp[:scripts_dir]
			
			descriptions = files.map { |f| wp.publish_description_factory.create(f) }.compact
			descriptions.each { |desc|
				publisher = @plugin_registory[desc.by]
				
				unless publisher
					ex = StandardError.new("publisher not found `#{desc.by}'")
					@listeners.each { |l| l.error(desc, ex) }
					failed_descs << desc
					next
				end
				
				if ex = @plugin_registory.registed_error(desc.by)
					@listeners.each { |l| l.error(desc, ex) }
					failed_descs << desc
					next
				end
				
				begin
					to = desc.publish_to
					if to
						to = File.dirname(to) unless /\/$/ === to
						FileUtils.mkdir_p(to)
					end
					publisher.publish(desc)
					
					@listeners.each { |l| l.successed(desc) }
					ok_descs << desc
				rescue Exception => ex
					@listeners.each { |l| l.error(desc, ex) }
					failed_descs << desc
				end
			}
			$:.delete(wp[:scripts_dir])
		}
		
		[ok_descs, failed_descs]
	end

end

end
