/*
 * $Id: WebProjectPage.java,v 1.9 2004/06/07 06:46:15 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.io.File;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.PropertyPage;

import com.narucy.webpub.core.*;
import com.narucy.webpub.ui.*;


/**
 * WebProjectPage.
 */
public class WebProjectPage extends PropertyPage {
	
	final static String[] defaultEncodings = new String[]{
		"ISO-8859-1",
		"US-ASCII",
		"UTF-16",
		"UTF-16BE",
		"UTF-16LE",
		"UTF-8",
	};
	
	WebProject webProject;

	WebProjectStructureBlock block;
	Combo encodingsCombo;
	ListViewer extsListViewer, mappedUrlViewer;
	Button addExtButton, removeExtButton, editExtButton, addUrlButton, removeUrlButton, editUrlButton;
	
	Listener buttonListener = new Listener() {
		public void handleEvent(Event e) {
			if(e.widget == addExtButton) {
				handleAddExtension();
			}else if(e.widget == removeExtButton){
				handleRemoveExtension();
			}else if(e.widget == editExtButton) {
				handleRemoveExtension();
			}else if (e.widget == addUrlButton) {
				handleAddUrl();
			}else if(e.widget == removeUrlButton) {
				handleRemoveUrl();
			}else if(e.widget == editUrlButton){
				handleEditUrl();
			}
		}
	};

	/*
	 * Note: HTML extension used to template synchronization target,
	 * and inner document publish way definication
	 * (as '<?publish by=\"...\" ?>').
	 */
	protected Control createContents(Composite parent) {
		setDescription("Web Project directory structure setting");
		IProject proj = (IProject)getElement().getAdapter(IProject.class);
		try {
			webProject = (WebProject)proj.getNature(WebProject.ID_NATURE);
		} catch (CoreException ex) {
			throw new RuntimeException(ex);
		}
		
		// create base widget
		block = new WebProjectStructureBlock(new IStatusChangeListener() {
			public void statusChanged(IStatus status) {
				int serverity = status.getSeverity();
				switch (serverity) {
					case IStatus.ERROR:
						setErrorMessage(status.getMessage());
						break;
					case IStatus.WARNING:
					case IStatus.INFO:
						setMessage(status.getMessage(), serverity);
						break;
					case IStatus.OK:
						setErrorMessage(null);
						setMessage(null, serverity);
						break;
				}
				
				updateApplyButton();
			}
		});
		
		Composite base = null;
		try {
			base = block.createControl(parent, proj);
		} catch (CoreException ex) {
			throw new RuntimeException(ex);
		}
		
		// char code input controls
		new Label(base, SWT.NONE).setText("HTML char code:");
		encodingsCombo = new Combo(base, SWT.BORDER | SWT.DROP_DOWN);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalIndent = 5;
		encodingsCombo.setLayoutData(gd);
		
		encodingsCombo.setItems(defaultEncodings);
		
		// extensions widgets
		Label labe = new Label(base, SWT.NONE);
		labe.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
		labe.setText("HTML file &type extension:");
		createExtensionViewer(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		// Mapped URL definication
		labe = new Label(base, SWT.NONE);
		labe.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING));
		labe.setText("&Mapped URL:");
		createMappedUrlViewer(base)
			.setLayoutData(new GridData(GridData.FILL_BOTH));		
		
		// apply web project values to control
		reload();
		
		return base;
	}
	
	Control createExtensionViewer(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		
		base.setLayout(new GridLayout(2, false));
		extsListViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 3;
		extsListViewer.getControl().setLayoutData(gd);
		extsListViewer.setContentProvider(new ArrayContentProvider());
		extsListViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean selectedItem = !event.getSelection().isEmpty();
				editExtButton.setEnabled(selectedItem);
				removeExtButton.setEnabled(selectedItem);
			}
		});
		
		addExtButton = createButton(base, "&Add");
		editExtButton = createButton(base, "&Edit");
		removeExtButton = createButton(base, "&Remove");
		
		return base;
	}
	
	Control createMappedUrlViewer(Composite parent){
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(2, false));
		
		mappedUrlViewer = new ListViewer(base);
		GridData gd = new GridData(GridData.FILL_BOTH);
		gd.verticalSpan = 3;
		mappedUrlViewer.getControl().setLayoutData(gd);
		mappedUrlViewer.setContentProvider(new ArrayContentProvider());
		mappedUrlViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				boolean selectedItem = !event.getSelection().isEmpty();
				editUrlButton.setEnabled(selectedItem);
				removeUrlButton.setEnabled(selectedItem);
			}
		});
		
		addUrlButton = createButton(base, "A&dd");
		editUrlButton = createButton(base, "Ed&it");
		removeUrlButton = createButton(base, "Re&move");
		
		return base;
	}
	
	
	Button createButton(Composite parent, String text){
		Button button = new Button(parent, SWT.PUSH);
		button.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.HORIZONTAL_ALIGN_FILL));
		button.setText(text);
		button.addListener(SWT.Selection, buttonListener);
		return button;
	}

	String inputExtension(String initialExt){
		InputDialog dialog = new InputDialog(
			getShell(),
			"File Extension",
			"Input HTML file type extension",
			initialExt,
			new FileExtensionValidator());
		
		return (dialog.open() == InputDialog.OK) ? dialog.getValue() : null;
	}
	
	URL inputUrl(String initialUrl){
		InputDialog dialog = new InputDialog(
			getShell(),
			"URL",
			"Input project public document mapped url",
			initialUrl,
			null);
		
		return (dialog.open() == InputDialog.OK) ? toURL(dialog.getValue()) : null;
	}
	
	static URL toURL(String urlStr){
		try {
			return new URL(urlStr);
		} catch (MalformedURLException e) {
			try {
				return new File(urlStr).toURL();
			} catch (MalformedURLException e1) {
			}
		}
		return null;
	}

	void reload(){
		// init char code settings value
		setEncoding(webProject.getString(WebProject.ENCODING));
		
		extsListViewer.setInput( new ArrayList(Arrays.asList(webProject.getArray(WebProject.HT_EXTENSIONS))) );
		extsListViewer.refresh();
		
		mappedUrlViewer.setInput( new ArrayList(Arrays.asList(webProject.getArray(WebProject.MAPPED_URL))) );
		mappedUrlViewer.refresh();
	}
	
	void setEncoding(String encoding){
		int index = encodingsCombo.indexOf(encoding);
		if( index != -1){
			encodingsCombo.select(index);
		}else{
			encodingsCombo.add(encoding, 0);
			encodingsCombo.select(0);
		}
	}
	
	void handleAddExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		Object obj = inputExtension(null);
		if(obj != null){
			Object sel = ((IStructuredSelection)extsListViewer.getInput()).getFirstElement();
			if(sel == null){
				list.add(obj);
			}else{
				list.add(list.indexOf(sel), obj);
			}
		}
		extsListViewer.refresh();
	}
	
	void handleRemoveExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		list.removeAll( ((IStructuredSelection)extsListViewer.getSelection()).toList() );
		extsListViewer.refresh();
	}
	
	void handleEditExtension(){
		ArrayList list = (ArrayList)extsListViewer.getInput();
		String sel = (String)((IStructuredSelection)extsListViewer.getSelection()).getFirstElement();
		String newExt = inputExtension(sel);
		if( newExt != null){
			int index = list.indexOf(sel);
			list.remove(index);
			list.add(index, newExt);
			extsListViewer.refresh();
		}
	}
	
	void handleAddUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		
		URL url = inputUrl(null);
		if(url != null){
			Object sel = ((IStructuredSelection)mappedUrlViewer.getSelection()).getFirstElement();
			if(sel == null){
				list.add(url.toString());
			}else{
				list.add(list.indexOf(sel), url.toString());
			}
			mappedUrlViewer.refresh();
		}
	}
	
	void handleRemoveUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		list.removeAll( ((IStructuredSelection)mappedUrlViewer.getSelection()).toList() );
		mappedUrlViewer.refresh();
	}
	
	void handleEditUrl(){
		ArrayList list = (ArrayList)mappedUrlViewer.getInput();
		String sel = (String)((IStructuredSelection)mappedUrlViewer.getSelection()).getFirstElement();
		URL newUrl = inputUrl(sel);
		
		if( newUrl != null){
			int index = list.indexOf(sel);
			list.remove(index);
			list.add(index, newUrl.toString());
			mappedUrlViewer.refresh();
		}
	}

	protected void performDefaults() {
		setEncoding(webProject.getString(WebProject.ENCODING));
		extsListViewer.setInput( new ArrayList(Arrays.asList(webProject.getDefaultArray(WebProject.HT_EXTENSIONS))) );
		extsListViewer.refresh();
		
		mappedUrlViewer.setInput( new ArrayList(Arrays.asList(webProject.getDefaultArray(WebProject.MAPPED_URL))) );
		mappedUrlViewer.refresh();
	}

	public boolean performOk() {
		webProject.setValue(
			WebProject.HTSOURCES_FOLDER,
			block.getPath(WebProject.HTSOURCES_FOLDER));
		
		webProject.setValue(
			WebProject.SCRIPTS_FOLDER,
			block.getPath(WebProject.SCRIPTS_FOLDER));

		webProject.setValue(
			WebProject.PUBLISH_FOLDER,
			block.getPath(WebProject.PUBLISH_FOLDER));

		webProject.setValue(WebProject.ENCODING, encodingsCombo.getText() );
		
		ArrayList exts = ((ArrayList)extsListViewer.getInput());
		webProject.setValue(WebProject.HT_EXTENSIONS, (String[])exts.toArray(new String[exts.size()]) );
		
		ArrayList urls = (ArrayList)mappedUrlViewer.getInput();
		webProject.setValue(WebProject.MAPPED_URL, (String[])urls.toArray(new String[urls.size()]) );
		
		try {
			webProject.storePreferences();
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		}
		
		return true;
	}

}
