/*
 * $Id: PublishMappingPage.java,v 1.7 2004/06/07 06:46:16 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.properties;

import java.io.IOException;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.IWorkbenchPropertyPage;
import org.eclipse.ui.dialogs.PropertyPage;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.IllegalConfigurationException;
import com.narucy.webpub.ui.WebpubUIPlugin;


/**
 * PublishDescriptorDefinitionPage provides publish description editing ui.
 * @see PropertyPage
 */
public class PublishMappingPage extends PropertyPage implements IWorkbenchPropertyPage {

	IFile editTarget;

	Button
		noPublishButton,
		useParentButton,
		openParentButton,
		useCurrentButton;
	
	PublishMappingBlock block;

	SelectionListener radioSelectionListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e) {
			refreshControls();
		}

		public void widgetDefaultSelected(SelectionEvent e) {}
	};

	public PublishMappingPage() {
	}
	
	protected Control createContents(Composite parent)  {
		final IFolder folder = (IFolder)getElement().getAdapter(IFolder.class);
		editTarget = folder.getFile(".publish");
		
		boolean fileExist = editTarget.exists();
		
		// create controls
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(1, false));
		
		noPublishButton = createButton(base, "&No Publish");
		
		useParentButton = createButton(base, "Use &Parent Setting");
		openParentButton = new Button(base, SWT.PUSH);
		openParentButton.setText("&Open Parent Description");
		
		// init button state for publish description state.
		if(hasPublishSourceMember()){
			noPublishButton.setSelection(!fileExist);
			useParentButton.setEnabled(false);
			openParentButton.setEnabled(false);
		}else{
			useParentButton.setSelection(!fileExist);
			noPublishButton.setEnabled(false);

			final IFile mapFile = folder.getFile(".publish");
			if(mapFile.exists()){
				openParentButton.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						openPublishMappingDialog(mapFile, getShell());
					}
				});
			}else{
				openParentButton.setEnabled(false);
			}
		}
		
		// define in current folder settings control
		
		useCurrentButton = createButton(base, "Use &Current Folder Settings");
		useCurrentButton.setSelection(fileExist);
		block = new PublishMappingBlock(base, editTarget, new IStatusChangeListener() {
			public void statusChanged(IStatus status) {
				setErrorMessage( status.getSeverity() == IStatus.ERROR ? status.getMessage() : null);
			}
		});
		block.getControl().setLayoutData( new GridData(GridData.FILL_BOTH));
		
		new Text(base, SWT.READ_ONLY | SWT.MULTI | SWT.WRAP)
			.setText("Note: If you selection parent setting, remove a publish description file of this folder.");

		refreshControls();
		return base;
	}
	
	boolean hasPublishSourceMember(){
		try{
			IContainer folder = editTarget.getParent();
			WebProject wp = (WebProject)folder.getProject().getNature(WebProject.ID_NATURE);
			IResource[] files = folder.members();
			for (int i = 0; i < files.length; i++) {
				if(files[i] instanceof IFile && wp.getPublishDescriptionFactory().create((IFile)files[i]) != null ){
					return true;
				}
			}
		}catch(CoreException ex){
			WebpubUIPlugin.handleException(ex);
		} catch (IOException ex) {
			WebpubUIPlugin.handleException(ex);
		} catch (IllegalConfigurationException ex) {
			WebpubUIPlugin.handleException(ex);
		}
		return false;
	}
	
	void initButtonStatue(){
		
	}

	Button createButton(Composite base, String name){
		Button button = new Button(base, SWT.RADIO);
		button.addSelectionListener(radioSelectionListener);
		button.setText(name);
		return button;
	}
	
	void refreshControls(){
		block.setEnabled( useCurrentButton.getSelection() );
	}
	
	/**
	 * Open a publish descriptor definication dialog.
	 * Specifiy publish description file is modifing target for user.
	 */
	public static void openPublishMappingDialog(final IFile pubDescFile, Shell parentShell){
		Dialog dialog = new Dialog(parentShell) {
			
			PublishMappingBlock block;
			
			protected Point getInitialSize(){
				Point p = super.getInitialSize();
				return new Point( (int)(p.x * 1.25), (int)(p.y * 1.5) );
			}

			protected Control createDialogArea(Composite parent) {
				Composite base = (Composite)super.createDialogArea(parent);
				
				Shell newShell = base.getShell();
				newShell.setText("Publish Descriptor Definition - " + pubDescFile.getFullPath());
				
				base.setLayout(new FillLayout());
				block = new PublishMappingBlock(base, pubDescFile, null);
				
				return base;
			}
			
			protected void okPressed() {
				try {
					block.distribute();
				} catch (CoreException e) {
					WebpubUIPlugin.handleException(e);
				}
				super.okPressed();
			}
			
			protected int getShellStyle(){
				return super.getShellStyle() | SWT.MAX | SWT.RESIZE;
			}
		};
		
		dialog.open();
	}

	/**
	 * Store the edited document instance.
	 */
	public boolean performOk() {
		super.performOk();
		try{
			if( useCurrentButton.getSelection() ){
				return block.distribute();
			}else{
				IFile target = block.getEditTarget();
				if( target != null && target.exists() ){
					target.delete(true, null);
				}
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		return true;
	}

}
