package com.narucy.webpub.ui.preferences;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.*;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.preference.*;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.*;

import com.narucy.webpub.core.*;
import com.narucy.webpub.ui.WebpubUIPlugin;

/**
 * RubyInterpreterPreferencePage
 */
public class RubyInterpreterPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

	final static String[]
		COLUMNS = {"Name", "Message", "State"};
	
	InterpreterCheckerManager checkerManager = null;
	
	Button interpreterBrowseButton, useRdtInterpreterButton;
	Text interpreterLocation;
	
	Table infoTable;
	Text infoNote;
	
	public RubyInterpreterPreferencePage(){
		setPreferenceStore(WebpubUIPlugin.getDefault().getPreferenceStore());
	}
	
	public void init(IWorkbench workbench) {
	}

	protected Control createContents(Composite parent) {
		Composite base = new Composite(parent, SWT.NONE);
		
		base.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		base.setLayout(new GridLayout(1, false));
		
		createInterpreterSelectionControl(base);
		
		new Label(base, SWT.NONE).setText("Ruby infomation:");
		
		// create ruby infomation state.
		infoTable = new Table(base, SWT.V_SCROLL | SWT.H_SCROLL | SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION);
		infoTable.setData("RubyInterpreterPreferencePage.infoTable");
		for(int i=0; i<COLUMNS.length; i++){
			TableColumn col = new TableColumn(infoTable, SWT.LEFT);
			col.setText(COLUMNS[i]);
		}
		infoTable.setLinesVisible(true);
		infoTable.setHeaderVisible(true);
		infoTable.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event ev) {
				TableItem item = (TableItem)ev.item;
				String key = item.getText();
				String desc = checkerManager.getDescription(key);
				setNoteText(desc);
			}
		});
		
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.heightHint = 100;
		infoTable.setLayoutData(gd);
		
		// create note
		infoNote = new Text(base, SWT.READ_ONLY | SWT.NO_FOCUS | SWT.MULTI | SWT.READ_ONLY | SWT.WRAP);
		infoNote.setData("RubyInterpreterPreferencePage.infoNote");
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.heightHint = 100;
		infoNote.setLayoutData(gd);
		
		// initial values
		load();
		
		return base;
	}
	
	/**
	 * Create ruby interperter setting controls.
	 */
	Composite createInterpreterSelectionControl(Composite parent){
		checkerManager = new InterpreterCheckerManager();
		
		Composite base = new Composite(parent, SWT.NONE);
		base.setLayout(new GridLayout(3, false));
		base.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		
		new Label(base, SWT.NONE).setText("Interpreter:");
		
		interpreterLocation = new Text(base, SWT.BORDER);
		interpreterLocation.setData("RubyInterpreterPreferencePage.interpreterLocation");
		interpreterLocation.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		interpreterLocation.addListener(SWT.Modify, new Listener() {
			public void handleEvent(Event ev) {
				interpreterCheck();
			}
		});
		
		interpreterBrowseButton = new Button(base, SWT.PUSH);
		interpreterBrowseButton.setText("&Browse...");
		interpreterBrowseButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event ev) {
				callInterperterBrowseDialog();
			}
		});
		interpreterBrowseButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		
		new Label(base, SWT.NONE);
		new Label(base, SWT.NONE);
		
		useRdtInterpreterButton = new Button(base, SWT.PUSH);
		useRdtInterpreterButton.setText("Use &RDT Settings");
		useRdtInterpreterButton.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event ev) {
				useRdtInterpreterLocation();
			}
		});
		
		useRdtInterpreterButton.setEnabled(WebpubUIPlugin.getRDTSelectedInterpreterLocation() != null);
		useRdtInterpreterButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		
		return base;
	}

	void callInterperterBrowseDialog(){
		FileDialog dialog = new FileDialog(getShell(), SWT.OPEN);
		
		String commandName = System.getProperty("os.name").indexOf("Windows") != -1 ? "rubyw.exe" : "ruby";
		dialog.setFilterNames(new String[]{"Ruby Interpreter (" + commandName + ")"});
		dialog.setFilterExtensions(new String[]{commandName});
		
		String file = dialog.open();
		if( file != null ){
			interpreterLocation.setText(file);
		}
	}

	void useRdtInterpreterLocation(){
		String path = WebpubUIPlugin.getRDTSelectedInterpreterLocation();
		if( path != null){
			interpreterLocation.setText(path);
		}
	}

	void setNoteText(String text){
		infoNote.setText( text != null ? "Note: " + text : "");
	}
	
	void interpreterCheck() {
		try {
			final String interpreterPath = interpreterLocation.getText();
			
			InterpreterCheckResult result = checkerManager.fastCheck(interpreterPath);
			if(result != null){
				applyErrorResult(result);
				return;
			}
			
			new ProgressMonitorDialog(getShell()).run(false, true, new IRunnableWithProgress(){
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException{
					monitor.beginTask("checking ruby interpreter enviornment", 1);
					try{
						setErrorMessage(null);
						infoTable.removeAll();
						try {
							applyErrorResult( checkerManager.check(interpreterPath) );
						} catch (CoreException e) {
							setErrorMessage(e.getLocalizedMessage());
						}
					}finally{
						monitor.done();
					}
				}
			});
		}
		catch (InvocationTargetException e) {
			setErrorMessage(e.getMessage());
		}
		catch (InterruptedException e) {
			setErrorMessage(e.getMessage());
		}
	}
	
	void applyErrorResult(InterpreterCheckResult result){
		TableItem[] items = infoTable.getItems();
		for (int i = 0; i < items.length; i++) {
			items[i].dispose();
		}
		String[] keys = result.getKeys();
		for (int i = 0; i < keys.length; i++) {
			String k = keys[i];
			String[] r = { k, result.getMessage(k), toStateString(result.getState(k))};
			new TableItem(infoTable, SWT.NONE).setText(r);
		}
		TableColumn[] column = infoTable.getColumns();
		for (int i = 0; i < column.length; i++) {
			column[i].pack();
		}
	}
	
	static String toStateString(int v){
		switch (v) {
			case NONE:
				return "OK";
			case INFORMATION:
				return "Infomation";
			case WARNING:
				return "Warning";
			case ERROR:
				return "Error";
			default :
				throw new IllegalArgumentException("Invalid value:" + v);
		}
	}

	public boolean performOk() {
		IPreferenceStore store = getPreferenceStore();
		String path = interpreterLocation.getText();
		store.setValue(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION, path);
		
		return true;
	}

	protected void performDefaults() {
		IPreferenceStore store = getPreferenceStore();
		interpreterLocation.setText(
			store.getDefaultString(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION) );
	}
	
	void load(){
		IPreferenceStore store = getPreferenceStore();
		interpreterLocation.setText(
			 store.getString(WebpubUIPlugin.RUBY_INTERPRETER_LOCATION) );
	}
	

}
