/*
 * $Id: TemplateManager.java,v 1.8 2004/06/07 06:46:37 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.template;

import java.io.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * <p>
 * TemplateManager roles are to keep template datas for a project,
 * and modify templates.
 */
public class TemplateManager {

	// project par one instance.

	final public static String
		MARKER = WebpubPlugin.ID_PLUGIN + ".template.marker",
		MARKER_PARSE = WebpubPlugin.ID_PLUGIN + ".template.parsermarker",
		MARKER_SYNCMISS = WebpubPlugin.ID_PLUGIN + "template.syncmissmarker";

	static HashMap registedProjects = new HashMap();

	TemplateParser parser = new TemplateParser();
	WebProject webProject;

	HashMap
		globalTemplates = new HashMap(),
		templateDefinedLocation = new HashMap();
	
	ArrayList listeners = new ArrayList();

	
	public TemplateManager(WebProject webProj) throws CoreException {
		webProject = webProj;
		researchAll(null);
	}

	public WebProject getWebProject() {
		return webProject;
	}
	
	static IMarker getMarker(IResource res, String type) throws CoreException {
		IMarker[] markers = res.findMarkers(type, false, IResource.DEPTH_ZERO);
		return markers.length > 0 ? markers[0] : res.createMarker(type);
	}

	/**
	 * Template defined files modifier for synchronize in project templates
	 * consistent.
	 */
	class ModifyTemplateProc implements IWorkspaceRunnable {
	
		IFile[] checksFiles;
		ArrayList distedFiles = new ArrayList();
		
		ModifyTemplateProc(IFile[] checksFiles) throws CoreException{
			this.checksFiles = checksFiles;
		}
		
		public void run(IProgressMonitor monitor) throws CoreException {
			monitor.beginTask("Research templates", 10000);
			try{
				int ratio = 10000 / checksFiles.length;
				for (int i = 0; i < checksFiles.length; i++) {
					IFile f = checksFiles[i];
					if( f.isAccessible()){
						try {
							autoModify(f, new SubProgressMonitor(monitor, ratio));
							addGlobalTemplate(f);
							deleteMakers(f);
						} catch (TemplateParseException e) {
							handleParseException(f, e);
						} catch (TemplateManageException e) {
							handleManageException(f, e);
						} catch( IOException e){
							throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, e.getMessage(), e));
						}
					}
				}
			}
			finally{
				monitor.done();
			}
		}

		void autoModify(IFile file, IProgressMonitor monitor) throws TemplateParseException, IOException, CoreException, TemplateManageException{
			monitor.beginTask(
				"Modify that content changed file has template that relational resources",
				1);
			
			try {
				IProject proj = file.getProject();
				if( !file.isSynchronized(IResource.DEPTH_ZERO) ){
					file.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, 1));
				}
				Template[] maybeModifiedTemplates = parser.parse(
					file.getContents(),
					webProject.getString(WebProject.ENCODING) ).getAllTemplates();

				// search modify target files.
				ArrayList relationFiles = new ArrayList();
				ArrayList changeTemplates = new ArrayList();
				tmplLoop: for (int i=0; i<maybeModifiedTemplates.length; i++) {
					Template maybeNewTmpl = maybeModifiedTemplates[i];
					
					// ignore template type is jump.
					String name = maybeNewTmpl.getName();
					if( name.equals(Template.NAME_CONTENTS) ){
						continue;
					}
					
					// search changed tmplates.
					Template globalSideTmpl = getGlobalTemplate(name);
					if( globalSideTmpl == null){
						// new template
						globalTemplates.put(name, maybeNewTmpl);
						templateDefinedLocation.put(name, new ArrayList());
					} else if( !globalSideTmpl.equals(maybeNewTmpl) ){
						// If changed file has templates is different with global template
						// and, already changed flags is on, batting a changed file, thrown
						// exception.
						for(int j=0; j<changeTemplates.size(); j++){
							Template t = (Template)changeTemplates.get(j);
							if( t.getName().equals(name) ){
								if(!t.equals(maybeNewTmpl)){
									IMarker marker = getMarker(file, MARKER_SYNCMISS);
									marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
									marker.setAttribute(IMarker.MESSAGE, "Updated template found more two points in same template name:" + name);
									throw new TemplateManageException(file, name);
								}
								continue tmplLoop;
							}
						}
						// modify template. get modify target template relational files.
						IFile[] definedFiles = getTemplateDefinedFiles(name);
						for (int j=0; j<definedFiles.length; j++) {
							IFile f = definedFiles[j];
							if( !relationFiles.contains(f) ){
								relationFiles.add(f);
							}
						}
						changeTemplates.add(maybeNewTmpl);
					}
				}
				
				// regist to global template.
				if( !changeTemplates.isEmpty() && !relationFiles.isEmpty() ){
					Template[] changeTmpls = (Template[])changeTemplates.toArray(new Template[changeTemplates.size()]);
					for (int i = 0; i < changeTmpls.length; i++) {
						Template t = changeTmpls[i];
						globalTemplates.put( t.getName(), t);
					}
					doModify(
						(IFile[])relationFiles.toArray(new IFile[relationFiles.size()]),
						changeTmpls,
						new SubProgressMonitor(monitor, 1));
				}
			} finally {
				monitor.done();
			}
		}
		
		IFile[] getDistributedFiles(){
			return (IFile[])distedFiles.toArray( new IFile[distedFiles.size()]);
		}
		
		void doModify(IFile[] relFiles, Template[] changeTmpls, IProgressMonitor monitor) throws TemplateParseException, IOException, CoreException{
			monitor.beginTask("Do modify for template synchronization", 1000);
			try{
				for (int i = 0, task = 1000 / relFiles.length; i < relFiles.length; i++) {
					IFile f = relFiles[i];
					deleteMakers(f);
					
					Template tmpl = parser.parse(f.getContents(), webProject.getString(WebProject.ENCODING));
					for (int j = 0; j < changeTmpls.length; j++) {
						tmpl.modifyTmpl( changeTmpls[j] );
					}
					distribute(f, tmpl);
					distedFiles.add(f);
					f.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, task));
	        	}
			}finally{
				monitor.done();
			}
		}
		
		void p(Object o) {
			System.out.println(getClass() + ":" + o);
		}
	}
	
	/**
	 * Delete after appends attributes in template manager.
	 */
	static void deleteMakers(IResource res) throws CoreException{
		IMarker[] markers = res.findMarkers(MARKER , true, IResource.DEPTH_ZERO);
		for (int j = 0; j < markers.length; j++) {
			markers[j].delete();
		}
	}
	
	void handleParseException(IFile file, TemplateParseException ex) throws CoreException{
		IMarker marker = getMarker(file, MARKER_PARSE);
		marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
		marker.setAttribute(IMarker.MESSAGE, ex.getMessage());
		marker.setAttribute(IMarker.LINE_NUMBER, ex.getLineNum());
	}
	
	void handleManageException(IFile file, TemplateManageException ex) throws CoreException{
		IMarker marker = getMarker(file, MARKER_SYNCMISS);
		marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
		marker.setAttribute(IMarker.MESSAGE, ex.getMessage());
	}
	
	void distribute(IFile file, Template tmpl) throws IOException, CoreException {
		PrintWriter writer = null;
		try {
			String encoding = webProject.getString(WebProject.ENCODING);
			writer = new PrintWriter(new OutputStreamWriter(new FileOutputStream(file.getLocation().toFile()),encoding));
			String[] lines = tmpl.getLines();
			for (int i = 0; i < lines.length; i++) {
				writer.println(lines[i]);
			}
		}finally{
			if(writer != null){
				writer.close();
			}
		}
	}
	
	/**
	 * modify templates cache.
	 */
	public IFile[] syncronizeTemplate(IFile[] changedFiles, IFile[] removedFile, IProgressMonitor monitor) throws CoreException {
		if(monitor == null){
			monitor = new NullProgressMonitor();
		}
		monitor.beginTask("Search Template", 3);
		try{
			IFile[] distributed;
			for(int i=0; i<removedFile.length; i++){
				removeGlobalTemplate(removedFile[i] );
			}
			monitor.worked(1);
			
			if( changedFiles.length > 0){
				ModifyTemplateProc proc = new ModifyTemplateProc(changedFiles);
				ResourcesPlugin.getWorkspace().run(proc, new SubProgressMonitor( monitor, 1) );
				distributed = proc.getDistributedFiles();
			}else{
				distributed = new IFile[0];
			}
			monitor.worked(1);
			
			fireTemplateModified();
			
			return distributed;
		}finally{
			monitor.done();
		}
	}
	
	/**
	 * Throws CoreException that thrown when in occse search html template
	 * resoruces.
	 */
	public void researchAll(IProgressMonitor monitor) throws CoreException{
		if(monitor == null){
			monitor = new NullProgressMonitor();
		}
		monitor.beginTask("researching templates", IProgressMonitor.UNKNOWN);
		try{
			IFile[] files = webProject.getFileMembers(WebProject.HTSOURCES_FOLDER);
			for (int i = 0; i < files.length; i++) {
				IFile f = files[i];
				try {
					if( webProject.isHTExtension(f.getFileExtension()) && f.exists() ){
						f.refreshLocal(IResource.DEPTH_ZERO, new SubProgressMonitor(monitor, 1));
						
						deleteMakers(f);
						addGlobalTemplate(f);
						
						monitor.worked(1);
					}
				} catch (TemplateManageException ex) {
					handleManageException(f, ex);
				} catch (TemplateParseException ex) {
					handleParseException(f, ex);
				} catch (IOException e) {
					throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, e.getMessage(), e));
				}
			}
			fireTemplateModified();
		}
		finally{
			monitor.done();
		}
	}

	/**
	 * Load templat from speicfy file into global templates stack.
	 */
	void addGlobalTemplate(IFile file) throws TemplateManageException, TemplateParseException, IOException, CoreException {
		
		HashMap notifyTargets = new HashMap();
		
		Template rootTmpl = parser.parse( file.getContents(), webProject.getString(WebProject.ENCODING) );
		Template[] tmpls = rootTmpl.getAllTemplates();
		
		for (int i=0; i<tmpls.length; i++) {
			Template tmpl = tmpls[i];
			String name = tmpl.getName();
			// none addition name of CONTENTS.
			if( name.equals(Template.NAME_CONTENTS)){
				continue;
			}
			if( !isContainGlobalTemplate(name) ){
				globalTemplates.put(name, tmpl.clone() );
				templateDefinedLocation.put(name, new ArrayList() );
			}
			List list = (List)templateDefinedLocation.get(name);
			if( !list.contains(file)){
				list.add(file);
			}
			// if same name template but contain is different,
			// raise exception.
			Template addedTmpl = getGlobalTemplate(name);
			if( !addedTmpl.equals(tmpl)){
				throw new TemplateManageException(file, name);
			}
		}
	}
	
	/**
	 * Remove referer target specify file form templates cache.
	 */
	void removeGlobalTemplate(IFile file){
		Template[] tmpls = getTemplates(file);
		for (int i = 0; i < tmpls.length; i++) {
			Template template = tmpls[i];
			
			String tmplName = template.getName();
			List files = (List)templateDefinedLocation.get(tmplName);
			files.remove(file);
			if( files.size() == 0){
				globalTemplates.remove(tmplName);
				templateDefinedLocation.remove(tmplName);
			}
		}
	}
	
	/**
	 * Returns specifed file has template.
	 * None parse process, get from on memory cache.
	 */
	public Template[] getTemplates(IFile file){
		ArrayList tmpls = new ArrayList();
		Template[] globalTmpls = getGlobalTemplates();
		for (int i = 0; i < globalTmpls.length; i++) {
			Template tmpl = globalTmpls[i];
			IFile[] definedFiles = getTemplateDefinedFiles(tmpl.getName());
			for (int j = 0; j < definedFiles.length; j++) {
				if( definedFiles[j].equals(file) ){
					tmpls.add(tmpl);
				}
			}
		}
		return (Template[])tmpls.toArray(new Template[tmpls.size()]);
	}

	/**
	 * Returns is resisted specify template name in this template manager.
	 */
	public boolean isContainGlobalTemplate(String name){
		// generally both HashMap instances are same.
		return globalTemplates.containsKey(name) && templateDefinedLocation.containsKey(name);
	}
	
	public Template getGlobalTemplate(String name){
		return (Template)globalTemplates.get(name);
	}
	
	public String[] getGlobalTemplateNames(){
		return (String[])globalTemplates.keySet().toArray(new String[globalTemplates.size()]);
	}
	
	public Template[] getGlobalTemplates(){
		return (Template[])globalTemplates.values().toArray(new Template[globalTemplates.size()]);
	}
	
	public IFile[] getTemplateDefinedFiles(String name){
		List files = (List)templateDefinedLocation.get(name);
		return (files != null) ? (IFile[])files.toArray( new IFile[files.size()] ) : new IFile[0];
	}

	public void addListener( TemplateManagerListener l){
		listeners.add(l);
	}
	
	public void removeListener( TemplateManagerListener l){
		listeners.remove(l);
	}
	
	protected void fireTemplateModified(){
		for (int i = 0; i < listeners.size(); i++) {
			((TemplateManagerListener)listeners.get(i)).templateModified(webProject.getProject());
		}
	}	
	
	/**
	 * Return true if specified file is ignore html template search target
	 * in web project.
	 */	
	public boolean isTemplateFile(IFile file) throws CoreException {
		// need to be a web project file.
		if(!webProject.isHTExtension(file.getFileExtension()) ){
			return false;
		}
		
		IContainer srcFolder = webProject.getFolder(WebProject.HTSOURCES_FOLDER);
		for(IContainer res = file.getParent(); !(res instanceof IProject); res = res.getParent() ){
			if ( res.equals(srcFolder) ) {
				return true;
			}
		}
		
		return false;
	}

}
