/*
 * $Id: SourceFileSeeker.java,v 1.14 2004/06/07 06:46:35 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.publish;
import java.io.File;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * PublishFromSeeker provides search publish from content file function.
 */
public class SourceFileSeeker {
	
	public static IFile findSource(String urlStr) throws CoreException {
		IFile f = urlToFile(urlStr);
		return f != null ? findSource(f) : null;
	}

	public static IFile findSource(URL url) throws CoreException {
		IFile f = urlToFile(url);
		return f != null ? findSource(f) : null;
	}
	
	public static IFile findSource(IFile f) throws CoreException {
		return new SourceFileSeeker(f).getPublishFrom();
	}
	
	IFile publishedFile;
	WebProject webProject = null;
	
	public static IFile urlToFile(String urlStr) throws CoreException {
		// for cgi value.
		urlStr = urlStr.replaceFirst("\\?.*", "");
		URL url = null;
		try {
			File f = new File(urlStr);
			if (f.exists()) {
				url = f.toURL();
			}
			url = new URL(urlStr);
		} catch (MalformedURLException e) {
		}
		return url != null ? urlToFile(url) : null;
	}
	
	/**
	 * URL to ht resource location. returns publish file.
	 */
	public static IFile urlToFile(URL url) throws CoreException {
		// to workspace file
		Path path = new Path(new File(url.getFile()).toString());
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(path);
		if (file != null) {
			return file;
		}
		
		// find from mapped url.
		String urlStr = url.toString();
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			WebProject wp = wps[i];
			IContainer pubFolder = wp.getFolder(WebProject.PUBLISH_FOLDER);
			String[] urls = wp.getArray(WebProject.MAPPED_URL);
			for (int j = 0; j < urls.length; j++) {
				String u = urls[j];
				if (urlStr.toLowerCase().startsWith(u.toLowerCase())) {
					String relPath = urlStr.substring(u.length());
					if (relPath.endsWith("/")) {
						String[] indexFileNames = wp.getArray(WebProject.DIRECTORY_INDEX);
						if(indexFileNames.length > 0){
							for (int k = 0; k < indexFileNames.length; k++) {
								IFile f = pubFolder.getFile(new Path(indexFileNames[k]));
								if(f.exists()){
									// if exist a file, return first.
									return f;
								}
							}
							// if not found accessible items, returns first directory index
							return pubFolder.getFile(new Path(indexFileNames[0]));
						}
					} else {
						// returns file
						return pubFolder.getFile(new Path(relPath));
					}
				}
			}
		}
		return null;
	}

	public SourceFileSeeker(IFile pubFile) throws CoreException {
		publishedFile = pubFile;
		
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			WebProject wp = wps[i];
			IContainer pubFolder = wp.getFolder(WebProject.PUBLISH_FOLDER);
			if (pubFolder.getFullPath().isPrefixOf(pubFile.getFullPath())) {
				webProject = wp;
				break;
			}
		}
	}

	/**
	 * <p>
	 * Returns publish source file of constructer specified url published file.
	 * <ul>
	 * <li>If specify file is not inclulded in publish folder of web project.
	 * <li>Next, search way to simple mapping "ht source folder"/"public
	 * folder".
	 * <li>If source file not found in this method work, search publish
	 * descriptor, and map a file.
	 * </ul>
	 */
	public IFile getPublishFrom() throws CoreException {
		if (webProject == null) {
			return null;
		}
		IPath publishFolderPath = webProject.getFolder(WebProject.PUBLISH_FOLDER).getFullPath();
		IPath sourcePath = publishedFile.getFullPath();
		if (!publishFolderPath.isPrefixOf(sourcePath) ){
			return null;
		}
		// first, direct match
		IPath relPath = sourcePath.removeFirstSegments(publishFolderPath.segmentCount());
		IContainer htFolder = webProject.getFolder(WebProject.HTSOURCES_FOLDER);
		IFile htSrc = htFolder.getFile(relPath);
		if (htSrc.exists()) {
			return htSrc;
		}
		// search by publish location specified mapppings.
		IFile f = getPublishFromByMappings(true);
		if(f == null){
			f = getPublishFromByMappings(false);
		}
		if(f != null){
			htSrc = f;
		}
		return htSrc;
	}
	
	IFile getPublishFromByMappings(boolean mustExist) throws CoreException {
		// if not found ht source mapping, search from publish descriptors.
		IFile[] publishMappingFiles = findPublishMappingFiles();
		for (int i = 0; i < publishMappingFiles.length; i++) {
			IFile mappingFile = publishMappingFiles[i];
			
			IContainer srcBaseFolder = mappingFile.getParent();
			WebProject sourceWebProject = (WebProject)srcBaseFolder.getProject().getNature(WebProject.ID_NATURE);
			IContainer pubFolder = webProject.getFolder(WebProject.PUBLISH_FOLDER);
			if(!isValidPublishResource(pubFolder)){
				// publish location is contain search target file.
				continue;
			}
			
			PublishMapping[] mappings = PublishMappingStore.getInstance().getMapping(mappingFile);
			for (int j = 0; j < mappings.length; j++) {
				PublishMapping mapping = mappings[j];
				IPath pubTo = mapping.getPublishTo();
				if (pubTo == null) {
					// ignore publish location is not specified. (already chacked direct match)
					continue;
				}
				// if publish to path last char is '/', it represents specify directory,
				// in search section, appends file name to publish to directory.
				if(pubTo.toString().endsWith("/")){
					pubTo = pubTo.append(publishedFile.getName());
				}
				
				// define publish base folder
				IPath pubToBaseFolderPath;
				
				IPath srcFolderPath = sourceWebProject.getFolder(WebProject.HTSOURCES_FOLDER).getFullPath();
				IPath srcBaseFolderPath = srcBaseFolder.getFullPath();
				if(srcFolderPath.isPrefixOf(srcBaseFolderPath) && !pubTo.isAbsolute()){
					IPath relPath = srcBaseFolderPath.removeFirstSegments(srcFolderPath.segmentCount());
					pubToBaseFolderPath = pubFolder.getFullPath().append(relPath);
				} else {
					pubToBaseFolderPath = pubFolder.getFullPath();
				}
				
				IPath publishFilePath = publishedFile.getFullPath();
				if(!pubToBaseFolderPath.isPrefixOf(publishFilePath)){
					// not match by container location
					continue;
				}
				
				// match file check
				IPath pubRel = publishFilePath.removeFirstSegments(pubToBaseFolderPath.segmentCount());
				String[][] pubToWildcardMatchResult = null;
				if(pubTo instanceof Wildcard){
					pubToWildcardMatchResult = ((Wildcard)pubTo).matchWithGroup(pubRel);
					if(pubToWildcardMatchResult == null){
						continue;
					}
				}else{
					if(!pubTo.makeAbsolute().equals(pubRel.makeAbsolute())){
						continue;
					}
				}
				// find matching publish source file
				IPath pattern = mapping.getMatchPattern();
				if(pattern instanceof Wildcard){
					if(pubToWildcardMatchResult != null){
						// source maching result apply to publish source matching pattern
						pattern = ((Wildcard)pattern).makePathByMatchResult(pubToWildcardMatchResult);
					}
					// makePathByMatchResult may returns Wildcard, if must exist flag is true
					// find exist resource by that returned wildcard.
					if(pattern instanceof Wildcard && mustExist){
						IFile[] files = WebProject.getFileMembers(srcBaseFolder);
						for (int k = 0; k < files.length; k++) {
							IFile f = files[k];
							IPath rp = f.getFullPath().removeFirstSegments(srcBaseFolderPath.segmentCount());
							if(((Wildcard)pattern).match(rp)){
								return f;
							}
						}
					}
				}
				IFile sourceFile = srcBaseFolder.getFile(pattern);
				if(!mustExist || sourceFile.exists()){
					return sourceFile;
				}
			}
		}
		
		return null;
	}
	
	int[] wildcardDoubleAsteriskIndex(String[][] matchResults){
		int begin = -1, end = -1;
		for(int i=0; i<matchResults.length; i++){
			String[] r = matchResults[i];
			if(r.length == 2 && r[1] == null){
				if(begin == -1){
					begin = i;
				}
			}else{
				if(begin != -1){
					return new int[]{begin, i};
				}
			}
		}
		return null;
	}
	
	boolean isValidPublishResource(IContainer pubFolder){
		// check the target 
		for(IContainer c = publishedFile.getParent(); !(c instanceof IWorkspaceRoot); c = c.getParent()){
			if(c.equals(pubFolder)){
				return true;
			}
		}
		return false;
	}
	
	IFile[] findPublishMappingFiles() throws CoreException {
		ArrayList dist = new ArrayList();
		collectPublishMappingFiles(webProject.getProject(), dist);
		IFile[] files = (IFile[]) dist.toArray(new IFile[dist.size()]);
		final IContainer htFolder = webProject.getFolder(WebProject.HTSOURCES_FOLDER);
		Arrays.sort(files, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
			public int compare(Object a, Object b) {
				IPath ap = ((IFile) a).getFullPath();
				IPath bp = ((IFile) b).getFullPath();
				
				IPath htFolderPath = htFolder.getFullPath();
				boolean inHtSourceA = htFolderPath.isPrefixOf(ap);
				boolean inHtSourceB = htFolderPath.isPrefixOf(bp);
				if (inHtSourceA != inHtSourceB) {
					return inHtSourceA ? 1 : -1;
				}
				int as = ap.segmentCount();
				int bs = bp.segmentCount();
				return (as != bs) ? (as < bs ? 1 : -1) : 0;
			}
		});
		return files;
	}
	
	void collectPublishMappingFiles(IContainer folder, List dist) throws CoreException {
		WebProject[] wps = WebProject.getWebProjects();
		for (int i = 0; i < wps.length; i++) {
			if (wps[i].getFolder(WebProject.PUBLISH_FOLDER).equals(folder)) {
				return;
			}
		}
		IResource[] resources = folder.members();
		for (int i = 0; i < resources.length; i++) {
			IResource r = resources[i];
			if (r instanceof IFile && ((IFile) r).getName().equals(".publish")) {
				dist.add(r);
			} else if (r instanceof IContainer) {
				collectPublishMappingFiles((IContainer) r, dist);
			}
		}
	}
	
}
