/*
 * $Id: ZippedPreset.java,v 1.5 2004/06/07 06:46:38 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.preset;

import java.io.*;
import java.net.URL;
import java.util.*;
import java.util.zip.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

public class ZippedPreset extends Preset {
	
	ZipFile zipFile;
	ZipEntry[] zipEntries;

	String rootZipEntryPath;
	
	public ZippedPreset(URL baseDirectory, String presetName, String description) throws CoreException {
		super(baseDirectory, presetName, description);
		Exception error = null;
		try{
			File file = new File( Platform.asLocalURL(baseDirectory).getFile() );
			zipFile = new ZipFile(file);
			
			// collect all zip entries
			ArrayList entries = new ArrayList();
			Enumeration enu = zipFile.entries();
			while (enu.hasMoreElements()) {
				ZipEntry entry = (ZipEntry)enu.nextElement();
				if(entry.isDirectory()){
					continue;
				}
				
				String p = entry.getName();
				
				if(p.indexOf(".webproject") != -1){
					rootZipEntryPath = p.substring(0, p.length() - ".webproject".length());
					InputStream stream = zipFile.getInputStream(entry);
					try{
						loadProperty(stream);
					}finally{
						stream.close();
					}
				}else{
					entries.add(entry);
				}
			}
			if(getWebProjectProperty() == null){
				error = new IllegalArgumentException("sepcify archive is not contain .webproject file:" + baseDirectory);
			}
			
			zipEntries = (ZipEntry[]) entries.toArray(new ZipEntry[entries.size()]);
		}catch(IOException e){
			error = e;
		}
		if(error != null){
			throw new CoreException(new Status(
					IStatus.ERROR,
					WebpubPlugin.ID_PLUGIN,
					IStatus.OK,
					"appearance error in read zip archive, can't read .webproject: " + baseDirectory,
					error));
		}
	}

	public Map getDistributeEntries(WebProject webProject){
		Map targets = new HashMap();
		Properties prop = getWebProjectProperty();
		
		String srcDir = rootZipEntryPath + prop.getProperty(WebProject.HTSOURCES_FOLDER);
		ZipEntry[] htSources = getEntries(srcDir);
		for (int i=0; i<htSources.length; i++) {
			ZipEntry entry = htSources[i];
			Path relPath = new Path( entry.getName().substring(srcDir.length()) );

			IContainer htSourceFolder = webProject.getFolder(WebProject.HTSOURCES_FOLDER);
			IResource r = htSourceFolder.getFile(relPath);
			if(!r.exists()){
				targets.put(entry, r);
			}
		}
		
		String scriptsDir = rootZipEntryPath + prop.getProperty(WebProject.SCRIPTS_FOLDER);
		ZipEntry[] scripts = getEntries(scriptsDir);
		for (int i=0; i < scripts.length; i++) {
			ZipEntry entry = scripts[i];
			Path relPath = new Path(entry.getName().substring(scriptsDir.length()));
			
			IContainer scriptsFolder = webProject.getFolder(WebProject.SCRIPTS_FOLDER);
			IResource r = scriptsFolder.getFile(relPath);
			if(!r.exists()){
				targets.put(entry, r);
			}
		}
		
		String pubDir = rootZipEntryPath + prop.getProperty(WebProject.PUBLISH_FOLDER);
		
		ZipEntry[] etcs = getEntries(rootZipEntryPath, new String[]{srcDir, scriptsDir, pubDir});
		for (int i=0; i<etcs.length; i++) {
			ZipEntry entry = etcs[i];
			String name = entry.getName();
			String path = name.substring(rootZipEntryPath.length());
			IProject proj = webProject.getProject();
			
			IResource r = proj.getFile(path);
			if(!r.exists()){
				targets.put(entry, r);
			}
		}
		
		return targets;
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	public ZipEntry[] getDistributeZipEntries(Map distTargets){
		ZipEntry[] froms = (ZipEntry[])distTargets.keySet().toArray(new ZipEntry[distTargets.size()]);
		Arrays.sort(froms, new Comparator() {
			public boolean equals(Object obj) {
				return false;
			}
			public int compare(Object a, Object b) {
				return ((ZipEntry)a).getName().compareTo(((ZipEntry)b).getName());
			}
		});
		return froms;
	}
	
	protected void distribute(WebProject webProject, IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("distribute project files", 10000);
		try{
			Map distTargets = getDistributeEntries(webProject);
			
			// copy scripts file
			ZipEntry[] froms = getDistributeZipEntries(distTargets);
			
			int ratio = 10000 / froms.length;
			for (int i=0; i < froms.length; i++) {
				ZipEntry entry = froms[i];
				IFile file = (IFile)distTargets.get(entry);
				copy( entry, file, new SubProgressMonitor(monitor, ratio) );
			}
		}catch(IOException e){
			throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, e.getMessage(), e));
		}finally{
			monitor.done();
		}
	}
	
	void copy(ZipEntry from, IFile to, IProgressMonitor monitor) throws IOException, CoreException{
		monitor.beginTask("distribute archive entry", 2);
		
		InputStream input = null;
		try{
			ArrayList parents = new ArrayList();
			for(IContainer c = to.getParent(); c instanceof IFolder && !c.exists(); c = c.getParent()){
				parents.add(c);
			}
			if(!parents.isEmpty()){
				SubProgressMonitor subMonitor = new SubProgressMonitor(monitor, 1);
				subMonitor.beginTask("create parents", parents.size());
				try{
					for(int i=parents.size()-1; i>=0; i--){
						((IFolder)parents.get(i)).create(true, true, new SubProgressMonitor(subMonitor, 1));
					}
				}finally{
					subMonitor.done();
				}
			}else{
				monitor.worked(1);
			}
			input = zipFile.getInputStream(from);
			to.create(input, true, new SubProgressMonitor(monitor, 1));
		}finally{
			try{
				if(input != null) input.close();
			}finally{
				monitor.done();
			}
		}
	}

	ZipEntry[] getEntries(String dirName){
		return getEntries(dirName, null);
	}
	
	ZipEntry[] getEntries(String dirName, String[] excludeDirs){
		ArrayList dist = new ArrayList();
		ZipEntryLoop: for (int i = 0; i < zipEntries.length; i++) {
			ZipEntry entry = zipEntries[i];
			String name = entry.getName();
			if(name.indexOf(dirName) == 0 && dirName.length() + 1 < name.length() ){
				if(excludeDirs != null){
					for(int j=0; j<excludeDirs.length; j++){
						if(entry.getName().indexOf(excludeDirs[j]) == 0){
							continue ZipEntryLoop;
						}
					}
				}
				dist.add(entry);
			}
		}
		return (ZipEntry[]) dist.toArray(new ZipEntry[dist.size()]);
	}

	public String getRootZipEntryPath() {
		return rootZipEntryPath;
	}
	
	public ZipFile getZipFile() {
		return zipFile;
	}
}