/*
 * $Id: FilePreset.java,v 1.5 2004/06/07 06:46:38 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.preset;

import java.io.*;
import java.net.URL;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * 
 */
public class FilePreset extends Preset {

	final static String[] ignoreFileNames = {
		"CVS",
	};

	static FileFilter filter = new FileFilter() {
		public boolean accept(File path) {
			return Arrays.binarySearch(ignoreFileNames, path.getName()) < 0;
		}
	};

	File rootDir;
	
	public FilePreset(URL baseDirectory, String name, String description) throws CoreException {
		super(baseDirectory, name, description);
		Exception error = null;
		try {
			rootDir = new File(Platform.asLocalURL(baseDirectory).getFile());
			FileInputStream stream = new FileInputStream(new File(rootDir, ".webproject"));
			try{
				loadProperty(stream);
			}finally{
				stream.close();
			}
		} catch (IOException e) {
			error = e;
		}
		if(getWebProjectProperty() == null){
			error = new IllegalArgumentException("specify directory is not contain .webproject file:" + baseDirectory);
		}
		if( error != null){
			throw new CoreException(new Status(IStatus.ERROR, WebpubPlugin.ID_PLUGIN, IStatus.OK, error.getMessage(), error));
		}
	}
	
	public Map getDistributeFiles(WebProject webProject){
		HashMap targets = new HashMap();
		
		Properties prop = getWebProjectProperty();
		File htSourcesDir = new File(rootDir, prop.getProperty(WebProject.HTSOURCES_FOLDER));
		IContainer htSourceFolder = webProject.getFolder(WebProject.HTSOURCES_FOLDER);
		
		File[] htSourceFiles = collectFiles(htSourcesDir);
		for(int i=0; i<htSourceFiles.length; i++){
			File f = htSourceFiles[i];
			Path rel = new Path(f.toString().substring(htSourcesDir.toString().length()));
			targets.put(f, htSourceFolder.getFile(rel));
		}
		
		File scriptsDir = new File(rootDir, prop.getProperty(WebProject.SCRIPTS_FOLDER));
		IContainer scriptsFolder = webProject.getFolder(WebProject.SCRIPTS_FOLDER);
		File[] scriptFiles = collectFiles(scriptsDir);
		for(int i=0; i<scriptFiles.length; i++){
			File f = scriptFiles[i];
			Path rel = new Path(f.toString().substring(scriptsDir.toString().length()));
			targets.put(f, scriptsFolder.getFile(rel));
		}
		
		File[] files = collectFiles(rootDir);
		for (int j=0; j<files.length; j++) {
			File f = files[j];
			if(!targets.containsKey(f)){
				String path = f.toString().substring(rootDir.toString().length());
				if(path.length() > 0){
					IProject proj = webProject.getProject();
					targets.put(f, proj.getFile(path));
				}
			}
		}
		
		return targets;
	}
	
	File[] collectFiles(File base){
		ArrayList dist = new ArrayList();
		collectFiles(base, dist);
		return (File[]) dist.toArray(new File[dist.size()]);
	}
	
	void collectFiles(File file, List dist){
		if(file.isFile()){
			dist.add(file);
		}
		if(file.isDirectory()){
			File[] files = file.listFiles(filter);
			for (int i = 0; i < files.length; i++) {
				collectFiles( files[i], dist);
			}
		}
	}

	protected void distribute(WebProject webProject, IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("distribute preset files", 10000);
		IProject proj = webProject.getProject();
		try{
			Map distTargets = getDistributeFiles(webProject);
			if(distTargets.isEmpty()){
				return;
			}
			Object[] orgFiles = distTargets.keySet().toArray();
			Arrays.sort(orgFiles);
			
			int ratio = 9000 / orgFiles.length;
			for (int i=0; i<orgFiles.length; i++) {
				File orgFile = (File)orgFiles[i];
				IFile distFile = (IFile)distTargets.get(orgFile);
				if(distFile.exists()){
					continue;
				}
				IContainer parent = distFile.getParent();
				if(parent instanceof IFolder && !parent.exists()){
					createFolder((IFolder)parent, new SubProgressMonitor(monitor, ratio / 2));
					ratio -= ratio / 2;
				}
				distFile.create(new FileInputStream(orgFile), true, new SubProgressMonitor(monitor, ratio));
				
			}
			proj.refreshLocal(IResource.DEPTH_INFINITE, new SubProgressMonitor(monitor, 1000));
			
		} catch (IOException e) {
			throw new CoreException(new Status(
					IStatus.ERROR,
					WebpubPlugin.ID_PLUGIN,
					IStatus.OK,
					"appearance error in file coping: " + baseDirectory,
					e));
		}finally{
			monitor.done();
		}
	}
	
	
}
