/*
 * $Id: Wildcard.java,v 1.13 2004/06/07 06:46:36 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core;

import java.util.regex.*;

import org.eclipse.core.runtime.*;

/**
 * Wildcard
 * "**" is incremental directory match. "?" is one char, * is zero or occurences pattern.
 */
public class Wildcard extends Path {

	final static Pattern INFINITY = null;

	Pattern[] matchPatterns;

	public Wildcard(IPath patternPath) throws PatternSyntaxException {
		this(patternPath.toString());
	}
	
	public Wildcard(String orgPattern) throws PatternSyntaxException {
		super(orgPattern);
		
		int segCount = segmentCount();
		matchPatterns = new Pattern[segCount];
		
		for(int i=0; i<segCount; i++){
			String ptn = segment(i);
			if(ptn.indexOf("**") != -1){
				if(ptn.length() == 2){
					for(int j=0; j<i; j++){
						if(matchPatterns[j] == INFINITY){
							throw new PatternSyntaxException("** mark must by only one segument", orgPattern, orgPattern.indexOf("**"));
						}
					}
					matchPatterns[i] = INFINITY;
				}else{
					throw new PatternSyntaxException("** mark didn't append other chars", orgPattern, orgPattern.lastIndexOf("**"));
				}
			}else{
				String p = ptn.
					replaceAll("\\.", "\\\\.").
					replaceAll("\\?", "(.?)").
					replaceAll("\\*", "(.*?)");
				
				matchPatterns[i] = Pattern.compile(p);
			}
		}
	}
	
	public String[][] matchWithGroup(String path){
		return matchWithGroup(new Path(path));
	}

	public String[][] matchWithGroup(IPath path){
		int segCount = path.segmentCount();
		if(segCount < matchPatterns.length){
			return null;
		}
		String[][] groupEntries = new String[segCount][];
		for (int i=0, j=0; i<segCount;) {
			String segEntry = path.segment(i);
			if(j < matchPatterns.length){
				Pattern ptn = matchPatterns[j++];
				if(ptn == INFINITY){
					while(i < segCount - (matchPatterns.length - j)){
						groupEntries[i++] = new String[]{ segEntry, null };
						segEntry = path.segment(i);
					}
				} else {
					Matcher m = ptn.matcher(segEntry);
					if (m.matches()){
						int groupCount = m.groupCount()+1;
						String[] entry = new String[groupCount];
						for(int k=0; k<groupCount; k++){
							entry[k] = m.group(k);
						}
						groupEntries[i++] = entry;
					}else{
						return null;
					}
				}
			}else{
				return null;
			}
		}
		return groupEntries;
	}
	
	public boolean match(String path){
		return match(new Path(path));
	}
	
	public boolean match(IPath path){
		return matchWithGroup(path) != null;
	}
	
	public IPath makePathByMatchResult(String[][] srcMatchResult){
		IPath path = new Path("");
		
		int len = toString().indexOf("**") != -1 ? srcMatchResult.length : Math.min(srcMatchResult.length, segmentCount());
		if(segmentCount() > len){
			path = removeLastSegments(len);
		}
		
		for(int i=len-1; i>=0; i--){
			String[] matchEntry = srcMatchResult[srcMatchResult.length-1-i];
			if(matchEntry.length == 2 && matchEntry[1] == null){
				path = path.append(matchEntry[0]);
				continue;
			}
			
			String seg = segment(segmentCount()-1-i);
			for(int j=1;  j<matchEntry.length && matchEntry[j] != null; j++){
				if(seg.indexOf('*') != -1){
					seg = seg.replaceFirst("\\*", matchEntry[j]);
				}else if (seg.indexOf('?') != -1){
					seg = seg.replaceFirst("\\?", matchEntry[j].length() > 0 ? matchEntry[j].substring(0,1) : "");
				}else {
					break;
				}
			}
			path = path.append(seg);
		}
		
		if(containWildcardSymble(path)){
			path = new Wildcard(path);
		}
		return path;
	}

	public static boolean containWildcardSymble(IPath path){
		return containWildcardSymble(path.toString());
	}
	
	public static boolean containWildcardSymble(String pathStr){
		return pathStr.indexOf('*') != -1 || pathStr.indexOf('?') != -1;
	}
	
	public boolean equals(Object obj) {
		return obj instanceof Wildcard &&  super.equals(obj);
	}

}
