/*
 * $Id: PublishDescriptionFactoryTest.java,v 1.13 2004/06/07 06:46:50 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests.publish;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.Path;

import com.narucy.webpub.core.*;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.core.tests.WebpubTestPlugin;

/**
 * 
 */
public class PublishDescriptionFactoryTest extends TestCase {

	public PublishDescriptionFactoryTest(String name) throws Exception {
		super(name);
	}
	
	IProject project;
	PublishDescriptionFactory fac;

	void createProject(String path) throws Exception {
		project = WebpubTestPlugin.createTestProject(path).getProject();
		fac = ((WebProject)project.getNature(WebProject.ID_NATURE)).getPublishDescriptionFactory();
	}
	
	public void testWildcardProject() throws Exception {
		createProject("test_scripts/webpub/wildcard_test_project");
		
		IFile file = project.getFile("ht_sources/a/a1/index.html");
		assertTrue(file.exists());
		PublishDescription desc = fac.create(file);
		assertEquals(new Wildcard("/a/a1/index?.html"), desc.getPublishTo());
		
		file = project.getFile("ht_sources/a/a2/index.html");
		assertTrue(file.exists());
		desc = fac.create(file);
		assertEquals(new Wildcard("/a/a2/index?.html"), desc.getPublishTo());
	}

	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	public void testCreate() throws Exception {
		createProject("test_scripts/webpub/test_workspace/web_project");
		
		// tests isPublish
		assertNull( fac.create(project.getFile("xxx/yyy/zzz/nofile") ) );
		assertNull( fac.create(project.getFile("ht_sources/nofile") ) );
		assertNull( fac.create(project.getFile("scripts/base.rb") ) );
		assertNull( fac.create(project.getFile("public_html/home.html") ) );
		assertNull( fac.create(project.getFile(".project") ) );
		assertNull( fac.create(project.getFile(".webproject") ) );

		assertNotNull( fac.create(project.getFile("ht_sources/home.html") ) );
		assertNotNull( fac.create(project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-1.html") ) );
		assertNotNull( fac.create(project.getFile("erbs/a.erb") ) );
		assertNotNull( fac.create(project.getFile("erbs/c/a/c-a-a.erb") ) );
		assertNotNull( fac.create(project.getFile("rd/a.rd") ) );
		assertNotNull( fac.create(project.getFile("rd/with_tmpl/c.rd") ) );
	}

	public void testUseWildcard() throws Exception {
		createProject("test_scripts/webpub/test_workspace/web_project");

		String[] files = {
			"erbs/a.erb",
			"erbs/b.erb",
			"erbs/c/a/c-a-a.erb",
			"erbs/c/a/c-a-b.erb",
		};
		
		PublishDescription[] descs = new PublishDescription[4];
		for (int i=0; i<files.length; i++) {
			IFile a = project.getFile(files[i]);
			descs[i] = fac.create(a);
			assertEquals("erb", descs[i].getPublishBy());
			assertTrue(descs[i].getArguments().isEmpty());
		}
		
		assertEquals(new Path("/erbs/a.html"), descs[0].getPublishTo());
		assertEquals(new Path("/erbs/b.html"), descs[1].getPublishTo());
		assertEquals(new Path("/erbs/c-a-a.html"), descs[2].getPublishTo());
		assertEquals(new Path("/erbs/c-a-b.html"), descs[3].getPublishTo());
	}
	
	/**
	 * test creation descriptor as a file.
	 */
	public void testCreateDescriptionFromFile() throws Exception {
		createProject("test_scripts/webpub/test_workspace/web_project");
		
		assertNull(fac.create(project.getFile("ht_sources/.publish")));
		assertNull(fac.create(project.getFile("ht_sources/CSV")));
		
		IFile file = project.getFile("ht_sources/home.html");
		assertTrue(file.exists());
		PublishDescription desc = fac.create(file);
		
		assertEquals("home.rb", desc.getArgument("script") );
		assertEquals("Home", desc.getArgument("class"));
		assertNull(desc.getArgument("method"));
		assertEquals( new Path("/home.html"), desc.getPublishTo() );
	}

	/**
	 * Create description form parents folder.
	 */	
	public void testCreateDescription() throws Exception {
		createProject("test_scripts/webpub/test_workspace/web_project");
		IFile file = project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-1.html");
		
		PublishDescription desc = fac.create(file);
		assertEquals("copy", desc.getPublishBy());
		assertEquals("aaa", desc.getArgument("opt_a"));
		assertEquals("a b c d e", desc.getArgument("opt_b"));
		
		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-2.html");
		desc = fac.create(file);
		assertEquals("amrita", desc.getPublishBy());
		assertTrue(desc.argumentsEquals(fac.create(file)));
		assertEquals(
				new Path("/folder/1/1/1/2.html"),
				desc.getPublishTo());
		
		desc = fac.create(project.getFile("ht_sources/folder/subfolder/subsubfolder/1-1-1-3.html"));
		assertEquals("amrita", desc.getPublishBy());
		assertEquals(
				new Path("/folder/3.html"),
				desc.getPublishTo());
		
		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/s-1.html");
		assertEquals("amrita", desc.getPublishBy());
		desc = fac.create(file);
		assertEquals("s.rb", desc.getArgument("script"));
		assertNull(desc.getArgument("class"));
		assertEquals("s", desc.getArgument("method"));
		assertEquals(
				new Path("/folder/subfolder/subsubfolder/s-1.html"),
				desc.getPublishTo());

		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/t-1.html");
		assertEquals("amrita", desc.getPublishBy());
		desc = fac.create(file);
		assertEquals("t.rb", desc.getArgument("script"));
		assertEquals("T", desc.getArgument("class"));
		assertNull(desc.getArgument("method"));
		assertEquals("x y z", desc.getArgument("opt"));
		assertEquals(
				new Path("/foo_folder/"),
				desc.getPublishTo());
		
		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/u-1.html");
		assertEquals("amrita", desc.getPublishBy());
		desc = fac.create(file);
		assertEquals("u.rb", desc.getArgument("script"));
		assertEquals("U", desc.getArgument("class"));
		assertEquals("u", desc.getArgument("method"));
		assertEquals(
				new Path("/uuu.html"),
				desc.getPublishTo());

		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/a.png");
		desc = fac.create(file);
		assertEquals("copy", desc.getPublishBy());
		assertEquals("A", desc.getArgument("opt"));
		assertEquals(
				new Path("/folder/subfolder/subsubfolder/a.png"),
				desc.getPublishTo());
		
		file = project.getFile("ht_sources/folder/subfolder/subsubfolder/b.png");
		desc = fac.create(file);
		assertEquals("copy", desc.getPublishBy());
		assertEquals("B", desc.getArgument("opt"));
		assertEquals(
				new Path("/folder/subfolder/subsubfolder/b.png"),
				desc.getPublishTo());
	}

	protected void tearDown() throws Exception {
		if(project != null){
			WebpubTestPlugin.forceDelete(project);
			project = null;
		}
	}

}
