/*
 * $Id: WebResourceLinkModelTest.java,v 1.7 2004/06/07 06:46:50 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.core.tests;

import junit.framework.TestCase;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;

import com.narucy.webpub.core.*;

/**
 * <ul>
 * <li>Input text
 * <li>If resource not found, not returned reource object,
 * and nodify warning
 * <li>If resource found, nodify error is not nothing and nodify
 * error is closed.
 * <li>If absolute enabled, transration path is absolute and disabled when
 * return relative path.
 * </ul>
 */
public class WebResourceLinkModelTest extends TestCase {

	WebProject webProject = null;
	
	public WebResourceLinkModelTest(String name) {
		super(name);
	}
	
	public void testMakeRelativePath() throws Exception {
		IPath pa = new Path("/root/a/b");
		IPath pb = new Path("/root/a/b/c/d");
		IPath pc = new Path("/root/a/x/y/z/y/x");
		
		int index = pa.matchingFirstSegments(pb);
		assertTrue(pa.segmentCount() == index);
		
		IPath pd = new Path("/root/a/../a/../b/../a/x/y/../../z/../x/y/k/../z/y/x");
		assertEquals( pd.removeTrailingSeparator(), pc);
		
		// relative path
		assertEquals(
			"../../../../../b/c/d",
			WebResourceLinkModel.makeRelativePath(pd, pb).toString() );
		
		assertEquals(
			"../../../../../b/c/d",
			WebResourceLinkModel.makeRelativePath(pc, pb).toString() );
	}

	/**
	 * Tests input work. WebResourceLinkModel must implements:
	 * 
	 * <ul>
	 * <li>nottify input result to user.
	 * <li>transrate relative path to absolute path and reverse that.
	 * </ul>
	 */
	public void testInput() throws Exception{
		webProject = WebpubTestPlugin.createTestProject("test_scripts/webpub/test_workspace/web_project");
		
		final int[] flags = new int[2];
		WebResourceLinkModel model = new WebResourceLinkModel(webProject.getProject().getFolder("public_html"));
		
		model.addListener(new WebResourceSelectionListener() {
			public void resourceChanged(IPath path, IResource existResource) {
				flags[0]++;
			}
			public void statusChanged(IStatus status) {
				flags[1]++;
			}
		});
		
		IProject proj = webProject.getProject();
		// first, path is not setted. return null.
		assertNull(model.getWebResource());
		assertNull(model.getPath());
		
		// tests get path
		model.setPath("home.html");
		assertEquals("home.html", model.getPath().toString());
		assertEquals(proj.getFile("public_html/home.html"), model.getWebResource());
		
		// test transrate path absolute, relative
		IFile file = proj.getFile("public_html/folder/subfolder/subsubfolder/1-1-1-1.html");
		
		model.setPath("folder/subfolder/subsubfolder/1-1-1-1.html");
		
		model.setAbsolute(true);
		assertTrue(model.isAbsolute());
		assertEquals("/folder/subfolder/subsubfolder/1-1-1-1.html", model.getPath().toString());
		assertEquals(file, model.getWebResource());
		
		model.setAbsolute(false);
		assertFalse(model.isAbsolute());
		assertEquals("folder/subfolder/subsubfolder/1-1-1-1.html", model.getPath().toString());
		assertEquals(file, model.getWebResource());
		
	}
	
	public void testRelativePath() throws Exception{
		webProject = WebpubTestPlugin.createTestProject(
				"test_scripts/webpub/test_workspace/web_project");
		
		final int[] flags = new int[2];
		IProject proj = webProject.getProject();
		WebResourceLinkModel model = new WebResourceLinkModel(
			proj.getProject().getFolder("ht_sources/folder/subfolder"));
		
		model.addListener(new WebResourceSelectionListener() {
			public void resourceChanged(IPath path, IResource existResource) {
				flags[0]++;
			}
			public void statusChanged(IStatus status) {
				flags[1]++;
			}
		});
		
		// up
		String[] pathes = new String[]{"../../home.html", "/home.html"};
		for(int i=0; i<2; i++){
			model.setPath(pathes[i]);
			
			IFile homeFile = proj.getFile("public_html/home.html");
			assertEquals(homeFile, model.getWebResource());
			
			model.setAbsolute(true);
			assertEquals("/home.html", model.getPath().toString());
			
			model.setAbsolute(false);
			assertEquals("../../home.html", model.getPath().toString() );
		}
		
		// down
		pathes = new String[]{"subsubfolder/1-1-1-1.html", "/folder/subfolder/subsubfolder/1-1-1-1.html"};
		for(int i=0; i<2; i++){
			model.setPath(pathes[i]);
			
			IFile homeFile = proj.getFile("public_html/folder/subfolder/subsubfolder/1-1-1-1.html");
			assertEquals(homeFile, model.getWebResource());
			
			model.setAbsolute(true);
			assertEquals("/folder/subfolder/subsubfolder/1-1-1-1.html", model.getPath().toString());
			
			model.setAbsolute(false);
			assertEquals("subsubfolder/1-1-1-1.html", model.getPath().toString() );
		}

		// up and down
		pathes = new String[]{"../../imgs/a.png", "/imgs/a.png"};
		for(int i=0; i<2; i++){
			model.setPath(pathes[i]);
			
			IFile imgFile = proj.getFile("public_html/imgs/a.png");
			assertEquals(imgFile, model.getWebResource());
			
			model.setAbsolute(true);
			assertEquals("/imgs/a.png", model.getPath().toString());
			
			model.setAbsolute(false);
			assertEquals("../../imgs/a.png", model.getPath().toString() );
		}
	}

	protected void tearDown() throws Exception {
		if(webProject != null){
			WebpubTestPlugin.forceDelete(webProject.getProject());
			webProject = null;
		}
	}

}
