/*
 * $Id: HTPreviewLinkWithEditorManager.java,v 1.7 2004/04/20 00:44:14 hn Exp $
 * Copyright Narushima Hironori. All rights reserved.
 */
package com.narucy.webpub.ui.views;

import java.io.IOException;
import java.net.*;
import java.util.*;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.*;
import org.eclipse.jface.action.*;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.*;
import org.eclipse.ui.ide.IDE;

import com.narucy.webpub.core.WebProject;
import com.narucy.webpub.core.publish.*;
import com.narucy.webpub.ui.*;

/**
 * PreviewAutoActivedManager provides editor part bridge.
 * 
 */
public class HTPreviewLinkWithEditorManager {

	final public static String
		M_PREVIEW_TARGETS = "previewTargets",
		M_PREVIEW_BEHAVIOR = "previewBehavior";

	HTPreView preview;
	
	MnemonicNamer mnemonicNamer = new MnemonicNamer() {
		public String toLabel(Object obj) {
			IFile file = (IFile)obj;
			String name = file.getName();
			String fullPath = file.getFullPath().toString().substring(1);
			int nameLen = name.length();
			int fullLen = fullPath.length();
			if( nameLen + fullLen > 32){
				fullPath = fullPath.substring(0, fullLen-nameLen) + "...";
			}
			return name + "  [" + fullPath + "]";
		}
	};

	HashMap fileActions = new HashMap();
	Action linkWithEditorAction;

	IResourceChangeListener resourceListener = new IResourceChangeListener() {
		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta delta = event.getDelta();
			if(delta != null){
				handleResourceChanged(delta);
			}
		}
	};
	
	/**
	 * Call back from part.
	 */
	IPartListener partListener = new IPartListener() {
		public void partBroughtToTop(IWorkbenchPart part) {}
		public void partDeactivated(IWorkbenchPart part) {}


		public void partClosed(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				unregistObserveEditor( (IEditorPart)part);
			}
		}

		public void partOpened(IWorkbenchPart part) {
			if( part instanceof IEditorPart){
				registObserveEditor( (IEditorPart)part);
			}
		}
		
		public void partActivated(IWorkbenchPart part) {
			if(isLinkWithEditor() && part instanceof IEditorPart){
				IFile editFile = getFile( (IEditorPart)part);
				if(editFile != null){
					changeBrowseFile(editFile);
				}
			}
		}
	};
	
	/**
	 * Construct PrivewAutoActivedManager.
	 */
	public HTPreviewLinkWithEditorManager(HTPreView view) {
		preview = view;
		IMenuManager mgr = preview.getMenuManager();

		// addition editor menu item. 
		mgr.add( new Separator(M_PREVIEW_TARGETS));
		// create auto mate preview changer action
		mgr.add( new Separator(M_PREVIEW_BEHAVIOR));
		linkWithEditorAction = new Action("&Link with Editor", IAction.AS_CHECK_BOX ){
			public void run(){
				handleLinkWithEditorCheck();
			}
		};
		linkWithEditorAction.setToolTipText("Link with Editor");
		linkWithEditorAction.setImageDescriptor(WebpubUIPlugin.getImageDescriptor("clcl16/synced.gif"));
		
		mgr.add(linkWithEditorAction);
		
		// init for current open editors 
		IEditorReference[] refs = preview.getSite().getPage().getEditorReferences(); 
		for (int i = 0; i < refs.length; i++) {
			registObserveEditor( refs[i].getEditor(true) );
		}

		view.getSite().getPage().addPartListener(partListener);
		ResourcesPlugin.getWorkspace().addResourceChangeListener(resourceListener);
	}
	
	void handleResourceChanged(IResourceDelta delta){
		final ArrayList removes = new ArrayList();
		try{
			delta.accept(new IResourceDeltaVisitor() {
				public boolean visit(IResourceDelta d) throws CoreException {
					IResource r = d.getResource();
					if(r instanceof IFile && d.getKind() == IResourceDelta.REMOVED && fileActions.containsKey(r)){
						removes.add(r);
					}
					return true;
				}
			});
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
		if(removes.size() > 0){
			final Display disp = preview.getSite().getShell().getDisplay();
			disp.asyncExec(new Runnable() {
				public void run() {
					if(!disp.isDisposed()){
						for(int i=0; i<removes.size(); i++){
							undefinePreviewAction((IFile)removes.get(i));
						}
					}
				}
			});
		}
	}

	void handleLinkWithEditorCheck(){
		boolean checked = linkWithEditorAction.isChecked();
		if(!checked){
			return;
		}
		
		IFile[] registedFiles = getRegistedFiles();
		if(registedFiles.length > 0){
			IFile editFile = getFile( preview.getSite().getPage().getActiveEditor() );
			
			for (int i=0; i<registedFiles.length; i++) {
				IFile r = registedFiles[i];
				if( editFile.equals(r) ){
					changeBrowseFile(r);
					break;
				}
			}
		}else{
			// link with editor button pressed, but editor is not opened.
			preview.browse(null);
		}
	}
	
	public void handleWebBorwserUrlChanged() {
		try{
			IFile browseFile = preview.getCurrentPublishFrom();
			if(isLinkWithEditor()){
				// auto activiation browser url changed, notify workbench part.
				if(fileActions.containsKey(browseFile) ){
					IDE.openEditor(preview.getSite().getPage(), browseFile, true);
				}
			}
			// update open action radio state
			if(browseFile == null){
				String url = preview.getUrl();
				if(url != null){
					browseFile = SourceFileSeeker.findSource(url);
				}
			}
			IFile[] fs = getRegistedFiles();
			for (int i=0; i<fs.length; i++) {
				IFile f = fs[i];
				getBrowseAction(f).setChecked(f.equals(browseFile));
			}
		}catch(CoreException e){
			WebpubUIPlugin.handleException(e);
		}
	}
	
	void p(Object o) {
		System.out.println(getClass() + ":" + o);
	}
	
	public IFile[] getRegistedFiles(){
		return (IFile[])fileActions.keySet().toArray(new IFile[fileActions.size()]);
	}
	
	/**
	 * @param openFile specify null, check is all disabled.
	 */
	public void changeBrowseFile(IFile openFile){
		if(getBrowseAction(openFile) != null && preview.isPreviewble(openFile)){
			// do browse and checke state changing.
			PublishDescription desc = getPublishDescription(openFile);
			if( desc != null ){
				IResource res = desc.getPublishTo();
				if( res instanceof IFile){
					openFile = (IFile)res;
				}
			}
			IPath localPath = openFile.getLocation();
			if(localPath != null){
				try {
					preview.browse(localPath.toFile().toURL().toString());
				} catch (MalformedURLException ex) {
					WebpubUIPlugin.handleException(ex);
				}
			}
		}
	}
	
	void checkOpenAction(IFile openFile){
		IFile[] fs = getRegistedFiles();
		for (int i = 0; i < fs.length; i++) {
			IFile f = fs[i];
			getBrowseAction(f).setChecked( f.equals(openFile) );
		}
	}

	public void definePreviewAction(final IFile file){
		if(!fileActions.containsKey(file)){
			Action action = new Action(mnemonicNamer.getLabel(file), IAction.AS_RADIO_BUTTON){
				public void run(){
					changeBrowseFile(file);
				}
			};
			ActionContributionItem actionItem = new ActionContributionItem(action);
			fileActions.put(file, actionItem);
			preview.getMenuManager().appendToGroup(M_PREVIEW_TARGETS , actionItem);
		}
	}
	
	public IAction getBrowseAction(IFile file){
		return fileActions.containsKey(file) ?
			((ActionContributionItem)fileActions.get(file)).getAction() :
			null;
	}

	static IFile getFile(IEditorPart editor){
		IEditorInput input = editor.getEditorInput();
		return (input instanceof IFileEditorInput) ?
			((IFileEditorInput)input).getFile() :
			null;
	}

	void registObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null && preview.isPreviewble(file)){
			definePreviewAction(file);
		}
	}
	
	void unregistObserveEditor(IEditorPart editor){
		IFile file = getFile(editor);
		if(file != null){
			undefinePreviewAction(file);
		}
		if( isLinkWithEditor() && fileActions.isEmpty() ){
			preview.browse(null);
		}
	}
	
	void undefinePreviewAction(IFile file){
		IMenuManager menuMgr = preview.getMenuManager();
		ActionContributionItem actionItem = (ActionContributionItem)fileActions.remove(file);
		if(actionItem != null){
			menuMgr.remove(actionItem);
		}
		mnemonicNamer.release(file);
	}
	
	static URL getPublishUrl(IFile file){
		if( file != null){
			PublishDescription desc = getPublishDescription(file);
			if( desc != null){
				try {
					return desc.getPublishTo().getLocation().toFile().toURL();
				} catch (MalformedURLException e) {
					WebpubUIPlugin.handleException(e);
				}
			}
		}
		
		return null;
	}
	
	static PublishDescription getPublishDescription(IFile file){
		try{
			WebProject wp = (WebProject)file.getProject().getNature(WebProject.ID_NATURE);
			if(wp != null){
				return wp.getPublishDescriptionFactory().create(file);
			}
		} catch (IllegalConfigurationException e) {
			WebpubUIPlugin.handleException(e);
		} catch (CoreException e) {
			WebpubUIPlugin.handleException(e);
		} catch (IOException e) {
			WebpubUIPlugin.handleException(e);
		}
		return null;
	}
	
	boolean isLinkWithEditor(){
		return linkWithEditorAction != null && linkWithEditorAction.isChecked();
	}

	public Action getLinkWithEditorAction() {
		return linkWithEditorAction;
	}

	public void dispose(){
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(resourceListener);
	}
}
